#!/usr/bin/python3

#
# config_file_generator.py
#
#  Author: avida, mmerolli
#  Created: Apr 2018
#  Purpose: generates a config YAML file from an XML 'master' descriptor file.
#

import csv
import xmltodict
import sys
import argparse
import xml.etree.ElementTree as ET
import os
import filecmp

#
# Local import
#

from yaml_config_file_generator import YamlConfigFileGenerator
from commented_tree_builder import CommentedTreeBuilder
from schema_validator import SchemaValidator

from common_helpers import AppLogger

#
# GLOBAL VARIABLES
#

g_options = {}


#
# HELPER FUNCTIONS
#


def parse():
    parser = argparse.ArgumentParser()

    # TODO: add XLSX to generate documetnation
    parser.add_argument(
        "--output_type",
        help="output type (YAML)",
        default="YAML",
    )

    # basic input and output
    parser.add_argument("--input", nargs="?", help="XML input file path")
    parser.add_argument("--output", nargs="?", help="output file path")

    # advanced:
    parser.add_argument(
        "--schema",
        nargs="?",
        help="XSD schema file path",
    )
    parser.add_argument(
        "--verbose", help="be verbose", default=False, action="store_true"
    )

    parse_result = parser.parse_args()
    if parse_result.input == None or parse_result.output == None:
        print(
            "ERROR: to generate a config file both the input and output parameters are required.. quitting."
        )
        sys.exit(128)
    return parse_result


def get_initial_xml_comment(filename):
    target = CommentedTreeBuilder()
    cparser = ET.XMLParser(target=target)
    ET.parse(filename, parser=cparser)
    # print('found first comment: {}'.format(FIRST_XML_COMMENT))
    return target.get_first_xml_comment()


def validate_namespace(xml_data):
    valid_namespace = "http://www.empirix.com/master/xml/config"
    if xml_data.get("config-file").get("@xmlns") != valid_namespace:
        print(
            "MALFORMED XML NAMESPACE REFERENCE: the XML file must reference '%s'. Aborting."
            % (valid_namespace)
        )
        sys.exit(255)


def replace_output_file_if_changed(logger, out_file, tmp_out_file):
    if os.path.isfile(out_file) and filecmp.cmp(tmp_out_file, out_file, shallow=False):
        # the output file already exists and is identical to what we just generated:
        # do not modify it, e.g. by changing its timestamp:
        logger.log(
            "Output file '%s' is up to date. No change written to disk." % out_file
        )
        os.remove(tmp_out_file)
    else:
        logger.log("Written output file '%s'." % out_file)
        os.rename(tmp_out_file, out_file)


#
# MAIN
#

if __name__ == "__main__":
    if sys.version_info < (3, 9):
        sys.stdout.write("Sorry, requires Python 3.9\n")
        sys.exit(1)

    g_options = parse()

    thelogger = AppLogger(g_options.verbose)
    thelogger.log(f"Starting conversion of {g_options.input} into {g_options.output}")

    # before processing the XML file try to validate it against the schema:
    if g_options.schema:
        v = SchemaValidator(thelogger)
        v.validate_xml(g_options.input, g_options.schema)

    # read input XML file
    xml_data = None
    try:
        with open(g_options.input) as fd:
            xml_data = xmltodict.parse(fd.read())
            validate_namespace(xml_data)
    except IOError as err:
        print(f"Error reading input XML file: {err}")
        sys.exit(128)

    first_xml_comment = get_initial_xml_comment(g_options.input)

    # now traverse XML data and produed output file:
    if g_options.output_type.upper() == "YAML":
        tmp_out_file = g_options.output + ".tmp"
        json_schema_file = g_options.output + ".schema.json"
        json_schema_file_tmp = g_options.output + ".schema.json.tmp"

        output_gen = YamlConfigFileGenerator(thelogger)
        output_gen.print_yaml_config_file(
            xml_data, first_xml_comment, tmp_out_file, json_schema_file_tmp
        )

        replace_output_file_if_changed(thelogger, g_options.output, tmp_out_file)
        replace_output_file_if_changed(
            thelogger, json_schema_file, json_schema_file_tmp
        )
    else:
        print(f"Unsupported output type: {g_options.output_type.upper()}")
        sys.exit(100)

    sys.exit(0)  # success
