from common_helpers import listify, throw_error, throw_malformed_error


class OptionProperties(object):
    def __init__(self, prop):
        """Parse all XML attributes of a <option> node"""
        # Mandatory attributes
        self.name = prop["@name"]
        self.short_desc = prop["@short_desc"]
        self.long_desc = prop.get("@long_desc")
        self.type_list = [x.strip() for x in prop["@type"].split()]

        # Optional attributes
        self.available_values = prop.get("@available_values", None)
        self.required = self.str_to_pytype(prop.get("@required", "false"), "boolean")
        # If no default_type is given, take the first type in type_list
        self.default_type = prop.get("@default_type", self.type_list[0])

        # support nested options
        self.children = prop.get("option", None)

        self.default_value = None
        self.min_value = None
        self.max_value = None

        self.fail_malformed_error(
            "@default_value" in prop and "default_values" in prop,
            "Cannot have both format default_value and default_values.",
        )

        if "list" in self.default_type:
            if "@default_value" in prop:
                if prop["@default_value"] == "":
                    default_value_str = []
                else:
                    default_value_str = [
                        x.strip() for x in prop["@default_value"].split(",")
                    ]
            else:
                default_value_str = listify(
                    prop["default_values"].get("default_value", [])
                )
        else:
            default_value_str = prop["@default_value"].strip()

        self.fail_malformed_error(
            ("@min_value" in prop or "@max_value" in prop)
            and any(
                type not in ["integer", "uint32", "float"] for type in self.type_list
            ),
            "[min,max]_value allowed only with numeric types",
        )

        # MIN/MAX VALUE INTERPRETATION
        generic_num_type = "float" if "float" in self.type_list else "integer"
        min_value_str = prop.get("@min_value", None)
        max_value_str = prop.get("@max_value", None)
        if min_value_str:
            self.min_value = self.str_to_pytype(min_value_str, generic_num_type)
        if max_value_str:
            self.max_value = self.str_to_pytype(max_value_str, generic_num_type)
        if "uint32" in self.type_list:
            self.min_value = max(0, self.min_value) if self.min_value else 0
            self.max_value = (
                min(0xFFFFFFFF, self.max_value) if self.max_value else 0xFFFFFFFF
            )

        # BACKGROUND CHECKS
        self.fail_malformed_error(
            len(self.type_list) > 1 and "object_list" in self.type_list,
            "object_list not allowed with multiple type property",
        )
        self.fail_malformed_error(
            any("list" in type for type in self.type_list)
            and not all("list" in type for type in self.type_list),
            "int_list and string_list are only compatible with each other in case of a multiple type property",
        )
        self.fail_malformed_error(
            "enum" in self.type_list and not self.available_values,
            'No "available_values" for "enum" type.',
        )
        self.fail_malformed_error(
            "enum" in self.type_list
            and (
                self.default_type == "enum"
                and default_value_str not in self.available_values
            ),
            '"default_value" not in "available_values" for "enum" type.',
        )
        self.fail_malformed_error(
            self.default_type == "object_list" and default_value_str,
            "Only empty default_value is accepted for object_list",
        )

        self.default_value = self.str_to_pytype(default_value_str, self.default_type)
        self.fail_malformed_error(isinstance(self.default_value, (int, float)) and self.min_value and self.default_value < self.min_value, "default_value < min_value")  # type: ignore
        self.fail_malformed_error(isinstance(self.default_value, (int, float)) and self.max_value and self.default_value > self.max_value, "default_value > max_value")  # type: ignore

    def fail_malformed_error(self, cond, message=""):
        if cond:
            throw_malformed_error(self, message)

    def str_to_pytype(self, input_str, input_type):
        # TODO key error if not into XML_TO_TYPE_MAP: unexpected type
        XML_TO_TYPE_MAP = {
            "integer": int,
            "uint32": int,
            "float": float,
            "boolean": bool,
            "enum": str,
            "string": str,
            "object_list": list,  # It should actually be an EMPTY list
            "string_list": list,
            "int_list": list,
        }
        XML_LIST_TO_TYPE_MAP = {
            "string_list": str,
            "int_list": int,
            "object_list": type(None),
        }
        try:
            end_type = XML_TO_TYPE_MAP[input_type]
            if end_type is list:
                assert isinstance(input_str, list)
                list_el_type = (
                    XML_LIST_TO_TYPE_MAP[input_type]
                    if isinstance(input_type, str)
                    else type(input_str[0])
                )
                return [list_el_type(x) for x in input_str]
            elif end_type is bool:
                return {"true": True, "false": False}[input_str.lower()]
            elif not input_str:
                return input_str  # Keeps None and empty strings/lists unchanged
            else:
                return end_type(input_str)
        except:
            throw_malformed_error(self)

    def type_to_json_schema(self, input_type):
        xml_type_to_json_schema_type_map = {
            "object": {"type": "object"},
            "boolean": {"type": "boolean"},
            "enum": {
                "enum": self.available_values.split(",")
                if self.available_values
                else ""
            },
            "float": {"type": "number"},
            "integer": {"type": "integer"},
            "uint32": {"type": "integer"},
            "string": {"type": "string"},
            "int_list": {"type": "array", "items": {"type": "integer"}},
            "string_list": {"type": "array", "items": {"type": "string"}},
            "object_list": {"type": "array", "items": {"type": "object"}},
        }
        try:
            ret_json_schema = xml_type_to_json_schema_type_map[input_type]
        except KeyError:
            throw_error(
                f"UNEXPECTED TYPE WARNING: {type} is not an allowed type. Allowed types: {xml_type_to_json_schema_type_map.keys()}"
            )

        if self.min_value is not None:
            ret_json_schema |= {"minimum": self.min_value}
        if self.max_value is not None:
            ret_json_schema |= {"maximum": self.max_value}

        return ret_json_schema

    def get_json_schema(self):
        if len(self.type_list) == 1:
            return self.type_to_json_schema(self.default_type)
        else:
            return {
                "anyOf": [
                    self.type_to_json_schema(input_type)
                    for input_type in self.type_list
                ]
            }

    def get_comment(self):
        """Returns a comment suitable for INI and YAML outputs that makes sense to describe this <option>"""
        return (
            f"{self.type_list} option, {self.long_desc}"
            + (f", min_value={self.min_value}" if hasattr(self, "min_value") else "")
            + (f", max_value={self.max_value}" if hasattr(self, "max_value") else "")
        ).rstrip()
