import pathlib
import hashlib
import os

import pytest

import generate_helm_chart_reference_xlsx as generator


class TestGenerateHelmChartReferenceXlsx:
    SCRIPT_LOCATION = pathlib.Path(__file__).parent.resolve()
    XML_LOCATION = f"{SCRIPT_LOCATION}/input_files/"
    EXPECTED_FILES_LOCATION = f"{SCRIPT_LOCATION}/expected_files/"

    class CommandlineArgs:
        def __init__(self, parent, helm_chart_xmls, app, version):
            self.helm_chart_xmls = sorted(helm_chart_xmls)
            self.output_dir = f"{parent.SCRIPT_LOCATION}"
            self.app = app
            self.version = version

    def get_check_sum(self, file_name):
        md5_hash = None
        hasher = hashlib.md5()
        with open(file_name, "rb") as f:
            buf = f.read()
            hasher.update(buf)
            md5_hash = hasher.hexdigest()
        return md5_hash

    def verify_check_sums(self, actual_file, expected_file, remove_actual_file=True):
        actual_check_sum = self.get_check_sum(actual_file)
        assert actual_check_sum, f"'{actual_file}' should have a valid check sum"

        expected_check_sum = self.get_check_sum(expected_file)
        assert expected_check_sum, f"'{expected_file}' should have a valid check sum"

        assert expected_check_sum == actual_check_sum, "Check sum should match"

        if remove_actual_file:
            os.remove(actual_file)

    def test_xml_to_documentation_generation_common_kpis(self):
        print(
            "Running test xml to documentation generation for an example tcap-loadbalancer app"
        )
        args = self.CommandlineArgs(
            self,
            helm_chart_xmls=[
                f"{self.XML_LOCATION}/helm-chart-reference-{config_file_no}.xml"
                for config_file_no in [1, 2, 3]
            ],
            app="helm-test",
            version="1.0.0-local",
        )

        visible_parsed_infos = [
            generator.parse_xml(xml_filename, args.app)
            for xml_filename in args.helm_chart_xmls
        ]

        # generate xlsx
        output_filename = os.path.join(
            os.path.abspath(args.output_dir),
            "-".join(
                filter(
                    None, (args.app, args.version, "configparams-documentation.xlsx")
                )
            ),
        )
        generator.export_xlsx(
            args.helm_chart_xmls,
            visible_parsed_infos,
            output_filename,
            args.app,
            args.version,
            unit_test=True,
        )

        test_file_name = f"{args.app}-{args.version}-configparams-documentation"
        self.verify_check_sums(
            f"{self.SCRIPT_LOCATION}/{test_file_name}.xlsx",
            f"{self.EXPECTED_FILES_LOCATION}/{test_file_name}.xlsx",
        )
