#
# This is the file which is typically included by top-level-folder makefiles of a git repository
# using Automated Test Framework (ATF)
#

# ------------------------------------------------------------------------------------------
# Calling makefile parameters
#
# REQUIRED parameters defined by the caller Makefile
#     ATF_DOCKER_REMOTE_NEXUS_DOCKER_REGISTRY, ATF_DOCKER_GROUP
#         The registry name (e.g. empirixartifactory-docker.jfrog.io) and the docker group (e.g. empirix-eva-ecc/atf-docker)
#         for the atf docker to use in the "automated_tests" target implementation.
#         (this typically comes from AutomatedTestsFramework-support.mk)
#     ATF_DOCKER_IMAGE_NAME
#         Name of the ATF docker image, possibly including the tag/version (e.g. atf-docker:2.1.3-master). 
#         (this typically comes from AutomatedTestsFramework-support.mk)
#
# OPTIONAL GNU make variables (or environment variables) considered by this Makefile are:
#
# * TESTSUITE=<whatever testsuite name you want to run>:<testcase name>:<testrun>
# * TEAMCITY_TESTSUITE=identical to TESTSUITE
# * CFG=release/debug/debug-gcov
# * ATF_ENABLE=0/1/false/true
# * SKIP_STOP_TESTCASE=0/1/true
# * SKIP_TESTBUNDLE_UPDATE=0/1/false/true
# * RUN_SLEEP_INSIDE=<svc name>   (e.g. 'capture')
# * SKIP_FORMAT_CHECK=0/1/false/true
# * ATF_BACKEND=kubernetes/docker-compose
# * ATF_DOCKER_IMAGE_VARIANT
# * ATF_DEBUG=0/1/true
# * ATF_MONITOR_PERFORMANCE=false/true
# * ATF_KUBE_CONTEXT=
# * ATF_KUBE_NAMESPACE=
# * ATF_CPUS=<someNumber>
# * ATF_TEST_CONAN_VERSION/ATF_TEST_CONAN_CHANNEL
# * ATF_ADDITIONAL_OPTIONS=<any additional CLI option supported by ATF>
# * ATF_ADDITIONAL_ENV=<1 extra env var that can be exported by e.g. build agent and used inside testcases>
# * APT_DEBUG_MODE=0/1/false/true
# * APT_BUILD=0/1/false/true    # Used to inform the build that a 'performance' build is started and some build parts (debug & gcov) will be skipped
# * APT_SPECIFIC_RUN=<someNumber>
# ------------------------------------------------------------------------------------------

ifndef CONAN_VERSION
$(error Please define CONAN_VERSION before including this snippet)
endif

ifndef SUPPORTS_HELP
# default value
SUPPORTS_HELP=1
endif

ifeq ($(FUNCTIONAL_TESTS_DIR),)
FUNCTIONAL_TESTS_DIR=Testing/AutomatedTests
endif
ifeq ($(PERFORMANCE_TESTS_DIR),)
PERFORMANCE_TESTS_DIR=Testing/PerformanceTests
endif

ifeq ($(AUTOMATED_TESTS_CONANFILE),)
AUTOMATED_TESTS_CONANFILE=$(TOPDIR)/$(TESTS_DIR)/conandata.j2
endif

ifeq ($(ATF_ENABLE),)
SKIP_TESTBUNDLE_UPDATE=0
endif
ifeq ($(ATF_ENABLE),)
# default value
ATF_ENABLE=1
endif
ifneq ($(ATF_ENABLE), true)
	ifneq ($(ATF_ENABLE), 1)
		ifneq ($(ATF_ENABLE), 0)
			ifneq ($(ATF_ENABLE), false)
$(error Unsupported value for ATF_ENABLE variable: $(ATF_ENABLE))
			endif
		endif
	endif
endif

# convert from true/false to 1/0
ifeq ($(ATF_ENABLE), false)
ATF_ENABLE=0
endif
ifeq ($(ATF_ENABLE), true)
ATF_ENABLE=1
endif

ifeq ($(TEST_BUNDLES_DIR),)
TEST_BUNDLES_DIR=$(TOPDIR)/$(TESTS_DIR)/test-bundles
endif

ifeq ($(SKIP_TESTBUNDLE_UPDATE),)
# default value
SKIP_TESTBUNDLE_UPDATE=0
endif
# convert from true/false to 1/0
ifeq ($(SKIP_TESTBUNDLE_UPDATE),false)
SKIP_TESTBUNDLE_UPDATE=0
endif
ifeq ($(SKIP_TESTBUNDLE_UPDATE),true)
SKIP_TESTBUNDLE_UPDATE=1
endif

ifeq ($(SKIP_FORMAT_CHECK),)
# default value
SKIP_FORMAT_CHECK=0
endif
# convert from true/false to 1/0
ifeq ($(SKIP_FORMAT_CHECK),false)
SKIP_FORMAT_CHECK=0
endif
ifeq ($(SKIP_FORMAT_CHECK),true)
SKIP_FORMAT_CHECK=1
endif

ifeq ($(ATF_DOCKER_IMAGE_VARIANT),)
ifeq ($(BUILD_AGENT),1)
# for Teamcity build agents we want to have code coverage as well as address sanitizer checks in automated tests, so we use GCOV-instrumented docker images which have coverage flag enabled
ATF_DOCKER_IMAGE_VARIANT:=_G
else
# for developer machines by default we use docker image variants defined via CFG variable
ATF_DOCKER_IMAGE_VARIANT:=$(DEBUG_TAG)
endif
endif

# below flag will be used in case of downloading the source code of upstream projects from the artifactory
DOWNLOAD_UPSTREAM_SOURCE_CODE:=0

# ATF debug mode means: run sequentially (no parallelism) and print on screen everything that is done
# NOTE: old GNU make variable name was "ATFDEBUG", now for coherency it's ATF_DEBUG, we accept both:
ifneq ($(ATFDEBUG),)
ATF_DEBUG:=$(ATFDEBUG)
endif

# inside ATF_EXPANDED_OPTIONS we expand several high-level ATF_* parameters exposed at GNU make level into specific ATF option names;
# please note that the developer should always set only ATF_ADDITIONAL_OPTIONS; ATF_EXPANDED_OPTIONS is internal and should not be
# set or reset by the developer directly!
ATF_EXPANDED_OPTIONS:=

ifeq ($(ATF_DEBUG),1)
ATF_EXPANDED_OPTIONS+=--debug 
# fmon: Oct 2022: disabled this feature because I believe nobody is using it and is just bugging about more credentials
DOWNLOAD_UPSTREAM_SOURCE_CODE:=0
endif
ifeq ($(ATF_DEBUG),true)
ATF_EXPANDED_OPTIONS+=--debug 
# fmon: Oct 2022: disabled this feature because I believe nobody is using it and is just bugging about more credentials
DOWNLOAD_UPSTREAM_SOURCE_CODE:=0
endif

ifeq ($(CFG),debug)
# fmon: Oct 2022: disabled this feature because I believe nobody is using it and is just bugging about more credentials
DOWNLOAD_UPSTREAM_SOURCE_CODE:=0
endif

# ATF skip stop tetscase option is good during debugging/testcase-development
ifeq ($(SKIP_STOP_TESTCASE),1)
ATF_EXPANDED_OPTIONS+=--skip_stop_testcase
endif
ifeq ($(SKIP_STOP_TESTCASE),true)
ATF_EXPANDED_OPTIONS+=--skip_stop_testcase
endif

ifneq ($(ATF_CPUS),)
ATF_EXPANDED_OPTIONS+=--num_parallel_workers=$(ATF_CPUS)
endif

# ATF monitor performance option is used to enable to docker performance monitoring
# Report would be generated in 'TeamcityArtifacts/automated-tests-performance-monitor/'
# This feature is based on the utility "cmonitor_collector" installed inside ATF docker image
ifeq ($(ATF_MONITOR_PERFORMANCE),)
ATF_MONITOR_PERFORMANCE:=true
endif

ifeq ($(ATF_PERFORMANCE_SAMPLE_INTERVAL),)
ATF_PERFORMANCE_SAMPLE_INTERVAL:=1
endif

ifeq ($(ATF_MONITOR_PERFORMANCE),true)
ATF_EXPANDED_OPTIONS+=--monitor_performance --performance_sample_interval=$(ATF_PERFORMANCE_SAMPLE_INTERVAL)
endif

ifneq ($(ATF_KUBE_CONTEXT),)
ATF_EXPANDED_OPTIONS+=--atf_kube_context=$(ATF_KUBE_CONTEXT)
endif

ifneq ($(ATF_KUBE_NAMESPACE),)
ATF_EXPANDED_OPTIONS+=--atf_kube_namespace=$(ATF_KUBE_NAMESPACE)
endif

ifneq ($(APT_SPECIFIC_RUN),)
ATF_EXPANDED_OPTIONS+=--apt_specific_testrun=$(APT_SPECIFIC_RUN)
endif

ifneq ($(ATFCOMA_PYPI_VERSION),)
$(error Please remove ATFCOMA_PYPI_VERSION version definition)
endif

# APT_DEBUG_MODE var. is usefull on Automated Performance Test (APT) to run a small set of testruns, just for debug
# convert from true/false to 1/0
ifeq ($(APT_DEBUG_MODE),false)
APT_DEBUG_MODE=0
endif
ifeq ($(APT_DEBUG_MODE),true)
APT_DEBUG_MODE=1
endif

ifndef ATF_TEST_CONAN_VERSION
# default value: test the same version that is shown by e.g. "make info" command: the version automatically computed by git
override ATF_TEST_CONAN_VERSION=$(CONAN_VERSION)
endif
ifndef ATF_TEST_CONAN_CHANNEL
# default value: test the same version that is shown by e.g. "make info" command: the version automatically computed by git
override ATF_TEST_CONAN_CHANNEL=$(CONAN_CHANNEL)
endif

ifeq ($(ATF_TEST_CONAN_VERSION),auto)
# default value: test the same version that is shown by e.g. "make info" command: the version automatically computed by git
override ATF_TEST_CONAN_VERSION=$(CONAN_VERSION)
endif
ifeq ($(ATF_TEST_CONAN_CHANNEL),auto)
# default value: test the same version that is shown by e.g. "make info" command: the version automatically computed by git
override ATF_TEST_CONAN_CHANNEL=$(CONAN_CHANNEL)
endif

ifeq ($(ATF_BACKEND),)
# default value
ATF_BACKEND=docker-compose
endif

ifeq ($(ATF_KERNEL_COREPATTERN),)
# default value: as of Nov 2021, this value is perfectly aligned with the default value used by Klerity in production
# systems thanks to the "worker-nodes-configurator" Kube Daemonset
ATF_KERNEL_COREPATTERN=/debug/core.%t.%e.%p.%h
endif

ifeq ($(LAB),billerica)
else
ifeq ($(LAB),modena)
else
ifeq ($(LAB),massy)
else
$(error Invalid value for LAB variable: $(LAB))
endif
endif
endif

ifeq ($(TESTSUITE),)
# default value
TESTSUITE=@functional_tests
endif
# TEAMCITY_TESTSUITE is the alternative name for TESTSUITE
ifneq ($(TEAMCITY_TESTSUITE),)
override TESTSUITE=$(TEAMCITY_TESTSUITE)
endif

ifeq ($(KUBECONFIG),)
# the env var KUBECONFIG is used by 'kubectl' and by Python kubernetes library to find out the config file to talk with Kube API server;
# there's a case where the KUBECONFIG var does NOT exist at all on the host; the code below will however create an empty one in the environment
# of the ATF docker... this will trigger a failure inside Python kubernetes library like:
#
#  File "/usr/local/lib/python3.9/site-packages/kubernetes/config/kube_config.py", line 792, in load_kube_config
#    loader = _get_kube_config_loader(
#  File "/usr/local/lib/python3.9/site-packages/kubernetes/config/kube_config.py", line 751, in _get_kube_config_loader
#    raise ConfigException(
# kubernetes.config.config_exception.ConfigException: Invalid kube-config file. No configuration found.
#
# For this reason we actually avoid passing an empty KUBECONFIG inside ATF docker; the default value is harmless anyhow:
KUBECONFIG=/root/.kube/config
endif

# this env. var. is used in build agents by functional automated tests 
# to understand if the kubeconfig file must be removed or just checked;
# reason for removing the kubeconfig file is that when a test fail inside the 'kind' kubernetes-on-the-fly, the kube config of that 'kind'
# instance is left around (to allow debugging) and it must be cleaned up BEFORE starting a new build / new automated_tests step.
ifeq ($(ATF_BUILDAGENT_REMOVE_KUBECONFIG),)
ATF_BUILDAGENT_REMOVE_KUBECONFIG=1
endif

ifeq ($(ATF_BUILDAGENT_REJECT_ISOLCPUS),)
# machines having isolcpus in their boot options are typically machines where ICEK has been run and they are not suitable
# for running automated tests (reason being: typically only very few CPUs are available outside isolcpu pool).
# A notable exception is for HW validation build agents that will need to have isolcpus turned on... the caller makefile
# must adjust this variable in that case
ATF_BUILDAGENT_REJECT_ISOLCPUS=1
endif

ifeq ($(ATF_NUM_REPETITIONS),)
# repeating the same testcase(s) more than once is useful to spot flaky tests...
# when ATF_NUM_REPETITIONS>1 ATF will be invoked multiple times with the same options until the first failure is hit
ATF_NUM_REPETITIONS=1
endif

# About REALTOP:REALTOP bind-mount:
#  ATF will launch docker-compose commands inside the ATF docker exploiting the so-called "docker in docker" feature;
#  however ATF needs to first render the docker-compose YAML files and write inside them volume absolute paths.
#  The docker compose YAML files are translated into commands that, through the docker-in-docker socket, reach the
#  docker daemon running on the host. Then it's the docker daemon that will need to actually create the bind-mounts.
#  IOW the docker volumes of each docker are defined and finalized from within the ATF docker but are actually mounted by the 
#  docker daemon running OUTSIDE the ATF docker. To avoid problems we thus need ATF to be able to access exactly the same
#  absolute path that the docker daemon can access. That explains why $(REALTOP) is mounted... on $(REALTOP) in ATF docker
# About /sys:/sys bind-mount:
#  ATF launches the "cmonitor_collector" utility that requires the full /sys hierarchy available to be able to monitor other
#  dockers' cgroups. Mounting /sys on /sys is what e.g. also cAdvisor suggests to do.
# About /root/.kube:/root/.kube bind-mount:
#  Intuitively you might think 'ro' is enough; instead this bind mount must be rw because utilities like 'kind' or 'kubectl'
#  need to acquire locks and write stuff there
# About /root/.docker/:/root/.docker/ bind-mount:
#  This is required to share the docker authentication tokens; this is useful to avoid docker pull rate limitations;
#  see https://www.docker.com/increase-rate-limits/, as of May2022, without authentication we can pull max 16 images/hour !!!
ATF_DOCKER_VOLUMES_PARAMS:=\
	--volume $(REALTOP):$(REALTOP):rw \
	--volume $(CONAN_DATA_DIR):$(CONAN_DATA_DIR):ro \
	--volume $(CONAN_LOCAL_PROFILE_DIR):$(CONAN_DOCKER_PROFILE_DIR):ro \
	--volume $(CONAN_REMOTES_CONFIG_FILE):$(CONAN_REMOTES_CONFIG_FILE):ro \
	--volume /sys:/sys:ro \
	--volume /run/atf:/run/atf:rw \
	--volume /root/.docker/:/root/.docker/:ro \
	--volume /root/.kube:/root/.kube:rw \
	--volume /root/.ssh:/root/.ssh:ro \
	--volume /var/run/docker.sock:/var/run/docker.sock:rw \
	--volume /etc/docker/daemon.json:/etc/docker/daemon.json:ro \
	--volume /root/.config/helm/repositories.yaml:/root/.config/helm/repositories.yaml \
	--mount type=tmpfs,destination=/home/builder/rpmbuild

ATF_DOCKER_MAKEFILES_ENV_VARS_PARAMS:=\
	--env TEAMCITY_VERSION="$(TEAMCITY_VERSION)" \
	--env TC_PULLREQUEST_SOURCE_BRANCH="$(TC_PULLREQUEST_SOURCE_BRANCH)" \
	--env BUILD_AGENT="$(BUILD_AGENT)" \
	--env CFG="$(CFG)" \
	--env LAB="$(LAB)" \
	--env V="$(V)" \
	--env KUBECONFIG="$(KUBECONFIG)" \
	--env TESTSUITE="$(TESTSUITE)" \
	--env TEAMCITY_TESTSUITE="$(TEAMCITY_TESTSUITE)" \
	--env TESTS_DIR=$(TESTS_DIR) \
	--env ATF_COMMAND=$(ATF_COMMAND) \
	--env ATF_MODE=$(ATF_MODE) \
	--env NEXUS_PCAPBOT_USER="$(NEXUS_PCAPBOT_USER)" \
	--env NEXUS_PCAPBOT_PW="$(NEXUS_PCAPBOT_PW)" \
	--env SKIP_STOP_TESTCASE=$(SKIP_STOP_TESTCASE) \
	--env SKIP_FORMAT_CHECK=$(SKIP_FORMAT_CHECK) \
	--env ATF_BACKEND=$(ATF_BACKEND) \
	--env ATF_REMOVE_CONTAINER_LIMITS=$(ATF_REMOVE_CONTAINER_LIMITS) \
	--env ATF_KUBE_CONTEXT=$(ATF_KUBE_CONTEXT) \
	--env ATF_KUBE_NAMESPACE=$(ATF_KUBE_NAMESPACE) \
	--env ATF_DOCKER_IMAGE_VARIANT=$(ATF_DOCKER_IMAGE_VARIANT) \
	--env ATF_DEBUG=$(ATF_DEBUG) \
	--env ATF_MONITOR_PERFORMANCE=$(ATF_MONITOR_PERFORMANCE) \
	--env ATF_KEEP_PERFORMANCE_MONITOR_REPORT=$(ATF_KEEP_PERFORMANCE_MONITOR_REPORT) \
	--env ATF_CPUS=$(ATF_CPUS) \
	--env ATF_TEST_CONAN_VERSION="$(ATF_TEST_CONAN_VERSION)" \
	--env ATF_TEST_CONAN_CHANNEL="$(ATF_TEST_CONAN_CHANNEL)" \
	--env ATF_NUM_REPETITIONS="$(ATF_NUM_REPETITIONS)" \
	--env ATF_ADDITIONAL_OPTIONS="$(ATF_ADDITIONAL_OPTIONS)" \
	--env ATF_ADDITIONAL_ENV="$(ATF_ADDITIONAL_ENV)" \
	--env APT_DEBUG_MODE="$(APT_DEBUG_MODE)" \
	--env APT_SPECIFIC_RUN="$(APT_SPECIFIC_RUN)" \
	--env ATF_REMOTE_REGISTRY_DOCKER_FOR_PULL="$(REMOTE_REPO_NEXUS_DOCKER_FOR_PULL)" \
	--env ATF_REMOTE_REGISTRY_DOCKER_GROUP="$(DOCKER_GROUP)" \
	--env ATF_CONTAINER_IMAGE_PREFIX="$(REMOTE_REPO_NEXUS_DOCKER_FOR_PULL)/$(DOCKER_GROUP)/"

ifeq ($(BUILD_AGENT),0)
# we must make it possible to enter the Nexus PCAP user/pw during local automated tests:
ATF_DOCKER_ALLOW_NEXUS_PW_INPUT:=\
	--interactive --tty
endif

ATF_DOCKER_ALLOW_CTRLC_PARAMS:=--stop-timeout=0 

# About --network=host: it is required to let cmonitor_collector in ATF docker access other containers' network interfaces
# About --pid=host: it is required to let cmonitor_collector in ATF docker access other containers' process stats
# NET_ADMIN permissions are required to allow "ifconfig up" to work from within the ATF docker (HW validation pipeline)
# privileged is required to allow changing the "vm.nr_hugepages" setting through sysctl 
ATF_DOCKER_PERMISSIONS:=\
	--network=host \
	--pid=host \
	--cap-add NET_ADMIN \
	--privileged

#
# If ATF_DOCKER_IMAGE_FOR_RUN is not defined, we create it here, taking the contents of some variables;
# note that these variables are typically defined by the support makefile generated by the docker-base repo
#

ifeq ($(ATF_DOCKER_IMAGE_FOR_RUN),)
# calculate a default value for the atf docker image to pass to the "docker run"

ifneq ($(ATF_DOCKER_IMAGE_NAME),)
ifneq ($(ATF_DOCKER_GROUP),)
ifneq ($(ATF_DOCKER_REMOTE_NEXUS_DOCKER_REGISTRY),)

# let's build here the name of the atf docker to pass to "docker run"
ATF_DOCKER_IMAGE_FOR_RUN=$(ATF_DOCKER_REMOTE_NEXUS_DOCKER_REGISTRY)/$(ATF_DOCKER_GROUP)/$(ATF_DOCKER_IMAGE_NAME)

endif
endif
endif

endif


# ------------------------------------------------------------------------------------------
# Dependency checks for test bundles
# ------------------------------------------------------------------------------------------

ifneq ($(AUTOMATED_TESTS_CONANFILE),)
update_conan_test_bundles::
	@[ ! -z "$(TEAMCITY_VERSION)" ] && echo "##teamcity[blockOpened name='Regenerating the conandata.yml']" || true
	@echo "============================================================================"
	@echo "Regenerating the conandata.yml for automated tests"
	@echo "============================================================================"
	# run Jinja2 to obtain the actual ConanFile
	THIS_REPO_CONAN_VERSION=$(ATF_TEST_CONAN_VERSION) \
	THIS_REPO_CONAN_CHANNEL=$(ATF_TEST_CONAN_CHANNEL) \
	DOCKER_IMAGE_VARIANT=$(ATF_DOCKER_IMAGE_VARIANT) \
		j2 $(AUTOMATED_TESTS_CONANFILE) -o $(subst .j2,.yml,$(AUTOMATED_TESTS_CONANFILE))
	@[ ! -z "$(TEAMCITY_VERSION)" ] && echo "##teamcity[blockClosed name='Regenerating the conandata.yml']" || true

# Fetch requirements for integration tests via Conan:
check_deps_automated_tests::
	@[ ! -z "$(TEAMCITY_VERSION)" ] && echo "##teamcity[blockOpened name='Retrieving test-bundles using Conan']" || true
	@echo "============================================================================"
	@echo "Retrieving test-bundles using Conan"
	@echo "============================================================================"
	@# cleanup test bundles installed from previous runs (if any)
	@rm -rf $(TEST_BUNDLES_DIR)/conan* \
			$(TEST_BUNDLES_DIR)/graph_info.json \
			$(TEST_BUNDLES_DIR)/deploy_manifest.txt \
			$(TEST_BUNDLES_DIR)/test-bundle-*
	# deploy new test bundles into the "test-bundles" folder
	TOPDIR=$(REALTOP) conan install $(dir $(AUTOMATED_TESTS_CONANFILE)) -g deploy --install-folder $(TEST_BUNDLES_DIR) --options runtime=$(RUNTIME)
	@# don't leave Conan garbage around:
	@rm -rf $(TEST_BUNDLES_DIR)/conan* \
			$(TEST_BUNDLES_DIR)/graph_info.json \
			$(TEST_BUNDLES_DIR)/deploy_manifest.txt
	@[ ! -z "$(TEAMCITY_VERSION)" ] && echo "##teamcity[blockClosed name='Retrieving test-bundles using Conan']" || true

# Apply docker-compose overlay to debug components; for now we support only 1 overlay:
#    RUN_SLEEP_INSIDE to replace the entrypoint inside the services under test
apply_docker_compose_overlays::
	@echo "============================================================================"
	@echo "Apply docker-compose overlays on test-bundles using yq"
	@echo "============================================================================"
ifneq ($(RUN_SLEEP_INSIDE),)
	@echo "Changing with 'sleep' the entrypoint inside services [$(RUN_SLEEP_INSIDE)]"
	yq --inplace eval '.services."$(RUN_SLEEP_INSIDE)$${SERVICE_NAME_POSTFIX}".entrypoint=["/usr/bin/sleep", "10000"]' \
		$(TEST_BUNDLES_DIR)/test-bundle-$(RUN_SLEEP_INSIDE)/docker-compose.$(RUN_SLEEP_INSIDE).yml
endif
endif

# ------------------------------------------------------------------------------------------
# FUNCTIONAL/PERFORMANCE AUTOMATED test targets
# !!!! IMPORTANT: THESE TARGETS DO NOT SUPPORT PARALLEL MODE !!!!
#
# How to debug automated tests manually:
#  1) run: make docker_fulldist
#  2) run: make docker_fullpackaging
#  3) run:
#      $ make automated_tests
#   to run ALL testsuites (as Teamcity would do) but using the docker images with your modifications, or:
#      $ make automated_tests TESTSUITE=eva-allinone
#                                       ^^^^^^^^^^^^
#                                       or any other testsuite you want
# ------------------------------------------------------------------------------------------

# start_atf is meant to be used INSIDE the ATF docker
start_atf::
	@echo "$(shell date +%F-%T): starting make target from inside the ATF docker"
ifeq ($(SKIP_FORMAT_CHECK),1)
	@echo "Skipping JSON/YAML linting as requested by SKIP_FORMAT_CHECK=1."
else
	@echo "Checking formatting of JSON/YAML config files before automated_tests."
	@echo "Use SKIP_FORMAT_CHECK=1 as CLI option to skip this check if you want to save some time."
	@[ ! -z "$(TEAMCITY_VERSION)" ] && echo "##teamcity[blockOpened name='Checking JSON and YAML formatting']" || true
	$(MAKE) format_check_json_yaml
	@[ ! -z "$(TEAMCITY_VERSION)" ] && echo "##teamcity[blockClosed name='Checking JSON and YAML formatting']" || true
endif # SKIP_FORMAT_CHECK
ifndef TESTS_DIR
	@echo "*** ERROR: please call this makefile target supplying explicitly the TESTS_DIR variable"
	@exit 1
endif
ifndef ATF_COMMAND
	@echo "*** ERROR: please call this makefile target supplying explicitly the ATF_COMMAND variable"
	@exit 1
endif
ifndef ATF_MODE
	@echo "*** ERROR: please call this makefile target supplying explicitly the ATF_MODE variable"
	@exit 1
endif
	mkdir -p $(TEAMCITY_ARTIFACTS_DIR_MAIN)
	cd $(TESTS_DIR) && \
		for repIdx in $$(seq 1 $(ATF_NUM_REPETITIONS)); do \
			echo ; \
			echo "============================================================================" ; \
			echo "Starting automated tests repetition $$repIdx/$(ATF_NUM_REPETITIONS)" ; \
			echo "============================================================================" ; \
			AUTOMATED_TEST_FRAMEWORK_DIR=$(AUTOMATED_TEST_FRAMEWORK_DIR) \
			python3 main.py \
				--logfile="$(AUTOMATED_TESTS_LOGFILE)" \
				--automated_test_framework_dir="$(AUTOMATED_TEST_FRAMEWORK_DIR)" \
				--artifacts_dir="$(TEAMCITY_ARTIFACTS_DIR_MAIN)" \
				--gcov_tarball_dir="$(TEAMCITY_ARTIFACTS_DIR_COVERAGE_GCOV_REPORT)" \
				--pcap_gpb_repo="$(REMOTE_REPO_NEXUS_PCAP_FOR_PULL)" \
				--test_results_repo="$(REMOTE_REPO_NEXUS_TESTRESULTS_FOR_PUSH)" \
				--docker_image_registry=$(REMOTE_REPO_NEXUS_DOCKER_FOR_PULL) \
				--helm_repository=$(REMOTE_REPO_NEXUS_HELM_ROOT_FOR_PULL) \
				--docker_image_group=$(DOCKER_GROUP) \
				--automated_tests_docker_image_variant="$(ATF_DOCKER_IMAGE_VARIANT)" \
				--gdb_source_code_volume="$(REALTOP)" \
				--gdb_upstream_dependencies_conanfile="$(REALTOP)/conandata.yml" \
				--atf_mode=$(ATF_MODE) \
				--backend_mode=$(ATF_BACKEND) \
				$(ATF_EXPANDED_OPTIONS) $(ATF_ADDITIONAL_OPTIONS) \
				$(ATF_COMMAND) $(TESTSUITE) 2>&1 ; \
			atf_exit_code=$$?; \
			if [[ $$atf_exit_code -ne 0 ]]; then \
				echo "-- AUTOMATED TESTS FAILED DURING REPETITION $$repIdx/$(ATF_NUM_REPETITIONS) --"; \
				exit $$atf_exit_code; \
			fi \
		done
	@if (( repIdx == $(ATF_NUM_REPETITIONS) )); then \
		echo "-- ALL $(ATF_NUM_REPETITIONS) AUTOMATED TESTS REPETITIONS PASSED SUCCESSFULLY --" ; \
	fi

# This target is useful for debugging automated tests
bash_in_atf_docker:
	docker run --rm \
		--interactive --tty \
		$(ATF_DOCKER_ALLOW_CTRLC_PARAMS) \
		$(ATF_DOCKER_VOLUMES_PARAMS) \
		$(ATF_DOCKER_MAKEFILES_ENV_VARS_PARAMS) \
		$(ATF_DOCKER_PERMISSIONS) \
		--workdir=$(REALTOP) \
		$(ATF_DOCKER_ADDITIONAL_PARAMS) \
		$(ATF_DOCKER_IMAGE_FOR_RUN) \
		--entrypoint=/bin/bash \
			-c 'git config --global --add safe.directory $(REALTOP) && /bin/bash'

# start_atf is meant to be used OUTSIDE the ATF docker
start_atf_docker:
ifeq ($(SKIP_TESTBUNDLE_UPDATE),1)
	@echo "Skipping update of testbundles as requested by SKIP_TESTBUNDLE_UPDATE=1."
else
	@echo "Updating test bundles"
	$(MAKE) update_conan_test_bundles
	$(MAKE) check_deps_automated_tests
endif # SKIP_TESTBUNDLE_UPDATE
	$(MAKE) apply_docker_compose_overlays
ifeq ($(DOWNLOAD_UPSTREAM_SOURCE_CODE),1)
	@echo "============================================================================"
	@echo "Downloading source code of upstream repositories to allow easier debugging"
	@echo "============================================================================"
	@${EMPIRIX_PIPELINE_FRAMEWORK_SCRIPT_SUPPORT_DIR}/conan-download-upstream-source-code.py \
		--conandata_path=$(REALTOP)/conandata.yml \
		--url_prefix=$(REMOTE_REPO_ARTIFACTORY_CONAN_FOR_PULL) \
		--conan_dir=$(CONAN_DATA_DIR)
endif # DOWNLOAD_UPSTREAM_SOURCE_CODE
	@echo "Adjusting kernel corepattern to [$(ATF_KERNEL_COREPATTERN)] before entering ATF docker"
	@echo $(ATF_KERNEL_COREPATTERN) >/proc/sys/kernel/core_pattern
ifneq ($(ATF_DOCKER_IMAGE_FOR_RUN),)
	@echo "============================================================================"
	@echo "$(shell date +%F-%T): Running 'start_atf' makefile target inside ATF docker"
	@echo "============================================================================"
	docker run --rm --name automated_tests_docker \
		$(ATF_DOCKER_ALLOW_CTRLC_PARAMS) \
		$(ATF_DOCKER_VOLUMES_PARAMS) \
		$(ATF_DOCKER_ALLOW_NEXUS_PW_INPUT) \
		$(ATF_DOCKER_MAKEFILES_ENV_VARS_PARAMS) \
		$(ATF_DOCKER_PERMISSIONS) \
		--workdir=$(REALTOP) \
		--entrypoint=/bin/bash \
		$(ATF_DOCKER_ADDITIONAL_PARAMS) \
		$(ATF_DOCKER_IMAGE_FOR_RUN) \
			-c 'git config --global --add safe.directory $(REALTOP) && /usr/bin/make start_atf'
else
	@echo "Please provide a valid ATF_DOCKER_IMAGE_FOR_RUN to start automated tests"
endif

# ------------------------------------------------------------------------------------------
# 1. Check YAML/YML and JSON config file format. Stop if format not correct or warning.
# 2. Correct the YAML/YML and JSON config file format. Stop if any syntactical error.
# 3. Formatter works only within Docker container.
# docker_format_check_json_yaml
# docker_format_inplace_json_yaml
# ------------------------------------------------------------------------------------------

docker_format_check_json_yaml::
ifeq ($(ATF_DOCKER_IMAGE_FOR_RUN),)
	@echo "Please provide a valid ATF_DOCKER_IMAGE_FOR_RUN... was AutomatedTestsFramework-support.mk included in the Makefile of this directory?"
	@exit 100
endif
	@echo "Check formatting of JSON and YAML files."
	docker run --rm --name automated_tests_docker \
		$(ATF_DOCKER_PERMISSIONS) \
		$(ATF_DOCKER_ALLOW_CTRLC_PARAMS) \
		$(ATF_DOCKER_VOLUMES_PARAMS) \
		--workdir=$(REALTOP) \
		--entrypoint=/usr/bin/make \
		$(ATF_DOCKER_MAKEFILES_ENV_VARS_PARAMS) \
		$(ATF_DOCKER_ADDITIONAL_PARAMS) \
		$(ATF_DOCKER_IMAGE_FOR_RUN) \
		format_check_json_yaml 2>&1
	@echo "Successfully completed checking of JSON and YAML format."

docker_format_inplace_json_yaml::
ifeq ($(ATF_DOCKER_IMAGE_FOR_RUN),)
	@echo "Please provide a valid ATF_DOCKER_IMAGE_FOR_RUN... was AutomatedTestsFramework-support.mk included in the Makefile of this directory?"
	@exit 100
endif
	@echo "Reformatting JSON and YAML files."
	docker run --rm --name automated_tests_docker \
		$(ATF_DOCKER_PERMISSIONS) \
		$(ATF_DOCKER_ALLOW_CTRLC_PARAMS) \
		$(ATF_DOCKER_VOLUMES_PARAMS) \
		--workdir=$(REALTOP) \
		--entrypoint=/usr/bin/make \
		$(ATF_DOCKER_MAKEFILES_ENV_VARS_PARAMS) \
		$(ATF_DOCKER_ADDITIONAL_PARAMS) \
		$(ATF_DOCKER_IMAGE_FOR_RUN) \
		format_inplace_json_yaml 2>&1
	@echo "Reformatting JSON and YAML files completed successfully!"


# ------------------------------------------------------------------------------------------
# FUNCTIONAL automated tests
# ------------------------------------------------------------------------------------------

__check_kubeconfig_file:
ifeq ($(BUILD_AGENT),1)
ifeq ($(ATF_BUILDAGENT_REMOVE_KUBECONFIG),1)
	@[ -f $(KUBECONFIG) ] && { \
		echo "============================================================================" ; \
		echo "!!! STALE KUBECONFIG FILE DETECTED... DISABLING IT !!!" ; \
		echo "============================================================================" ; \
		mv -f $(KUBECONFIG) $(KUBECONFIG)_$(shell date +%F-%T) ; \
	} || echo "The kubeconfig file $(KUBECONFIG) is not present... it will be created by ATF later on when needed."
else
	# the caller makefile asked to NOT delete the kubeconfig, if already present; the use case is: this build agent
	# is running a custom kube instance (typically a Rancher instance which is long-lived and does not get recreated on
	# each Teamcity build); in that case we just want to check that the kubeconfig is pointing to such local instance
	# of Kubernetes (not a remote Kubernetes!) and that's it:
	python3 $(EMPIRIX_PIPELINE_FRAMEWORK_SCRIPT_SUPPORT_DIR)/check_cluster_name.py $(KUBECONFIG) local	
endif
endif

__check_isolcpus:
ifeq ($(BUILD_AGENT),1)
ifeq ($(ATF_BUILDAGENT_REJECT_ISOLCPUS),1)
	@grep -q isolcpus /proc/cmdline && { \
		echo "============================================================================" ; \
		echo "!!! ISOLCPUS DETECTED... REFUSING TO START AUTOMATED TESTS !!!" ; \
		echo "============================================================================" ; \
		exit 66 ; \
	 } || echo "No isolcpus detected... proceeding."
else
	@echo "Skipping isolcpu check since ATF_BUILDAGENT_REJECT_ISOLCPUS=$(ATF_BUILDAGENT_REJECT_ISOLCPUS)"
endif
endif

docker_automated_tests::
ifeq ($(ATF_ENABLE), 1)
	$(MAKE) __check_kubeconfig_file
	$(MAKE) __check_isolcpus
	$(MAKE) TESTS_DIR=$(FUNCTIONAL_TESTS_DIR) ATF_MODE=functional ATF_COMMAND=run-pytest    start_atf_docker
endif # ATF_ENABLE

# old synonym
automated_tests:: docker_automated_tests

automated_tests_list::
	$(MAKE) TESTS_DIR=$(FUNCTIONAL_TESTS_DIR) SKIP_TESTBUNDLE_UPDATE=1 SKIP_FORMAT_CHECK=1 ATF_MODE=functional ATF_COMMAND=list-testcases    start_atf_docker

automated_tests_help::
	$(MAKE) TESTS_DIR=$(FUNCTIONAL_TESTS_DIR) SKIP_TESTBUNDLE_UPDATE=1 SKIP_FORMAT_CHECK=1 ATF_MODE=functional ATF_COMMAND=--help            start_atf_docker

# stop all automated tests: this is useful to free resources (CPU and RAM) before actually running the C++ builds which will thus will be
# able to use all available resources to speed up the build:
automated_tests_stop::
	@systemctl start docker || echo "No docker daemon is available"
	@sleep 10                                                        # give time docker daemon to start
	-docker stop automated_tests_docker   # in case it was hanging since a previous build perhaps
	$(MAKE) TESTS_DIR=$(FUNCTIONAL_TESTS_DIR) SKIP_TESTBUNDLE_UPDATE=1 SKIP_FORMAT_CHECK=1 \
		ATF_MODE=functional ATF_COMMAND=stop-all ATF_ADDITIONAL_OPTIONS= ATF_EXPANDED_OPTIONS= ATF_CPUS=  \
		start_atf_docker


# ------------------------------------------------------------------------------------------
# PERFORMANCE automated tests
# ------------------------------------------------------------------------------------------
docker_performance_tests::
ifeq ($(APT_BUILD), 1)
	$(MAKE) TESTS_DIR=$(PERFORMANCE_TESTS_DIR) ATF_DOCKER_IMAGE_VARIANT="" ATF_MODE=performance ATF_COMMAND=run-pytest ATF_BACKEND=kubernetes   start_atf_docker
endif # APT_BUILD

# old synonym
performance_tests:: docker_performance_tests

performance_tests_list::
	$(MAKE) TESTS_DIR=$(PERFORMANCE_TESTS_DIR) ATF_MODE=performance ATF_COMMAND=list-testcases    start_atf_docker

performance_tests_stop::
	$(MAKE) TESTS_DIR=$(PERFORMANCE_TESTS_DIR) SKIP_TESTBUNDLE_UPDATE=1 SKIP_FORMAT_CHECK=1 \
		ATF_MODE=performance ATF_BACKEND=kubernetes ATF_COMMAND=stop-all \
		ATF_KUBE_CONTEXT=$(ATF_KUBE_CONTEXT) ATF_KUBE_NAMESPACE=$(ATF_KUBE_NAMESPACE) \
		ATF_ADDITIONAL_OPTIONS=  \
		start_atf_docker



# ------------------------------------------------------------------------------------------
# Help target
# NOTE1: descriptions must be all aligned at column 54 as standard (including the initial TAB)
# NOTE2: only most useful targets are documented; internal/obscure targets must NOT be listed
# ------------------------------------------------------------------------------------------

ifeq ($(SUPPORTS_HELP),1)
help::
	@echo
	@echo "Integration/Performance/Format tests targets (to run OUTSIDE the builder docker):"
	@echo "    automated_tests:                      runs FUNCTIONAL integration tests"
	@echo "    automated_tests_list:                 provide a list of available integration tests"
	@echo "    automated_tests_help:                 provide detailed help about all integration tests options"
	@echo "    automated_tests_stop:                 forcefully stop any possible integration test still running/stuck"
	@echo "    docker_format_check_json_yaml:        run the [format_check_json_yaml] target inside the atf docker"
	@echo "    docker_format_inplace_json_yaml:      run the [format_inplace_json_yaml] target inside the atf docker"
	@echo "    performance_tests:                    runs PERFORMANCE tests"
	@echo
	@echo "List of GNU make variables supported by 'automated_tests' target: TESTSUITE,CFG,RUN_SLEEP_INSIDE,SKIP_STOP_TESTCASE,SKIP_TESTBUNDLE_UPDATE,SKIP_FORMAT_CHECK,ATF_BACKEND,ATF_DOCKER_IMAGE_VARIANT,ATF_DEBUG,ATF_MONITOR_PERFORMANCE,ATF_NUM_REPETITIONS,ATF_KUBE_CONTEXT,ATF_KUBE_NAMESPACE,ATF_CPUS,ATF_ADDITIONAL_OPTIONS,APT_DEBUG_MODE,APT_SPECIFIC_RUN"
endif # SUPPORTS_HELP
