#
# This is the file which is included by all Makefiles that wants to
# get definitions of
#  - architecture-related variables
#  - toolchain-related variables
#  - some 3rd party libs
#
# The calling Makefile must define:
#     TOPDIR
#         The directory containing this makefile
# This makefile defines several useful variables, among which:
#     CFLAGS
#         A collection of flags specific to C.  This is passed to GCC during both
#         compilation and linking of C code.  Do not overwrite this variable;
#         if really needed, append contents to it.
#     CPPFLAGS
#         A collection of flags specific to C++.  This is passed to GCC during both
#         compilation and linking of C++ code.  Do not overwrite this variable;
#         if really needed, append contents to it.
#     COMMONFLAGS
#         A collection of flags containing architecture-related flags and
#         build-mode-flags (debug/release), etc. This is passed to GCC during both
#         compilation and linking of both C and C++ code.  Do not overwrite this variable;
#         if really needed, append contents to it.
#     LINKER_FLAGS
#         A collection of linker-specific flags. This is passed to GCC only during
#         linking.  Do not overwrite this variable; if really needed, append contents to it.
#

# ---------------------------------------------
# dependencies
# ---------------------------------------------

ifndef TOPDIR
$(error Please define TOPDIR first)
endif

# ---------------------------------------------
# environment variables
# ---------------------------------------------

# if ARCH is not provided, use the current machine architecture
HOST_ARCH:=$(shell uname -m)
ifndef ARCH
    # if inside docker container set arch=docker
    PATH_TO_DOCKERENV=/.dockerenv
    ifneq ("$(wildcard $(PATH_TO_DOCKERENV))","")
        ARCH:=docker
    else
    	ARCH:=${HOST_ARCH}
    endif
#$(info Assuming ARCH=$(ARCH))
endif

ifdef NOISY
$(info "##Profile: pwd=$(CURDIR), ARCH=$(ARCH), time=$(shell date)")
endif

# ---------------------------------------------
# check command-line variables
# ---------------------------------------------

# default value is the default runtime used in EVA/ECC
# FC33    used between 2020-2022
# FC35    IN-58492, adopted in June 2022
# OL9     Azure-DevOpsID-342491, adopted in May 2024
RUNTIME=OL9
ifneq ($(BUILDER_RUNTIME),)
	RUNTIME=${BUILDER_RUNTIME}
endif

ifndef CFG
	CFG=release
#$(info Assuming CFG=$(CFG))
else
	ifneq ($(CFG), debug-gcov)
		ifneq ($(CFG), debug-tsan)
			ifneq ($(CFG), debug)
				ifneq ($(CFG), release)
$(error Unsupported value for CFG variable: $(CFG))
				endif
			endif
		endif
	endif
endif

# Look at the value of the environment variable "LAB": that's set on all Teamcity agents as env variable
# with either the value 'modena', 'billerica' or 'massy'
ifeq ($(LAB),)
	# default value
	LAB=billerica
endif
ifneq ($(LAB), billerica)
	ifneq ($(LAB), modena)
		ifneq ($(LAB), massy)
$(error Unsupported value for LAB variable: $(LAB))
		endif
	endif
endif

ifeq ($(ARCH), ind)				 # ind is a shortcut for 'independent'
	override ARCH=independent
endif

ifneq ($(ARCH), x86_64)
	ifneq ($(ARCH), tile)
		ifneq ($(ARCH), independent)
			ifneq ($(ARCH),docker)
$(error Unsupported value for ARCH variable: $(ARCH))
			endif
		endif
	endif
endif

ifndef V
	# verbosity variable: V=1 shows all C++ build output, V=0 shows very condensed output
	V=0
endif

ifeq ($(BUILD_AGENT),)
	ifneq ($(TEAMCITY_VERSION),)
		# by default we enable build-agent logics only when running under Teamcity
		BUILD_AGENT=1
	else
		BUILD_AGENT=0
	endif
endif
ifneq ($(BUILD_AGENT), 1)
	ifneq ($(BUILD_AGENT), 0)
$(error Unsupported value for BUILD_AGENT variable: $(BUILD_AGENT))
	endif
endif

ifeq ($(APT_BUILD),)
# default value
APT_BUILD=0
endif
ifneq ($(APT_BUILD), true)
	ifneq ($(APT_BUILD), 1)
		ifneq ($(APT_BUILD), 0)
			ifneq ($(APT_BUILD), false)
$(error Unsupported value for APT_BUILD variable: $(APT_BUILD))
			endif
		endif
	endif
endif

# convert from true/false to 1/0
ifeq ($(APT_BUILD), false)
APT_BUILD=0
endif
ifeq ($(APT_BUILD), true)
APT_BUILD=1
endif

ifndef ENABLE_LTO
	ENABLE_LTO=0
endif

ifeq ($(NUM_CPUS),)
	# NOTE: it's important NOT to use -j alone: -j alone means "infinite jobs" and we've seen a lot
	#       of cases where using -j alone triggers a lot of OOM for GCC
	NUM_CPUS:=$(shell grep -c ^processor /proc/cpuinfo)
endif

# ---------------------------------------------
# system tools
# ---------------------------------------------

RM = rm -f
RMDIR = rm -rf
MKDIR = mkdir -p
CP = cp -f
LN = ln -sfv

# if recursive behavior for 'cp' is needed use 'cp -r $(CP_OPTS)'
CP_OPTS:=-af
ifeq ($(V),1)
CP_OPTS:=-afv
endif

LN_OPTS:=-sf
ifeq ($(V),1)
LN_OPTS:=-sfv
endif

# if recursive behavior for 'rm' is needed use 'rm -r $(RM_OPTS)'
RM_OPTS:=-f
ifeq ($(V),1)
RM_OPTS:=-fv
endif

TAR_CREATE_OPTS:=-cf
ifeq ($(V),1)
TAR_CREATE_OPTS:=-cvf
endif

# beautysh tool location in the builder docker:
BEAUTYSH_BIN:=/usr/local/bin/beautysh
BEAUTYSH_CMDLINE_OPTS:=--force-function-style=fnpar

# clang-format tool location in the builder docker:
CLANG_FORMAT_BIN:=/usr/bin/clang-format-v11

# cppcheck tool location in the builder docker:
CPPCHECK_BIN:=/usr/bin/cppcheck

# cppcheck config files in the builder docker:
# note that the crazy packagers of CppCheck like to change
# their default configuration folder from release to release
# in subtle ways, e.g. changing one of the two "C" letters to "c"
# or viceversa
CPPCHECK_CFG_DIR:=/usr/share/Cppcheck

# maven binary in the builder docker:
MAVEN:=/usr/bin/mvn -T1C

# Sonarqube tools in the builder docker:
SONARQUBE_WRAPPER_OUTDIR:=SonarqubeArtifacts/
SONARQUBE_MAKE_WRAPPER_BIN:=/opt/build-wrapper-linux-x86/build-wrapper-linux-x86-64 --out-dir $(SONARQUBE_WRAPPER_OUTDIR) /usr/bin/make -j$(NUM_CPUS)
SONARQUBE_CLI_SCAN_BIN:=/usr/local/bin/sonar-scanner

# ---------------------------------------------
# build type
# ---------------------------------------------

# !! NOTE: Tilera ignores CFG=debug/release and always builds in release mode!!
ifeq ($(ARCH), tile)
ifeq ($(CFG), debug)
override CFG=release
$(warning Tilera supports only CFG=release, overriding CFG variable)
endif
endif

#
# C/C++ flags applied to ALL build types:
#  -pipe: use pipes rather than temporary files for communication between the various stages of compilation
#  -fmessage-length=0: do not line-wrap errors/warnings (for better IDE integration)
#  -MMD and -MP: instruct GCC to cooperate with GNU make by producing .d files for dependency calculations
#  -fno-semantic-interposition: do not create stubs that allows for semantic interposition of dynamic symbols (e.g. LD_PRELOAD tricks);
#                               these stubs are typically named @plt in GDB and have a performance hit plus make debugging less easy
# Linker-specific:
#  --build-id: this is required to be able to build -debuginfo RPM packages
#  --as-needed: emit a DT_NEEDED tag only for a library that satisfies an undefined symbol reference from a regular object file 
CFLAGS :=
CPPFLAGS :=
COMMONFLAGS := -pipe -fmessage-length=0 -MMD -MP -fno-semantic-interposition
LINKER_FLAGS := -pipe -fmessage-length=0 -Wl,--build-id -Wl,--as-needed


#
# TODO: unify debug settings below that are currently split between CFG=debug/debug-gcov/debug-tsan;
#       the reason why that's important is because we need to mix CFG=debug and CFG=debug-gcov in our pipelines
#       (to build coverage reports of the current repo only) and thus we must ensure perfect binary interoperability;
#       this is hard to achieve diffing manually the below sections
#

ifeq ($(CFG), debug)
	BUILD := debug
	DEBUG_TAG := _D
	DEBUG_BUILD := yes
	CONAN_INSTALL_BUILD_TYPE := Debug
	CONAN_CREATE_BUILD_TYPE := Debug

	#  -Wstack-usage: we need to raise from 300kB default to 5MB (!!!!!) because ASAN instrumentation is increasing a lot stack usage
	# -ftree-vrp is useful when building with -O0 to allow the compiler to do array bound checking anyway (it's enabled normally only at O2)
	COMMONFLAGS += -Wstack-usage=5000000 -Werror -O0 -ggdb -fno-inline -ftree-vrp
	LINKER_FLAGS += -O0
	EXTRA_DEFS := DEBUG _DEBUG

	# By default debug builds will include asan instrumentation. If someone needs to disable it explicitly, set SKIP_ASAN_INSTRUMENTATION to 1
	ifneq ($(SKIP_ASAN_INSTRUMENTATION), 1)
		# -fsanitize=address activates the Google Address Sanitizer, very useful to spot memory problems: https://github.com/google/sanitizers/wiki/AddressSanitizer
		# -fno-omit-frame-pointer is required to get proper stack trace in case of failures
		COMMONFLAGS += -fsanitize=address -fno-omit-frame-pointer
		LINKER_FLAGS += -fsanitize=address
		# detect_odr_violation: To avoid false detection of one definition rule. 
		# E.g.: when a variable is defined in base-libs and is available to downstream via common-libs & protocol-events, without this option ASAN would fail with odr failures
		ASAN_OPTIONS += detect_odr_violation=1
	endif

	# Boost extra-defines to help debugging:
	# Any debug config MUST include BOOST_CB_ENABLE_DEBUG flag for compilation. 
	# Mixing together code built with & without  this flag will lead to unexplained behaviors with the classes which declare boost::circular_buffers
	# like mismatch in sizeof of class objects and missing values of variables which are present after boost::circular_buffers variables etc.
	#     BOOST_CB_ENABLE_DEBUG is to enable debugging of circular buffers
	EXTRA_DEFS += BOOST_CB_ENABLE_DEBUG
else
ifeq ($(CFG), debug-tsan)
	BUILD := debug-tsan
	DEBUG_TAG := _T
	DEBUG_BUILD := yes
	CONAN_INSTALL_BUILD_TYPE := DebugTsan
	CONAN_CREATE_BUILD_TYPE := DebugTsan

	#  -Wstack-usage: warn against pieces of code trying to use a lot of stack memory; 300kB works with all our code and is a reasonable value
	# -ftree-vrp is useful when building with -O0 to allow the compiler to do array bound checking anyway (it's enabled normally only at O2)
	# -fsanitize=thread activates the Google Thread Sanitizer, very useful to spot race conditions: https://github.com/google/sanitizers/wiki/ThreadSanitizerCppManual
	COMMONFLAGS += -Wstack-usage=500000 -Werror -O0 -ggdb -fno-inline -ftree-vrp -fsanitize=thread
	LINKER_FLAGS += -O0 -fsanitize=thread
	EXTRA_DEFS := DEBUG _DEBUG

	# Boost extra-defines to help debugging:
	# Any debug config MUST include BOOST_CB_ENABLE_DEBUG flag for compilation. 
	# Mixing together code built with & without  this flag will lead to unexplained behaviors with the classes which declare boost::circular_buffers
	# like mismatch in sizeof of class objects and missing values of variables which are present after boost::circular_buffers variables etc.
	#     BOOST_CB_ENABLE_DEBUG is to enable debugging of circular buffers
	EXTRA_DEFS += BOOST_CB_ENABLE_DEBUG
else
ifeq ($(CFG), debug-gcov)
	BUILD := debug-gcov
	DEBUG_TAG := _G
	DEBUG_BUILD := yes

	# XXX: if you want to add support to debug-gcov for upstream,
	#      change CONAN_INSTALL_BUILD_TYPE to DebugGcov
	# GCOV is a special case: when using CFG=debug-gcov we assume that
	# the developer wants to fetch regular debug binaries of upstream
	# dependencies (without gcov instrumentation) and instead will build
	# with gcov instrumentation only the code of the repo he's working on.
	# Of course any Conan package created in this mode and containing gcov-instrumented code needs to be created with build_type=DebugGcov 

	CONAN_INSTALL_BUILD_TYPE := DebugGcov
	CONAN_CREATE_BUILD_TYPE := DebugGcov

	# By default debug-gcov will ALWAYS include asan instrumentation. This would be required to catch memory related issues while running unit/automated tests and it cannot be disabled

	#  -Wstack-usage: we need to raise from 300kB default to 5MB (!!!!!) because ASAN instrumentation is increasing a lot stack usage
	# -ftree-vrp is useful when building with -O0 to allow the compiler to do array bound checking anyway (it's enabled normally only at O2)
	# -fsanitize=address activates the Google Address Sanitizer, very useful to spot memory problems: https://github.com/google/sanitizers/wiki/AddressSanitizer
	# -fno-omit-frame-pointer is required to get proper stack trace in case of failures
	# --coverage is added to generate code coverage information when unit tests and ATF tests run with ASAN binaries
	COMMONFLAGS += -Wstack-usage=5000000 -Werror -O0 -ggdb -fno-inline -ftree-vrp -fsanitize=address -fno-omit-frame-pointer --coverage
	LINKER_FLAGS += -O0 -fsanitize=address --coverage
	EXTRA_DEFS := DEBUG _DEBUG GCOV_INSTRUMENTATION

	# detect_odr_violation: To avoid false detection of one definition rule. 
	# E.g.: when a variable is defined in base-libs and is available to downstream via common-libs & protocol-events, without this option ASAN would fail with odr failures
	ASAN_OPTIONS += detect_odr_violation=1

	# Boost extra-defines to help debugging:
	# Any debug config MUST include BOOST_CB_ENABLE_DEBUG flag for compilation. 
	# Mixing together code built with & without  this flag will lead to unexplained behaviors with the classes which declare boost::circular_buffers
	# like mismatch in sizeof of class objects and missing values of variables which are present after boost::circular_buffers variables etc.
	#     BOOST_CB_ENABLE_DEBUG is to enable debugging of circular buffers
	EXTRA_DEFS += BOOST_CB_ENABLE_DEBUG
else
	BUILD := release
	DEBUG_TAG :=
	DEBUG_BUILD := no
	CONAN_INSTALL_BUILD_TYPE := Release
	CONAN_CREATE_BUILD_TYPE := Release

	#  -Wstack-usage: warn against pieces of code trying to use a lot of stack memory; 300kB works with all our code and is a reasonable value
	# NOTE1: Intel nehalem (nov 2008) was the first architecture to support SSE4.2 instruction set which is a requirement for modern
	#        DPDK versions so that we cannot really run our C++ code on older architecture anyway. Thus optimize at least for this
	#        type of CPU:
	# NOTE2: -ggdb is put even in release builds because we later use, at RPM packaging time, the %debug_package directive to strip
	#        the debug symbols and place them in a different RPM:
	# NOTE3: -fno-omit-frame-pointer is important when O1, O2 or O3 are used to make it possible to always reconstruct correctly the backtrace;
	#        it has a very little performance impact anyway
	COMMONFLAGS += -Wstack-usage=300000 -Werror -O3 -ggdb -fno-omit-frame-pointer -march=nehalem -mtune=nehalem

	LINKER_FLAGS += -O3
	# in case LTO is enabled, it is important to provide the optimization flag also to the linker stage:
	EXTRA_DEFS :=
endif
endif
endif

EXTRA_DEFS += ARCH_$(ARCH)
COMMONFLAGS += -fPIC
CPPFLAGS += -std=c++20
LINKER_FLAGS += -rdynamic -shared-libgcc -fPIC
USR_LIBS += numa

ifeq ($(ENABLE_LTO),1)
# Note: last time it was tested (Jan 2017) it was found that LTO provided no big improvement;
#       the size of the generated binaries (.so and binaries) decreases between 5% to 15% with LTO enabled but linking is terribly slowed down
COMMONFLAGS += -flto -fuse-linker-plugin -fvisibility=hidden
LINKER_FLAGS += -flto -fuse-linker-plugin
endif

ifdef NEED_PERMISSIVE_FLAG_HACK
#$(warning Adding PERMISSIVE HACK to compile options)
	PERMISSIVE_FLAG := -fpermissive
	WARNING_FLAGS :=
else
	# by default, do not include -fpermissive, rather add gcc flags that help you spot problems!
	PERMISSIVE_FLAG :=

	# enable as many useful warnings as possible:
	WARNING_FLAGS := -Wall -Wextra -Warray-bounds -Wreturn-type

	# promote some warnings to errors
	ERRORS_FLAGS := -Werror=nonnull

	# further warnings can be promoted to errors:
	# -Werror=non-virtual-dtor -Werror=array-bounds -Werror=int-to-pointer-cast -Werror=return-type

	#
	# explanation for disabled warnings:
	# -Wno-reorder:                 this is just because we have a BUNCH of wrongly ordedered parameters in ctors
	# -Wno-unused-parameter:        this is just because we have a BUNCH of unused parameters
	# -Wno-sign-compare:            this is just because we have a BUNCH of wrong sign comparisons
	# -Wno-unused-variable:         this is just because we have a BUNCH of unused variables
	#
	# -Wno-invalid-offsetof:        ????
	#
	# -Wno-unused-local-typedefs:   ASSERT_STATIC() creates a typedef to catch at compile-time invalid situations. The typedef is never used
	#                               in reality and thus would generate a warning.
	# -Wno-address-of-packed-member:	this is because we need to use packed structs and reference to struct members
	#									but gcc 9.2.0 (used inside docker toolchain) raise warnings. taking address of packed member
	#								 	may result in an unaligned pointer value and performance gap
	#
	# check using below url if common warning or c, c++ specific
	# https://gcc.gnu.org/onlinedocs/gcc/Option-Index.html
	DISABLED_WARNING_FLAGS_COMMON := -Wno-unused-parameter -Wno-sign-compare -Wno-unused-variable -Wno-unused-local-typedefs -Wno-address-of-packed-member
	DISABLED_WARNING_FLAGS_CPP := $(DISABLED_WARNING_FLAGS_COMMON) -Wno-reorder -Wno-invalid-offsetof -Wno-register -Wno-deprecated-declarations
	DISABLED_WARNING_FLAGS_C := $(DISABLED_WARNING_FLAGS_COMMON)
endif


# ---------------------------------------------
# toolchain vars for current $(ARCH) value
# ---------------------------------------------

ifeq ($(ARCH), x86_64)
	# map the ARCH-independent values to the x86_64 toolchain
	TOOLCHAIN=$(TOOLCHAIN_x86_64)
	TOOLCHAIN_VERSION=$(TOOLCHAIN_VERSION_x86_64)
	TOOLCHAIN_RELEASE=$(TOOLCHAIN_RELEASE_x86_64)
	TOOLCHAIN_GCC_VERSION=$(TOOLCHAIN_GCC_VERSION_x86_64)

	# derived values
	CROSSTOOLDIR=/opt/empirix/toolchains/x86_64-$(TOOLCHAIN_x86_64)-$(TOOLCHAIN_VERSION_x86_64)-$(TOOLCHAIN_RELEASE_x86_64)
	RUNTIME_DIR=/lib/empirix/x86_64-$(TOOLCHAIN_VERSION_x86_64)-$(TOOLCHAIN_RELEASE_x86_64)
	CROSSTOOL_RPM=empirix-toolchain-$(TOOLCHAIN_x86_64)-$(TOOLCHAIN_VERSIONED_SUFFIX)
	RUNTIME_RPM=empirix-toolchain-$(TOOLCHAIN_x86_64)-runtime-libs-$(TOOLCHAIN_VERSIONED_SUFFIX)
	DEBUGTOOLS_RPM=empirix-toolchain-$(TOOLCHAIN_x86_64)-debug-tools-$(TOOLCHAIN_VERSIONED_SUFFIX)
	TOOLBIN=$(CROSSTOOLDIR)/cross-tools/bin

	# Placing the version in the RPM name is the official way to allow simultaneous installations of multiple versions
	# of the same actual RPM. In this case it allows us to keep installed e.g.
	#   empirix-toolchain-gcc-7.3-glibc-2.25-runtime-libs-NFV4
	#   empirix-toolchain-gcc-7.3-glibc-2.25-runtime-libs-NFV5
	# which have identical GCC and glibc version but are still different:
	TOOLCHAIN_VERSIONED_SUFFIX=$(TOOLCHAIN_VERSION_x86_64)$(TOOLCHAIN_RELEASE_x86_64)-$(TOOLCHAIN_VERSION_x86_64)-$(TOOLCHAIN_RELEASE_x86_64)
	
	# EMPIRIX_LDFLAGS are used when building 3rd party libs:
	EMPIRIX_LDFLAGS=-Wl,--dynamic-linker -Wl,$(RUNTIME_DIR)/ld-linux.so.2
	
	# NOTE: adding $(CROSSTOOLDIR)/tools/include to EXTRA_INCS would be more correct but that would break
	#       cppcheck analysis tool, so use COMMONFLAGS:
	COMMONFLAGS+=-I$(CROSSTOOLDIR)/tools/include
	DEBUGTOOLS_DIR=$(CROSSTOOLDIR)/debug-tools
	ARCH_IS_DOCKER_OR_x86_64=1
endif

ifeq ($(ARCH), docker)
	TOOLBIN=/usr/bin
	DEBUGTOOLS_DIR=/usr
	RUNTIME_DIR=
	EMPIRIX_LDFLAGS=
	ARCH_IS_DOCKER_OR_x86_64=1
endif



# ---------------------------------------------
# compiler and related tools
# ---------------------------------------------

# if a toolchain is NOT available (e.g. ARCH=tile), skip compiler definition
ifneq ($(TOOLBIN),)
	# Empirix toolchain tools:
	CPP:=$(TOOLBIN)/g++
	CXX:=$(TOOLBIN)/g++
	CC:=$(TOOLBIN)/gcc
	LINKER:=$(TOOLBIN)/g++
	LD:=$(TOOLBIN)/ld
	AR:=$(TOOLBIN)/gcc-ar
	RANLIB:=$(TOOLBIN)/gcc-ranlib
	GDB:=$(DEBUGTOOLS_DIR)/bin/gdb
	VALGRIND:=$(DEBUGTOOLS_DIR)/bin/valgrind
	# IMPORTANT: there is already a GCOV variable, call this differently:
	GCOV_BIN:=$(CROSSTOOLDIR)/cross-tools/bin/gcov
	LCOV_BIN:=$(CROSSTOOLDIR)/tools/bin/lcov
	LCOV_GENHTML:=$(CROSSTOOLDIR)/tools/bin/genhtml
	
	LCOV_OPTS:=--gcov-tool $(GCOV_BIN) --no-external --quiet
endif

# toolchain tools native on the host machine:
HOSTCC:=gcc -pipe
HOSTCPP:=g++ -pipe
HOSTCXX:=g++ -pipe
HOSTLD:=ld

#
# NOTE: what we do here is to pass to the compiler as C preprocessor define the size of 
#       the L1 cache of the machine building our code. This is the best we can do as some
#       of the source code depends on build-time optimizations and sizing choices done based 
#       on L1 cache size. This means that this code will actually be optimized for the CPU
#       of the build agent producing the build, and not for the CPU actually running the product
#       (unknown at this time). Anyway this is kind of useless exercise: on all CPUs I ever used
#       the L1 cache size is always 64B...
#
EXTRA_DEFS += _BUILD_TIME_CACHELINE_SIZE=$(shell getconf LEVEL1_DCACHE_LINESIZE)


BUILDID:=${ARCH}-${RUNTIME}-${BUILD}
TOPBUILDDIR:=./Build
BUILDDIR:=${TOPBUILDDIR}/$(BUILDID)

# BUILDID_UPSTREAM is like BUILDID restricted to just the $(ARCH)-debug/release modes; every other BUILDID string
# (e.g. "x86_64-debug-gcov" or "x86_64-release-gcov") is mapped back to just "x86_64-debug" or "x86_64-release"
# This is useful to identify folders for thirdparty/upstream repositories which provide binaries only in debug/release mode, not any other fancy build configuration.
BUILDID_UPSTREAM:=$(BUILDID)
#BUILDID_UPSTREAM:=$(subst -gcov,,$(BUILDID_UPSTREAM))

CFG_UPSTREAM:=$(CFG)
#CFG_UPSTREAM:=$(subst -tsan,,$(CFG_UPSTREAM))
#CFG_UPSTREAM:=$(subst -gcov,,$(CFG_UPSTREAM))
CFG_UPSTREAM:=$(CFG)

# synonims for compatibility with older makefiles
BUILDID_THIRDPARTY:=$(BUILDID_UPSTREAM)
CFG_THIRDPARTY:=$(CFG_UPSTREAM)


# ---------------------------------------------
# Teamcity integration
# ---------------------------------------------

TEAMCITY_ARTIFACTS_DIR_MAIN:=$(abspath $(TOPDIR)/TeamcityArtifacts)
TEAMCITY_ARTIFACTS_DIR_UNITTESTS_COREDUMPS:=$(TEAMCITY_ARTIFACTS_DIR_MAIN)/unit-tests-coredumps
TEAMCITY_ARTIFACTS_DIR_UNITTESTS_JUNIT:=$(TEAMCITY_ARTIFACTS_DIR_MAIN)/unit-tests-junit
TEAMCITY_ARTIFACTS_DIR_UNITTESTS_LOGS:=$(TEAMCITY_ARTIFACTS_DIR_MAIN)/unit-tests-logfiles
TEAMCITY_ARTIFACTS_DIR_COVERAGE_HTML_REPORT:=$(TEAMCITY_ARTIFACTS_DIR_MAIN)/cpp-code-coverage-html

# this is the folder where tarballs of GCDA files will be placed; for some repositories, this folder
# will be huge and thus it has been moved away from TeamcityArtifacts:
TEAMCITY_ARTIFACTS_DIR_COVERAGE_GCOV_REPORT:=$(abspath $(TOPDIR))/CppCodeCoverageTarballs


# ---------------------------------------------
# Macros to produce more readable output
# see https://www.gnu.org/software/make/manual/make.html#Canned-Recipes
# ---------------------------------------------

COLOR_RED:=\033[0;31m
COLOR_GREEN:=\033[0;32m
COLOR_YELLOW:=\033[0;33m
COLOR_OFF:=\033[0m

define open_section
	@[ ! -z "$(TEAMCITY_VERSION)" ] && echo "##teamcity[blockOpened name='$(1)']" || echo -e "\n>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>\n>> $(COLOR_YELLOW)START $(1)$(COLOR_OFF) >>"
endef
define close_section
	@[ ! -z "$(TEAMCITY_VERSION)" ] && echo "##teamcity[blockClosed name='$(1)']" || echo -e "<< $(COLOR_YELLOW)FINISHED $(1)$(COLOR_OFF) <<\n<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<\n"
endef
define print_red
	@[ ! -z "$(TEAMCITY_VERSION)" ] && echo "$(1)" || echo -e "$(COLOR_RED)$(1)$(COLOR_OFF)"
endef
define print_green
	@[ ! -z "$(TEAMCITY_VERSION)" ] && echo "$(1)" || echo -e "$(COLOR_GREEN)$(1)$(COLOR_OFF)"
endef
define print_yellow
	@[ ! -z "$(TEAMCITY_VERSION)" ] && echo "$(1)" || echo -e "$(COLOR_YELLOW)$(1)$(COLOR_OFF)"
endef


# ---------------------------------------------
# Tilera support
# ---------------------------------------------

DISABLE_TILERA:=0
TILERA_MDE_VERSION:=4.3.4.193953
ifneq (,$(wildcard /opt/tilera/TileraMDE-$(TILERA_MDE_VERSION)/tilegx/.))
    export TILERA_ROOT=/opt/tilera/TileraMDE-$(TILERA_MDE_VERSION)/tilegx
endif

TILERA_MDE := /opt/tilera/TileraMDE-$(TILERA_MDE_VERSION)/tilegx

# many source code files enable some specific debugging routines when DEBUG is defined
# NOTE: TILE_ASSERT() has effect only when DEBUG is defined; assert() and art_assert() always have effects
#TILERA_DEBUG_FLAGS := -DDEBUG
TILERA_DEBUG_FLAGS := -DNDEBUG

# some other possible optimization flags include (see Tilera Optimization Guide):
#    -floop-block -fprefetch-loop-arrays -funsafe-loop-optimizations
# and for linker:
#    -flto
TILERA_OPTIMIZATION := -O2
#TILERA_OPTIMIZATION := -O0 -g
TILERA_WARNING_FLAGS := -Wall -Werror -Wstrict-prototypes -Wundef

# system libraries:
TILERA_USR_CFLAGS := -I$(TILERA_MDE)/tile/usr/include \
					-I$(TILERA_MDE)/tile/usr/include/libxml2 \
					-I$(TILERA_MDE)/tile/usr/include/glib-2.0 \
					-I$(TILERA_MDE)/tile/usr/lib/glib-2.0/include
TILERA_USR_LIBS := -lglib-2.0 -levent -lpthread -lrt -ltmc -lgxpci -lgxio -lpcap

# global C/linker flags:
TILERA_CFLAGS := -std=gnu99 -fno-strict-aliasing -pthread -DARCH_tile \
					$(TILERA_PROF_FLAGS) $(TILERA_DEBUG_FLAGS) $(TILERA_WARNING_FLAGS) $(TILERA_OPTIMIZATION) $(TILERA_FEEDBACK_COMPILER) \
					$(TILERA_USR_CFLAGS) -I.


# ---------------------------------------------
# Docker builder image vars
# ---------------------------------------------
TILERA_TOOLCHAIN_DIR=/opt/tilera
CONAN_DATA_DIR=/opt/empirix/conan
CONAN_LOCAL_PROFILE_DIR=$(HOME)/.conan/profiles/
CONAN_DOCKER_PROFILE_DIR=/root/.conan/profiles/
CONAN_REMOTES_CONFIG_FILE=/root/.conan/remotes.json
# FIXME IN-41192: enable the following values to run Conan as the "builder" user inside the builder docker instead of "root" user:
#CONAN_LOCAL_PROFILE_DIR=$(HOME)/.conan/
#CONAN_DOCKER_PROFILE_DIR=/home/builder/.conan/
# note that DOCKER_TARBALL_DIR is chosen in a such a way that Makefiles stored in different subfolders
# will use a different directory located in "DockerArtifacts"; in this way even if the tarballs
# produced by 2 Makefiles are both named "docker-binaries.tar.gz" they will not conflict because they
# get stored in 2 different locations:
DOCKER_TARBALL_DIR:=$(TOPDIR)/DockerArtifacts-$(RUNTIME)/$(shell realpath --relative-to=$(TOPDIR) .)

# ------------------------------------------------------------------------------------------
# Documentation
# ------------------------------------------------------------------------------------------

DOCUMENTATION_GENERATOR_DIR:=${EMPIRIX_PIPELINE_FRAMEWORK_MAKEFILE_SUPPORT_DIR}/../config-file-generator/
DOCUMENTATION_GENERATOR_SCRIPT:=$(DOCUMENTATION_GENERATOR_DIR)/generate_helm_chart_reference_xlsx.py
SYSTEM_HEALTH_KPI_GENERATOR_DIR:=${EMPIRIX_PIPELINE_FRAMEWORK_MAKEFILE_SUPPORT_DIR}/../systemhealth-kpi-generator/
SYSTEM_HEALTH_KPI_GENERATOR_SCRIPT:=$(SYSTEM_HEALTH_KPI_GENERATOR_DIR)/systemhealth_kpi_generator.py
