#
# This is the file which is included by
#   - Application-targets.mk
#   - SharedLibrary-targets.mk
#   - StaticLibrary-targets.mk
#   - Recursive-targets.mk
# Note that this file is included AFTER the definition of main makefile targets.
#
# The calling Makefile must define:
#     SUPPORTS_CPPCHECK
#         If 2 then the "cppcheck" target is not defined at all; the caller makefile must define it if needed.
#         If 1 (default) the 'cppcheck' target will be provided for $(TARGS_SRC) and $(TARGS_HDRS) source and header files.
#         If 0 (default) then the "cppcheck" target will be implemented as no-op.
#     SUPPORTS_FORMAT_CHECK
#         If 1 (default) the "format_check" target provided for $(TARGS_SRC) and $(TARGS_HDRS) source and header files.
#         If 0 then the "format_check" target will be implemented as no-op.
#         Same applies to "format_inplace"
#
# This makefile provides the following targets:
#  - .cpp and .c to .o files


# ------------------------------------------------------------------------------------------
# Variables from caller makefile
# ------------------------------------------------------------------------------------------

ifndef ENABLE_DEEP_CPPCHECK
ifeq ($(BUILD_AGENT),1)
#$(info Apparently this is a build machine, enabling deep but slow CppChecks)
# default value
#ENABLE_DEEP_CPPCHECK=1

# keep disabled even on build machines -- it's damn slow
ENABLE_DEEP_CPPCHECK=0
else
# default value
ENABLE_DEEP_CPPCHECK=0
endif
endif

ifndef HAS_CLEAN_SUPPORT
# default value
HAS_CLEAN_SUPPORT=1
endif


# ------------------------------------------------------------------------------------------
# Targets to produce object files
# ------------------------------------------------------------------------------------------

ifeq ($(ARCH), tile)

$(BUILDDIR)/%.o: %.cpp  $(TOPDIR)/conandata.yml
	$(TILERA_MDE)/bin/tile-gcc -MMD -MP -x c $(TILERA_CFLAGS) $(TILERA_EXTRA_CFLAGS) -o $@ -c $<

$(BUILDDIR)/%.o: %.c  $(TOPDIR)/conandata.yml
	$(TILERA_MDE)/bin/tile-gcc -MMD -MP $(TILERA_CFLAGS) $(TILERA_EXTRA_CFLAGS) -o $@ -c $<

else		# $(ARCH)=tile

$(BUILDDIR)/%.o: %.cpp $(TOPDIR)/conandata.yml
ifeq ($(V),0)
	@echo "CXX [$(CFG)] $<"
	@$(CXX) -c $(CPPFLAGS) $(COMMONFLAGS) $(PERMISSIVE_FLAG) $(WARNING_FLAGS) $(DISABLED_WARNING_FLAGS_CPP) $(ERRORS_FLAGS) $< -o $@
else
	$(CXX) -c $(CPPFLAGS) $(COMMONFLAGS) $(PERMISSIVE_FLAG) $(WARNING_FLAGS) $(DISABLED_WARNING_FLAGS_CPP) $(ERRORS_FLAGS) $< -o $@
endif


$(BUILDDIR)/%.o: %.cxx $(TOPDIR)/conandata.yml
ifeq ($(V),0)
	@echo "CXX [$(CFG)] $<"
	@$(CXX) -c $(CPPFLAGS) $(COMMONFLAGS) $(PERMISSIVE_FLAG) $(WARNING_FLAGS) $(DISABLED_WARNING_FLAGS_C) $(DISABLED_WARNING_FLAGS_CPP) $(ERRORS_FLAGS) $< -o $@
else
	$(CXX) -c $(CPPFLAGS) $(COMMONFLAGS) $(PERMISSIVE_FLAG) $(WARNING_FLAGS) $(DISABLED_WARNING_FLAGS_C) $(DISABLED_WARNING_FLAGS_CPP) $(ERRORS_FLAGS) $< -o $@
endif

$(BUILDDIR)/%.o: %.cc $(TOPDIR)/conandata.yml
ifeq ($(V),0)
	@echo "CXX [$(CFG)] $<"
	@$(CXX) -c $(CPPFLAGS) $(COMMONFLAGS) $(PERMISSIVE_FLAG) $(WARNING_FLAGS) $(DISABLED_WARNING_FLAGS_C) $(DISABLED_WARNING_FLAGS_CPP) $(ERRORS_FLAGS) $< -o $@
else
	$(CXX) -c $(CPPFLAGS) $(COMMONFLAGS) $(PERMISSIVE_FLAG) $(WARNING_FLAGS) $(DISABLED_WARNING_FLAGS_C) $(DISABLED_WARNING_FLAGS_CPP) $(ERRORS_FLAGS) $< -o $@
endif

$(BUILDDIR)/%.o: %.c $(TOPDIR)/conandata.yml
ifeq ($(V),0)
	@echo "CC $<"
	@$(CC) -c $(CFLAGS) $(COMMONFLAGS) $(WARNING_FLAGS) $(DISABLED_WARNING_FLAGS_C) $(ERRORS_FLAGS) $< -o $@
else
	$(CC) -c $(CFLAGS) $(COMMONFLAGS) $(WARNING_FLAGS) $(DISABLED_WARNING_FLAGS_C) $(ERRORS_FLAGS) $< -o $@
endif

endif		# $(ARCH)=tile



# ------------------------------------------------------------------------------------------
# Targets to produce format checking files
# NOTE: explicit dependency from 'clang-format' is to force rechecking if we change something there
# ------------------------------------------------------------------------------------------

$(BUILDDIR)/%.sourcefmtcheck: %.c $(TOPDIR)/.clang-format
	@echo "FMTCHECK $<"
	@$(CLANG_FORMAT_BIN) --style=file $< >$(BUILDDIR)/$(notdir $<).formatted
	@diff -u $< $(BUILDDIR)/$(notdir $<).formatted >$(BUILDDIR)/$(notdir $<).diff || true
	@if [ -s $(BUILDDIR)/$(notdir $<).diff ] ; then \
		echo " ...failed format check for $<:" ; \
		cat $(BUILDDIR)/$(notdir $<).diff ; \
		false ; \
	else \
		touch $@ ; \
	fi

$(BUILDDIR)/%.sourcefmtcheck: %.cpp $(TOPDIR)/.clang-format
	@echo "FMTCHECK $<"
	@$(CLANG_FORMAT_BIN) --style=file $< >$(BUILDDIR)/$(notdir $<).formatted
	@diff -u $< $(BUILDDIR)/$(notdir $<).formatted >$(BUILDDIR)/$(notdir $<).diff || true
	@if [ -s $(BUILDDIR)/$(notdir $<).diff ] ; then \
		echo " ...failed format check for $<:" ; \
		cat $(BUILDDIR)/$(notdir $<).diff ; \
		false ; \
	else \
		touch $@ ; \
	fi

$(BUILDDIR)/%.sourcefmtcheck: %.cxx $(TOPDIR)/.clang-format
	@echo "FMTCHECK $<"
	@$(CLANG_FORMAT_BIN) --style=file $< >$(BUILDDIR)/$(notdir $<).formatted
	@diff -u $< $(BUILDDIR)/$(notdir $<).formatted >$(BUILDDIR)/$(notdir $<).diff || true
	@if [ -s $(BUILDDIR)/$(notdir $<).diff ] ; then \
		echo " ...failed format check for $<:" ; \
		cat $(BUILDDIR)/$(notdir $<).diff ; \
		false ; \
	else \
		touch $@ ; \
	fi

$(BUILDDIR)/%.headerfmtcheck: %.h $(TOPDIR)/.clang-format
	@echo "FMTCHECK $<"
	@$(CLANG_FORMAT_BIN) --style=file $< >$(BUILDDIR)/$(notdir $<).formatted
	@diff -u $< $(BUILDDIR)/$(notdir $<).formatted >$(BUILDDIR)/$(notdir $<).diff || true
	@if [ -s $(BUILDDIR)/$(notdir $<).diff ] ; then \
		echo " ...failed format check for $<:" ; \
		cat $(BUILDDIR)/$(notdir $<).diff ; \
		false ; \
	else \
		touch $@ ; \
	fi

$(BUILDDIR)/%.bashcheck: %.sh
	@echo "FMTCHECK $<"
	@mkdir -p $(BUILDDIR)/
	@$(BEAUTYSH_BIN) $(BEAUTYSH_CMDLINE_OPTS) --check --files $< ; \
	if [ $$? -ne 0 ]; then \
		echo "Invalid formatting for $<" ; \
		false ; \
	else \
		echo "Valid formatting for $<" ; \
		touch $@ ; \
	fi


# ------------------------------------------------------------------------------------------
# Targets to produce CppCheck pseudo target files
# ------------------------------------------------------------------------------------------

#
# ABOUT PREPROCESSOR SYMBOLS:
#   since there are possibly a ton of them, we don't pass them at command line but rather provide an "empirix.cfg" library

CPPCHECKS_FLAGS := \
	--platform=unix64 --enable=warning,portability \
	--std=c++20 --inline-suppr --max-configs=1 --error-exitcode=2 --language=c++ \
	--library=/usr/share/Cppcheck/empirix
ifeq ($(V),0)
CPPCHECKS_FLAGS += --quiet
endif

STATIC_ANALYSIS_CFG_DIR=${EMPIRIX_PIPELINE_FRAMEWORK_MAKEFILE_SUPPORT_DIR}/../static-analysis

ifneq ($(DISABLE_SUPPRESSION_LIST),1)
CPPCHECKS_FLAGS += --suppressions-list=$(STATIC_ANALYSIS_CFG_DIR)/cppcheck-suppression-list.cfg
endif

ifeq ($(ENABLE_DEEP_CPPCHECK),1)
# by adding all include folders to CppCheck flags, we greatly increase the ability of CppCheck to spot bugs (it will be 
# able to follow all class/function declarations. Unfortunately this makes it also VERY slow.
# For this reason we keep this as an optional behavior.
#
# ABOUT INCLUDE PATHS:
#   we pass only the ones in EXTRA_INCS
CPPCHECKS_FLAGS += $(foreach inc, $(EXTRA_INCS), -I $(abspath $(inc)))
endif

$(CPPCHECK_CFG_DIR)/empirix.cfg:
	@echo "Installing Empirix CppCheck configuration file in $(CPPCHECK_CFG_DIR)"
	@cp -f $(STATIC_ANALYSIS_CFG_DIR)/empirix.cfg $(CPPCHECK_CFG_DIR)


$(BUILDDIR)/%.cppcheck: %.cpp $(CPPCHECK_CFG_DIR)/empirix.cfg
ifeq ($(V),0)
	@echo "CPPCHECK $<"
	@$(CPPCHECK_BIN) $(CPPCHECK_INCS) $(CPPCHECKS_FLAGS) $(EXTRA_CPPCHECKS_FLAGS) $<
else
	$(CPPCHECK_BIN) $(CPPCHECK_INCS) $(CPPCHECKS_FLAGS) $(EXTRA_CPPCHECKS_FLAGS) $<
endif
	@touch $@

$(BUILDDIR)/%.cppcheck: %.cxx $(CPPCHECK_CFG_DIR)/empirix.cfg
ifeq ($(V),0)
	@echo "CPPCHECK $<"
	@$(CPPCHECK_BIN) $(CPPCHECK_INCS) $(CPPCHECKS_FLAGS) $(EXTRA_CPPCHECKS_FLAGS) $<
else
	$(CPPCHECK_BIN) $(CPPCHECK_INCS) $(CPPCHECKS_FLAGS) $(EXTRA_CPPCHECKS_FLAGS) $<
endif
	@touch $@

$(BUILDDIR)/%.cppcheck: %.c $(CPPCHECK_CFG_DIR)/empirix.cfg
ifeq ($(V),0)
	@echo "CPPCHECK $<"
	@$(CPPCHECK_BIN) $(CPPCHECK_INCS) $(CPPCHECKS_FLAGS) $(EXTRA_CPPCHECKS_FLAGS) $<
else
	$(CPPCHECK_BIN) $(CPPCHECK_INCS) $(CPPCHECKS_FLAGS) $(EXTRA_CPPCHECKS_FLAGS) $<
endif
	@touch $@

$(BUILDDIR)/%.cppcheck: %.h $(CPPCHECK_CFG_DIR)/empirix.cfg
ifeq ($(V),0)
	@echo "CPPCHECK $<"
	@$(CPPCHECK_BIN) $(CPPCHECK_INCS) $(CPPCHECKS_FLAGS) $(EXTRA_CPPCHECKS_FLAGS) $<
else
	$(CPPCHECK_BIN) $(CPPCHECK_INCS) $(CPPCHECKS_FLAGS) $(EXTRA_CPPCHECKS_FLAGS) $<
endif
	@touch $@


# ------------------------------------------------------------------------------------------
# CppCheck target
# do implement the target if SUPPORTS_CPPCHECK=1
# ------------------------------------------------------------------------------------------

.PHONY: cppcheck

ifeq ($(SUPPORTS_CPPCHECK), 1)

cppcheck:: $(TARGS_CPPCHECKS)
	@echo "CPP check completed for $(TARGS_SRCS) $(TARGS_HDRS)"
	
cppcheck_clean::
	rm -f $(TARGS_CPPCHECKS)

endif
ifeq ($(SUPPORTS_CPPCHECK), 0)

cppcheck::
	@echo "This directory does not support cppcheck static code analysis."

endif



# ------------------------------------------------------------------------------------------
# Format check target
# ------------------------------------------------------------------------------------------

.PHONY: format_check format_inplace

ifeq ($(SUPPORTS_FORMAT_CHECK), 1)

format_check:: $(TARGS_FORMATCHECKS)
	@echo "C/C++ formatting style check completed for $(TARGS_SRCS) $(TARGS_HDRS)"

format_inplace::
	@echo "Reformatting C/C++ files using clang-format utility"
	@for file in $(TARGS_SRCS) $(TARGS_HDRS); do \
		echo "Reformatting C/C++ file $$file" ; \
		$(CLANG_FORMAT_BIN) --style=file -i $$file ; \
	done

endif
ifeq ($(SUPPORTS_FORMAT_CHECK), 0)

format_check::
	@echo "This directory does not support clang format check."

format_inplace::
	@echo "This directory does not support clang reformatting."

endif


# ------------------------------------------------------------------------------------------
# Clean target
# ------------------------------------------------------------------------------------------

ifeq ($(HAS_CLEAN_SUPPORT), 1)

clean::
	-$(RMDIR) $(BUILDDIR) cpp-code-coverage-results
	-$(RM) *.gcno *.gcda *.lcov
ifeq (1,$(RM_OUTDIR))
	@# just remove the entire output directory and that's it
	-$(RMDIR) $(OUTDIR)
else
ifneq (,$(OUTLIB))
	@echo "Removing the library file $(OUTLIB) and its possible versioned companion files"
	-$(RM) $(OUTLIB)
	-$(RM) $(OUTLIB).?
	-$(RM) $(OUTLIB).?.?
	-$(RM) $(OUTLIB).?.?.?
endif
ifneq (,$(OUTEXE))
	@echo "Removing the executable file $(OUTEXE)"
	-$(RM) $(OUTEXE)
endif
endif
	
fullclean::
	$(MAKE) CFG=release clean
	$(MAKE) CFG=debug-gcov clean
ifneq ($(TEAMCITY_ARTIFACTS_DIR_MAIN),)
	@-$(RM) -rf $(TEAMCITY_ARTIFACTS_DIR_MAIN)
endif

endif	# HAS_CLEAN_SUPPORT=1



# ------------------------------------------------------------------------------------------
# Help target
# NOTE1: descriptions must be all aligned at column 54 as standard (including the initial TAB)
# NOTE2: only most useful targets are documented; internal/obscure targets must NOT be listed
# ------------------------------------------------------------------------------------------

ifeq ($(SUPPORTS_HELP),1)
help::
ifeq ($(SUPPORTS_CPPCHECK), 1)
	@echo    
	@echo "C/C++ static analysis targets (to run INSIDE the builder docker):"    
	@echo "    cppcheck:                             run cppcheck static code analysis"
	@echo "    cppcheck_clean:                       clean cppcheck static code analysis intermediate files"
endif
ifeq ($(SUPPORTS_FORMAT_CHECK), 1)
	@echo    
	@echo "C/C++ formatting targets (to run INSIDE the builder docker):"    
	@echo "    format_check:                         recursively check the style of C/C++ code with clang-format"
	@echo "    format_inplace:                       recursively re-format the style of C/C++ code with clang-format"
endif
endif # SUPPORTS_HELP
