# ------------------------------------------------------------------------------------------
# GIT targets
# ------------------------------------------------------------------------------------------

# 14days = 1209600sec
# 28days = 2419200sec
OLD_THRESHOLD_SEC=2419200

ifeq ($(FORCE),)
FORCE=0
endif

ifndef SUPPORTS_HELP
# default value
SUPPORTS_HELP=1
endif

git_clean:
	git clean -fdx -e .project -e .cproject -e .settings

git_show_all_branches:
	for branch in `git branch --remotes | grep -v HEAD`; do \
		echo -e `git show --format="%cr,%an," $$branch | head -n 1` $$branch ; \
	done | sort -r

git_show_remote_old_branches:
	@echo "Following git branches are older than $(OLD_THRESHOLD_SEC)sec:"
	@OLD_THRESHOLD_SEC=$(OLD_THRESHOLD_SEC) ; \
	TS_NOW=$$(date +%s) ; \
	for branch in `git branch --remotes | grep -v HEAD`; do \
		if [[ "$$branch" = "origin/master" || "$$branch" = "origin/develop" || $$branch =~ origin/support/* ]]; then \
			echo "Skipping stable branch $$branch" ; \
		else \
			TS_BRANCH=`git show --format="%ct" $$branch | head -n 1`; \
			if (( (TS_NOW - TS_BRANCH) > OLD_THRESHOLD_SEC )); then \
				echo -e `git show --format="%cr,%an," $$branch | head -n 1` $$branch ; \
			fi ; \
		fi ; \
	done | sort -r

git_delete_remote_old_branches:
	git fetch --prune
	@echo
	@echo
	@$(MAKE) --quiet git_show_remote_old_branches
	@echo
ifeq ($(FORCE),0)
	@echo "Are you sure you want to delete all branches listed above? If no hit CTRL+C"
	@read -n1
else
	@echo "FORCE=1 means we are proceeding without confirmation"
endif
	@OLD_THRESHOLD_SEC=$(OLD_THRESHOLD_SEC) ; \
	TS_NOW=$$(date +%s) ; \
	for branch in `git branch --remotes | grep -v HEAD`; do \
		if [[ "$$branch" = "origin/master" || "$$branch" = "origin/develop" || $$branch =~ origin/support/* ]]; then \
			echo "Skipping stable branch $$branch" ; \
		else \
			TS_BRANCH=`git show --format="%ct" $$branch | head -n 1`; \
			if (( (TS_NOW - TS_BRANCH) > OLD_THRESHOLD_SEC )); then \
				BRANCH_NO_REMOTE_PREFIX="$${branch//origin\//}" ; \
				echo "Now deleting branch [$$BRANCH_NO_REMOTE_PREFIX] from remote" ; \
				git push origin --delete $$BRANCH_NO_REMOTE_PREFIX 2>&1 ; \
			fi ; \
		fi ; \
	done | sort -r

GIT_BRANCH_NAME_TO_REMOVE_REGEX := (feature|bugfix|hotfix)\/.*

git_delete_local_feature_branches:
	@for br in `git branch --quiet --list | egrep --only-matching '$(GIT_BRANCH_NAME_TO_REMOVE_REGEX)'`; do \
		echo "Found [$$br] to delete...."  ; \
	done
	@echo
ifeq ($(FORCE),0)
	@echo "Are you sure you want to delete all branches listed above? If no hit CTRL+C"
	@read -n1
else
	@echo "FORCE=1 means we are proceeding without confirmation"
endif
	@echo "Now deleting:"
	@for br in `git branch --quiet --list | egrep --only-matching '$(GIT_BRANCH_NAME_TO_REMOVE_REGEX)'`; do \
		git branch -D $$br ; \
	done
	
git_release:
ifeq ($(GITLAB_TOKEN),)
	@echo "Please provide a Gitlab token GITLAB_TOKEN=xxxxxx to run this target"
	exit 2
endif
ifeq ($(EMAILTO),)
	@echo "Please provide a valid email address EMAILTO=xxxxxx@infovista.com to run this target"
	exit 2
endif
ifeq ($(GIT_REPO_NAME),)
	@echo "Please provide a valid git repository name as GIT_REPO_NAME=capture|statemachine|correlator to run this target"
	exit 2
endif
	docker run --publish 8080:8080 -it repo-billerica.perform.empirix.com:5000/devops-tools/kramer:latest --gitlabToken $(GITLAB_TOKEN) --emailTo $(EMAILTO) --vaultPassword 123Empirix! --branch develop --project empirix-eng/klerity-ecc-eva/$(GIT_REPO_NAME)


git_show_contributors:
	# from https://stackoverflow.com/questions/4589731/git-blame-commit-statistics
	git ls-tree -r -z --name-only HEAD | sed 's/^/.\//' | xargs -0 -n1 git blame --line-porcelain HEAD | grep -ae "^author "|sort|uniq -c|sort -nr


# ------------------------------------------------------------------------------------------
# Help target
# NOTE1: descriptions must be all aligned at column 54 as standard (including the initial TAB)
# NOTE2: only most useful targets are documented; internal/obscure targets must NOT be listed
# ------------------------------------------------------------------------------------------

ifeq ($(SUPPORTS_HELP),1)
help::
	@echo
	@echo "Git helper targets (to run OUTSIDE the builder docker):"
	@echo "    git_clean:                            clean all files that are not tracked by git"           
	@echo "    git_show_all_branches:                show all branches in the remote, with owner and creation date"
	@echo "    git_show_remote_old_branches:         show only branches from the remote that are older than 28days"
	@echo "    git_delete_remote_old_branches:       delete branches from the remote that are older than 28days"
	@echo "    git_delete_local_feature_branches:    delete local feature branches"
	@echo "    git_release:                          run Kramer tool to create a release tag, provide GITLAB_TOKEN=xyzw"
endif # SUPPORTS_HELP
