#!/usr/bin/env python3

"""
This Python module allows to check the size of a Teamcity artifacts
folder and, if size is too large than a given threshold, make a copy
of artifacts and in old folder leave just a README file with a 
'too large' message.  
"""

import os, shutil, socket, argparse


def remove_file(path: str, filename: str) -> None:
    """Remove given file if exists"""
    f = os.path.join(path, filename)
    if os.path.exists(f):
        os.remove(f)


def get_size(path: str = ".") -> int:
    """Calcultate bytes size of a folder"""
    total_size = 0
    for dirpath, dirnames, filenames in os.walk(path):
        for f in filenames:
            fp = os.path.join(dirpath, f)
            total_size += os.path.getsize(fp)
    return total_size


def check_size(threshold: int, path: str, too_big_path: str) -> None:
    """Check if folder size is more than a threshold (in MB), eventually rename folder and"""
    mb_size = get_size(path) / 1024 ** 2
    if mb_size >= threshold:
        # rename TeamCity artifacts folder
        if os.path.exists(too_big_path) and os.path.isdir(too_big_path):
            shutil.rmtree(too_big_path)
        shutil.move(path, too_big_path)
        os.makedirs(path)

        # generate disk consumption report
        generate_disk_consumption_report(path,too_big_path)

        # add a readme with a message in old artifacts folder
        hostname = socket.gethostname()
        ip = socket.gethostbyname(hostname)
        wrn_msg = f"This build generated more than {threshold}MB of artifacts (about {round(mb_size, 2)}MB). Please log on the build agent ({ip}) to check which artifacts is that large in {too_big_path} folder."
        with open(os.path.join(path, "README.txt"), "w+") as readme:
            readme.write(wrn_msg)
        print(wrn_msg)
    else:
        print(f"{path} folder is smaller than {threshold}MB. Proceed")

def convert_size_human_readable_format(size_bytes:int):
    if size_bytes < 1024:
        return f"{size_bytes} bytes"
    elif size_bytes < 1024**2:
        return f"{size_bytes / 1024:.2f} KB"
    elif size_bytes < 1024**3:
        return f"{size_bytes / 1024**2:.2f} MB"
    else:
        return f"{size_bytes / 1024**3:.2f} GB"

def generate_disk_consumption_report(path: str, too_big_path: str):
    report = {}
    for dirpath, dirnames, filenames in os.walk(too_big_path):
        report[dirpath] = {'size': get_size(dirpath), 'files': {}}
        for filename in filenames:
            file_path = os.path.join(dirpath, filename)
            report[dirpath]['files'][filename] = os.path.getsize(file_path)

    sorted_report = sorted(report.items(), key=lambda x: x[1]['size'], reverse=True)
    report_file_path = os.path.join(path, "Artifacts_Disk_Consumption.txt")
    with open(report_file_path, "w+") as f:
        for folder, info in sorted_report:
            folder_size = convert_size_human_readable_format(info['size'])
            f.write(f"{folder}: {folder_size}\n")
            for filename, size in sorted(info['files'].items(), key=lambda x: x[1], reverse=True):
                file_size = convert_size_human_readable_format(size)
                f.write(f"  {filename}: {file_size}\n")

if __name__ == "__main__":
    # parse the arguments
    parser = argparse.ArgumentParser(
        description="Check size of artifacts folder and truncate it if too large",
    )
    parser.add_argument(
        "--threshold",
        help="Set threshold (in MB) beyond which folder is too large",
        type=int,
        default=500,
    )
    parser.add_argument(
        "--artifacts",
        help="TeamCity Artifacts folder",
        type=str,
        default="TeamcityArtifacts",
    )
    parser.add_argument(
        "--too_big_path",
        help="Set destination folder if input folder is too big",
        type=str,
        default="TeamcityArtifactsTooBig",
    )
    parser.add_argument(
        "--output_file",
        help="Where to save 'too large' message",
        type=str,
        default="README.txt",
    )
    args = parser.parse_args()

    # just in case, remove the README.txt if already present
    remove_file(args.artifacts, args.output_file)

    # now check the size of all artifacts and eventually avoid to upload a huge size to Teamcity server
    check_size(args.threshold, args.artifacts, args.too_big_path)
