#!/bin/bash

#
# Smart conan deploy on remote repositories
# Created: Aug 2020
# Author: giuliano
#
# Supports the following features:
# - Registry login/authentication
# - Logging of pushed conan packages
#
# Usage:
#   ./conan-deploy-wrapper.sh [repo_user] [repo_password] [repo_url] [deployfile] [conan_pkg1] [conan_pkg2] ...
#
# To ease aggregation of multiple runs of the script, the logfile will NOT be truncated, just used in APPEND mode.
#
# Exit code:
#  0: success
#  100: login failure
#  200: tag failure
#  300: push failure
#

# parse arguments

set -u

USER="$1"; shift
PWD="$1"; shift
ARTIFACTORY_REMOTE_NAME="$1"; shift
REMOTE_REPO_URL="$1"; shift
DEPLOY_FILELIST_LOG="$1"; shift
CONAN_PKG_LIST="$*"
LOCAL_TAG="0.0-local"

# functions

function check_conan_remotes()
{
    local REPO=$(conan remote list | grep $ARTIFACTORY_REMOTE_NAME)
    local USER_LOGGED=$(conan user -r $ARTIFACTORY_REMOTE_NAME | grep $USER)

    if [[ "$REPO" == *"$ARTIFACTORY_REMOTE_NAME"* ]]; then
        echo "Conan repository is already present in the list of remotes"
    else
        conan remote add $ARTIFACTORY_REMOTE_NAME $REMOTE_REPO_URL False
    fi

    if [[ "$USER_LOGGED" == *"Authenticated"* ]]; then
        echo "Conan user it's already logged"
    else
        conan user $USER -p $PWD -r $ARTIFACTORY_REMOTE_NAME
        if [ $? -ne 0 ]; then echo "Failed authentication against $ARTIFACTORY_REMOTE_NAME / $REMOTE_REPO_URL" ; exit 100 ; fi

        echo "** Login succeeded to remote repository $ARTIFACTORY_REMOTE_NAME"
    fi
}

function error_local_tag()
{
    PKG=$1
    echo "-----------------------------------------------------------------------------------------------------"
    echo "-----------------------------------------------------------------------------------------------------"
    echo "ERROR *** Package $PKG can't be pushed to the repo because version $LOCAL_TAG is just for local usage ***"
    echo "-----------------------------------------------------------------------------------------------------"
    echo "-----------------------------------------------------------------------------------------------------"
    exit 200
}

# main

NCORRECTLY_DEPLOYED=0
NSKIPPED=0

# ensure folder for log file exists
mkdir -p $(dirname $DEPLOY_FILELIST_LOG)

check_conan_remotes

for PKG in $CONAN_PKG_LIST; do

    if [[ "$PKG" == *"$LOCAL_TAG"@* ]]; then
        error_local_tag $PKG
    fi

    echo "Checking if the Conan package [$PKG] already exists in the remote..."
    conan search --remote=$ARTIFACTORY_REMOTE_NAME $PKG >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        echo "WARNING: Apparently the Conan package [$PKG] was already deployed: check [$ARTIFACTORY_REMOTE_NAME/$REMOTE_REPO_URL]. Redeploying is forbidden. Skipping this deploy."
        DEPLOY_RESULT="skipped"
        (( NSKIPPED++ ))
    else

        # NOTE: we use the --all flag in case the Conan package includes multiple binary "builds" inside; typical case is when
        #       "conan create" is invoked multiple times (prior to the upload) with different --options or --settings flags
        echo "Proceeding with upload on repo: $ARTIFACTORY_REMOTE_NAME [$REMOTE_REPO_URL] of all package IDs for the Conan package with reference [$PKG]"
        TOPDIR=$TOPDIR conan upload  --no-overwrite --confirm --remote=$ARTIFACTORY_REMOTE_NAME --all $PKG
        if [ $? -ne 0 ]; then echo "Failed upload to $ARTIFACTORY_REMOTE_NAME / $REMOTE_REPO_URL" ; exit 200 ; fi

        DEPLOY_RESULT="deployed"
        (( NCORRECTLY_DEPLOYED++ ))
    fi

    echo "conan,${REMOTE_REPO_URL},${PKG},${DEPLOY_RESULT}" | tee -a $DEPLOY_FILELIST_LOG
done


