#!/bin/bash

# Author: F. Montorsi
# Created: Sep 2015
# Last modification: Dec 2020
#
# HISTORY
# v2: changed sources_on_probe="/mnt/sources" to sources_on_probe="/home/fmontorsi/sources"
# v3: changed default output path to $HOME, for non-root usage
# v4: added to Perforce of CSA-NFV-1 under Internal-Utilities/developer_gdb_utils/
# v5: change first parameter to take directly the absolute path for source files
# v6: updated to account for csa-nfv / third-party split
# v7: updated to account for csa-nfv splitting into new repos and docker-compose


# default config

sources="/source"					# this is the default path where mount_sources.sh puts sources
outputfile="/root/.gdbinit"
gdb_history="/tmp/gdb_shellhistory.log"


# go

function show_help()
{
    echo "$0 Usage:"
    echo "   $0 [source code folder]"
    echo "Purpose: generates a .gdbinit file to allow GDB to find Empirix EVA/ECC sources when debugging"
    echo "Default values:"
    echo " source code folder = absolute path of the directory containing the source code for GDB; default value is [$sources]"
    echo "The generated .gdbinit will be installed on the target IP in $outputfile"
}

current_dir=$(readlink -f .)

if [[ "$1" = "--help" || "$1" = "h" ]]; then
    show_help
    exit 0
fi
if [[ ! -z "$1" ]]; then
    sources="$(realpath $1)"
fi


preamble_text="""# auto-generated gdbinit script, by generate_gdbinit.sh on $(date)
set print pretty on
set print object on
set print static-members on
set print vtbl on
set print demangle on
set auto-load safe-path /
set history save

# avoid notifications from thread creation/deletion:
set print thread-events off

# SIGUSR1 is used in Empirix sw to wake up sleeping threads... don't block on that
handle SIGUSR1 nostop

# show a bit more context when issuing list command from CLI:
set listsize 30
"""

if [[ -d "$sources" ]]; then
	gdb_history="$HOME/gdb_shellhistory.log"
else
	echo "Invalid source code path $sources. Aborting."
	exit 1
fi



# start output file generation

echo "$preamble_text" >$outputfile
echo "set history filename $gdb_history">>$outputfile

echo
echo "# source folders">>$outputfile

dirs="$(find ${sources} -type d | grep -v Build | grep -v Lib | grep -v Bin | grep -v .git | grep -v TeamcityArtifacts | grep -v Testing | grep -v DockerArtifacts | grep -v HelmChart)"
for d in $dirs; do
    echo "Adding folder $d"
    echo "dir $d">>$outputfile
done

echo "Generated successfully $outputfile using folders found at $sources"
