#!/bin/bash

set -u

function pause()
{
    echo "Press any key to proceed or CTRL+C to stop if you feel like something is wrong"
    read -n1 -s
}

function ask_yes_no()
{
    local QUESTION="$1"

    read -p "$QUESTION " -n 1 -r
    echo    # (optional) move to a new line
    if [[ $REPLY =~ ^[Yy]$ ]]; then
        true
        return
    else
        false
        return
    fi
}

function print_section()
{
    echo "======================== $1 ======================="
}

function get_current_git_branch()
{
    echo "$(git rev-parse --abbrev-ref HEAD)"
}


function abort_if_branch_is_unclean()
{
    local STATUS="$(git status --short)"
    if [ ! -z "$STATUS" ]; then
        echo "Git status reports that the local branch [$(get_current_git_branch)] is not clean... please stash/commit/discard any work before launching this script:"
        git status
        exit 2
    fi

    echo "Apparently your local branch [$(get_current_git_branch)] is clean... proceeding"
}

function abort_if_branch_is_ahead()
{
    local AHEAD_STATUS="$(git status --short --branch | grep ahead)"
    if [ ! -z "$AHEAD_STATUS" ]; then
        echo "Git status reports that the local branch [$(get_current_git_branch)] has commits never pushed to remote... please commit/discard any work before launching this script:"
        git status
        exit 2
    fi

    echo "Apparently your local branch [$(get_current_git_branch)] is in sync with remote... proceeding"
}

function parse_args()
{
    NEW_GIT_TAG="${1:-}"
    if [ -z "$NEW_GIT_TAG" ]; then
        echo "Usage: $0 NEW_GIT_TAG"
        echo "Example: $0 1.9.0"
        echo
        echo "IMPORTANT: this script will perform double-tagging; in other words the '-develop' and '-master' postfix will"
        echo "           be appended to the the NEW_GIT_TAG to generate the actual GIT tag names"  
        exit 2
    fi

    if [[ $NEW_GIT_TAG =~ .*develop.* ]]; then
        echo "Wrong TAG name: the branch name should not be included in the tag: it will be added automatically by this script"
        exit 3
    fi
    if [[ $NEW_GIT_TAG =~ .*master.* ]]; then
        echo "Wrong TAG name: the branch name should not be included in the tag: it will be added automatically by this script"
        exit 3
    fi

    local pat='([0-9]+).([0-9]+).([0-9]+)'
    if [[ $NEW_GIT_TAG =~ $pat ]]; then
        echo ""
    fi

    NEW_GIT_TAG_DEVELOP="${NEW_GIT_TAG}-develop"
    NEW_GIT_TAG_MASTER="${NEW_GIT_TAG}-master"
}

parse_args $*

print_section "Aligning LOCAL master and develop branches"
git checkout develop && git pull
abort_if_branch_is_unclean
abort_if_branch_is_ahead

git checkout master && git pull
abort_if_branch_is_ahead

git checkout develop && git pull
pause

print_section "Tagging LOCALLY the develop branch with tag [$NEW_GIT_TAG_DEVELOP]"
git tag -a $NEW_GIT_TAG_DEVELOP -m "Release $NEW_GIT_TAG"

print_section "Merging master into develop to avoid conflicts"
git merge -s ours -m "commit to discard" master                            # this is VERY important to make below step less painful
pause

print_section "Merging LOCALLY master into develop with squash"
git checkout master && git merge --squash develop   # should merge with ZERO conflicts always
pause

print_section "Please edit the checkin commit msg for the master branch"
grep 'IN-' .git/SQUASH_MSG >/tmp/commitmsg          # to find out all Jira items mentioned in the squashed merge message
nano /tmp/commitmsg                                 # to eventually edit the commit message / tag message

print_section "Committing LOCALLY the changes on the master branch using tag [$NEW_GIT_TAG_MASTER]"
git commit -a -F /tmp/commitmsg
git tag -F /tmp/commitmsg -a $NEW_GIT_TAG_MASTER

print_section "Please do the final checks on the LOCAL MASTER branch before pushing to the REMOTE"
git log -n3                  # to check everything is ok
pause

print_section "Pushing master branch to the REMOTE"

ask_yes_no "Shall I proceed pushing to the remote? This operation cannot be undone easily. Press 'y' to proceed or CTRL+C to exit"
git push                 # to push the single checkin of alignment
git push origin --tags   # to push the new tag as well

print_section "Resetting develop branch LOCALLY"
git checkout develop && git reset --hard origin/develop   # discard all activities done on the develop branch

print_section "Release to MASTER branch completed"
