#!/bin/bash

#
# Smart Python deploy on remote repositories
# Created: Oct 2020
# Author: fmontorsi
#
# Supports the following features:
# - Registry login/authentication
# - Logging of pushed pipy packages
#
# Usage:
#   ./python-deploy-wrapper.sh [repo_user] [repo_password] [repo_url] [deployfilelog] [pydir1] [pydir2] ...
#
# To ease aggregation of multiple runs of the script, the logfile will NOT be truncated, just used in APPEND mode.
#
# Exit code:
#  0: success
#  100: login failure
#  200: tag failure
#  300: push failure
#

# parse arguments

set -u

USER="$1"; shift
PWD="$1"; shift
REMOTE_REPO_URL="$1"; shift
DEPLOY_FILELIST_LOG="$1"; shift
PYTHON_DIRS="$*"

REMOTE_REPO_URL_FOR_CHECK="$REMOTE_REPO_URL"


# functions



# main

NCORRECTLY_DEPLOYED=0
NSKIPPED=0

# ensure folder for log file exists
mkdir -p $(dirname $DEPLOY_FILELIST_LOG)

# FIXME FIXME: we should avoid touching the pypirc this way but rather put the configuration of the remote pypi repo in setup-devel
# just like for all other repos
cp -f $HOME/.pypirc $HOME/.pypirc.backup || true
cat <<EOF > $HOME/.pypirc
[distutils]
index-servers = remote_pypi_repo

[remote_pypi_repo]
repository: $REMOTE_REPO_URL
username: $USER
password: $PWD

[global]
index = https://repo-billerica.perform.empirix.com/repository/pypi/pypi-local
index-url = https://repo-billerica.perform.empirix.com/repository/pypi-local/simple

EOF


CURL_CMD="curl -v --fail --user $USER:$PWD"
CURL_CMD_PRESENCE_CHECK="$CURL_CMD --output /dev/null --silent"

NCORRECTLY_DEPLOYED=0
NSKIPPED=0
for PKG in $PYTHON_DIRS; do

    echo "Proceeding with upload on repo [$REMOTE_REPO_URL] of the pypi package [$PKG]"
    pushd $PKG
    rm -rf dist *.egg-info

    # this upload procedure works in Nexus but not in Artifactory:
    ##python3 setup.py sdist
    ##twine upload --verbose --repository-url $REMOTE_REPO_URL --username $USER --password $PWD dist/*

    # first of all produce the "source distribution"
    python3 setup.py sdist --quiet

    # acquire a few info required to check if the package already exists in the remote
    PKG_TARBALL="$(ls dist/)"
    if [ -z "$PKG_TARBALL" ]; then   echo "ERROR in getting Pypi package tarball. Aborting." ; exit 100; fi

    PKG_NAME_FOR_CHECK="$(python3 setup.py --name)"

    PKG_NAME="$(echo *.egg-info | sed 's@.egg-info@@')"
    if [ ! -d "${PKG_NAME}.egg-info" ]; then   echo "ERROR in getting Pypi package name. Aborting." ; exit 100; fi

    PKG_VERSION="$(cat ${PKG_NAME}.egg-info/PKG-INFO | grep '^Version' | cut -f2 -d' ')"
    if [ -z "$PKG_VERSION" ]; then   echo "ERROR in getting Pypi package version. Aborting." ; exit 100; fi

    echo "========================================"
    echo "Package information"
    echo "PKG_TARBALL=$PKG_TARBALL"
    echo "PKG_NAME=$PKG_NAME"
    echo "PKG_NAME_FOR_CHECK=$PKG_NAME_FOR_CHECK"
    echo "PKG_VERSION=$PKG_VERSION"
    echo "========================================"

    # anti re-deploy check:
    echo "Checking if the $REMOTE_REPO_URL_FOR_CHECK/$PKG_NAME_FOR_CHECK/$PKG_VERSION/$PKG_TARBALL is already on the repo"
    if $CURL_CMD_PRESENCE_CHECK $REMOTE_REPO_URL_FOR_CHECK/$PKG_NAME_FOR_CHECK/$PKG_VERSION/$PKG_TARBALL; then
        echo "WARNING: Apparently the PyPi package [$PKG_TARBALL] was already deployed: check [$REMOTE_REPO_URL/$PKG_VERSION/$PKG_TARBALL]. Redeploying is forbidden. Skipping this deploy."
        DEPLOY_RESULT="skipped"
        (( NSKIPPED++ ))
    else

        # for Artifactory or Nexus we use directly setup.py
        python3 setup.py sdist upload -r remote_pypi_repo

        upload_result=$?
        rm -rf dist *.egg-info setup.py
        popd

        if [ $upload_result -ne 0 ]; then echo "Failed upload to $REMOTE_REPO_URL" ; exit 200 ; fi

        DEPLOY_RESULT="deployed"
        (( NCORRECTLY_DEPLOYED++ ))
    fi

    echo "pypi,${REMOTE_REPO_URL},${PKG_NAME}==${PKG_VERSION},${DEPLOY_RESULT}" | tee -a $DEPLOY_FILELIST_LOG
done

echo "Count of Pypi packages deployed: $NCORRECTLY_DEPLOYED. Skipped Pypi packages: $NSKIPPED"

if [ -f $HOME/.pypirc.backup ]; then
    mv $HOME/.pypirc.backup $HOME/.pypirc
fi

