#!/bin/bash

#
# Smart RPM deploy on remote repositories
# Created: Feb 2020
# Author: fmontorsi
#
# Supports the following features:
# - HTTP authentication
# - REDEPLOY check & block (to avoid checksum mismatches!)
# - CHECKSUM computation during upload
#
# Usage:
#   ./rpm-deploy-wrapper.sh [rpm_folder] [repo_user] [repo_password] [repo_url] [output_logfile]
# E.g.:
#   ./rpm-deploy-wrapper.sh myprj/RpmArtifacts "$(NEXUS_USER)" "$(NEXUS_PW)" "https://repo-billerica.perform.empirix.com/repository/ecc-eva-billerica-yum/" outputLog.txt
#
# To ease aggregation of multiple runs of the script, the logfile will NOT be truncated, just used in APPEND mode.
#
# Exit code:
#  0: success
#  100: deploy failure
#  150: version failure
#  200: redeploy check failure


set -u

DIR="$1"
USER="$2"
PW="$3"
REMOTE_REPO_URL_PUSH="$4"
DEPLOY_FILELIST_LOG="$5"
LOCAL_TAG="0.0_local"

CURL_CMD="curl -v --fail --user $USER:$PW"
CURL_CMD_PRESENCE_CHECK="$CURL_CMD --output /dev/null --silent --head"

function error_local_tag()
{
  RPM=$1
  echo "-----------------------------------------------------------------------------------------------------"
  echo "-----------------------------------------------------------------------------------------------------"
  echo "ERROR *** rpm $RPM can't be pushed to the repo because version $LOCAL_TAG is just for local usage ***"
  echo "-----------------------------------------------------------------------------------------------------"
  echo "-----------------------------------------------------------------------------------------------------"
  exit 150
}

# ensure folder for log file exists
mkdir -p $(dirname $DEPLOY_FILELIST_LOG)

NCORRECTLY_DEPLOYED=0
NSKIPPED=0
if ls $DIR/x86_64/*.rpm >/dev/null 2>&1; then
    for rpmfile in $DIR/x86_64/*.rpm; do
        RPM_FILENAME_ONLY="$(basename $rpmfile)"
        if [[ "$RPM_FILENAME_ONLY" == *"$LOCAL_TAG"* ]]; then
          error_local_tag $RPM_FILENAME_ONLY
        fi

        if $CURL_CMD_PRESENCE_CHECK $REMOTE_REPO_URL_PUSH/$RPM_FILENAME_ONLY; then
            echo "WARNING: Apparently the RPM file [$RPM_FILENAME_ONLY] was already deployed: check [$REMOTE_REPO_URL_PUSH/$RPM_FILENAME_ONLY]. Redeploying is forbidden. Skipping this deploy."
            DEPLOY_RESULT="skipped"
            (( NSKIPPED++ ))
        else
            echo "Looks like the RPM file [$RPM_FILENAME_ONLY] has never been deployed before. Proceeding."
            SHA1SUM="$(sha1sum $rpmfile | awk '{print $1}')"
            SHA256SUM="$(sha256sum $rpmfile | awk '{print $1}')"
            MD5SUM="$(md5sum $rpmfile | awk '{print $1}')"

            # exit immediately in case of network failure:
            $CURL_CMD -XPUT $REMOTE_REPO_URL_PUSH \
                -H "X-Checksum-Sha1:$SHA1SUM" \
                -H "X-Checksum-Sha256:$SHA256SUM" \
                -H "X-Checksum-MD5:$MD5SUM" \
                -i -T $rpmfile || exit 100

            DEPLOY_RESULT="deployed"
            (( NCORRECTLY_DEPLOYED++ ))
        fi

        echo "rpm,${REMOTE_REPO_URL_PUSH},${RPM_FILENAME_ONLY},${DEPLOY_RESULT}" | tee -a $DEPLOY_FILELIST_LOG
    done
else
    echo "No RPM artifacts in $DIR directory to deploy."
fi

echo "Count of RPMs deployed: $NCORRECTLY_DEPLOYED. Skipped RPMs: $NSKIPPED"
