#!/usr/bin/env python3

import os, sys
import argparse
import datetime

from hashlib import md5
import xlsxwriter
from systemhealth_xml_loader import SystemhealthXMLLoader
from typing import Tuple


class SystemhealthCppCodeGenerator:
    def __init__(self, verbose: bool, code_formatter_binary: str):
        """constructor"""
        self.verbose = verbose
        self.kpi_counter = 0

        self.code_formatter = code_formatter_binary

    def __get_parent_of_master_format_directory(self, system_health_xmls: list):
        master_format = "/master-format/"
        for xml in system_health_xmls:
            if master_format in xml:
                return xml.split(master_format)[0]
        sys.exit("None of the xml from ")

    def __format_generated_header_file(
        self, code_formater: str, source_file: str
    ) -> None:
        """
        Format generated header files according to some code formatting rules.

        Here, we use cpp code formatter binary same one used in `make format_inplace` to keep
        the formatting in sync with update in cpp formatting tool in product

        Args:
            code_formater: This is the code formatting binary used in 'make format_inplace'
            source_file: Source file to format

        Returns:
            None

        Raises:
            If the inplace formatting fails then exits the process with same error code
        """
        if self.verbose:
            print(f"Formatting {source_file} inplace using {code_formater}")
        exit_code = os.system(f"{code_formater} --style=file -i {source_file}")
        if exit_code:
            print(
                f"Failed inplace formatting of {source_file}: exit code {exit_code} returned by {code_formater}"
            )
            sys.exit(exit_code)
        else:
            if self.verbose:
                print(f"Successfully formatted {source_file} inplace")

    def __get_header_includes(self) -> str:
        return """\
            //------------------------------------------------------------------------------------------
            // This header file was automatically generated by the SystemHealthKpiGenerator utility
            // from pipeline-framework repository.
            //
            // !! DO NOT EDIT THIS FILE MANUALLY !!
            //
            //------------------------------------------------------------------------------------------\n\n
            #pragma once\n
            #include <string>
            #include <SystemHealthDefs.h> // this is defined in base-libs\n"""

    def __get_header_defines(self, process_name, kpi_count) -> str:
        process_name = process_name.upper()
        return f"""
            // automatically-generated constants
            #define {process_name}_KPI_MAX {kpi_count}                 /* deprecated constant name; please use the one below */
            #define SYSHEALTH_NUM_KPIS_{process_name} {kpi_count}\n
        """

    def __get_systemhealth_process_kpi_enum(
        self,
        kpi_counter: int,
        kpi_list: list,
        processname: str,
        first_xml_being_parsed: bool,
    ) -> Tuple[str, int]:
        systemhealth_process_kpi_enum = f"""
        // KPI IDs allocated by SystemHealthKpiGenerator and that can be used in the SystemHealthKPICollectorBase::SetKpiValue() API
        typedef enum
        {{
        """
        fieldname_extension = "KPI"
        enum_template = """\t{name}_{extension}{value},\n"""
        for i in range(len(kpi_list)):
            field = kpi_list[i]
            if field["type"] == "string":
                continue
            enum_entry = {
                "name": field["name"],
                "extension": fieldname_extension,
                "value": "",
            }
            if i == 0:
                enum_entry["value"] = f" = {kpi_counter}"
            systemhealth_process_kpi_enum += enum_template.format(**enum_entry)
            kpi_counter = kpi_counter + 1

        systemhealth_process_kpi_enum += f"}} systemhealth_{processname}_kpis_t;\n\n"

        return systemhealth_process_kpi_enum, kpi_counter

    def __get_static_process_kpi_descriptor_array(
        self, processname: str, kpi_list: list
    ) -> str:
        static_process_kpi_descriptor = (
            f"""
            // array of KPI descriptors to be provided to the SystemHealthKPICollectorBase::InitSystemHealthKpi() API
            static const kpi_descriptor_t g_{processname}_kpi_descriptors[] = {{
            """
        )
        array_template = """{{ {name}_{extension}, "{name_lower}", {kpi_type}, "{description}", {service_impact}, {requires_labels}, {thread_aggregation_function} }},\n"""
        fieldname_extension = "KPI"
        for field in kpi_list:
            if field["type"] == "string":
                continue
            array_entry = {
                "name": field["name"],
                "extension": fieldname_extension,
                "name_lower": field["name"].lower(),
                "kpi_type": field["kpi_type"].upper(),
                "description": field["description"],
                "service_impact": field["service_impact"],
                "requires_labels": field["requires_labels"],
                "thread_aggregation_function": field["thread_aggregation_function"],
            }
            static_process_kpi_descriptor += array_template.format(**array_entry)
        static_process_kpi_descriptor += "};"
        return static_process_kpi_descriptor

    def __replace_file(self, src_file: str, dst_file: str) -> None:
        # if ".h" does not exists then simply move ".h.tmp.h" to ".h".
        # if ".h" already exists, Compare md5 sum of ".h" and just ".h.tmp.h" and overwrite ".h" if the md5sum missmatch.
        digests = {src_file: None, dst_file: None}
        for filename in digests:
            if os.path.exists(filename):
                with open(filename, "rb") as f:
                    digests[filename] = md5(f.read()).hexdigest()

        if digests[src_file] == None:
            print("ERROR : header file was not generated from xml file")
            sys.exit(1)

        if digests[dst_file] == digests[src_file]:
            os.remove(src_file)
            if self.verbose:
                print(f"File {dst_file} is up to date")
        else:
            os.replace(src_file, dst_file)
            if self.verbose:
                print(f"Updated {dst_file}")

    def generate_cpp_header_file(
        self,
        output_dir: str,
        xml_filename: str,
        processname: str,
        kpi_list: list,
        first_xml_being_parsed: bool,
        last_xml_being_parsed: bool,
    ):
        """Generates the header (.h) file from the master-format *.xml file defining System Health KPI's.

        Args:
            xml_filename: XML file defining System Health KPI's
            kpi_list: should be a 3D array: [XML files; fields; attributes]
            first_xml_being_parsed: is "true" if the xml_filename is the first xml.
            last_xml_being_parsed: is "true" if the xml_filename is the last xml to be processed.
        """

        _kpi_counter = self.kpi_counter

        (
            systemhealth_process_kpi_enum,
            _kpi_counter,
        ) = self.__get_systemhealth_process_kpi_enum(
            _kpi_counter, kpi_list, processname, first_xml_being_parsed
        )
        header_source_code = self.__get_header_includes()

        header_source_code += self.__get_header_defines(processname, _kpi_counter)

        header_source_code += systemhealth_process_kpi_enum

        header_source_code += self.__get_static_process_kpi_descriptor_array(
            processname, kpi_list
        )

        # Write into temporary header file
        file_without_extension = f"{output_dir}/{xml_filename.split('.')[0]}"
        temp_header_file = f"{file_without_extension}.tmp.h"
        with open(temp_header_file, "w") as file:
            file.write(f"{header_source_code}\n")

        self.__format_generated_header_file(self.code_formatter, temp_header_file)

        final_header_file = f"{file_without_extension}.h"
        self.__replace_file(temp_header_file, final_header_file)

        self.kpi_counter = _kpi_counter
