#
# todo:
# split this unit test in unit tests for each submodule: XML loader, CPP code generator, DOC generator
#

import pathlib
import hashlib
import os

import pytest

from systemhealth_kpi_generator import SystemhealthKpiGenerator


class TestSystemHealthKpiGenerator:
    SCRIPT_LOCATION = pathlib.Path(__file__).parent.resolve()
    XML_LOCATION = f"{SCRIPT_LOCATION}/master-format/"
    EXPECTED_FILES_LOCATION = f"{SCRIPT_LOCATION}/expected_files/"

    class CommandlineArgs:
        def __init__(
            self,
            parent,
            system_health_xmls,
            export_type="header",
            system_health_prefix="unittest_",
            unit_test_mode=True,
        ):
            self.export_type = export_type
            self.system_health_xmls = system_health_xmls
            self.system_health_prefix = system_health_prefix
            self.output_dir = f"{parent.SCRIPT_LOCATION}"
            self.code_formatter = "/tmp/clang-format"
            self.unit_test_mode = unit_test_mode
            self.verbose = True

    def get_check_sum(self, file_name):
        md5_hash = None
        hasher = hashlib.md5()
        with open(file_name, "rb") as f:
            buf = f.read()
            hasher.update(buf)
            md5_hash = hasher.hexdigest()
        return md5_hash

    def verify_check_sums(self, actual_file, expected_file, remove_actual_file=True):
        actual_check_sum = self.get_check_sum(actual_file)
        assert actual_check_sum, f"'{actual_file}' should have a valid checksum"

        expected_check_sum = self.get_check_sum(expected_file)
        assert expected_check_sum, f"'{expected_file}' should have a valid checksum"

        assert (
            expected_check_sum == actual_check_sum
        ), f"checksum should match between {actual_file} and {expected_file}"

        if remove_actual_file:
            os.remove(actual_file)

    def test_headers_not_replaced_if_check_sum_match(self):
        print("Running test headers are no replaced if checksum matches")
        test_file_name = "eva_ecc_kpis"
        args = self.CommandlineArgs(
            self, system_health_xmls=[f"{self.XML_LOCATION}/{test_file_name}.xml"]
        )
        sysheathkpi = SystemhealthKpiGenerator(args)
        sysheathkpi.generate()

        generated_file = f"{self.SCRIPT_LOCATION}/{test_file_name}.h"
        expected_file = f"{self.EXPECTED_FILES_LOCATION}/{test_file_name}.h"
        self.verify_check_sums(generated_file, expected_file, remove_actual_file=False)

        # record creation and modification timestamp of a generated file
        c_time_run1 = os.path.getctime(generated_file)
        m_time_run1 = os.path.getmtime(generated_file)

        # Attempt to re-generate same file without changing the xml
        args = self.CommandlineArgs(
            self, system_health_xmls=[f"{self.XML_LOCATION}/{test_file_name}.xml"]
        )
        sysheathkpi = SystemhealthKpiGenerator(args)
        sysheathkpi.generate()

        # Obtain creation and modification timestamp of a generated file
        c_time_run2 = os.path.getctime(generated_file)
        m_time_run2 = os.path.getmtime(generated_file)

        assert (
            c_time_run1 == c_time_run2
        ), "Generated file should not replace if check-sum matches (create timestamp changed)"

        assert (
            m_time_run1 == m_time_run2
        ), "Generated file should not replace if check-sum matches (modify timestamp changed)"

        self.verify_check_sums(generated_file, expected_file)

    def test_xml2cpp_generation(self):
        # Verify header generation for eva/ecc component (like Capture, SM ...)
        print("Running test xml to headers generation for non common kpis")

        args = self.CommandlineArgs(
            self,
            system_health_xmls=[f"{self.XML_LOCATION}/eva_ecc_kpis.xml"],
        )
        sysheathkpi = SystemhealthKpiGenerator(args)
        sysheathkpi.generate()

        self.verify_check_sums(
            f"{self.SCRIPT_LOCATION}/eva_ecc_kpis.h",
            f"{self.EXPECTED_FILES_LOCATION}/eva_ecc_kpis.h",
        )

        # Since 'eva_ecc_kpix.xml' includes 'test_import_1_kpis.xml' and 'test_import_2_kpis.xml' separate header
        # 'test_import_1_kpis.h' and 'test_import_2_kpis.h' are generated, lets verify it as well
        self.verify_check_sums(
            f"{self.SCRIPT_LOCATION}/test_import_1_kpis.h",
            f"{self.EXPECTED_FILES_LOCATION}/test_import_1_kpis.h",
        )

        self.verify_check_sums(
            f"{self.SCRIPT_LOCATION}/test_import_2_kpis.h",
            f"{self.EXPECTED_FILES_LOCATION}/test_import_2_kpis.h",
        )

        # Since 'test_import_1_kpis.xml' includes and 'test_recursive_import_kpis.xml' separate header
        # 'test_recursive_import_kpis.h' is generated, lets verify it as well
        self.verify_check_sums(
            f"{self.SCRIPT_LOCATION}/test_recursive_import_kpis.h",
            f"{self.EXPECTED_FILES_LOCATION}/test_recursive_import_kpis.h",
        )

    def test_xml2cpp_generation_with_no_imports(self):
        # Verify header generation for eva/ecc component (like Capture, SM ...)
        print(
            "Running test xml to headers generation for non common kpis and having no imports direct / recursive"
        )

        args = self.CommandlineArgs(
            self,
            system_health_xmls=[f"{self.XML_LOCATION}/eva_ecc_no_imports_kpis.xml"],
        )
        sysheathkpi = SystemhealthKpiGenerator(args)
        sysheathkpi.generate()

        self.verify_check_sums(
            f"{self.SCRIPT_LOCATION}/eva_ecc_no_imports_kpis.h",
            f"{self.EXPECTED_FILES_LOCATION}/eva_ecc_no_imports_kpis.h",
        )

    def test_xml2cpp_generation_with_few_requires_labels_field(self):
        # Verify header generation for eva/ecc component (like Capture, SM ...) where xml
        # has some fields with 'requires_labels' ture, some with false and some does not
        # define it (defaults to fasle)
        print("Running test xml to headers generation with few requires_lables set")

        args = self.CommandlineArgs(
            self,
            system_health_xmls=[
                f"{self.XML_LOCATION}/eva_ecc_with_requires_labels_kpis.xml"
            ],
        )
        sysheathkpi = SystemhealthKpiGenerator(args)
        sysheathkpi.generate()

        self.verify_check_sums(
            f"{self.SCRIPT_LOCATION}/eva_ecc_with_requires_labels_kpis.h",
            f"{self.EXPECTED_FILES_LOCATION}/eva_ecc_with_requires_labels_kpis.h",
        )

    def test_xml2cpp_generation_with_thread_aggregation_functions(self):
        # Verify header generation for eva/ecc component (like Capture, SM ...) where xml
        # has different fields using different thread aggregation functions through
        # 'thread_aggregation_function' attribute if not present then defaults to 'none'
        print(
            "Running test xml to headers generation with fields atrributes thread aggregation_function set"
        )
        TEST_FILE_NAME = "eva_ecc_with_thread_aggregation_function_kpis"
        args = self.CommandlineArgs(
            self,
            system_health_xmls=[f"{self.XML_LOCATION}/{TEST_FILE_NAME}.xml"],
        )
        sysheathkpi = SystemhealthKpiGenerator(args)
        sysheathkpi.generate()

        self.verify_check_sums(
            f"{self.SCRIPT_LOCATION}/{TEST_FILE_NAME}.h",
            f"{self.EXPECTED_FILES_LOCATION}/{TEST_FILE_NAME}.h",
        )

    #
    # DOCUMENTATION TESTS
    #

    def test_xml2documentation_generation(self):
        # Verify document generation for eva/ecc component (like Capture, SM ...)
        print("Running test xml to documentation generation for non common kpis")
        args = self.CommandlineArgs(
            self,
            system_health_xmls=[f"{self.XML_LOCATION}/eva_ecc_kpis.xml"],
            export_type="documentation",
            system_health_prefix="test2_",
            unit_test_mode=True,
        )
        sysheathkpi = SystemhealthKpiGenerator(args)
        sysheathkpi.generate()

        self.verify_check_sums(
            f"{self.SCRIPT_LOCATION}/test2_prometheus_metrics_docs.xlsx",
            f"{self.EXPECTED_FILES_LOCATION}/test2_prometheus_metrics_docs.xlsx",
        )
