#
# Empirix Conan file to package C++ artifacts generated by this repository
# as a Conan package
#
# A few important points on how Conan is used here:
# 1) Conan is used only for the PACKAGING and does NOT drive the C++ build.
#    GNU make drives the C++ compilation, linking and installation steps.
# 2) Conan versioning is obtained from GNU make, through the use of the
#    "make get_conan_version" command... in this way the version declared by
#    GNU make will always be perfectly aligned to the one declared inside the
#    Conan package.
# 3) The installation of the artifacts into the Conan package is done through
#    "make install" command, instead of hardcoding here a long list of "copy"
#    commands.
from conans import ConanFile
from conans import AutoToolsBuildEnvironment
import os
import yaml


def get_conan_requirements():
    with open(r"conandata.yml") as file:
        conandata = yaml.load(file, Loader=yaml.FullLoader)
        return conandata["conan_requirements"]


class ThirdPartyCppLibsConan(ConanFile):
    # metadata specific to this repo:
    name = "third-party-cpp-libs"
    description = "third party C++ libraries for EVA-ECC components"
    # metadata common to all Empirix C++ repos:
    generators = "make"
    author = "Empirix EVA/ECC team"
    settings = "os", "compiler", "build_type", "arch"
    license = "Empirix Commercial License"
    topics = None
    # IMPORTANT: since we don't drive the build via Conan we need to export together with the C++ sources also the static/shared libs
    #            produced by the GNU make-driven pipeline: the "sources" exported using "exports_sources" need to be self-sufficient
    #            to support all "make install" commands... so we need to export as "sources" also all build artifact results like
    #            "x86_64-release" and "docker-release" folders.
    #            We can just exclude stuff, like RPMs eventually present, that clearly is the intermediate result of the build pipeline
    exports_sources =  (
        "*",
        "!*.rpm",
    )
    build_policy = "missing"  # see comment in conan-base repo: for now we cannot use 'never'
    # The python_requires feature is a way to share files and code between different recipes
    python_requires = get_conan_requirements()
    # python_requires_extend is used to inform Conan about additional Python base classes.
    # The resulting inheritance is equivalent to declare our Pkg class as class Pkg(pyreq.MyBase, ConanFile)
    # To see more details go to https://docs.conan.io/en/latest/extending/python_requires.html
    python_requires_extend = "conan-base.EmpirixConanBase"

    def package(self):
        """
        This function is called by Conan in 2 cases:
        a) when building the package with the 'conan create' command; at that
        b) when installing a header-only package just-downloaded from a Conan repository, i.e. when using "conan install --build"

        In both cases this function executes inside the so-called "package folder" which has no git information (.git is missing)
        and thus we cannot use tools.Git() here. The package folder will contain all the sources of the original git repository
        matched by the "exports_sources" variable (by default everything except hidden folders like the .git one).

        Having no .git folder means having a smaller Conan package but also means that we cannot derive from git the value of 2
        env vars required by our makefiles: GIT_BRANCH_NAME and GIT_DESCRIBE.
        However typically Empirix makefiles will not use any version variable defined by Makefile.version
        so that we can actually provide some dummies for these git-based informations just to make GNU make happy and get the "install" done.

        In order to avoid hardcoding lots of "self.copy" commands, as done in several conanfile examples, we rely on the availability
        of a "make install" target.
        """

        self.output.info("Executing package() method")
        env_vars_dict = self.empirix_read_env_vars()
        gnu_make_wrapper = AutoToolsBuildEnvironment(self)

        # "docker" is now the only possible ARCH used by eva-ecc makefiles
        # Note that this is a completely different variable from conan settings.arch
        empirix_arch = "docker"

        env_vars_dict["CONAN_CPU_COUNT"] = "1"  # do not run GNU make commands in parallel mode (-j)
        env_vars_dict["DESTDIR"] = os.path.join(self.package_folder, "include")
        env_vars_dict["ARCH"] = empirix_arch
        gnu_make_wrapper.make(target="install_headers", vars=env_vars_dict)

        env_vars_dict["DESTDIR"] = os.path.join(self.package_folder, "makefiles")
        gnu_make_wrapper.make(target="install_support_makefiles", vars=env_vars_dict)

        env_vars_dict["DESTDIR"] = os.path.join(self.package_folder, "etc")
        gnu_make_wrapper.make(target="install_ixengine_license", vars=env_vars_dict)

        env_vars_dict["DESTDIR"] = os.path.join(self.package_folder, "etc")
        gnu_make_wrapper.make(target="install_highsee_config_files", vars=env_vars_dict)
        cfg = self.empirix_get_cfg()

        runtime = self.empirix_get_runtime()
        buildid = self.empirix_get_buildid(empirix_arch, runtime, cfg)

        self.output.info("Packaging STATIC/SHARED LIBRARIES with CFG=%s, ARCH=%s, RUNTIME=%s" % (cfg, empirix_arch, runtime))
        env_vars_dict["DESTDIR"] = os.path.join(self.package_folder, "lib/" + buildid)
        env_vars_dict["CFG"] = cfg
        gnu_make_wrapper.make(target="install_libs", vars=env_vars_dict)
