/*
 * This file is a part of the Qosmos ixEngine.
 *
 * Copyright ENEA Qosmos Division 2000-2021 - All rights reserved
 *
 * This computer program and all its components are protected by
 * authors' rights and copyright law and by international treaties.
 * Any representation, reproduction, distribution or modification
 * of this program or any portion of it is forbidden without
 * Qosmos explicit and written agreement and may result in severe
 * civil and criminal penalties, and will be prosecuted
 * to the maximum extent possible under the law.
*/

#ifndef __QMDPI_STRUCT_H__
#define __QMDPI_STRUCT_H__

#include <stdint.h>


/***************************************
*
* Opaque Objects
*
****************************************/

/** Engine instance */
struct qmdpi_engine;

/** Worker instance */
struct qmdpi_worker;

/** Potocol Bundle instance */
struct qmdpi_bundle;

/** DPI processing result */
struct qmdpi_result;

/** DPI flow context */
struct qmdpi_flow;

/** Signature info */
struct qmdpi_signature;

/** Signature tune info */
struct qmdpi_tune;

/** Attribute info */
struct qmdpi_attr;

/** Extra library instance */
struct qmdpi_extralib;

/***************************************
*
* Non-opaque Objects
*
****************************************/

/** Classification path */
struct qmdpi_path {
    uint32_t qp_len;
    uint32_t qp_value[16];
};

/** **DO NOT ACCESS DIRECTLY**; PLEASE USE ACCESSORS IN qmdpi.h */
struct qmdpi_result_flags {
    union {
        struct {
uint32_t qrf_classified_state_changed   :
            1; /**< classified state has changed */
uint32_t qrf_offloaded_state_changed    :
            1; /**< offloaded state has changed  */
uint32_t qrf_pdu_dir_changed            :
            1; /**< packet directions are now inverted (CTS <-> STC) */
            uint32_t qrf_classified_state           : 1; /**< classified state */
            uint32_t qrf_offloaded_state            : 1; /**< offloaded state  */
uint32_t qrf_pdu_dir_index              :
            1; /**< packet direction index QMDPI_DIR0, QMDPI_DIR1 */
uint32_t qrf_pdu_dir                    :
            2; /**< packet direction: QMDPI_DIR_CTS or QMDPI_DIR_STC */
            uint32_t qrf_cacheable_state            : 1; /**< currently not used */
uint32_t qrf_path_changed               :
            1; /**< path has changed on the current direction */
uint32_t qrf_flow_expired               :
            1; /**< flow is going to be destroyed  */
uint32_t qrf_flow_created               :
            1; /**< flow has just been created     */
uint32_t qrf_tcp_reass_queue_full       :
            1; /**< TCP reassembly queue overflowed */
uint32_t qrf_attr_buf_full              :
            1;  /**< set to 1 if attribute buffer is full */
uint32_t qrf_classif_ext                :
            1;  /**< classification extension possible with TDM */
uint32_t qrf_classified_final_state     :
            1;  /**< flow is fully classified and final */
uint32_t qrf_classified_final_state_changed :
            1;  /**< flow final classification state has changed*/
            uint32_t qrf_reserved                   : 1; /**< reserved */
            uint32_t qrf_cacheable_path_len         : 5; /**< longest_cacheable_path_len */
            uint32_t qrf_unused                     : 9; /**< for later use */
        } values;
        uint32_t qrf_flags; /**< opaq value */
    } qrf;
};

/** **DO NOT ACCESS DIRECTLY**; PLEASE USE ACCESSORS IN qmdpi.h */
#pragma pack(1)
struct qmdpi_flow_flags {
    uint32_t          qf_offload     : 1;  /**< flow is offloaded */
    uint32_t          qf_classified  : 1;  /**< flow is classified */
    uint32_t          qf_pathchanged : 2;  /**< internal use */
    uint32_t          qf_basicdpi    : 1;  /**< internal use */
    uint32_t          qf_needdpi     : 1;  /**< internal use */
    uint32_t          qf_l3l4        : 4;  /**< internal use */
    uint32_t          qf_pdatalvl    : 4;  /**< internal use */
uint32_t          qf_used        :
    1;  /**< flow is used by the engine or flow manager */
    uint32_t          qf_needqextract: 1;  /**< internal use */
    uint32_t          qf_waychanged  : 1;  /**< internal use */
    uint32_t          qf_initialized : 1;  /**< internal use */
    uint32_t          qf_expirating  : 1;  /**< internal use */
    uint32_t          qf_established : 1;  /**< internal use */
    uint32_t          qf_usehw       : 1;
    uint32_t          qf_tcp_handshake_seen: 1; /**< tcp 3 way handshake was seen */
uint32_t          qf_classif_cached :
    1;  /**< flow is classified through classification cache */
uint32_t          qf_classif_cache_added :
    1;    /**< the flow path has already been added in
                                                              the classification cache layerstore, avoid adding the same entry */
uint32_t          qf_classified_final_state :
    1; /**< flow is fully classified and final */
uint32_t          qf_classif_cache_blocked :
    1;  /**< we have found an entry in the classification cache layerstore, but the entry is blocked */
uint32_t          qf_classif_cache_partial :
    1;  /**< we have found an entry in the classification cache layerstore, but the entry is a partial classif */
    uint32_t          qf_fastpass    : 1; /**< internal use */
uint32_t          qf_ml_disabled :
    1; /**< internal use, ML categorization disabled on this flow */
    uint32_t          qf_unused      : 3; /**< for later use */
};

/** Flow flags (qmdpi_flow_flags), pointers, and qmdpi_path */
struct qmdpi_flow_info {
    struct qmdpi_flow_flags qf_flags;

    void             *qf_user_flow;
    void             *qf_reserved2;

    struct qmdpi_path  *qf_dpi_path; /** classification path */
};
#pragma pack()

/**
 * @brief IP4 address structure, data type: QMDPI_DATA_IP_ADDR
 **/
#pragma pack(1)
typedef struct qmdpi_ip_addr {
    uint32_t addr; /**< IP4 address on 4 bytes. */
} qmdpi_ip_addr_t;
#pragma pack()

/**
 * @brief IP6 address structure, data type: QMDPI_DATA_IP6_ADDR
 **/
#pragma pack(1)
typedef struct qmdpi_ip6_addr {
    uint8_t addr[16]; /**< IP6 address on 16 bytes. */
} qmdpi_ip6_addr_t;
#pragma pack()

/**
 * @brief IPv4/v6 address type
 **/
typedef enum qmdpi_ul3l4_l3_family {
    QMDPI_UL3L4_UNKNOWN = 0,  /**< Unknown address type. */
    QMDPI_UL3L4_IP4,          /**< ipv4 address type.    */
    QMDPI_UL3L4_IP6,          /**< ipv6 address type.    */
} qmdpi_ul3l4_l3_family_t;

/**
 * @brief IPv4/v6 address
 **/
typedef struct qmdpi_ul3l4_addr {
    uint8_t family;               /**< l3l4 address family (ipv6, ipv4). */
    union {
        uint32_t addr4;           /**< Address value for ipv4.           */
        qmdpi_ip6_addr_t addr6;   /**< Address value for ipv6.           */
    };
} qmdpi_ul3l4_addr_t;

/**
 * @brief Defines a classification prediction for a given peer, data type: QMDPI_DATA_PEER
 **/
typedef struct qmdpi_peer {
    qmdpi_ul3l4_addr_t  peer_addr;        /**< peer address.                     */
    uint32_t            peer_path[16];    /**< classification prediction's path. */
    uint32_t            peer_l4_proto;    /**< layer 4 protocol.                 */
    uint16_t            peer_port;        /**< layer 4 port.                     */
} qmdpi_peer_t;

/**
 * MAC (ethernet) address, data type: QMDPI_DATA_MAC_ADDR
 **/
#pragma pack(1)
typedef struct qmdpi_mac_addr {
    uint8_t  addr[6];   /**< Ethernet address on 6 bytes. */
} qmdpi_mac_addr_t;
#pragma pack()

/**
 * @brief IP4 network address, data type: QMDPI_DATA_IP_NET
 **/
typedef struct qmdpi_ip_net {
    uint32_t  addr;     /**< Network address on 4 bytes. */
    uint32_t  mask;     /**< Network mask on 4 bytes.    */
} qmdpi_ip_net_t;

/**
 * @brief Defines binary data, data type: QMDPI_DATA_BINARY
 **/
typedef struct qmdpi_binary {
    uint32_t  len;      /**< Length in bytes of binary data. */
    uint8_t   data[64]; /**< Binary data.                    */
} qmdpi_binary_t;

/**
 * @brief Defines an index type, data type: QMDPI_DATA_STRING_INDEX
 **/
typedef uint32_t qmdpi_index_t;

/**
 * @brief Defines internal data of a parent, data type: QMDPI_DATA_PARENT
 **/
typedef struct qmdpi_parent {
    struct uevent
            *first_child; /**< Pointer to the first child for navigation purpose.           */
    int is_end_sent;            /**< Is the MPA_END event already sent for this parent ?          */
} qmdpi_parent_t;

/**
 * @brief Defines statemachine anomaly, data type: QMDPI_DATA_SM_ANOMALY
 **/
typedef struct qmdpi_sm_anomaly {
    const char
    *machine_name;      /**< INTERNAL State machine name that raised an anomaly.  */
    uint32_t
    node_id;           /**< INTERNAL Node id that raised an anomaly.             */
    uint32_t
    transnode_id;      /**< INTERNAL transnode that raised an anomaly.           */
    uint8_t
    anomaly_kind;      /**< The kind of anomaly raised (see kind of statemachine anomaly for different values).                          */
} qmdpi_sm_anomaly_t;

/**
 * @brief Defines a kind of statemachine anomaly
 **/
enum {
    QMDPI_ANOMALY_ENDED_BEFORE_FINAL_NODE = 1,  /**< Performs cleanup on a statemachine before it reaches the final node. */
    QMDPI_ANOMALY_TRANSNODE_BUFFER_FULL = 2,    /**< transnode buffer_full.                                               */
};


/**
 * @brief Defines a union of Qosmos data types
 **/
typedef union qmdpi_data {
    qmdpi_ip_addr_t      ip_addr;       /**< ip network address constant value.*/
    qmdpi_ip_net_t       ip_net;        /**< ip network address and mask data. */
    qmdpi_mac_addr_t     mac_addr;      /**< ethernet mac address data.        */
    qmdpi_ip6_addr_t     ip6_addr;      /**< ip6 address data.                 */
    qmdpi_binary_t       binary;        /**< binary data.                      */
    qmdpi_index_t
    string_index;  /**< string index data (an association between a string and an integer). */
    qmdpi_parent_t       parent;        /**< when the event is a parent.       */
    qmdpi_peer_t
    peer;          /**< information related to a predicted classification <IP address, L4 port, L4 protocol, path>. */
    qmdpi_sm_anomaly_t   sm_anomaly;    /**< statemachine anomaly attributes.  */
} qmdpi_data_u;

#endif /* __QMDPI_STRUCT_H__ */
