/*
*   This file is a part of Qosmos ixEngine.
*   Copyright  Qosmos 2000-2016 - All rights reserved
*/

/* standard headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* libpcap header for capture */
#include <pcap.h>

/* Qosmos ixEngine header */
#include <qmdpi.h>
/* Bundle header */
#include <qmdpi_bundle_api.h>

/* Qosmos ixEngine main objects */
struct qmdpi_engine *engine = NULL;
struct qmdpi_worker *worker = NULL;
struct qmdpi_bundle *bundle = NULL;


static unsigned packet_number;
static int file_number = 0;

struct file_ctx {
    FILE *ptr;
    int size;
};

/* build GIF files from HTTP content */
static void ae_play_with_result(struct qmdpi_result *result)
{
    struct file_ctx *fc = NULL;
    char const *attr_value;
    int proto_id, attr_id, attr_len, attr_flags;

    struct qmdpi_flow *f = qmdpi_result_flow_get(result);
    struct qmdpi_result_flags const *result_flags = qmdpi_result_flags_get(result);

    if (f == NULL || result_flags == NULL) {
        return;
    }

    while (qmdpi_result_attr_getnext(result, &proto_id, &attr_id,
                                     &attr_value, &attr_len, &attr_flags) == 0) {
        switch (attr_id) {
            case Q_HTTP_MIME_TYPE:
                /* detect MIME type and if it is a GIF file,
                 * open a file and store the descriptor in the flow user handle */
                if ((sizeof("image/gif") - 1 == attr_len) &&
                        (strncmp(attr_value, "image/gif", attr_len) == 0)) {
                    char buff[16];
                    snprintf(buff, sizeof(buff) - 1, "img%03d.gif", file_number++);
                    fc = (struct file_ctx *)malloc(sizeof(struct file_ctx));
                    if (fc == NULL) {
                        fprintf(stderr, "error: malloc\n");
                        return;
                    }
                    fc->ptr = fopen(buff, "w");
                    fc->size = 0;
                    qmdpi_flow_user_handle_set(f, (void *)fc);
                }
                break;
            case Q_HTTP_CONTENT:
                /* write content to file */
                fc = (struct file_ctx *)qmdpi_flow_user_handle_get(f);
                if (fc != NULL) {
                    fwrite(attr_value, attr_len, 1, fc->ptr);
                    fc->size += attr_len;
                }
                break;
            case Q_HTTP_REQUEST:
                /* if QMDPI_ATTR_PARENT_END flag is set then HTTP response is complete
                 * and we can close the file */
                if (attr_flags & QMDPI_ATTR_PARENT_END) {
                    fc = (struct file_ctx *)qmdpi_flow_user_handle_get(f);
                    if (fc != NULL) {
                        printf("GIF file successfully created (%d bytes)\n", fc->size);
                        fclose(fc->ptr);
                        free(fc);
                        qmdpi_flow_user_handle_set(f, NULL);
                    }
                }
                break;
            default:
                printf("unknown attribute ID %d\n", attr_id);

        }
    }

    /* close file if flow is expired and we did not see an end flag for HTTP request */
    if (QMDPI_RESULT_FLAGS_FLOW_EXPIRED(result_flags)) {
        fc = (struct file_ctx *)qmdpi_flow_user_handle_get(f);
        if (fc == NULL) {
            return;
        }
        fclose(fc->ptr);
        free(fc);
        qmdpi_flow_user_handle_set(f, NULL);
    }
}

static pcap_t *ae_pcap_open(const char filename[])
{
    pcap_t *fd;
    char errbuf[PCAP_ERRBUF_SIZE];
    fd = pcap_open_offline(filename, errbuf);
    if (0 == fd) {
        fprintf(stderr, "cannot open pcap file: %s\n", errbuf);
        return NULL;
    }
    return fd;
}

static void ae_pcap_close(pcap_t *p)
{
    if (p) {
        pcap_close(p);
    }
}

/* initialize Qosmos ixEngine */
static int ae_engine_init(void)
{
    int ret = 0;

    /* create engine instance */
    engine = qmdpi_engine_create("injection_mode=packet;nb_workers=1;nb_flows=1000");
    if (engine == NULL) {
        fprintf(stderr, "cannot create engine instance\n");
        return -1;
    }

    /* create worker instance */
    worker = qmdpi_worker_create(engine);
    if (worker == NULL) {
        fprintf(stderr, "cannot create worker instance\n");
        goto error_engine;

    }

    /* create bundle instance */
    bundle = qmdpi_bundle_create_from_file(engine, NULL);
    if (bundle == NULL) {
        fprintf(stderr, "cannot create bundle instance\n");
        goto error_worker;
    }

    /* activate bundle */
    ret = qmdpi_bundle_activate(bundle);
    if (ret < 0) {
        fprintf(stderr, "cannot activate bundle\n");
        goto error_bundle;

    }

    /* enable all signatures on bundle */
    ret = qmdpi_bundle_signature_enable_all(bundle);
    if (ret < 0) {
        fprintf(stderr, "error enabling all protocols\n");
        goto error_bundle;
    }

    /* register attributes */
    ret = qmdpi_bundle_attr_register(bundle, "http", "request");
    if (ret < 0) {
        fprintf(stderr, "cannot add metadata http:request\n");
        goto error_bundle;
    }
    ret = qmdpi_bundle_attr_register(bundle, "http", "mime_type");
    if (ret < 0) {
        fprintf(stderr, "cannot add metadata http:mime_type\n");
        goto error_bundle;
    }
    ret = qmdpi_bundle_attr_register(bundle, "http", "content");
    if (ret < 0) {
        fprintf(stderr, "cannot add metadata http:content\n");
        goto error_bundle;
    }

    return 0;

error_bundle:
    qmdpi_bundle_destroy(bundle);
error_worker:
    qmdpi_worker_destroy(worker);
error_engine:
    qmdpi_engine_destroy(engine);

    return -1;

}

static void ae_engine_exit(void)
{
    struct qmdpi_result *result;

    qmdpi_bundle_destroy(bundle);

    /* expire remaining flow contexts */
    while (qmdpi_flow_expire_next(worker, NULL, &result) == 0) {
        ae_play_with_result(result);
    }

    qmdpi_worker_destroy(worker);
    qmdpi_engine_destroy(engine);
    qmdpi_license_destroy();
}

static void ae_process(u_char *user,
                       const struct pcap_pkthdr *phdr,
                       const u_char *pdata)
{
    struct qmdpi_result *result;
    int ret;

    (void)user;
    packet_number++;

    /* set PDU information to be processed by the worker */
    if (qmdpi_worker_pdu_set(worker, pdata, phdr->caplen, &phdr->ts,
                             QMDPI_PROTO_ETH, QMDPI_DIR_DEFAULT, 0) != 0) {
        return;
    }

    /* process packet with worker and provide DPI result */
    do {
        ret = qmdpi_worker_process(worker, NULL, &result);
        if (ret < 0) {
            fprintf(stderr, "DPI processing failure at packet #%u\n", packet_number);
        } else {
            /* play with result */
            ae_play_with_result(result);
        }
    } while (ret == QMDPI_PROCESS_MORE);

    /* destroy expired flows */
    int nb_remaining = 10;
    do {
        if (qmdpi_flow_expire_next(worker, &phdr->ts, &result) != 0) {
            break;
        }
        ae_play_with_result(result);
        nb_remaining --;
    } while (nb_remaining);


}

int main(int argc, const char **argv)
{
    pcap_t *pcap;
    int ret;

    /* check arguments */
    if (argc != 2) {
        fprintf(stderr, "Usage: %s <pcap_file>\n", *argv);
        return 1;
    }

    /* check pcap */
    pcap = ae_pcap_open(argv[1]);
    if (pcap == NULL) {
        return 1;
    }


    /* init Qosmos ixEngine */
    ret = ae_engine_init();
    if (ret < 0) {
        return 1;
    }

    /* DPI processing loop  */
    ret = pcap_loop(pcap, -1, ae_process, NULL);
    if (ret == -1) {
        fprintf(stderr, "error: pcap_loop\n");
        ae_engine_exit();
        ae_pcap_close(pcap);
        return 1;
    }

    ae_engine_exit();
    ae_pcap_close(pcap);

    return 0;
}
