/*
 *   This file is a part of Qosmos ixEngine.
 *   Copyright  Qosmos 2000-2016 - All rights reserved
 */

/* standard headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include <inttypes.h>

/* libpcap header for capture */
#include <pcap.h>

/* Qosmos ixEngine header */
#include <qmdpi.h>

/* Qosmos ixEngine main objects */
struct qmdpi_engine *engine = NULL;
struct qmdpi_worker *worker = NULL;
struct qmdpi_bundle *bundle = NULL;

static uint64_t packet_number;

#define FILENAME_SZ (256)
static int pdata_packet_number;
static unsigned dump_pdb = 0;
static char pdb_in_filename[FILENAME_SZ];
static char pdd_in_filename[FILENAME_SZ];
static char *pdb_filename_buf = NULL;

/**
 * Print custom_signature usage
 *
 * @param name: program name
 */
static void cs_usage(char *name)
{
    fprintf(stderr, "Usage:\n");
    fprintf(stderr, "\t1) %s [--pdd-filename filename] [--dump-pdb]\n",
            name);
    fprintf(stderr,
            "\t2) %s [--pdb-filename filename] [--packet-number N] [pcap_filename,...]\n",
            name);
    fprintf(stderr, "\tOR (All-at-once):\n");
    fprintf(stderr,
            "\t1) %s [--pdd-filename filename] [--packet-number N] [pcap_filename,...]\n",
            name);
    fprintf(stderr, "Options:\n");
    fprintf(stderr, "\t--help: Print usage\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "\t--dump-pdb: Produce pdb file.\n");
    fprintf(stderr,
            "\t--packet-number <N>: Add signature at packet N (with N > 0).\n");
    fprintf(stderr, "\t--pdb-filename <filename>: Name of the pdb file.\n");
    fprintf(stderr, "\t--pdd-filename <filename>: Name of the pdd file.\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "Examples:\n");
    fprintf(stderr, "\tCreate PDB from custom signature (PDD) file:\n");
    fprintf(stderr, "\t\t%s --pdd-filename my.pdd --dump-pdb\n", name);
    fprintf(stderr,
            "\tCreate PDB from custom signature (PDD) file and load it at packet 20:\n");
    fprintf(stderr, "\t\t%s --pdd-filename my.pdd --packet-number 20 custom.pcap\n",
            name);
    fprintf(stderr, "\tLoad PDB file at packet 20:\n");
    fprintf(stderr, "\t\t%s --pdb-filename my.pdb --packet-number 20 custom.pcap\n",
            name);
}

int cs_parse_param(int argc, char *argv[])
{
    static struct option opts[] = {
        {"packet-number", 1, 0, 0},
        {"pdb-filename", 1, 0, 0},
        {"pdd-filename", 1, 0, 0},
        {"dump-pdb", 0, 0, 0},
        {"help", 0, 0, 0},
        {0, 0, 0, 0},
    };
    int c, opti;
    pdata_packet_number = -1;
    memset(pdd_in_filename, 0, FILENAME_SZ);

    while ((c = getopt_long(argc, argv, "", opts, &opti)) != -1) {
        switch (c) {
            case 0 :
                if (strcmp(opts[opti].name, "packet-number") ==  0) {
                    if (sscanf(optarg, "%u", &pdata_packet_number) != 1) {
                        fprintf(stderr, "Unable to set packet_number\n");
                        return -1;
                    }
                    if (pdata_packet_number <= 0) {
                        fprintf(stderr, "packet number should be > 0\n");
                        return -1;
                    }
                } else if (strcmp(opts[opti].name, "help") ==  0) {
                    cs_usage(argv[0]);
                    return 0;
                } else if (strcmp(opts[opti].name, "pdb-filename") ==  0) {
                    if (strlen(optarg) < sizeof(pdb_in_filename)) {
                        memcpy(pdb_in_filename, optarg, strlen(optarg));
                        pdb_in_filename[strlen(optarg)] = '\0';
                    } else {
                        fprintf(stderr, "Too long pdb-filename\n");
                        return -1;
                    }
                } else if (strcmp(opts[opti].name, "pdd-filename") ==  0) {
                    if (strlen(optarg) < sizeof(pdd_in_filename)) {
                        memcpy(pdd_in_filename, optarg, strlen(optarg));
                        pdd_in_filename[strlen(optarg)] = '\0';
                    } else {
                        fprintf(stderr, "Too long pdd-filename\n");
                        return -1;
                    }
                } else if (strcmp(opts[opti].name, "dump-pdb") ==  0) {
                    dump_pdb = 1;
                }
                break;
            default :
                cs_usage(argv[0]);
                return -1;
        }
    }

    if (pdd_in_filename[0] == 0 && pdb_in_filename[0] == 0) {
        fprintf(stderr, "Error: either pdd-filename or pdb-filename is required.\n");
        return -1;
    }

    if (pdata_packet_number == -1 && !dump_pdb) {
        fprintf(stderr, "Error: packet_number is required.\n");
        return -1;
    }

    return 0;
}

static int cs_file_copy_to_buf(const char  *filename,
                               char **buf,
                               int  *buf_len)
{
    FILE      *file = NULL;
    int32_t  r_len = 0;

    if (strcmp(ENGINE_CONFIG_OS, "WINDOWS") == 0) {
        file = fopen(filename, "rb");
    } else {
        file = fopen(filename, "r");
    }

    if (file == NULL) {
        return -1;
    }

    fseek(file, 0L, SEEK_END);
    *buf_len = ftell(file);
    fseek(file, 0L, SEEK_SET);

    if ((*buf = malloc(*buf_len)) == NULL) {
        fclose(file);
        return -1;
    }

    r_len = fread(*buf, sizeof(uint8_t), *buf_len, file);

    if (*buf_len != r_len) {
        free(*buf);
        fclose(file);
        return -1;
    }

    fclose(file);
    return 0;
}

static int cs_buf_copy_to_file(const char  *filename,
                               char *buf,
                               int buf_len)
{
    FILE      *file = NULL;
    int32_t  w_len = 0;

    file = fopen(filename, "w");

    if (file == NULL) {
        return -1;
    }

    w_len = fwrite(buf, sizeof(uint8_t), buf_len, file);

    if (buf_len != w_len) {
        fprintf(stderr, "Error writing PDB.\n");
        fclose(file);
        return -1;
    }

    fclose(file);
    return 0;
}

static int cs_cleanup(char *buf)
{

    if (buf == NULL) {
        return 1;
    }

    free(buf);
    buf = NULL;

    return 0;
}

static void cs_error_dump(const char *csm_api,
                          int32_t err)
{
    fprintf(stderr, "Error CSM <%s>, %s:%i.\n",
            csm_api, qmdpi_error_get_string(bundle, err), err);
}

static uint32_t cs_pdb_swap(const char *pdb_filename)
{
    char             *pdb_buf     = NULL;
    int               pdb_buf_len = 0;
    int32_t           err         = 0;
    struct qmdpi_pdb *pdb;

    if (cs_file_copy_to_buf(pdb_filename, &pdb_buf,
                            &pdb_buf_len)) {
        fprintf(stderr, "%s not found.\n", pdb_filename);
        return -1;
    }

    /* load PDB */
    pdb = qmdpi_bundle_pdb_create_from_addr(bundle, pdb_buf, pdb_buf_len);
    if (pdb == NULL) {
        cs_error_dump("qmdpi_bundle_pdb_create_from_addr", qmdpi_error_get());
        cs_cleanup(pdb_buf);
        return -1;
    }

    /* set PDB as active */
    if ((err = qmdpi_bundle_pdb_activate(pdb))) {
        cs_error_dump("qmdpi_bundle_pdb_activate", err);
        cs_cleanup(pdb_buf);
        return -1;
    }

    fprintf(stdout, "Loading custom signatures...OK\n");

    if (qmdpi_bundle_signature_enable_all(bundle) < 0) {
        fprintf(stderr, "Error enabling all protocols\n");
        cs_cleanup(pdb_buf);
        return -1;
    }

    pdb_filename_buf = pdb_buf;

    return 0;
}

static void cs_pdb_unload(void)
{
    cs_cleanup(pdb_filename_buf);
}

static uint32_t cs_add_signature(const char *pdd_filename)
{
    char  *pdd_buf     = NULL;
    int pdd_buf_len = 0;
    int ret;

    if (cs_file_copy_to_buf(pdd_filename, &pdd_buf,
                            &pdd_buf_len)) {
        fprintf(stderr, "%s not found.\n", pdd_filename);
        return -1;
    }

    ret = qmdpi_bundle_pdb_compile_and_activate(bundle, pdd_buf, pdd_buf_len, NULL,
                                                0);
    if (ret < 0) {
        cs_error_dump("qmdpi_bundle_pdb_compile_and_activate", ret);
        cs_cleanup(pdd_buf);
        return -1;
    }

    fprintf(stdout, "Adding custom signatures...OK\n");

    ret = qmdpi_bundle_signature_enable_all(bundle);
    if (ret < 0) {
        fprintf(stderr, "Error enabling all protocols: %s\n",
                qmdpi_error_get_string(bundle, ret));
        cs_cleanup(pdd_buf);
        return -1;
    }

    cs_cleanup(pdd_buf);

    return 0;
}

static uint32_t cs_compile(const char *pdd_filename)
{
    char     *pdd_buf          = NULL;
    int       pdd_buf_len      = 0;
    char     *pdb_out_buf      = NULL;
    int       pdb_out_len      = 0;
    char     *pdb_out_filename = "my.pdb";
    int   err              = 0;

    if (cs_file_copy_to_buf(pdd_filename, &pdd_buf,
                            &pdd_buf_len)) {
        fprintf(stderr, "%s not found.\n", pdd_filename);
        return 1;
    }

    if ((err = qmdpi_bundle_pdb_compile(bundle,
                                        pdd_buf, pdd_buf_len, NULL, 0,
                                        &pdb_out_buf, &pdb_out_len))) {
        cs_error_dump("qmdpi_bundle_pdb_compile", err);
        cs_cleanup(pdd_buf);
        return 1;
    }
    fprintf(stdout, "Adding custom signatures...OK\n");

    cs_cleanup(pdd_buf);

    if (cs_buf_copy_to_file(pdb_out_filename, pdb_out_buf,
                            pdb_out_len)) {
        fprintf(stderr, "Could not write PDB to file.\n");
        return 1;
    }

    if (qmdpi_bundle_pdb_compile_cleanup(bundle, &pdb_out_buf)) {
        return 1;
    }

    fprintf(stdout, "New PDB file created: %s\n", pdb_out_filename);

    return 0;
}

/* open pcap file */
static inline pcap_t *cs_pcap_open(const char filename[])
{
    pcap_t *fd;
    char errbuf[PCAP_ERRBUF_SIZE];

    fd = pcap_open_offline(filename, errbuf);
    if (fd == NULL) {
        fprintf(stderr, "Cannot open pcap file: %s\n", errbuf);
        return NULL;
    }
    return fd;
}

/* close pcap file */
static inline void cs_pcap_close(pcap_t *p)
{
    if (p) {
        pcap_close(p);
    }
}

static void cs_result_display(struct qmdpi_result *result)
{
    struct qmdpi_result_flags const *result_flags = qmdpi_result_flags_get(result);
    struct qmdpi_flow *flow;
    struct qmdpi_path *path;
    struct qmdpi_bundle *b;
    char buffer[4096];

    /* print classification only if protocol path has changed */
    if (!QMDPI_RESULT_FLAGS_PATH_CHANGED(result_flags)) {
        return;
    }

    flow = qmdpi_result_flow_get(result);
    path = qmdpi_result_path_get(result);
    b = qmdpi_flow_bundle_get(flow);

    qmdpi_data_path_to_buffer(b, buffer, sizeof(buffer), path);

    printf("%"PRIu64" f(%"PRIu64")/path:%s\n", packet_number,
           qmdpi_flow_id_get(flow), buffer);
}

/* initialize Qosmos ixEngine */
static int cs_engine_init(void)
{
    int ret = 0;

    /* create engine instance */
    engine = qmdpi_engine_create("injection_mode=packet;nb_workers=1;nb_flows=200000");
    if (engine == NULL) {
        fprintf(stderr, "Cannot create engine instance: %s\n",
                qmdpi_error_get_string(NULL, qmdpi_error_get()));
        return -1;
    }

    /* create worker instance */
    worker = qmdpi_worker_create(engine);
    if (worker == NULL) {
        fprintf(stderr, "Cannot create worker instance: %s\n",
                qmdpi_error_get_string(NULL, qmdpi_error_get()));
        goto error_engine;

    }

    /* create bundle instance */
    bundle = qmdpi_bundle_create_from_file(engine, NULL);
    if (bundle == NULL) {
        fprintf(stderr, "Cannot create bundle instance: %s\n",
                qmdpi_error_get_string(NULL, qmdpi_error_get()));
        goto error_worker;
    }

    /* activate bundle */
    ret = qmdpi_bundle_activate(bundle);
    if (ret < 0) {
        fprintf(stderr, "Cannot activate bundle: %s\n", qmdpi_error_get_string(bundle,
                                                                               ret));
        goto error_bundle;

    }

    /* enable all signatures on bundle */
    ret = qmdpi_bundle_signature_enable_all(bundle);
    if (ret < 0) {
        fprintf(stderr, "Error enabling all protocols: %s\n",
                qmdpi_error_get_string(bundle, ret));
        goto error_bundle;
    }

    return 0;

error_bundle:
    qmdpi_bundle_destroy(bundle);
error_worker:
    qmdpi_worker_destroy(worker);
error_engine:
    qmdpi_engine_destroy(engine);

    return -1;
}

static void cs_engine_exit(void)
{
    struct qmdpi_result *result;
    while (qmdpi_flow_expire_next(worker, NULL, &result) == 0) {
        cs_result_display(result);
    }
    qmdpi_bundle_destroy(bundle);
    qmdpi_worker_destroy(worker);
    qmdpi_engine_destroy(engine);
    qmdpi_license_destroy();
}

static void cs_process(u_char *user,
                       const struct pcap_pkthdr *phdr,
                       const u_char *pdata)
{
    struct qmdpi_result *result;
    int ret;

    (void)user;
    packet_number++;

    if (packet_number == (uint64_t)pdata_packet_number) {
        if (pdd_in_filename[0] != 0) {
            cs_add_signature(pdd_in_filename);
        } else if (pdb_in_filename[0] != 0) {
            cs_pdb_swap(pdb_in_filename);
        }
    }

    /* set PDU information to be processed by the worker */
    if (qmdpi_worker_pdu_set(worker, pdata, phdr->caplen, &phdr->ts,
                             QMDPI_PROTO_ETH, QMDPI_DIR_DEFAULT, 0) != 0) {
        return;
    }

    /* process packet with worker and provide DPI result */
    do {
        ret = qmdpi_worker_process(worker, NULL, &result);
        if (ret < 0) {
            fprintf(stderr, "DPI processing failure at packet #%"PRIu64"\n", packet_number);
            continue;
        }
        cs_result_display(result);
        /* loop if the packet contains multiple flows */
    } while (ret == QMDPI_PROCESS_MORE);

    if ((packet_number % 1000) != 0) {
        return;
    }

    int nb_remaining = 100;
    do {
        if (qmdpi_flow_expire_next(worker, &phdr->ts, &result) != 0) {
            break;
        }
        cs_result_display(result);

        nb_remaining --;
    } while (nb_remaining);
}

int main(int argc, char **argv)
{
    pcap_t *pcap;
    int ret;

    /* check arguments */
    if (argc < 4) {
        cs_usage(argv[0]);
        return 1;
    }

    /* command line parsing */
    ret = cs_parse_param(argc, argv);
    if (ret < 0) {
        return 1;
    }


    /* init Qosmos ixEngine */
    ret = cs_engine_init();
    if (ret < 0) {
        return 1;
    }

    if (dump_pdb) {
        ret = cs_compile(pdd_in_filename);
        if (ret) {
            cs_engine_exit();
            return 1;
        }
        cs_engine_exit();
        return 0;
    }

    /* check pcap */
    pcap = cs_pcap_open(argv[argc - 1]);
    if (pcap == NULL) {
        cs_engine_exit();
        return 1;
    }

    ret = pcap_loop(pcap, -1, cs_process, NULL);
    if (ret == -1) {
        fprintf(stderr, "Error: pcap_loop\n");
        cs_engine_exit();
        cs_pcap_close(pcap);
        return 1;
    }

    cs_pcap_close(pcap);
    cs_engine_exit();
    cs_pdb_unload();

    return 0;
}
