/*
*   This file is a part of Qosmos ixEngine.
*   Copyright  Qosmos 2018-2018 - All rights reserved
*/

/* standard headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <inttypes.h>
#include <errno.h>
#include <sys/time.h>

/* libpcap header for capture */
#include <pcap.h>

/* Qosmos ixEngine header */
#include <qmdpi.h>
/* Bundle header */
#include <qmdpi_bundle_api.h>

#include "common.h"

/* Qosmos ixEngine main objects */
struct qmdpi_engine *engine = NULL;
struct qmdpi_worker *worker = NULL;
struct qmdpi_bundle *bundle = NULL;

static unsigned int packet_number;
static unsigned int global_id;
/* Counter used to count the number of decoded file */
static unsigned int global_decoded_id;
static unsigned int create_mdata_file = 0;
static unsigned int debug_level = 0;

static const char *attr_mpa[] = {
    "file_generic_id",
    "file_generic_sot",
    "file_generic_eot",
    "file_generic_content",
    "file_generic_content_encoding",
    "file_generic_type",
    "file_generic_size",
    "file_generic_name",
    "file_generic_offset",
    "file_generic_extension",
    "file_generic_size_decoded",
    "file_generic_content_decoded",
};

static const uint8_t table_64[64] = {
    '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
    'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j',
    'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't',
    'u', 'v', 'w', 'x', 'y', 'z',
    'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J',
    'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T',
    'U', 'V', 'W', 'X', 'Y', 'Z',
    'a', 'A'
};

/* Convert an uint64 to a string of 8 characters (plus null) */
static void u64tostr8(uint64_t number, char *buffer)
{
    int i;
    for (i = 0; i < 8; i++) {
        unsigned int rem = number % 64;

        buffer[i] = table_64[rem];
        number = number / 64;
    }
    buffer[i] = '\0';
}

static inline void copy_add_null(char *dest, unsigned int size_d,
                                 const char *src, unsigned int size_s)
{
    unsigned int min = size_d > size_s ? size_s : size_d - 1;

    memcpy(dest, src, min);
    dest[min] = '\0';
}

void print_file_metadata(FILE *file, struct file_context *file_ctx)
{
    fprintf(file, "file id: %" PRIu32 "\n",          file_ctx->file_id);
    fprintf(file, "name: %s\n",                      file_ctx->name);
    fprintf(file, "extension: %s\n",                 file_ctx->extension);
    fprintf(file, "mime type: %s\n",                 file_ctx->mime_type);
    fprintf(file, "content encoding: %s\n",          file_ctx->encoding);
    fprintf(file, "file size: %" PRIu64 "\n",        file_ctx->file_size);
    fprintf(file, "current size: %" PRIu64 "\n",     file_ctx->current_size);
    fprintf(file, "chunks: %" PRIu32 "\n",           file_ctx->chunks);

    if (file_ctx->decoded_file_size) {
        fprintf(file, "decoded file size: %" PRIu64 "\n",
                file_ctx->decoded_file_size);
        fprintf(file, "current size decoded: %" PRIu64 "\n",
                file_ctx->current_size_decoded);
        fprintf(file, "decoded chunks: %" PRIu32 "\n",
                file_ctx->decoded_chunks);
    }
}

static int create_metadata_file(struct file_context *file_ctx)
{
    int ret;
    FILE *mdata_file;
    char buffer[64];

    ret = snprintf(buffer, ARRAY_SIZE(buffer), "%s_metadata",
                   file_ctx->fs_name);
    if (ret == -1 || (unsigned int)ret >= ARRAY_SIZE(buffer)) {
        return -1;
    }
    mdata_file = fopen(buffer, "w");
    if (mdata_file == NULL) {
        PRINT_ERR("ERROR: cannot create file %s (%s)\n", buffer, strerror(errno));
        return -1;
    }
    print_file_metadata(mdata_file, file_ctx);
    ret = fclose(mdata_file);
    if (ret) {
        PRINT_ERR("ERROR: cannot close metadata file for id %" PRIu32 " (%s) %s\n",
                  file_ctx->file_id, buffer, strerror(errno));
    }

    return ret;
}

static int fr_file_context_release(struct file_context *file_ctx)
{
    if (file_ctx && file_ctx->file) {
        int ret;

        DBG_PRINTF_1("Closing file %s\n", file_ctx->fs_name);

        fflush(file_ctx->file);

        ret = fclose(file_ctx->file);
        if (ret) {
            PRINT_ERR("ERROR: cannot close file id %" PRIu32 " (%s) %s\n",
                      file_ctx->file_id, file_ctx->name, strerror(errno));
        }

        if (file_ctx->decoded_file) {

            DBG_PRINTF_1("Closing file %s\n", file_ctx->fs_name_dec);

            fflush(file_ctx->decoded_file);

            ret = fclose(file_ctx->decoded_file);
            if (ret) {
                PRINT_ERR("ERROR: cannot close decoded file id %" PRIu32 " (%s) %s\n",
                          file_ctx->file_id, file_ctx->name, strerror(errno));
            }
        }

        if (create_mdata_file) {
            create_metadata_file(file_ctx);
        }

        file_ctx->file = NULL;
        file_ctx->decoded_file = NULL;
        free(file_ctx);

    } else if (file_ctx) {
        DBG_PRINTF_1("freeing file %u\n", file_ctx->file_id);
        free(file_ctx);
    }

    return 0;
}

/* The function creates a file context if it does not exist. */
int fr_file_context_new(struct file_context **file_ctx,
                        struct timeval const *ts, uint32_t id)
{
    struct file_context *fc;

    *file_ctx = NULL;

    fc = (struct file_context *)malloc(sizeof(*fc));
    if (fc == NULL) {
        return -1;
    }
    memset(fc, 0, sizeof(*fc));

    if (ts != NULL) {
        fc->ts = *ts;
    }

    /* Create file id */
    fc->file_id = id;

    uint64_t ts64 = fc->ts.tv_sec * 1000000ull + fc->ts.tv_usec;
    char fs_id[10];

    u64tostr8(ts64, fs_id);

    int ret = snprintf(fc->fs_name, ARRAY_SIZE(fc->fs_name),
                       "pcap_file_%s_%" PRIu32 "",
                       fs_id, fc->file_id);
    if (ret == -1 || (unsigned int)ret >= ARRAY_SIZE(fc->fs_name)) {
        free(fc);
        return -1;
    }
    ret = snprintf(fc->fs_name_dec, ARRAY_SIZE(fc->fs_name_dec),
                   "pcap_file_%s_%" PRIu32 "_dec",
                   fs_id, fc->file_id);
    if (ret == -1 || (unsigned int)ret >= ARRAY_SIZE(fc->fs_name_dec)) {
        free(fc);
        return -1;
    }


    *file_ctx = fc;

    return 0;
}

/**
 * \brief Create a file with the given file name.
 *
 * \param[in] file    file descriptor.
 * \param[in] fs_name file name
 *
 * \return  0 in case of success, 0 otherwise.
 */
int fr_create_file(FILE **file, char *fs_name)
{
    DBG_PRINTF_0("Creating file %s\n", fs_name);

    *file = fopen(fs_name, "w");
    if (*file == NULL) {
        PRINT_ERR("ERROR: cannot create file %s (%s)\n", fs_name, strerror(errno));
        return -1;
    }

    return 0;
}

int fr_file_chunk_set(struct file_context *file_ctx, const char *buf,
                      uint64_t len)
{
    if (file_ctx == NULL || (buf == NULL && len) || (buf && len == 0)) {
        return -1;
    }

    /* Create file when we get the first chunk. */
    if (file_ctx->chunks == 0) {
        int ret = fr_create_file(&file_ctx->file, file_ctx->fs_name);
        if (ret == -1) {
            return -1;
        }
        /* increase number of "raw" file created */
        global_id++;
    }
    if (file_ctx->offset_present) {
        file_ctx->offset_present = 0;
        int err = fseek(file_ctx->file, (long) file_ctx->content_offset, SEEK_SET);
        if (err < 0) {
            PRINT_ERR("ERROR: cannot fseek for file id %" PRIu32 ": %s\n",
                      file_ctx->file_id, strerror(errno));
        }
    }

    size_t num_w = fwrite(buf, 1, len, file_ctx->file);
    if (num_w != len) {
        return -1;
    }

    file_ctx->current_size += len;
    file_ctx->chunks += 1;

    DBG_PRINTF_2("file size: %" PRIu64 " current size: %" PRIu64 " chunks: %" PRIu32
                 "\n",
                 file_ctx->file_size, file_ctx->current_size, file_ctx->chunks);

    return 0;
}

/**
 * \brief Function used to extract in a file the decoded content.
 *
 * \param[in] file_ctx  file context.
 * \param[in] buf       decoded data.
 * \param[in] len       data length.
 *
 * \return  0 in case of success, -1 otherwise.
 */
int fr_file_decoded_chunk_set(struct file_context *file_ctx, const char *buf,
                              uint64_t len)
{
    if (file_ctx == NULL || (buf == NULL && len) || (buf && len == 0)) {
        return -1;
    }

    /* Create file when we get the first chunk. */
    if (file_ctx->decoded_chunks == 0) {
        int ret = fr_create_file(&file_ctx->decoded_file, file_ctx->fs_name_dec);
        if (ret == -1) {
            return -1;
        }
        /* increase number of decoded file created */
        global_decoded_id++;
    } else {
        /* append to the end of the file */
        int err = fseek(file_ctx->decoded_file, 0, SEEK_END);
        if (err < 0) {
            PRINT_ERR("ERROR: cannot fseek for decoded file id %" PRIu32 ": %s\n",
                      file_ctx->file_id, strerror(errno));
        }
    }

    size_t num_w = fwrite(buf, 1, len, file_ctx->decoded_file);
    if (num_w != len) {
        return -1;
    }

    file_ctx->current_size_decoded += len;
    file_ctx->decoded_chunks += 1;

    DBG_PRINTF_2(PRIu64 " decoded current size: %" PRIu64 " chunks: %" PRIu32 "\n",
                 file_ctx->current_size_decoded, file_ctx->decoded_chunks);

    return 0;
}

/*
 * The function reconstructs files from file generic attributes.
 * New context is created when SOT is seen and released when EOT is raised.
 */
static void fr_play_with_result(struct qmdpi_result *result,
                                struct timeval const *ts)
{
    char const *attr_value;
    int proto_id, attr_id, attr_len, attr_flags;
    struct qmdpi_result_flags const *result_flags;
    uint32_t file_id = ~0;
    struct file_context *file_ctx = NULL;
    int id_changed = 1;

    result_flags = qmdpi_result_flags_get(result);

    if (result_flags == NULL) {
        return;
    }

    while (qmdpi_result_attr_getnext(result, &proto_id, &attr_id,
                                     &attr_value, &attr_len, &attr_flags) == 0) {

        if (attr_id == Q_MPA_FILE_GEN_ID) {
            if (file_id != *(uint32_t *) attr_value) {
                file_id = *(uint32_t *) attr_value;
                id_changed = 1;
            }
            continue;
        } else if (attr_id == Q_MPA_FILE_GEN_SOT) {
            fr_file_context_new(&file_ctx, ts, file_id);
            if (file_ctx == NULL) {
                PRINT_ERR("ERROR: cannot create a new file context.\n");
                return ;
            }
            if (fr_table_add_entry(file_id, file_ctx)) {
                PRINT_ERR("ERROR: cannot create a new table entry.\n");
                fr_file_context_release(file_ctx);
                return ;
            }
            continue;
        }

        /* Get or create file context. */
        if (id_changed) {
            if (fr_table_get_entry(file_id, &file_ctx) < 0) {
                /* file context should have been created when sot was encountered. */
                PRINT_ERR("ERROR: the file %" PRIu32 " does not exist.\n", file_id);
                /**
                 * Due to http2 wrong behavior, we can have missing some decompressed chunk,
                 * new request on another stream id, will drive to think that the file is completed.
                 * so just skip them.
                 *
                 * return ;
                 */
                continue;
            }
        }

        /* Here, we should have file_ctx != NULL. (created or getted) */
        switch (attr_id) {
            case Q_MPA_FILE_GEN_CONTENT:
                fr_file_chunk_set(file_ctx, attr_value, attr_len);
                break;

            case Q_MPA_FILE_GEN_CONTENT_ENCODING:
                copy_add_null(file_ctx->encoding, ENCODING_SIZE, attr_value, attr_len);
                break;

            case Q_MPA_FILE_GEN_TYPE:
                copy_add_null(file_ctx->mime_type, MIME_TYPE_SIZE, attr_value, attr_len);
                break;

            case Q_MPA_FILE_GEN_OFFSET:
                file_ctx->content_offset = *(uint64_t *) attr_value;
                file_ctx->offset_present = 1;
                break;

            case Q_MPA_FILE_GEN_SIZE:
                file_ctx->file_size = *(uint64_t *) attr_value;
                break;

            case Q_MPA_FILE_GEN_NAME:
                copy_add_null(file_ctx->name, FILENAME_SIZE, attr_value, attr_len);
                break;

            case Q_MPA_FILE_GEN_EXTENSION:
                copy_add_null(file_ctx->extension, EXTENSION_SIZE, attr_value, attr_len);
                break;

            case Q_MPA_FILE_GEN_EOT:
                file_ctx->eot = 1;
                if (fr_table_remove_entry(file_id)) {
                    PRINT_ERR("ERROR: attempt to remove non existant file_id %" PRIu32 "\n",
                              file_id);
                }
                fr_file_context_release(file_ctx);
                file_ctx = NULL;
                break;
            case Q_MPA_FILE_GEN_SIZE_DECODED:
                file_ctx->decoded_file_size = *(uint64_t *) attr_value;
                break;
            case Q_MPA_FILE_GEN_CONTENT_DECODED:
                fr_file_decoded_chunk_set(file_ctx, attr_value, attr_len);
                break;

            default:
                PRINT_ERR("ERROR: attribute not handled: %s (%d,%d)\n", \
                          qmdpi_attr_name_get(qmdpi_bundle_attr_get_byid(bundle, \
                                                                         proto_id, attr_id)), proto_id, attr_id);
        }
    }
}

static pcap_t *fr_pcap_open(const char *filename)
{
    pcap_t *fd;
    char errbuf[PCAP_ERRBUF_SIZE];
    fd = pcap_open_offline(filename, errbuf);
    if (0 == fd) {
        PRINT_ERR("ERROR: cannot open pcap file: %s\n", errbuf);
        return NULL;
    }
    fprintf(stdout, "Opening %s\n", filename);
    return fd;
}

static void fr_pcap_close(pcap_t *p)
{
    if (p) {
        pcap_close(p);
    }
}

static int attributes_register_cb(struct qmdpi_bundle *bundle,
                                  struct qmdpi_signature *signature,
                                  struct qmdpi_attr *attr, void *arg)
{
    const char *proto_name = qmdpi_signature_name_get(signature);
    const char *attr_name  = qmdpi_attr_name_get(attr);

    (void)arg;
    if (qmdpi_bundle_attr_register(bundle, proto_name, attr_name) < 0) {
        PRINT_ERR("ERROR: cannot add attribute %s:%s\n", proto_name, attr_name);
        return -1;
    }

    return 0;
}

static int attributes_unregister_cb(struct qmdpi_bundle *bundle,
                                    struct qmdpi_signature *signature,
                                    struct qmdpi_attr *attr, void *arg)
{
    const char *proto_name = qmdpi_signature_name_get(signature);
    const char *attr_name  = qmdpi_attr_name_get(attr);

    (void)arg;
    if (qmdpi_bundle_attr_unregister(bundle, proto_name, attr_name) < 0) {
        PRINT_ERR("ERROR: cannot add attribute %s:%s\n", proto_name, attr_name);
        return -1;
    }

    return 0;
}

/* initialize Qosmos ixEngine */
static int fr_engine_init(void)
{
    int ret = 0;

    /* create engine instance */
    engine = qmdpi_engine_create("fm_tcp_reass_enable=1;nb_workers=1;nb_flows=1000");
    if (engine == NULL) {
        PRINT_ERR("ERROR: cannot create engine instance\n");
        return -1;
    }

    /* create worker instance */
    worker = qmdpi_worker_create(engine);
    if (worker == NULL) {
        PRINT_ERR("ERROR: cannot create worker instance\n");
        goto error_engine;

    }

    /* create bundle instance */
    bundle = qmdpi_bundle_create_from_file(engine, NULL);
    if (bundle == NULL) {
        PRINT_ERR("ERROR: cannot create bundle instance\n");
        goto error_worker;
    }

    /* activate bundle */
    ret = qmdpi_bundle_activate(bundle);
    if (ret < 0) {
        PRINT_ERR("ERROR: cannot activate bundle\n");
        goto error_bundle;

    }

    /* enable all signatures on bundle */
    ret = qmdpi_bundle_signature_enable_all(bundle);
    if (ret < 0) {
        PRINT_ERR("ERROR: cannot enable all protocols\n");
        goto error_bundle;
    }

    unsigned int i;
    for (i = 0; i < ARRAY_SIZE(attr_mpa); i++) {
        ret = qmdpi_bundle_signature_foreach_enabled_by_mpa(bundle, attr_mpa[i],
                                                            attributes_register_cb,
                                                            bundle);
        if (ret < 0) {
            PRINT_ERR("ERROR: cannot enable %s\n", attr_mpa[i]);
            goto error_bundle;
        }
    }
    return 0;

error_bundle:
    qmdpi_bundle_destroy(bundle);
error_worker:
    qmdpi_worker_destroy(worker);
error_engine:
    qmdpi_engine_destroy(engine);

    return -1;
}

static void fr_engine_exit(void)
{
    struct qmdpi_result *result;

    /* expire remaining flow contexts */
    while (qmdpi_flow_expire_next(worker, NULL, &result) == 0) {
        fr_play_with_result(result, NULL);
    }

    unsigned int i;
    for (i = 0; i < ARRAY_SIZE(attr_mpa); i++) {
        int ret;
        ret = qmdpi_bundle_signature_foreach_enabled_by_mpa(bundle, attr_mpa[i],
                                                            attributes_unregister_cb,
                                                            bundle);
        if (ret < 0) {
            PRINT_ERR("ERROR: cannot disable %s\n", attr_mpa[i]);
            break;
        }
    }

    qmdpi_bundle_destroy(bundle);
    qmdpi_worker_destroy(worker);
    qmdpi_engine_destroy(engine);
    qmdpi_license_destroy();
}

static void fr_process(u_char *user,
                       const struct pcap_pkthdr *phdr,
                       const u_char *pdata)
{
    struct qmdpi_result *result;
    int ret;

    (void)user;
    packet_number++;

    /* set PDU information to be processed by the worker */
    if (qmdpi_worker_pdu_set(worker, pdata, phdr->caplen, &phdr->ts,
                             QMDPI_PROTO_ETH, QMDPI_DIR_DEFAULT, 0) != 0) {
        return;
    }

    /* process packet with worker and provide DPI result */
    do {
        ret = qmdpi_worker_process(worker, NULL, &result);
        if (ret < 0 && ret != QMDPI_EFLOW_LOOKUP) {
            PRINT_ERR("ERROR: DPI processing failure at packet #%u (%i)\n", packet_number,
                      ret);
        } else {
            fr_play_with_result(result, &phdr->ts);
        }
    } while (ret == QMDPI_PROCESS_MORE);

    /* destroy expired flows */
    int nb_remaining = 10;
    do {
        if (qmdpi_flow_expire_next(worker, &phdr->ts, &result) != 0) {
            break;
        }
        fr_play_with_result(result, NULL);
        nb_remaining --;
    } while (nb_remaining);
}

int main(int argc, const char **argv)
{
    pcap_t *pcap;
    int ret;

    /* check arguments */
    if (argc != 2) {
        PRINT_ERR("Usage: %s <pcap_file>\n", *argv);
        return 1;
    }

    /* check pcap */
    pcap = fr_pcap_open(argv[1]);
    if (pcap == NULL) {
        return 1;
    }


    /* init Qosmos ixEngine */
    ret = fr_engine_init();
    if (ret < 0) {
        return 1;
    }

    /* DPI processing loop  */
    ret = pcap_loop(pcap, -1, fr_process, NULL);
    if (ret == -1) {
        PRINT_ERR("ERROR: cannot process pcap\n");
        fr_engine_exit();
        fr_pcap_close(pcap);
        return 1;
    }

    fr_engine_exit();
    fr_pcap_close(pcap);

    DBG_PRINTF_0("Number of files reconstructed: %u, Number of decoded file reconstructed: %u\n",
                 global_id, global_decoded_id);

    return 0;
}
