/*
  This file is a part of Qosmos ixEngine.

   Copyright  Qosmos 2016 - All rights reserved

  This computer program and all its components are protected by
  authors' rights and copyright law and by international treaties.
  Any representation, reproduction, distribution or modification
  of this program or any portion of it is forbidden without
  Qosmos explicit and written agreement and may result in severe
  civil and criminal penalties, and will be prosecuted
  to the maximum extent possible under the law.
*/


/* standard headers */
#include <stdio.h>
#include <string.h>
#include <dlfcn.h>
#include <ctype.h>
#include <inttypes.h>

/* libpcap header for capture */
#include <pcap.h>

/* Qosmos ixEngine header */
#include <qmdpi.h>

struct dpi_instance {
    struct qmdpi_engine *engine;
    struct qmdpi_worker *worker;
    struct qmdpi_bundle *bundle;

    struct qmdpi_result  *result;
};

static struct dpi_instance dpi_instance[2];

static int packet_number;

/**
 * Print instance result
 */
static void instance_display_result(struct dpi_instance *instance,
                                    struct qmdpi_result *result)
{
    struct qmdpi_flow *flow = qmdpi_result_flow_get(result);

    char flow_id[32];
    int pos = 0;
    snprintf(flow_id + pos, sizeof(flow_id) - pos, "%"PRIu64,
             qmdpi_flow_id_get(flow));

    struct qmdpi_result_flags const *result_flags = qmdpi_result_flags_get(result);

    if (QMDPI_RESULT_FLAGS_FLOW_CREATED(result_flags) ||
            QMDPI_RESULT_FLAGS_FLOW_EXPIRED(result_flags)) {
        fprintf(stdout, "%d/%d %d f(%s)/p/flow_created=%d,flow_expired=%d\n",
                qmdpi_engine_id_get(instance->engine),
                qmdpi_worker_id_get(instance->worker),
                packet_number,
                flow_id,
                QMDPI_RESULT_FLAGS_FLOW_CREATED(result_flags),
                QMDPI_RESULT_FLAGS_FLOW_EXPIRED(result_flags));
    }

    /* print classification only if protocol path has changed */
    if (QMDPI_RESULT_FLAGS_PATH_CHANGED(result_flags)) {
        char buffer[4096];
        unsigned int i;
        int pos = 0;
        struct qmdpi_bundle *b = qmdpi_flow_bundle_get(flow);
        struct qmdpi_path *path = qmdpi_result_path_get(result);
        struct qmdpi_signature *sig;
        if (path) {
            for (i = 0; i < path->qp_len; i++) {
                sig = qmdpi_bundle_signature_get_byid(b, path->qp_value[i]);
                pos += sprintf(buffer + pos, "%s%s", i > 0 ? "." : "",
                               qmdpi_signature_name_get(sig));
            }
            fprintf(stdout, "%d/%d %d f(%s)/p/path=%s\n",
                    qmdpi_engine_id_get(instance->engine),
                    qmdpi_worker_id_get(instance->worker),
                    packet_number,
                    flow_id,
                    buffer);
        }
    }

    char const *data;
    int datalen;
    int proto_id;
    int attr_id;
    struct qmdpi_signature *sig;
    struct qmdpi_attr *attr;
    int flags;
    while (qmdpi_result_attr_getnext(result, &proto_id, &attr_id,
                                     &data, &datalen, &flags) == 0) {

        char buffer[4096 * 2 + 1];
        sig = qmdpi_bundle_signature_get_byid(instance->bundle, proto_id);
        attr = qmdpi_bundle_attr_get_byid(instance->bundle, proto_id, attr_id);

        int i;
        for (i = 0; i < datalen && (unsigned int)i < sizeof(buffer) - 1; i++) {
            buffer[i] = isprint(data[i]) ? data[i] : '.';
        }
        buffer[i] = '\0';
        fprintf(stdout, "%d/%d %d f(%s)/p/%s:%s=%s\n",
                qmdpi_engine_id_get(instance->engine),
                qmdpi_worker_id_get(instance->worker),
                packet_number,
                flow_id,
                qmdpi_signature_name_get(sig),
                qmdpi_attr_name_get(attr),
                buffer);
    }
}

/**
 * ------------------------
 * Pcap functions
 * ------------------------
 */
/*
 * Open a pcap by its filename
 */
static pcap_t *mi_trace_open(const char filename[])
{
    pcap_t *pcap = NULL;
    char errbuf[PCAP_ERRBUF_SIZE];

    pcap = pcap_open_offline(filename, errbuf);

    if (pcap == NULL) {
        fprintf(stderr, "On trace %s, pcap_open: %s\n", filename, errbuf);
        return NULL;
    }
    fprintf(stdout, "Opening trace %s.\n", filename);

    return pcap;
}

/*
 * Close a pcap
 */
static void mi_pcap_close(pcap_t *p)
{
    if (p) {
        pcap_close(p);
    }
}

/**
 * ------------------------
 * DPI functions
 * ------------------------
 */
/**
 * Instance initialitation of the dpi process
 */
static int instance_init(struct dpi_instance *instance, const char bundle[])
{
    /* engine instance init */
    instance->engine =
        qmdpi_engine_create("injection_mode=packet;nb_workers=1;nb_flows=4000");
    if (instance->engine == NULL) {
        printf("Cannot create engine instance: %s\n", qmdpi_error_get_string(NULL,
                                                                             qmdpi_error_get()));
        return -1;
    }

    /* worker instance init */
    instance->worker = qmdpi_worker_create(instance->engine);
    if (instance->worker == NULL) {
        printf("cannot create worker instance\n");
        goto error_engine2;
    }
    instance->bundle = qmdpi_bundle_create_from_file(instance->engine, bundle);
    if (instance->bundle == NULL) {
        printf("cannot create bundle instance: %s\n", bundle);
        goto error_worker;
    }

    if (qmdpi_bundle_activate(instance->bundle) < 0) {
        printf("cannot activate bundle: %s\n", bundle);
    }

    /* initialize all protocols on bundle */
    if (qmdpi_bundle_signature_enable_all(instance->bundle) < 0) {
        printf("error enabling all protocols: %s\n", bundle);
        goto error_bundle;
    }

    return 0;

error_worker:
error_engine2:
error_bundle:
    return -1;
}

/**
 * Instance configuration of the dpi process
 */
static void instance_config(struct dpi_instance *instance)
{
    printf("Configuring instance %d\n", qmdpi_engine_id_get(instance->engine));

    switch (qmdpi_engine_id_get(instance->engine)) {
        case 1:
            if (qmdpi_bundle_attr_register(instance->bundle, "irc", "login") < 0) {
                printf("cannot register metadata\n");
            }
            if (qmdpi_bundle_attr_register(instance->bundle, "irc", "server") < 0) {
                printf("cannot register metadata\n");
            }
            break;

        case 2:
            if (qmdpi_bundle_signature_disable_all(instance->bundle) < 0) {
                printf("Cannot disable signature\n");
            }
            if (qmdpi_bundle_signature_enable(instance->bundle, "ip") < 0) {
                printf("Cannot enable signature\n");
            }
            if (qmdpi_bundle_signature_enable(instance->bundle, "udp") < 0) {
                printf("Cannot enable signature\n");
            }
            if (qmdpi_bundle_signature_enable(instance->bundle, "tcp") < 0) {
                printf("Cannot enable signature\n");
            }
            if (qmdpi_bundle_signature_enable(instance->bundle, "http") < 0) {
                printf("Cannot enable signature\n");
            }
            if (qmdpi_bundle_attr_register(instance->bundle, "http", "server") < 0) {
                printf("cannot register metadata\n");
            }
            if (qmdpi_bundle_attr_register(instance->bundle, "http", "uri") < 0) {
                printf("cannot register metadata\n");
            }
            break;
    }
}

/**
 * Instance cleanup of the dpi process
 */
static void instance_exit(struct dpi_instance *instance)
{
    qmdpi_bundle_destroy(instance->bundle);

    while (qmdpi_flow_expire_next(instance->worker,
                                  NULL,
                                  &instance->result) == 0) {
        instance_display_result(instance, instance->result);
    }

    qmdpi_worker_destroy(instance->worker);
    qmdpi_engine_destroy(instance->engine);
}

/**
 * Process packet by sending it to dpi process
 */
static void mi_process_packet(struct dpi_instance *instance,
                              const struct pcap_pkthdr *phdr,
                              const u_char *pdata)
{
    int                  ret;

    if (qmdpi_worker_pdu_set(instance->worker, pdata, phdr->caplen, &phdr->ts,
                             QMDPI_PROTO_ETH, 2, 0) != 0) {
        return;
    }

    do {
        /* Process packet with worker without specifying dpi flow (packet mode) */
        ret = qmdpi_worker_process(instance->worker, NULL, &instance->result);
        if (ret < 0) {
            printf("DPI processing failure at packet #%u\n", packet_number);
        } else {
            instance_display_result(instance, instance->result);
        }
    } while (ret == QMDPI_PROCESS_MORE);

    /* Update timers (destroy expire flows) and possibly do something upon expiration */
    /* Frequency : every 1000 packets (for example)*/
    if ((packet_number % 1000) != 0) {
        return;
    }

    /* expire max 100 flows every 1000 packets */
    int nb_remaining = 100;
    do {
        int ret = qmdpi_flow_expire_next(instance->worker, &phdr->ts,
                                         &instance->result);
        if (ret != 0) {
            break;
        }

        nb_remaining --;
        instance_display_result(instance, instance->result);
    } while (nb_remaining);
}

int main(int argc, const char **argv)
{
    pcap_t *pcap;
    int ret;

    /* check arguments */
    if (argc != 4) {
        printf("Usage: %s bundle1.so bundle2.so <pcap_file>\n", *argv);
        return 1;
    }

    /* check pcap */
    pcap = mi_trace_open(argv[3]);
    if (pcap == NULL) {
        return 1;
    }

    /* instances creation */
    ret = instance_init(&dpi_instance[0], argv[1]);
    if (ret < 0) {
        return 1;
    }
    ret = instance_init(&dpi_instance[1], argv[2]);
    if (ret < 0) {
        return 1;
    }

    /* instances configuration */
    instance_config(&dpi_instance[0]);
    instance_config(&dpi_instance[1]);

    /* packet processing */
    while (1) {
        struct pcap_pkthdr *pkthdr;
        u_char const       *data;

        ret = pcap_next_ex(pcap, &pkthdr, &data);
        if (ret < 0) {
            break;
        }

        packet_number++;

        mi_process_packet(&dpi_instance[0], pkthdr, data);
        mi_process_packet(&dpi_instance[1], pkthdr, data);
    }

    /* instances cleanup */
    instance_exit(&dpi_instance[0]);
    instance_exit(&dpi_instance[1]);

    mi_pcap_close(pcap);
    qmdpi_license_destroy();

    return 0;
}
