/*
  This file is a part of Qosmos ixEngine.

   Copyright  Qosmos 2014 - All rights reserved

  This computer program and all its components are protected by
  authors' rights and copyright law and by international treaties.
  Any representation, reproduction, distribution or modification
  of this program or any portion of it is forbidden without
  Qosmos explicit and written agreement and may result in severe
  civil and criminal penalties, and will be prosecuted
  to the maximum extent possible under the law.
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pcap.h>
#include <pthread.h>
#include <inttypes.h>
#ifdef __FreeBSD__
#include <pthread_np.h>
#endif
#include <unistd.h>
#include <sched.h>

#ifndef WIN32
#include <arpa/inet.h>
#else
#include <winsock2.h>
#endif

#ifdef __FreeBSD__
typedef cpuset_t cpu_set_t;
#endif

/* Qosmos ixEngine header */
#include <qmdpi.h>

#include <packet_helper.h>

#include "customize.h"

#define LLC_HEADER_SZ 2
#define PACKET_QUEUESZ (1 << 13)
#define PACKET_QUEUEMASK (PACKET_QUEUESZ - 1)

struct pi_dpi_stats {
    unsigned long processed;
    unsigned long errors;
    unsigned long dropped;
};
#define PI_DPI_STAT_INIT(s) do {                                           \
        (s)->processed = 0;                                                 \
        (s)->errors = 0;                                                    \
        (s)->dropped = 0;                                                    \
    } while(/*CONSTCOND*/0)

/*
 * List of packets for a thread
 */
struct pi_thread {
    struct pi_dpi_stats st;
    int thread_id;
    int cpu_id;
    pthread_t handle;         /* thread handle */
    size_t read_index;
    size_t write_index;
    struct pi_pkt *packets[PACKET_QUEUESZ];
    struct qmdpi_worker *worker;
    pthread_mutex_t lock;
};
#define PI_THREAD_INIT(s) do {                                         \
        (s)->thread_id = 0;                                                 \
        (s)->cpu_id    = -1;                                                \
        (s)->read_index = 0;                                                \
        (s)->write_index = 0;                                               \
        PI_DPI_STAT_INIT(&(s)->st);                                            \
    } while(/*CONSTCOND*/0)

#define PACKET_INDEX(_value) ((_value) & PACKET_QUEUEMASK)


static struct qmdpi_engine *engine;
struct qmdpi_bundle *bundle;

/*
 * Array of packet buffer for all threads
 */
static struct pi_thread *threads;

static int noprint;
static int monitoring;
static unsigned int nb_cpus = 1;
static unsigned long pktdrop;
static char **traces;
static unsigned int tracesnr;

static uint32_t cpu_mapping[128];
static uint32_t cpu_mapping_id;

static inline void usage(char const *p)
{
    fprintf(stderr, "Usage:\n%s [options] <trace1> [trace2...]\n"
            "Options:\n"
            "\t--enable-monitoring: Enable performance monitoring. Resulting statistics (in terms of memory and processing time) are displayed at exit.\n"
            "\t--no-print: Do not print classification\n"
            "\t-n <nb_thread>: Number of threads to launch\n"
            "\t--cpu-mapping <value>,<value>,...  Set cpu mapping for each created instance\n",
            p);
}

struct pi_pkt {
    uint8_t *data;
    int32_t len;
    struct timeval timestamp;
    int32_t link_mode;
    int32_t thread_id;
    uint64_t packet_number;
};

#define CONFIG_LINE_SIZE 512
#define CONFIG_LINE_INIT_OFFSET (sizeof("injection_mode=packet;nb_flows=100000;")-1)
char config_line[CONFIG_LINE_SIZE] = "injection_mode=packet;nb_flows=100000;";

/**
 * ------------------------
 * Packet functions
 * ------------------------
 */
/**
 * Alloc a new packet
 */
static inline struct pi_pkt *pi_pkt_alloc(uint32_t caplen)
{
    struct pi_pkt *packet;

    packet = (struct pi_pkt *)malloc(sizeof(*packet) + caplen);

    if (packet == NULL) {
        fprintf(stderr, "Can't malloc new packet\n");
        return NULL;
    }

    memset(packet, 0, sizeof(*packet));

    packet->len = caplen;
    packet->data = (unsigned char *)(packet + 1);

    return packet;
}

/**
 * Free a packet
 */
static inline void pi_pkt_free(struct pi_pkt *p)
{
    free(p);
}

/**
 * Initialize packet with data, remove_llc flags indicates if LLC header is
 * present
 */
static struct pi_pkt *
pi_pkt_build(struct pcap_pkthdr *phdr, const u_char *pdata, int remove_llc)
{
    struct pi_pkt *packet;
    uint32_t caplen;

    if (remove_llc && phdr->caplen >= LLC_HEADER_SZ) {
        caplen = phdr->caplen - LLC_HEADER_SZ;
        pdata += LLC_HEADER_SZ;
    } else {
        caplen = phdr->caplen;
    }

    packet = pi_pkt_alloc(caplen);
    if (packet == NULL) {
        return NULL;
    }

    packet->timestamp = phdr->ts;
    packet->link_mode = QMDPI_PROTO_ETH;
    packet->len = caplen;

    memcpy(packet->data, pdata, caplen);

    return packet;
}

/**
 * Enqueue packet in thread ring
 */
static inline void pi_pkt_queue(struct pi_thread *thread,
                                struct pi_pkt *packet)
{
    size_t next_index;

    if (packet) {
        packet->thread_id = thread->thread_id;
    }

    next_index = PACKET_INDEX(thread->write_index + 1);

    while (next_index == thread->read_index) {
        /**
         * Chances are that all thread queue are full. So going to sleep should
         * be ok, letting other thread a chance to process some packets
         */
        usleep(10);
    }

    thread->packets[thread->write_index] = packet;
    /**
     * This mutex lock is mainly used as a memory barrier to force reading cpu
     * to get the packet modification before the write_index one.
     */
    pthread_mutex_lock(&thread->lock);
    thread->write_index = next_index;
    pthread_mutex_unlock(&thread->lock);
}

/**
 * Dequeue packet in thread ring
 */
static inline struct pi_pkt *pi_pkt_dequeue(struct pi_thread *thread)
{
    struct pi_pkt *packet;

    pthread_mutex_lock(&thread->lock);
    while (thread->read_index == thread->write_index) {
        /**
         * Let other threads going
         */
        pthread_mutex_unlock(&thread->lock);
        sched_yield();
        pthread_mutex_lock(&thread->lock);
    }
    pthread_mutex_unlock(&thread->lock);

    packet = thread->packets[thread->read_index];

    /*
     * Mutex lock is only needed for write index. This is more used as a memory
     * barrier for packet coherency
     */
    thread->read_index = PACKET_INDEX(thread->read_index + 1);

    return packet;
}

/**
 * Send NULL packet to all thread to stop them
 */
static void pi_pkt_stop(unsigned int nb_cpus)
{
    unsigned int i;

    for (i = 0; i < nb_cpus; ++i) {
        pi_pkt_queue(&threads[i], NULL);
    }
}


/**
 * ------------------------
 * DPI functions
 * ------------------------
 */
static void pi_result_display(struct pi_thread *th,
                              struct qmdpi_result *result)
{
    struct qmdpi_result_flags const *result_flags = qmdpi_result_flags_get(result);

    if (QMDPI_RESULT_FLAGS_PATH_CHANGED(result_flags)) {
        char buffer[4096];
        struct qmdpi_flow *flow = qmdpi_result_flow_get(result);
        struct qmdpi_path *path = qmdpi_result_path_get(result);
        struct qmdpi_bundle *b = qmdpi_flow_bundle_get(flow);
        if (path) {
            qmdpi_worker_data_path_to_buffer(th->worker, b, buffer, sizeof(buffer), path);
        }

        fprintf(stdout, "Core #%u\t\tClassification Path: %s\n",
                th->thread_id, buffer);
    }
}

/**
 * Process packet by sending it to dpi process
 */
static void *pi_pkt_process(void *arg)
{
    struct pi_pkt *pkt;
    struct pi_thread *th;
    uint64_t local_packet_number = 0;

    th = (struct pi_thread *)arg;

    if (th->cpu_id >= 0) {
#if defined(__APPLE__) || defined(SUNCC) || defined(WIN32)
        fprintf(stdout, "Cannot set thread affinity on this platform.\n");
#else
        cpu_set_t cpuset;
        int       s;

        CPU_ZERO(&cpuset);
        CPU_SET(th->cpu_id, &cpuset);

        s = pthread_setaffinity_np(pthread_self(), sizeof(cpuset), &cpuset);
        if (s != 0) {
            fprintf(stderr, "Cannot set affinity for thread %u: %s\n", th->thread_id,
                    strerror(s));
            return NULL;
        }

        fprintf(stdout, "Thread %i uses CPU %d\n", th->thread_id, th->cpu_id);
#endif
    }

    fprintf(stdout, "Thread %i %p started\n", th->thread_id,
            (void *) th->handle);

    while ((pkt = pi_pkt_dequeue(th)) != NULL) {
        int ret;

        local_packet_number++;

        if (qmdpi_worker_pdu_set(th->worker, pkt->data, pkt->len, &pkt->timestamp,
                                 pkt->link_mode, QMDPI_DIR_DEFAULT, 0) != 0) {
            ++th->st.errors;
            pi_pkt_free(pkt);
            continue;
        }
        do {
            struct qmdpi_result *result;
            ret = qmdpi_worker_process(th->worker, NULL, &result);
            if (ret < 0) {
                break;
            }

            if (!noprint) {
                pi_result_display(th, result);
            }
        } while (ret == QMDPI_PROCESS_MORE);

        /**
         * Get DPI stats
         */
        if (ret < 0) {
            ++th->st.errors;
        } else {
            ++th->st.processed;
            if (ret == QMDPI_PROCESS_IP_DEFRAG_DROP ||
                    ret == QMDPI_PROCESS_BOUNDS_FAILURE ||
                    ret == QMDPI_PROCESS_BOUNDS_DROP) {
                ++th->st.dropped;
            }

        }

        if ((local_packet_number % 1000) == 0) {
            struct qmdpi_result *result;

            int nb_remaining = 100;
            do {
                ret = qmdpi_flow_expire_next(th->worker, &pkt->timestamp, &result);
                if (ret != 0) {
                    break;
                }

                nb_remaining --;
                if (!noprint) {
                    pi_result_display(th, result);
                }
            } while (nb_remaining);
        }

        pi_pkt_free(pkt);
    }

    struct qmdpi_result *result;
    while (qmdpi_flow_expire_next(th->worker, NULL, &result) == 0) {
        if (!noprint) {
            pi_result_display(th, result);
        }
    }
    qmdpi_worker_destroy(th->worker);

    return NULL;
}

/* add param to config line */
int pi_config_line_add(char *config_line, int *config_line_offset,
                       const char *param, uint64_t value)
{
    char param_string[128];
    int len = 0;
    len = snprintf(param_string, sizeof(param_string), "%s=%"PRIu64";", param,
                   value);
    if (len < 0 || (unsigned int)len > sizeof(param_string) ||
            CONFIG_LINE_SIZE - *config_line_offset <= len) {
        fprintf(stderr, "Error buffer for DPI configuration: %s\n", param);
        return 0;
    }

    strncat(config_line, param_string, CONFIG_LINE_SIZE - *config_line_offset - 1);
    *config_line_offset += len ;
    return 1;
}

/**
 * Initialize the dpi engine
 */
static int pi_dpi_init(void)
{
    int ret;
    int config_line_offset = CONFIG_LINE_INIT_OFFSET;

    if (pi_config_line_add(config_line, &config_line_offset, "nb_workers",
                           nb_cpus) == 0) {
        return -1;
    }
    if (monitoring) {
        if (pi_config_line_add(config_line, &config_line_offset, "monitoring_enable",
                               1) == 0) {
            return -1;
        }
    }

    if (pi_custom_engine_configuration() < 0) {
        return -1;
    }

    engine = qmdpi_engine_create(config_line);
    if (engine == NULL) {
        fprintf(stderr, "error initializing DPI engine: %s\n",
                qmdpi_error_get_string(NULL, qmdpi_error_get()));
        goto err_engine;
    }

    if (cpu_mapping_id > 0 && cpu_mapping_id != nb_cpus) {
        fprintf(stderr,
                "Invalid cpu mapping configuration (%d threads, %u cpu mapping configurations)\n",
                nb_cpus, cpu_mapping_id);
        goto err_engine;
    }

    bundle = qmdpi_bundle_create_from_file(engine, NULL);
    if (bundle == NULL) {
        fprintf(stderr, "Cannot load bundle: %s\n", qmdpi_error_get_string(NULL,
                                                                           qmdpi_error_get()));
        goto err_bundle;
    }

    qmdpi_bundle_activate(bundle);

    if ((ret = qmdpi_bundle_signature_enable_all(bundle)) < 0) {
        fprintf(stderr, "Failed to enable all protocols: %s\n",
                qmdpi_error_get_string(bundle, ret));
        goto err_sig;
    }

    if (pi_custom_tune_configuration() < 0) {
        return -1;
    }

    if (pi_custom_attribute_configuration() < 0) {
        return -1;
    }

    if (monitoring) {
        qmdpi_stats_init_start(engine);
    }
    return 0;

err_sig:
    qmdpi_bundle_destroy(bundle);

err_bundle:
    qmdpi_engine_destroy(engine);

err_engine:
    return -1;
}

/**
 * Cleanup the dpi engine
 */
static int pi_dpi_exit(void)
{
    if (monitoring) {
        qmdpi_stats_dump(engine, stdout, (qmdpi_output_fn_t)fprintf);
        qmdpi_stats_exit(engine);
    }

    qmdpi_bundle_destroy(bundle);
    qmdpi_engine_destroy(engine);
    qmdpi_license_destroy();
    return 0;
}


/**
 * ------------------------
 * Pcap functions
 * ------------------------
 */
/*
 * Open a pcap by its filename
 */
static pcap_t *pi_trace_open(const char *filename)
{
    pcap_t *pcap = NULL;
    char errbuf[PCAP_ERRBUF_SIZE];

    pcap = pcap_open_offline(filename, errbuf);

    if (pcap == NULL) {
        fprintf(stderr, "On trace %s, pcap_open: %s\n", filename, errbuf);
        return NULL;
    }
    fprintf(stdout, "Opening trace %s.\n", filename);

    return pcap;
}

/*
 * Close a pcap
 */
static void pi_trace_close(pcap_t *p)
{
    pcap_close(p);
}


/**
 * ------------------------
 * Thread functions
 * ------------------------
 */
/**
 * Launch all processing threads
 */
static int pi_thread_launch(unsigned int nb_cpus)
{
    int ret;
    unsigned int i;

    for (i = 0; i < nb_cpus; ++i) {
        threads[i].thread_id = i;
        threads[i].worker = qmdpi_worker_create(engine);
        pthread_mutex_init(&threads[i].lock, NULL);

        ret = pthread_create(&threads[i].handle, NULL, pi_pkt_process,
                             &threads[i]);
        if (ret != 0) {
            fprintf(stderr, "Starting threads error\n");
            return ret;
        }
    }
    if (monitoring) {
        qmdpi_stats_init_end(engine);
    }

    return 0;
}

/**
 * Wait for all threads to terminate
 */
static inline void pi_thread_wait(unsigned int nb_cpus)
{
    unsigned int i;

    for (i = 0; i < nb_cpus; i++) {
        pthread_join(threads[i].handle, NULL);
        pthread_mutex_destroy(&threads[i].lock);
    }
}

/**
 * Dispatch packets for each thread (should be run by master thread)
 */
static int pi_pkt_dispatch(pcap_t *pcap, unsigned int nb_cpus)
{
    struct pcap_pkthdr *phdr;
    struct pi_pkt *packet;
    const u_char *pdata;
    int remove_llc;
    uint64_t packet_number = 0;
    int ret;

    if (pcap_datalink(pcap) == DLT_LINUX_SLL) {
        remove_llc = 1;
    } else {
        remove_llc = 0;
    }

    while ((ret = pcap_next_ex(pcap, &phdr, &pdata)) >= 0) {
        uint32_t hashkey;

        packet_number++;

        packet = pi_pkt_build(phdr, pdata, remove_llc);
        if (packet == NULL) {
            ++pktdrop;
            continue;
        }
        hashkey = qmdpi_packet_hashkey_get(pdata, phdr->caplen, QMDPI_PROTO_ETH);
        packet->packet_number = packet_number;

        pi_pkt_queue(&threads[hashkey % nb_cpus], packet);
    }

    return 0;
}

/**
 * ------------------------
 * Application functions
 * ------------------------
 */
/**
 * Process all pcap packets
 */
static int pi_run(void)
{
    pcap_t *pcap = NULL;
    int ret;
    size_t i;

    ret = pi_thread_launch(nb_cpus);
    if (ret != 0) {
        return ret;
    }

    for (i = 0; i != tracesnr; ++i) {
        pcap = pi_trace_open(traces[i]);
        if (pcap == NULL) {
            continue;
        }

        ret = pi_pkt_dispatch(pcap, nb_cpus);
        if (ret != 0) {
            return ret;
        }

        pi_trace_close(pcap);
    }

    pi_pkt_stop(nb_cpus);
    pi_thread_wait(nb_cpus);

    return 0;
}

/**
 * Initialize the application
 */
static int pi_app_init(void)
{
    unsigned int i;

    threads = (struct pi_thread *)malloc(nb_cpus * sizeof(*threads));
    if (threads == NULL) {
        fprintf(stderr, "Can't malloc threads\n");
        return -1;
    }

    for (i = 0; i < nb_cpus; ++i) {
        PI_THREAD_INIT(&threads[i]);

        if (cpu_mapping_id > 0) {
            threads[i].cpu_id = cpu_mapping[i];
        }
    }

    return 0;
}

/**
 * Print general statistics
 */
static void pi_stats_print(unsigned int nb_cpus)
{
    unsigned int i;
    fprintf(stdout, "\n"
            " --------------------\n"
            "| Packet Statistics  |\n"
            " --------------------\n\n");

    fprintf(stdout, "DPI:\n\n");
    for (i = 0; i < nb_cpus; ++i) {
        fprintf(stdout, " Core #%u: %lu processed (%lu dropped) / %lu errors\n", i,
                threads[i].st.processed, threads[i].st.dropped, threads[i].st.errors);
    }
    fprintf(stdout, "\n");
}

/**
 * Cleanup the application
 */
static int pi_app_exit(void)
{
    pi_stats_print(nb_cpus);

    free(threads);
    return 0;
}

/**
 * Parse options
 */
static int pi_args_parse(int argc, char **argv)
{
    int ac;
    char **p;

    if (argc < 2) {
        return -1;
    }

    for (ac = argc - 1, p = argv + 1; ac > 0; --ac, ++p) {
        if (strcmp(*p, "-n") == 0) {
            --ac;
            if (ac == 0) {
                return -1;
            }
            ++p;
            nb_cpus = atoi(*p);
        } else if (strcmp(*p, "--enable-monitoring") == 0) {
            monitoring = 1;
        } else if (strcmp(*p, "--no-print") == 0) {
            noprint = 1;
        } else if (strcmp(*p, "--cpu-mapping") == 0) {
            --ac;
            if (ac == 0) {
                return -1;
            }
            ++p;
            char *mapping = strtok(*p, ",");
            do {
                cpu_mapping[cpu_mapping_id++] = atoi(mapping);

                mapping = strtok(NULL, ",");
            } while (mapping != NULL);
        } else {
            break;
        }
    }

    if (ac == 0) {
        return -1;
    }

    traces = p;
    tracesnr = ac;

    return 0;
}

int main(int argc, char **argv)
{
    int ret = 0;

    ret = pi_args_parse(argc, argv);
    if (ret != 0) {
        usage(argv[0]);
        ret = 1;
        goto exit;
    }


    ret = pi_app_init();
    if (ret != 0) {
        ret = 1;
        goto exit;
    }

    ret = pi_dpi_init();
    if (ret != 0) {
        ret = 1;
        goto appexit;
    }

    pi_run();

    pi_dpi_exit();

appexit:
    pi_app_exit();
exit:
    return ret;
}
