/*
 *   This file is a part of Qosmos ixEngine.
 *   Copyright  Qosmos 2016-2016 - All rights reserved
 */

/* standard headers */
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <getopt.h>
#include <inttypes.h>

/* libpcap header for capture */
#include <pcap.h>

/* Qosmos ixEngine header */
#include <qmdpi.h>

/* Qosmos ixEngine main objects */
struct qmdpi_engine *engine = NULL;
struct qmdpi_worker *worker = NULL;
struct qmdpi_bundle *bundle = NULL;

static unsigned int  packet_number;
static unsigned int  swapat;
static char         *pcapfilename;
static char         *pdb_name;
static uint64_t      last_packet_ts;

static void pdb_swap_usage(const char *prg_name)
{
    fprintf(stderr, "%s --swapat <pkt_num> <pdb> <pcap_file>\n"
            , prg_name);
    fprintf(stderr, "Options:\n");
    fprintf(stderr,
            "\t--swapat <pkt_num> : load and swap pdb at packet number <pkt_num>\n");
}

static int pdb_swap_parse_param(int argc, char *argv[])
{

    /* check arguments */
    if (argc != 5) {
        pdb_swap_usage(argv[0]);
        return -1;
    }

    static struct option long_options[] = {
        {"swapat", 1, 0, 0},
        {0, 0, 0, 0},
    };

    while (1) {
        int option_index = 0;
        int c = getopt_long(argc, argv, "", long_options, &option_index);
        if (c == -1) {
            break;
        }
        switch (c) {
            case 0:
                if (!strcmp(long_options[option_index].name, "swapat")) {
                    if (sscanf(optarg, "%u", &swapat) != 1) {
                        fprintf(stderr, "unable to read pkt number\n");
                        pdb_swap_usage(*argv);
                        return -1;
                    }
                }
        }
    }

    pcapfilename = argv[argc - 1];
    pdb_name     = argv[argc - 2];

    return 0;
}

static void pdb_swap_play_with_result(struct qmdpi_worker *worker,
                                      struct qmdpi_result *result)
{
    char path_string[512];
    struct qmdpi_path *path = qmdpi_result_path_get(result);
    struct qmdpi_bundle *b = qmdpi_flow_bundle_get(qmdpi_result_flow_get(result));
    struct qmdpi_result_flags const *result_flags = qmdpi_result_flags_get(result);

    if (path == NULL || b == NULL) {
        return;
    }

    if(QMDPI_RESULT_FLAGS_CLASSIFIED_STATE_CHANGED(result_flags) ||
       QMDPI_RESULT_FLAGS_OFFLOADED_STATE_CHANGED(result_flags) ||
       QMDPI_RESULT_FLAGS_PDU_DIR_CHANGED(result_flags) ||
       QMDPI_RESULT_FLAGS_PATH_CHANGED(result_flags)) {
        qmdpi_data_path_to_buffer(b, path_string, sizeof(path_string), path);

        printf("Packet #%d processed, flow(%p) path=%s\n",
               packet_number,
               qmdpi_result_flow_get(result),
               path_string);
    }
}



static int pdb_swap_file_copy_to_buf(const char  *filename,
                                     char       **buf,
                                     int         *buf_len)
{
    FILE      *file = NULL;
    int32_t  r_len = 0;

    file = fopen(filename, "r");

    if (file == NULL) {
        return -1;
    }

    fseek(file, 0L, SEEK_END);
    *buf_len = ftell(file);
    fseek(file, 0L, SEEK_SET);

    if ((*buf = malloc(*buf_len)) == NULL) {
        fclose(file);
        return -1;
    }

    r_len = fread(*buf, sizeof(uint8_t), *buf_len, file);

    if (*buf_len != r_len) {
        free(*buf);
        fclose(file);
        return -1;
    }

    fclose(file);
    return 0;
}

static int pdb_swap_new_pdb_process(void)
{
    char             *pdb_buf     = NULL;
    int               pdb_buf_len = 0;
    int               err         = 0;
    struct qmdpi_pdb *pdb;

    if (pdb_swap_file_copy_to_buf(pdb_name, &pdb_buf,
                                  &pdb_buf_len)) {
        fprintf(stderr, "%s not found.\n", pdb_name);
        return -1;
    }

    /* load PDB */
    pdb = qmdpi_bundle_pdb_create_from_addr(bundle, pdb_buf, pdb_buf_len);
    if (pdb == NULL) {
        fprintf(stderr, "qmdpi_bundle_pdb_create_from_addr error.\n");
        free(pdb_buf);
        return -1;
    }

    free(pdb_buf);

    /* set PDB as active */
    if ((err = qmdpi_bundle_pdb_activate(pdb))) {
        fprintf(stderr, "qmdpi_bundle_pdb_activate error.\n");
        return -1;
    }

    printf("Loading new PDB (%s)...OK\n", pdb_name);

    if (qmdpi_bundle_signature_enable_all(bundle) < 0) {
        fprintf(stderr, "Error enabling all protocols.\n");
        return -1;
    }

    return 0;
}

static void pdb_swap_process(u_char *user, const struct pcap_pkthdr *phdr,
                             const u_char *pdata)
{

    struct qmdpi_result *result;
    int ret;

    packet_number++;

    /* swap bundles */
    if (packet_number == swapat) {
        /* Load second PDB */
        if (pdb_swap_new_pdb_process()) {
            exit(1);
        }
    }

    /* set PDU information to be processed by the worker */
    if (qmdpi_worker_pdu_set(worker, pdata, phdr->caplen, &phdr->ts,
                             QMDPI_PROTO_ETH, QMDPI_DIR_DEFAULT, 0) != 0) {
        return;
    }

    /* process packet with worker and provide DPI result */
    do {
        ret = qmdpi_worker_process(worker, NULL, &result);
        if (ret < 0) {
            fprintf(stderr, "DPI processing failure at packet #%u\n", packet_number);
            continue;
        }

        /* play with result */
        pdb_swap_play_with_result(worker, result);

    } while (ret == QMDPI_PROCESS_MORE);

    /* destroy expired flows */
    if (last_packet_ts < phdr->ts.tv_sec) {

        int nb_remaining = 100;
        do {
            ret = qmdpi_flow_expire_next(worker, &phdr->ts, &result);
            if (ret != 0) {
                break;
            }

            nb_remaining --;

        } while (nb_remaining);

        last_packet_ts = phdr->ts.tv_sec;
    }
}


static int pdb_swap_init(void)
{
    int ret = 0;

    /* create engine instance */
    engine = qmdpi_engine_create("injection_mode=packet;nb_workers=1;nb_flows=1000");
    if (engine == NULL) {
        fprintf(stderr, "cannot create engine instance\n");
        return -1;
    }

    /* create worker instance */
    worker = qmdpi_worker_create(engine);
    if (worker == NULL) {
        fprintf(stderr, "cannot create worker instance\n");
        goto error_engine;
    }

    /* create bundle instance */
    bundle = qmdpi_bundle_create_from_file(engine, NULL);
    if (bundle == NULL) {
        fprintf(stderr, "Cannot create bundle instance: %s\n",
                qmdpi_error_get_string(NULL, qmdpi_error_get()));
        goto error_worker;
    }

    /* enable all signatures on bundle */
    ret = qmdpi_bundle_signature_enable_all(bundle);
    if (ret < 0) {
        fprintf(stderr, "error enabling all protocols\n");
        goto error_bundle;
    }

    /* activate bundle */
    ret = qmdpi_bundle_activate(bundle);
    if (ret < 0) {
        fprintf(stderr, "cannot activate bundle\n");
        goto error_bundle;
    }

    return 0;

error_bundle:
    qmdpi_bundle_destroy(bundle);
error_worker:
    qmdpi_worker_destroy(worker);
error_engine:
    qmdpi_engine_destroy(engine);

    return -1;

}

int main(int argc, char **argv)
{
    char errbuf[PCAP_ERRBUF_SIZE];
    pcap_t *pcap;
    int ret;

    packet_number = 0;

    /* command line parsing */
    if (pdb_swap_parse_param(argc, argv) < 0) {
        return 1;
    }


    /* init Qosmos ixEngine */
    ret = pdb_swap_init();
    if (ret < 0) {
        fprintf(stderr, "cannot initialize application\n");
        return 1;
    }

    /* open pcap */
    pcap = pcap_open_offline(pcapfilename, errbuf);
    if (!pcap) {
        fprintf(stderr, "pcap_open: %s\n", errbuf);
        return 1;
    }

    /* DPI processing loop  */
    ret = pcap_loop(pcap, -1, pdb_swap_process, NULL);
    if (ret == -1) {
        fprintf(stderr, "error: pcap_loop\n");
        return 1;
    }

    printf("Cleaning up...\n");
    pcap_close(pcap);

    qmdpi_bundle_destroy(bundle);

    /* expire remaining flow contexts, do not care about processing result */
    while (qmdpi_flow_expire_next(worker, NULL, NULL) == 0);

    qmdpi_worker_destroy(worker);
    qmdpi_engine_destroy(engine);

    return 0;
}
