/*
 *   This file is a part of Qosmos ixEngine.
 *   Copyright  Qosmos 2000-2020 - All rights reserved
 */

/* standard headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>

#ifndef WIN32
#include <arpa/inet.h>
#else
#include <winsock2.h>
#endif

/* libpcap header for capture */
#include <pcap.h>

/* Qosmos ixEngine header */
#include <qmdpi.h>
/* Bundle API */
#include <qmdpi_bundle_api.h>

#include <packet_helper.h>
#include <sys_queue.h>

int link_mode = Q_PROTO_ETH;

/* Packet metadata */
struct pkt_signature {
    int l3proto;
    uint32_t l3lo; /* lowest addr */
    uint32_t l3hi; /* highest addr */
    int l4proto;
    uint16_t l4lo; /* lowest port */
    uint16_t l4hi; /* highest port */
};

struct pkt_metadata {
    struct pkt_signature sig;
    void const *l3hdr;
    void const *l4hdr;
    int dir_index;
};

/* Qosmos ixEngine main objects */
struct qmdpi_engine *engine = NULL;
struct qmdpi_worker *worker = NULL;
struct qmdpi_bundle *bundle = NULL;
struct qmdpi_result *result = NULL;

static uint64_t packet_number;

/* TDM option management */
int tdm_use = 0;

/* flow data structure */
struct flow {
    TAILQ_ENTRY(flow)  link;    /**< lru next/prev pointer  */
    LIST_ENTRY(flow)   next;    /**< hashtable next/prev pointer */

    struct qmdpi_flow *df;      /**< qmdpi_flow structure */

    unsigned char used  : 1;    /**< is our flow free   */
unsigned char dir0  :
    2;    /**< association between packet metadata direction index (0) and
                                  the direction (CTS / STC) */
unsigned char dir1  :
    2;    /**< association between packet metadata direction index (1) and
                                  the direction (CTS / STC)
                                         */
    unsigned char buffer[sizeof(struct pkt_metadata)];  /** pkt metadata */
};


/**
 * flow linked list
 */
struct flow_line {
    LIST_HEAD(, flow) head; /**< flow head linked list */
};

#define FLOW_HASHSZ  512

/**
 * flow data hashtable and descriptor
 */
struct flow_hash {
    struct flow_line lines[FLOW_HASHSZ]; /**< current used flow hashtables */

    TAILQ_HEAD(, flow) lru;      /**< LRU flow descriptor */
    struct flow *buffer;         /**< flow allocated */
};

/**
 * hashtable initialization
 */
static int flow_hash_create(struct flow_hash *h, unsigned  nr)
{
    unsigned int i;
    struct flow *w;
    memset(h, 0, sizeof(*h));

    /* allocate an array of nr flow object */
    h->buffer = (struct flow *)malloc(nr * sizeof(*h->buffer));
    if (h->buffer == NULL) {
        return QMDPI_ENOMEM;
    }

    memset(h->buffer, 0, nr * sizeof(*h->buffer));

    w = h->buffer;
    /* insert all element of flow array in the LRU list */
    for (i = 0; i < nr; w++, i++) {
        w->df = NULL;
        TAILQ_INSERT_HEAD(&h->lru, w, link);
    }
    return 0;
}

static int flow_hash_expire_next(struct flow_hash *h)
{
    unsigned int i;
    for (i = 0; i < sizeof(h->lines) / sizeof(*h->lines); ++i) {
        struct flow *w = LIST_FIRST(&h->lines[i].head);
        if (w) {
            LIST_REMOVE(w, next);
            return 0;
        }
    }
    return 1;
}

/**
 * when a new flow is needed, we remove the LRU head, and push it to the end
 */
static void result_display(struct qmdpi_result *result);

static void flow_hash_destroy(struct flow_hash *h)
{
    struct flow *f;
    struct qmdpi_result *result;
    while ((f = TAILQ_FIRST(&h->lru)) != NULL) {
        TAILQ_REMOVE(&h->lru, f, link);
        if (f->df) {
            qmdpi_flow_destroy(worker, f->df, &result);
            result_display(result);
        }
    }

    if (h->buffer) {
        free(h->buffer);
    }
}

/**
 * when a flow is retrieved, it is queued at the end of the LRU
 * in addition, we mark the current packet as a CLIENT or SERVER one
 */
static void flow_use(struct flow_hash *h, struct flow *r)
{
    r->used = 1;
    TAILQ_REMOVE(&h->lru, r, link);
    TAILQ_INSERT_TAIL(&h->lru, r, link);
}

/**
 * This function is called only at flow creation. Association between
 * dir_index and direction is registered
 * inside the flow
 */
static void flow_dir(struct flow *r, struct pkt_metadata *pmdata)
{
    struct qm_tcp_hdr const *hdr = (struct qm_tcp_hdr const *)pmdata->l4hdr;

    /* we try to determine connection type only on tcp flows */
    if (pmdata->sig.l4proto == 205 && hdr) {
        if (hdr->syn) {
            /* server */
            if (hdr->ack) {
                if (pmdata->dir_index == 0) {
                    r->dir0 = QMDPI_DIR_STC;
                    r->dir1 = QMDPI_DIR_CTS;
                } else {
                    r->dir0 = QMDPI_DIR_CTS;
                    r->dir1 = QMDPI_DIR_STC;
                }
            }
        }
    }
}

/**
 * Create a new flow
 */
static struct flow *flow_create(struct flow_hash *h,
                                struct pkt_metadata *pmdata)
{
    struct flow *r;

    r = TAILQ_FIRST(&h->lru);
    TAILQ_REMOVE(&h->lru, r, link);
    if (r->used) {
        LIST_REMOVE(r, next);
        struct qmdpi_flow *f = r->df;
        if (f) {
            struct qmdpi_result *result = NULL;
            qmdpi_flow_destroy(worker, f, &result);
            result_display(result);
            r->df = NULL;
        }
    }
    memset(r, 0, sizeof(*r));
    r->used = 1;
    if (pmdata->dir_index == 0) {
        r->dir0 = QMDPI_DIR_CTS;
        r->dir1 = QMDPI_DIR_STC;
    } else {
        r->dir0 = QMDPI_DIR_STC;
        r->dir1 = QMDPI_DIR_CTS;
    }
    flow_dir(r, pmdata);

    /*
     * If r->df is null here after qmdpi_flow_create(), we keep the flow in used
     * state in lru to simultate an offloaded flow
     */
    r->df = qmdpi_flow_create(worker, pmdata->sig.l3proto, pmdata->sig.l4proto,
                              (r->dir0 == QMDPI_DIR_CTS ? (void const *)&pmdata->sig.l3lo :
                               (void const *)&pmdata->sig.l3hi),
                              (r->dir0 == QMDPI_DIR_CTS ? (void const *)&pmdata->sig.l4lo :
                               (void const *)&pmdata->sig.l4hi),
                              (r->dir0 == QMDPI_DIR_CTS ? (void const *)&pmdata->sig.l3hi :
                               (void const *)&pmdata->sig.l3lo),
                              (r->dir0 == QMDPI_DIR_CTS ? (void const *)&pmdata->sig.l4hi :
                               (void const *)&pmdata->sig.l4lo));
    TAILQ_INSERT_TAIL(&h->lru, r, link);
    return r;
}

static struct flow *flow_new(struct flow_hash *h, struct pkt_metadata *pmdata)
{
    struct flow *res;
    uint32_t hashkey = ntohl(pmdata->sig.l3lo ^ pmdata->sig.l3hi);
    hashkey %= FLOW_HASHSZ;

    res = flow_create(h, pmdata);

    LIST_INSERT_HEAD(&h->lines[hashkey].head, res, next);
    //copy sig
    memcpy(res->buffer, (uint8_t const *)&pmdata->sig, sizeof(pmdata->sig));

    return res;
}

/**
 * find a flow according a PDU object
 */
static struct flow *flow_find(struct flow_hash *h, struct pkt_metadata *pmdata)
{
    struct flow *walker;

    uint32_t hashkey = ntohl(pmdata->sig.l3lo ^ pmdata->sig.l3hi);

    hashkey %= FLOW_HASHSZ;

    LIST_FOREACH(walker, &h->lines[hashkey].head, next) {
        //match sig
        if (memcmp(walker->buffer, (uint8_t const *)&pmdata->sig,
                   sizeof(pmdata->sig)) == 0) {
            flow_use(h, walker);
            return walker;
        }
    }

    return NULL;
}

/**
 * just skip the ethernet layer since this isn't part of flow identification
 * in our example
 */
static int packet_eth(unsigned char **pkt, unsigned int *len,
                      struct pkt_metadata *pmdata)
{
    struct qm_eth_hdr const *hdr = (struct qm_eth_hdr const *)*pkt;
    if (*len < sizeof(*hdr)) {
        return QMDPI_EINVAL;
    }
    if (hdr->h_proto == htons(QM_ETHERTYPE_IP)) {
        *pkt    += sizeof(*hdr);
        *len    -= sizeof(*hdr);
        pmdata->sig.l3proto = Q_PROTO_IP;
        return pmdata->sig.l3proto;
    } else if (hdr->h_proto == htons(QM_ETHERTYPE_ETH1Q)) {
        *pkt    += sizeof(*hdr);
        *len    -= sizeof(*hdr);
        pmdata->sig.l3proto = Q_PROTO_8021Q;
        return pmdata->sig.l3proto;
    }
    return QMDPI_EINVAL;
}

/* skip vlan header */
static int packet_eth1q(unsigned char **pkt, unsigned int *len,
                        struct pkt_metadata *pmdata)
{
    struct qm_eth1q_hdr const *hdr = (struct qm_eth1q_hdr const *)*pkt;
    if (*len < sizeof(struct qm_eth1q_hdr)) {
        return QMDPI_EINVAL;
    }
    if (hdr->h_proto == htons(QM_ETHERTYPE_IP)) {
        *pkt    += sizeof(*hdr);
        *len    -= sizeof(*hdr);
        pmdata->sig.l3proto = Q_PROTO_IP;
        return pmdata->sig.l3proto;
    }
    return QMDPI_EINVAL;
}

/* extract and store ip signature */
static int packet_ip(unsigned char **pkt,
                     unsigned int *len,
                     struct pkt_metadata *pmdata)
{
    struct qm_ip_hdr const *hdr = (struct qm_ip_hdr const *)*pkt;
    unsigned int iphdr_len, totlen = *len,  iptotlen;
    int _ipoff;
    int ip_protocol;

    if (*len < sizeof(*hdr)) {
        return QMDPI_EINVAL;
    }

    if (ntohl(hdr->saddr) < ntohl(hdr->daddr)) {
        pmdata->sig.l3lo = hdr->saddr;
        pmdata->sig.l3hi = hdr->daddr;
        pmdata->dir_index = 0;
    } else if (ntohl(hdr->saddr) > ntohl(hdr->daddr)) {
        pmdata->sig.l3lo = hdr->daddr;
        pmdata->sig.l3hi = hdr->saddr;
        pmdata->dir_index = 1;
    } else {
        pmdata->sig.l3lo = hdr->saddr;
        pmdata->sig.l3hi = hdr->daddr;
        pmdata->dir_index = -1;
    }

    pmdata->l3hdr = hdr;

    iphdr_len = hdr->ihl << 2;
    iptotlen = ntohs(hdr->tot_len);
    if ((iptotlen <= iphdr_len) || (*len <= iphdr_len)) {
        return QMDPI_EINVAL;
    }
    _ipoff = ntohs(hdr->frag_off);
    /* no fragmentation management */
    if (_ipoff & 0x3fff) {
        return QMDPI_EINVAL;
    }

    MAP_IPPROTO_TO_QPROTO(ip_protocol, hdr->protocol);
    if (ip_protocol > 0) {
        (*pkt) += iphdr_len;

        if (totlen < iptotlen) {
            (*len) -= iphdr_len;
        } else {
            (*len) = iptotlen - iphdr_len;
        }
        pmdata->sig.l4proto = ip_protocol;
        return pmdata->sig.l4proto;
    }

    return 0;
}

/* extract and store udp signature */
static int packet_tcp(unsigned char **pkt,
                      unsigned int *len,
                      struct pkt_metadata *pmdata)
{
    struct qm_tcp_hdr const *hdr = (struct qm_tcp_hdr const *)*pkt;

    if ((*len < sizeof(*hdr)) || ((int)*len < (hdr->doff << 2))) {
        return QMDPI_EINVAL;
    }
    if (pmdata->dir_index == 0) {
        pmdata->sig.l4lo = hdr->source;
        pmdata->sig.l4hi = hdr->dest;
    } else if (pmdata->dir_index == 1) {
        pmdata->sig.l4lo = hdr->dest;
        pmdata->sig.l4hi = hdr->source;
    } else {
        if (ntohs(hdr->source) < ntohs(hdr->dest)) {
            pmdata->sig.l4lo = hdr->source;
            pmdata->sig.l4hi = hdr->dest;
            pmdata->dir_index = 0;
        } else {
            pmdata->sig.l4lo = hdr->dest;
            pmdata->sig.l4hi = hdr->source;
            pmdata->dir_index = 1;
        }
    }

    pmdata->l4hdr = hdr;

    (*pkt) += (hdr->doff << 2);
    (*len) -= (hdr->doff << 2);
    return 0;
}

/* extract and store udp signature */
static int packet_udp(unsigned char **pkt,
                      unsigned int *len,
                      struct pkt_metadata *pmdata)
{
    struct qm_udp_hdr const *hdr = (struct qm_udp_hdr const *)*pkt;

    if (*len < sizeof(*hdr)) {
        return QMDPI_EINVAL;
    }

    if (pmdata->dir_index == 0) {
        pmdata->sig.l4lo = hdr->source;
        pmdata->sig.l4hi = hdr->dest;
    } else if (pmdata->dir_index == 1) {
        pmdata->sig.l4lo = hdr->dest;
        pmdata->sig.l4hi = hdr->source;
    } else {
        if (ntohs(hdr->source) < ntohs(hdr->dest)) {
            pmdata->sig.l4lo = hdr->source;
            pmdata->sig.l4hi = hdr->dest;
            pmdata->dir_index = 0;
        } else {
            pmdata->sig.l4lo = hdr->dest;
            pmdata->sig.l4hi = hdr->source;
            pmdata->dir_index = 1;
        }
    }
    pmdata->l4hdr = hdr;

    (*pkt) += sizeof(*hdr);
    (*len) -= sizeof(*hdr);
    return 0;
}

#define SCTP_TYPE_DATA 0x00

/* extract and store sctp signature only if the first chunk is a data type and not manage reinjection */
static int packet_sctp(unsigned char **pkt,
                       unsigned int *len,
                       struct pkt_metadata *pmdata)
{
    struct qm_sctp_hdr const *hdr = (struct qm_sctp_hdr const *)*pkt;

    if (*len < sizeof(*hdr) + sizeof(struct qm_sctp_chunk_hdr)) {
        return QMDPI_EINVAL;
    }

    if (pmdata->dir_index == 0) {
        pmdata->sig.l4lo = hdr->source;
        pmdata->sig.l4hi = hdr->dest;
    } else if (pmdata->dir_index == 1) {
        pmdata->sig.l4lo = hdr->dest;
        pmdata->sig.l4hi = hdr->source;
    } else {
        if (ntohs(hdr->source) < ntohs(hdr->dest)) {
            pmdata->sig.l4lo = hdr->source;
            pmdata->sig.l4hi = hdr->dest;
            pmdata->dir_index = 0;
        } else {
            pmdata->sig.l4lo = hdr->source;
            pmdata->sig.l4hi = hdr->dest;
            pmdata->dir_index = 1;
        }
    }
    pmdata->l4hdr = hdr;

    (*pkt) += sizeof(*hdr);
    (*len) -= sizeof(*hdr);

    while (sizeof(struct qm_sctp_chunk_hdr) <= (*len)) {
        struct qm_sctp_chunk_hdr const *chunk_hdr = (struct qm_sctp_chunk_hdr const *)(
                                                        *pkt);

        unsigned short chunk_hdr_len = ntohs(chunk_hdr->len);
        if (chunk_hdr_len > *len || chunk_hdr_len == 0) {
            return QMDPI_EINVAL;
        }

        if (chunk_hdr->type != SCTP_TYPE_DATA) {
            (*pkt) += chunk_hdr_len;
            (*len) -= chunk_hdr_len;
        } else {
            if (*len < sizeof(struct qm_sctp_chunk_hdr) + sizeof(struct
                                                                 qm_sctp_chunk_data)) {
                return QMDPI_EINVAL;
            }

            (*pkt) += sizeof(*chunk_hdr) + sizeof(struct qm_sctp_chunk_data);
            (*len) = chunk_hdr_len - sizeof(*chunk_hdr) - sizeof(struct qm_sctp_chunk_data);
            return 0;
        }
    }
    return 0;
}

/* header processing : extract signature (N-tuple) and store it in the dpi_flow_sig */
static int packet_in(unsigned char **pkt,
                     unsigned int *len, struct pkt_metadata *pmdata, int link_mode)
{
    pmdata->sig.l3proto = link_mode;

    while (link_mode > 0) {
        switch (link_mode) {
            case Q_PROTO_ETH:
                link_mode = packet_eth(pkt, len, pmdata);
                break;
            case Q_PROTO_8021Q:
                link_mode = packet_eth1q(pkt, len, pmdata);
                break;
            case Q_PROTO_IP:
                link_mode = packet_ip(pkt, len, pmdata);
                break;
            case Q_PROTO_UDP:
                link_mode = packet_udp(pkt, len, pmdata);
                break;
            case Q_PROTO_TCP:
                link_mode = packet_tcp(pkt, len, pmdata);
                break;
            case Q_PROTO_SCTP:
                link_mode = packet_sctp(pkt, len, pmdata);
                break;
            default:
                link_mode = 0;
                break;
        }
    }
    return link_mode;
}

/* open pcap file */
static inline pcap_t *si_pcap_open(const char filename[])
{
    pcap_t *fd;
    char errbuf[2048];

    fd = pcap_open_offline(filename, errbuf);
    if (fd == NULL) {
        fprintf(stderr, "cannot open pcap file: %s\n", errbuf);
        return NULL;
    }
    return fd;
}

/* close pcap file */
static inline void si_pcap_close(pcap_t *p)
{
    if (p) {
        pcap_close(p);
    }
}

/* initialize Qosmos ixEngine */
static int engine_init(struct flow_hash *h)
{
    int ret = 0;

    /* create engine instance */
    engine = qmdpi_engine_create("injection_mode=stream;nb_workers=1");
    if (engine == NULL) {
        fprintf(stderr, "cannot create engine instance\n");
        goto error_license;
    }

    /* create worker instance */
    worker = qmdpi_worker_create(engine);
    if (worker == NULL) {
        fprintf(stderr, "cannot create worker instance: %s\n",
                qmdpi_error_get_string(NULL, qmdpi_error_get()));
        goto error_engine;

    }

    /* create bundle instance */
    bundle = qmdpi_bundle_create_from_file(engine, NULL);
    if (bundle == NULL) {
        fprintf(stderr, "cannot create bundle instance: %s\n",
                qmdpi_error_get_string(NULL, qmdpi_error_get()));
        goto error_worker;
    }

    /* enable all signatures on bundle */
    ret = qmdpi_bundle_signature_enable_all(bundle);
    if (ret < 0) {
        fprintf(stderr, "error enabling all protocols: %s\n",
                qmdpi_error_get_string(bundle, ret));
        goto error_bundle;
    }

    /* activate bundle */
    ret = qmdpi_bundle_activate(bundle);
    if (ret < 0) {
        fprintf(stderr, "cannot activate bundle: %s\n", qmdpi_error_get_string(bundle,
                                                                               ret));
        goto error_bundle;

    }

    /* create flow hashtable */
    if (flow_hash_create(h, FLOW_HASHSZ)) {
        goto error_bundle;
    }

    /* init tdm */
    if (tdm_use) {
        if ((ret = qmdpi_report_init(engine, NULL))) {
            fprintf(stderr, "tdm report init error: %s\n",
                    qmdpi_error_get_string(bundle, ret));
            goto error_hash;
        }
        if ((ret = qmdpi_report_start(engine))) {
            fprintf(stderr, "tdm report start error: %s\n",
                    qmdpi_error_get_string(bundle, ret));
            goto error_report;
        }
    }

    return 0;

error_report:
    qmdpi_report_cleanup(engine);
error_hash:
    flow_hash_destroy(h);
error_bundle:
    qmdpi_bundle_destroy(bundle);
error_worker:
    qmdpi_worker_destroy(worker);
error_engine:
    qmdpi_engine_destroy(engine);
error_license:
    qmdpi_license_destroy();

    return -1;
}

static void engine_exit(pcap_t *p, struct flow_hash *h)
{
    si_pcap_close(p);

    while (flow_hash_expire_next(h) == 0);
    flow_hash_destroy(h);

    if (tdm_use) {
        int ret;
        if ((ret = qmdpi_report_stop(engine))) {
            fprintf(stderr, "tdm report stop error: %s\n",
                    qmdpi_error_get_string(bundle, ret));
        }

        if ((ret = qmdpi_report_cleanup(engine))) {
            fprintf(stderr, "tdm report cleanup error: %s\n",
                    qmdpi_error_get_string(bundle, ret));
        }
    }

    qmdpi_worker_destroy(worker);
    qmdpi_bundle_destroy(bundle);
    qmdpi_engine_destroy(engine);
    qmdpi_license_destroy();
}

#define TDM_EXTENSION_LEN 32
static void result_display(struct qmdpi_result *result)
{
    struct qmdpi_flow *flow = qmdpi_result_flow_get(result);

    struct qmdpi_engine *engine = qmdpi_worker_engine_get(worker);
    struct qmdpi_bundle *b = qmdpi_flow_bundle_get(flow);

    struct qmdpi_result_flags const *result_flags = qmdpi_result_flags_get(result);

    /* print classification only if protocol path has changed
     * or if an extension lookup can be performed using tdm
     */
    int tdm_extension_lookup = tdm_use &&
                               QMDPI_RESULT_FLAGS_NEED_CLASSIF_EXT(result_flags);

    /* print classification only if protocol path has changed */
    if (QMDPI_RESULT_FLAGS_PATH_CHANGED(result_flags) || (tdm_extension_lookup)) {
        char buffer[4096];
        struct qmdpi_path *path = qmdpi_result_path_get(result);
        if (path) {
            qmdpi_data_path_to_buffer(b, buffer, sizeof(buffer), path);
            printf("%d/%d %"PRIu64" f(%"PRIu64")/p/path=%s\n",
                   qmdpi_engine_id_get(engine),
                   qmdpi_worker_id_get(worker),
                   packet_number,
                   qmdpi_flow_id_get(flow),
                   buffer);
        }

        if (tdm_extension_lookup) {
            char tdm_ext[TDM_EXTENSION_LEN];
            tdm_ext[0] = '\0';
            int res_lookup = qmdpi_flow_classif_ext_get(flow, tdm_ext, TDM_EXTENSION_LEN);
            if (res_lookup == QMDPI_EOVERFLOW) {
                printf("tdm truncated result, a buffer size of %d is too small\n",
                       TDM_EXTENSION_LEN);
            } else if (res_lookup > 0) {
                printf("%d/%d %"PRIu64" f(%"PRIu64")/p/tdm_extended_path=%s.%.*s\n",
                       qmdpi_engine_id_get(engine),
                       qmdpi_worker_id_get(worker),
                       packet_number,
                       qmdpi_flow_id_get(flow),
                       buffer,
                       res_lookup,
                       tdm_ext);
            } else {
                /* errors */
                fprintf(stderr, "cannot get tdm classification extension: %s\n",
                        qmdpi_error_get_string(NULL, res_lookup));
            }
        }
    }
}

static void process(u_char *user, const struct pcap_pkthdr *phdr, u_char *pdata)
{
    int ret;
    struct flow_hash *h = (struct flow_hash *)user;
    struct flow *flow;
    unsigned int len = phdr->caplen;
    struct pkt_metadata pmdata;

    memset(&pmdata, 0, sizeof(pmdata));

    /****** Step 1 - compute packet signature ******/
    ret = packet_in(&pdata, &len, &pmdata, link_mode);
    if (ret < 0) {
        fprintf(stderr, "cannot parse packet number %"PRIu64"\n", packet_number);
        return;
    }

    /****** Step 2 - flow lookup ******/
    flow = flow_find(h, &pmdata);
    //XXX if new flow, set flow 5 tuple once for all (cannot be set any more afterwards)
    //important: it is up to the user to manage his flows and make sure he uses the same flow
    //context for both directions
    if (flow == NULL) {
        flow = flow_new(h, &pmdata);
        if (flow == NULL) {
            fprintf(stderr, "Can't allocate new flow\n");
            return;
        }
    }

    /* do not feed DPI if flow is offloaded */
    if (flow->df == NULL) {
        printf("do not feed DPI if flow is offloaded\n");
        return;
    }

    struct qmdpi_flow_info *info = qmdpi_flow_info_get(flow->df);

    /* do not feed DPI engine if payload is null or if flow is to be offloaded */
    if (len == 0 || QMDPI_FLOW_INFO_OFFLOADED(info)) {
        return;
    }

    int dir_index = pmdata.dir_index == 0 ? flow->dir0 : flow->dir1;

    /****** Step 3 - feed PDU object with packet data and metadata ******/
    if (qmdpi_worker_pdu_set(worker, pdata, len, &phdr->ts, 0, dir_index, 0) != 0) {
        return;
    }

    /****** Step 4: Set l3_header and l4_header ******/
    if (qmdpi_worker_pdu_header_set(worker, (void *) pmdata.l3hdr,
                                    (void *) pmdata.l4hdr) != 0) {
        return;
    }

    /****** Step 5: Packet processing ******/
    ret = qmdpi_worker_process(worker, flow->df, &result);
    if (ret < 0) {
        printf("DPI processing failure at packet #%"PRIu64"\n", packet_number);
        return;
    }

    result_display(result);

    if (QMDPI_RESULT_FLAGS_OFFLOADED_STATE(qmdpi_result_flags_get(result))) {
        qmdpi_flow_offload(worker, flow->df, &result);
        result_display(result);
    }
}

static void print_usage(const char *argv)
{
    fprintf(stderr, "Usage: %s [option] <pcap_file>\n", argv);
    fprintf(stderr, "options:\n");
    fprintf(stderr,
            "\t--tdm-classif-ext : activate tdm and give classification extension on unknown protocols if they exist into tdm tables\n"
            "\t--link-mode value: specify the trace link-mode, value is 'eth' or 'ip'\n");
}

int main(int argc, const char **argv)
{
    pcap_t *pcap;
    uint8_t *data;
    struct pcap_pkthdr pkthdr;
    struct flow_hash h;
    int ret;
    int ac;
    const char **p;


    /* check arguments */
    if (argc < 2 || argc > 5) {
        print_usage(*argv);
        return 1;
    }

    for (ac = argc - 1, p = argv + 1; ac > 0; --ac, ++p) {
        if (strcmp(*p, "--tdm-classif-ext") == 0) {
            tdm_use = 1;
        } else if (strcmp(*p, "--link-mode") == 0) {
            --ac;
            if (ac == 0) {
                print_usage(*argv);
                return 1;
            }
            ++p;
            if (strcmp(*p, "eth") == 0) {
                link_mode = Q_PROTO_ETH;
            } else if (strcmp(*p, "ip") == 0) {
                link_mode = Q_PROTO_IP;
            } else {
                print_usage(*argv);
                return 1;
            }
        } else {
            break;
        }
    }

    if (ac == 0) {
        print_usage(*argv);
        return 1;
    }

    /* check pcap */
    pcap = si_pcap_open(*p);

    if (pcap == NULL) {
        return 1;
    }

    /* init Qosmos ixEngine */
    ret = engine_init(&h);
    if (ret < 0) {
        si_pcap_close(pcap);
        return 1;
    }

    while ((data = (uint8_t *)pcap_next(pcap, &pkthdr))) {
        packet_number++;
        /* DPI processing routine */
        process((u_char *)&h, &pkthdr, data);
    }

    engine_exit(pcap, &h);

    return 0;
}
