/*
*   This file is a part of Qosmos ixEngine.
*   Copyright  Qosmos 2000-2016 - All rights reserved
*/

/* standard headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>

/* libpcap header for capture */
#include <pcap.h>

/* Qosmos ixEngine headers */
#include <qmdpi.h>
#include <qmdpi_bundle_api.h>
#include <sys_queue.h>

static uint64_t packet_number;

static struct qmdpi_engine *engine;
static struct qmdpi_worker *worker;
static struct qmdpi_bundle *bundle;


struct entry {
    char *buf;
    LIST_ENTRY(entry) next;
};

/* struct that will store email */
struct email {
    char *date;
    char *subject;
    LIST_HEAD(, entry) senders;
    LIST_HEAD(, entry) receivers;
};

static struct email *sa_imap_email_new(void)
{
    struct email *email = (struct email *)malloc(sizeof(struct email));
    if (email == NULL) {
        return NULL;
    }
    memset(email, 0, sizeof(*email));
    LIST_INIT(&email->senders);
    LIST_INIT(&email->receivers);

    return email;

}

static void sa_imap_email_free(struct email *email)
{
    struct entry *entry;

    while ((entry = LIST_FIRST(&email->senders)) != NULL) {
        LIST_REMOVE(entry, next);
        free(entry->buf);
        free(entry);
    }
    while ((entry = LIST_FIRST(&email->receivers)) != NULL) {
        LIST_REMOVE(entry, next);
        free(entry->buf);
        free(entry);
    }
    free(email->date);
    free(email->subject);
    free(email);
}

/* dump email message header */
static void sa_imap_email_print(struct email *email)
{
    struct entry *entry;

    fprintf(stdout, "Email message:\n");
    fprintf(stdout, " Date:%s\n", (email->date != NULL) ? email->date : "(null)");
    fprintf(stdout, " Subject:%s\n",
            (email->subject != NULL) ? email->subject : "(null)");
    fprintf(stdout, " Sender:\n");
    LIST_FOREACH(entry, &email->senders, next) {
        fprintf(stdout, "\t%s\n", entry->buf);
    }
    fprintf(stdout, " Receiver:\n");
    LIST_FOREACH(entry, &email->receivers, next) {
        fprintf(stdout, "\t%s\n", entry->buf);
    }
}

/* contruct email message header */
static void sa_imap_email_process(struct qmdpi_flow *flow,
                                  int attr_id,
                                  char const *data,
                                  int datalen,
                                  int flags)
{
    struct email *email = (struct email *)qmdpi_flow_user_handle_get(flow);
    struct entry *entry;

    /* handle start of email structured attribute */
    if (attr_id == Q_MPA_EMAIL && QMDPI_ATTR_PARENT_START & flags) {
        email = sa_imap_email_new();
        if (email == NULL) {
            return;
        }
        qmdpi_flow_user_handle_set(flow, email);

        return;
    }

    if (email == NULL) {
        return;
    }

    /* handle other cases */
    switch (attr_id) {
        case Q_MPA_EMAIL:
            if (QMDPI_ATTR_PARENT_END & flags) {
                /* end of an email structured attribute: dump email struct and reset it*/
                sa_imap_email_print(email);
                sa_imap_email_free(email);
                qmdpi_flow_user_handle_set(flow, NULL);
            }
            break;

        case Q_MPA_DATE:
            email->date = (char *)malloc(datalen * sizeof(*email->date) + 1);
            strncpy(email->date, data, datalen);
            email->date[datalen] = '\0';
            break;

        case Q_MPA_SUBJECT:
            email->subject = (char *)malloc(datalen * sizeof(*email->subject) + 1);
            strncpy(email->subject, data, datalen);
            email->subject[datalen] = '\0';
            break;

        case Q_MPA_SENDER:
            entry = (struct entry *)malloc(sizeof(struct entry));
            if (entry == NULL) {
                return;
            }
            entry->buf = (char *)malloc(datalen * sizeof(*entry->buf) + 1);
            strncpy(entry->buf, data, datalen);
            entry->buf[datalen] = '\0';
            LIST_INSERT_HEAD(&email->senders, entry, next);
            break;

        case Q_MPA_RECEIVER:
            entry = (struct entry *)malloc(sizeof(struct entry));
            if (entry == NULL) {
                return;
            }
            entry->buf = (char *)malloc(datalen * sizeof(*entry->buf) + 1);
            strncpy(entry->buf, data, datalen);
            entry->buf[datalen] = '\0';
            LIST_INSERT_HEAD(&email->receivers, entry, next);
            break;
    }
}


static void sa_imap_result(struct qmdpi_result *result)
{
    char const *data;
    int datalen;
    int proto_id;
    int attr_id;
    int flags;

    struct qmdpi_flow *flow = qmdpi_result_flow_get(result);

    while (qmdpi_result_attr_getnext(result, &proto_id, &attr_id,
                                     &data, &datalen, &flags) == 0) {
        sa_imap_email_process(flow, attr_id, data, datalen, flags);
    }
}

/* open pcap file */
static pcap_t *sa_pcap_open(const char filename[])
{
    pcap_t *fd;
    char errbuf[PCAP_ERRBUF_SIZE];
    fd = pcap_open_offline(filename, errbuf);
    if (0 == fd) {
        fprintf(stderr, "cannot open pcap file: %s\n", errbuf);
        return NULL;
    }
    return fd;
}

/* close pcap file */
static void sa_pcap_close(pcap_t *p)
{
    if (p) {
        pcap_close(p);
    }
}

struct imap_proto_attr {
    char *proto;
    char *attr;
};

static struct imap_proto_attr imap_proto_attr[] = {
    { "imap", "email" },
    { "imap", "end" },
    { "imap", "subject" },
    { "imap", "date" },
    { "imap", "sender" },
    { "imap", "receiver" },
    { NULL, NULL },
};

static int sa_imap_metadata_add(void)
{
    int i;

    for (i = 0; imap_proto_attr[i].proto; i++) {
        if (qmdpi_bundle_attr_register(bundle,
                                       imap_proto_attr[i].proto,
                                       imap_proto_attr[i].attr) < 0) {
            fprintf(stderr, "cannot add metadata %s:%s\n",
                    imap_proto_attr[i].proto,
                    imap_proto_attr[i].attr);
            return -1;
        }
    }

    return 0;
}

__attribute__((unused))
static void sa_imap_metadata_del(void)
{
    int i;

    for (i = 0; imap_proto_attr[i].proto; i++) {
        if (qmdpi_bundle_attr_unregister(bundle,
                                         imap_proto_attr[i].proto,
                                         imap_proto_attr[i].attr) < 0) {
            fprintf(stderr, "cannot remove metadata %s:%s\n",
                    imap_proto_attr[i].proto,
                    imap_proto_attr[i].attr);
        }
    }
}

static int sa_engine_init(void)
{
    int ret = 0;

    /* create engine instance */
    engine = qmdpi_engine_create("injection_mode=packet;nb_workers=1;nb_flows=1000");
    if (engine == NULL) {
        fprintf(stderr, "cannot create engine instance: %s\n",
                qmdpi_error_get_string(NULL, qmdpi_error_get()));
        return -1;
    }

    /* create worker instance */
    worker = qmdpi_worker_create(engine);
    if (worker == NULL) {
        fprintf(stderr, "cannot create worker instance: %s\n",
                qmdpi_error_get_string(NULL, qmdpi_error_get()));
        goto error_engine;

    }

    /* create bundle instance */
    bundle = qmdpi_bundle_create_from_file(engine, NULL);
    if (bundle == NULL) {
        fprintf(stderr, "cannot create bundle instance: %s\n",
                qmdpi_error_get_string(NULL, qmdpi_error_get()));
        goto error_worker;
    }

    /* activate bundle */
    ret = qmdpi_bundle_activate(bundle);
    if (ret < 0) {
        fprintf(stderr, "cannot activate bundle: %s\n", qmdpi_error_get_string(bundle,
                                                                               ret));
        goto error_bundle;

    }

    /* enable all signatures on bundle */
    ret = qmdpi_bundle_signature_enable_all(bundle);
    if (ret < 0) {
        fprintf(stderr, "error enabling all protocols: %s\n",
                qmdpi_error_get_string(bundle, ret));
        goto error_bundle;
    }

    if (sa_imap_metadata_add() < 0) {
        goto error_bundle;
    }
    return 0;

error_bundle:
    qmdpi_bundle_destroy(bundle);
error_worker:
    qmdpi_worker_destroy(worker);
error_engine:
    qmdpi_engine_destroy(engine);

    return -1;
}

static void sa_engine_exit(void)
{
    struct qmdpi_result *result;

    while (qmdpi_flow_expire_next(worker, NULL, &result) == 0) {
        sa_imap_result(result);
    }

    sa_imap_metadata_del();

    qmdpi_bundle_destroy(bundle);
    qmdpi_worker_destroy(worker);
    qmdpi_engine_destroy(engine);
    qmdpi_license_destroy();
}

static void sa_process(u_char *user,
                       const struct pcap_pkthdr *phdr,
                       const u_char *pdata)
{
    struct qmdpi_result *result;
    int ret;

    (void)user;
    packet_number++;

    /* set PDU information to be processed by the worker */
    if (qmdpi_worker_pdu_set(worker, pdata, phdr->caplen, &phdr->ts,
                             QMDPI_PROTO_ETH, QMDPI_DIR_DEFAULT, 0) != 0) {
        return;
    }

    /* process packet with worker and provide DPI result */
    do {
        ret = qmdpi_worker_process(worker, NULL, &result);
        if (ret < 0) {
            fprintf(stderr, "DPI processing failure at packet #%"PRIu64"\n", packet_number);
        } else {
            sa_imap_result(result);
        }
        /* loop if the packet contains multiple flows */
    } while (ret == QMDPI_PROCESS_MORE);

    if ((packet_number % 1000) == 0) {
        /* destroy expired flows and provide DPI result for each of them */
        int nb_remaining = 100;
        do {
            if (qmdpi_flow_expire_next(worker, &phdr->ts, &result) != 0) {
                break;
            }

            nb_remaining --;
            /* play with result */
            sa_imap_result(result);
        } while (nb_remaining);
    }
}

int main(int argc, const char **argv)
{
    pcap_t *pcap;
    int ret;

    packet_number = 0;


    /* check arguments */
    if (argc != 2) {
        fprintf(stderr, "Usage: %s <pcap_file>\n", *argv);
        return 1;
    }

    /* check pcap */
    pcap = sa_pcap_open(argv[1]);
    if (pcap == NULL) {
        return 1;
    }

    /* init Qosmos ixEngine */
    ret = sa_engine_init();
    if (ret < 0) {
        return 1;
    }

    ret = pcap_loop(pcap, -1, sa_process, NULL);
    if (ret == -1) {
        fprintf(stderr, "error: pcap_loop\n");
        sa_engine_exit();
        sa_pcap_close(pcap);
        return 1;
    }

    sa_engine_exit();
    sa_pcap_close(pcap);

    return 0;
}

