/*
  This file is a part of Qosmos ixEngine.

   Copyright  Qosmos 2022 - All rights reserved

  This computer program and all its components are protected by
  authors' rights and copyright law and by international treaties.
  Any representation, reproduction, distribution or modification
  of this program or any portion of it is forbidden without
  Qosmos explicit and written agreement and may result in severe
  civil and criminal penalties, and will be prosecuted
  to the maximum extent possible under the law.
*/

#include <vnet/plugin/plugin.h>
#include <vppinfra/bihash_8_8.h>
#include <vppinfra/dlist.h>
#include <vppinfra/pool.h>
#include <vppinfra/types.h>
#include <vppinfra/vec.h>

#include "flowtable.h"
#include <vnet/plugin/plugin.h>

#define MAX_NB_WORKERS 48
flowtable_main_t flowtable_main;

int
flowtable_enable_disable(flowtable_main_t *fm,
                         u32 sw_if_index, u8 enable_disable)
{
    u32 node_index = enable_disable ? fm->flowtable_index : ~0;

    return vnet_hw_interface_rx_redirect_to_node(fm->vnet_main,
                                                 sw_if_index, node_index);
}

#define FLOWTABLE_ADD_DESC "flowtable add [interface <Interface>] [next-node <Node>]"
static clib_error_t *
flowtable_add_command_fn(vlib_main_t *vm,
                         unformat_input_t *input,
                         vlib_cli_command_t *cmd)
{
    flowtable_main_t *fm = &flowtable_main;
    unformat_input_t _line_input, *line_input = &_line_input;
    u32 sw_if_index = ~0;
    u32 next_node_index = ~0;
    unformat_input_t sub_input = { 0 };
    int rv = 0;

    /* Get a line of input. */
    if (!unformat_user(input, unformat_line_input, line_input)) {
        return clib_error_return(0,
                                 "Invalid argument\nUsage:\n\t"FLOWTABLE_ADD_DESC"\n");
    }

    while (unformat_check_input(line_input) != UNFORMAT_END_OF_INPUT) {
        if (unformat(line_input, "interface %U", unformat_vnet_sw_interface,
                     fm->vnet_main, &sw_if_index)) {
            continue;
        }
        if (unformat(line_input, "interface %U", unformat_input, &sub_input)) {
            return clib_error_create("Invalid interface '%U'", format_unformat_error,
                                     &sub_input);
        } else if (unformat(line_input, "next-node %U", unformat_vlib_node,
                            fm->vlib_main, &next_node_index)) {
            continue;
        } else {
            return clib_error_return(0,
                                     "Invalid argument\nUsage:\n\t"FLOWTABLE_ADD_DESC"\n");
        }
    }

    if (sw_if_index == ~0 && next_node_index == ~0) {
        return clib_error_return(0,
                                 "Invalid argument\nUsage:\n\t"FLOWTABLE_ADD_DESC"\n");
    }

    if (sw_if_index != ~0) {
        rv = flowtable_enable_disable(fm, sw_if_index, 1);

        switch (rv) {
            case 0:
                break;
            case VNET_API_ERROR_INVALID_SW_IF_INDEX:
                return clib_error_return(0, "Invalid interface");
            case VNET_API_ERROR_UNIMPLEMENTED:
                return clib_error_return(0,
                                         "Device driver doesn't support redirection");
            default:
                return clib_error_return(0, "flowtable_enable_disable returned %d",
                                         rv);
        }
    }

    /* by default, leave the packet follow its course */
    if (next_node_index != ~0) {
        u32 next_index = ~0;

        next_index = vlib_node_get_next(fm->vlib_main,
                                        fm->flowtable_index,
                                        next_node_index);
        if (next_index == (~0)) /* not found */
            next_index = vlib_node_add_next(fm->vlib_main,
                                            fm->flowtable_index,
                                            next_node_index);
        fm->next_node_index = next_index;
    }

    return 0;
}

#define FLOWTABLE_DEL_DESC "flowtable delete interface <Interface>"
static clib_error_t *
flowtable_delete_command_fn(vlib_main_t *vm,
                            unformat_input_t *input,
                            vlib_cli_command_t *cmd)
{
    flowtable_main_t *fm = &flowtable_main;
    unformat_input_t _line_input, *line_input = &_line_input;
    u32 sw_if_index = ~0;
    unformat_input_t sub_input = { 0 };
    int rv = 0;

    /* Get a line of input. */
    if (!unformat_user(input, unformat_line_input, line_input)) {
        return clib_error_return(0,
                                 "Invalid argument\nUsage:\n\t"FLOWTABLE_DEL_DESC"\n");
    }

    while (unformat_check_input(line_input) != UNFORMAT_END_OF_INPUT) {
        if (unformat(line_input, "interface %U", unformat_vnet_sw_interface,
                     fm->vnet_main, &sw_if_index)) {
            continue;
        }
        if (unformat(line_input, "interface %U", unformat_input, &sub_input)) {
            return clib_error_create("Invalid interface '%U'", format_unformat_error,
                                     &sub_input);
        } else {
            return clib_error_return(0,
                                     "Invalid argument\nUsage:\n\t"FLOWTABLE_DEL_DESC"\n");
        }
    }

    if (sw_if_index == ~0) {
        return clib_error_return(0,
                                 "Invalid argument\nUsage:\n\t"FLOWTABLE_DEL_DESC"\n");
    }

    rv = flowtable_enable_disable(fm, sw_if_index, 0);

    switch (rv) {
        case 0:
            break;
        case VNET_API_ERROR_INVALID_SW_IF_INDEX:
            return clib_error_return(0, "Invalid interface");
        case VNET_API_ERROR_UNIMPLEMENTED:
            return clib_error_return(0,
                                     "Device driver doesn't support redirection");
        default:
            return clib_error_return(0, "flowtable_enable_disable returned %d",
                                     rv);
    }

    return 0;
}

#if (defined DEBUG) && (DEBUG >= 1)
#define FLOWTABLE_LOG_DESC "flowtable log [enable|disable]"
static clib_error_t *
flowtable_log_command_fn(vlib_main_t          *vm,
                         unformat_input_t     *input,
                         vlib_cli_command_t   *cmd)
{
    while (unformat_check_input(input) != UNFORMAT_END_OF_INPUT) {
        if (unformat(input, "enable")) {
            flowtable_main.log_enable = 1;
        } else if (unformat(input, "disable")) {
            flowtable_main.log_enable = 0;
        } else {
            break;
        }
    }

    if (flowtable_main.log_enable) {
        ft_print(vm, "Flowtable logging is enabled\n");
    } else {
        ft_print(vm, "Flowtable logging is disabled\n");
    }
    return 0;
}
#endif /* DEBUG */

#define FLOWTABLE_DPI_DESC "flowtable dpi [enable|disable]"
static clib_error_t *
flowtable_dpi_command_fn(vlib_main_t          *vm,
                         unformat_input_t     *input,
                         vlib_cli_command_t   *cmd)
{
    while (unformat_check_input(input) != UNFORMAT_END_OF_INPUT) {
        if (unformat(input, "enable")) {
            flowtable_main.dpi_enable = 1;
        } else if (unformat(input, "disable")) {
            flowtable_main.dpi_enable = 0;
        } else {
            break;
        }
    }

    if (flowtable_main.dpi_enable) {
        ft_print(vm, "Flowtable forwarding to dpi is enabled\n");
    } else {
        ft_print(vm, "Flowtable forwarding to dpi is disabled\n");
    }
    return 0;
}

#define FLOWTABLE_STATS_DESC "flowtable stats [enable|disable]"
static clib_error_t *
flowtable_stats_command_fn(vlib_main_t          *vm,
                           unformat_input_t     *input,
                           vlib_cli_command_t   *cmd)
{
    while (unformat_check_input(input) != UNFORMAT_END_OF_INPUT) {
        if (unformat(input, "enable")) {
            flowtable_main.stats_enable = 1;
            flowtable_stats_reset(vm, &flowtable_main);
        } else if (unformat(input, "disable")) {
            flowtable_main.stats_enable = 0;
        } else {
            break;
        }
    }

    if (flowtable_main.stats_enable) {
        ft_print(vm, "Flowtable stats are enabled\n");
    } else {
        ft_print(vm, "Flowtable stats are disabled\n");
    }
    return NULL;
}

#define FLOWTABLE_SHOW_STATS_DESC "flowtable show-stats"
static clib_error_t *
flowtable_show_stats_command_fn(vlib_main_t     *vm,
                                unformat_input_t     *input,
                                vlib_cli_command_t   *cmd)
{
    if (!flowtable_main.stats_enable)
        return clib_error_return(0, "Stats are disabled. Run "
                                 "'flowtable stats enable' to "
                                 "start aggregating data.");

    return flowtable_stats_show(vm, &flowtable_main);
}

static u32
get_ht_size(BVT(clib_bihash) *h)
{
    u32 active_buckets = 0;
    BVT(clib_bihash_bucket) * b;
    int i;

    for (i = 0; i < h->nbuckets; i++) {
        b = BV(clib_bihash_get_bucket)(h, i);
        if (BV(clib_bihash_bucket_is_empty)(b)) {
            continue;
        }
        active_buckets++;
    }
    return active_buckets;
}

#define FLOWTABLE_INFO_DESC "flowtable info"
static clib_error_t *
flowtable_info_command_fn(vlib_main_t          *vm,
                          unformat_input_t     *input,
                          vlib_cli_command_t   *cmd)
{
    u32 cpu_index;
    vlib_thread_main_t *tm = vlib_get_thread_main();
    flowtable_main_t *fm = &flowtable_main;
    flowtable_main_per_cpu_t *fmt;
    vlib_node_t *node;

    (void)input;
    (void)cmd;

    ft_print(vm,
             "The flowtable plugin uses ixEngine flow-level API to construct a stateful \n"
             "flow management functionality in a VPP node.");

    ft_print(vm, "\n");

    ft_print(vm, "Flowtable plugin version: %s\n", FLOWTABLE_PLUGIN_VER);

    ft_print(vm, "\n");

    ft_print(vm, "Hash table sizes:\n");
    for (cpu_index = 0; cpu_index < tm->n_vlib_mains; cpu_index++) {
        fmt = &fm->per_cpu[cpu_index];
        ft_print(vm, "Thread %d: buckets %d, elements %d\n", cpu_index,
                 get_ht_size(&fmt->flows_ht),
                 vec_len(fmt->ht_lines));
    }

    ft_print(vm, "\n");

    ft_print(vm, "Flowtable DPI: %s\n", fm->dpi_enable ? "enable" : "disable");
#if (defined DEBUG) && (DEBUG >= 1)
    ft_print(vm, "Flowtable log: %s\n", fm->log_enable ? "enable" : "disable");
#endif /* DEBUG */
    ft_print(vm, "Flowtable stats: %s\n", fm->stats_enable ? "enable" : "disable");

    /* Next node */
    node = vlib_get_next_node(vm, fm->flowtable_index, fm->next_node_index);
    if (node) {
        ft_print(vm, "Next-node: %s\n", node->name);
    }

    ft_print(vm, "\n");

    ft_print(vm, "CLI commands:\n");
    ft_print(vm, "\t"FLOWTABLE_ADD_DESC"\n");
    ft_print(vm, "\t"FLOWTABLE_DEL_DESC"\n");
    ft_print(vm, "\t"FLOWTABLE_DPI_DESC"\n");
    ft_print(vm, "\t"FLOWTABLE_INFO_DESC"\n");
#if (defined DEBUG) && (DEBUG >= 1)
    ft_print(vm, "\t"FLOWTABLE_LOG_DESC"\n");
#endif /* DEBUG */
    ft_print(vm, "\t"FLOWTABLE_STATS_DESC"\n");
    ft_print(vm, "\t"FLOWTABLE_SHOW_STATS_DESC"\n");

    ft_print(vm, "\n");

    return 0;
}

#if (defined DEBUG) && (DEBUG >= 1)
VLIB_CLI_COMMAND(flowtable_log_command, static) = {
    .path = "flowtable log",
    .short_help = "flowtable log [enable|disable]",
    .function = flowtable_log_command_fn,
};
#endif  /* DEBUG */

VLIB_CLI_COMMAND(flowtable_add_command) = {
    .path = "flowtable add",
    .short_help = FLOWTABLE_ADD_DESC,
    .function = flowtable_add_command_fn,
};

VLIB_CLI_COMMAND(flowtable_delete_command) = {
    .path = "flowtable delete",
    .short_help = FLOWTABLE_DEL_DESC,
    .function = flowtable_delete_command_fn,
};

VLIB_CLI_COMMAND(flowtable_dpi_command) = {
    .path = "flowtable dpi",
    .short_help = FLOWTABLE_DPI_DESC,
    .function = flowtable_dpi_command_fn,
};

VLIB_CLI_COMMAND(flowtable_stats_command) = {
    .path = "flowtable stats",
    .short_help = FLOWTABLE_STATS_DESC,
    .function = flowtable_stats_command_fn,
};

VLIB_CLI_COMMAND(flowtable_show_stats_command) = {
    .path = "flowtable show-stats",
    .short_help = FLOWTABLE_SHOW_STATS_DESC,
    .function = flowtable_show_stats_command_fn,
};

VLIB_CLI_COMMAND(flowtable_info_command) = {
    .path = "flowtable info",
    .short_help = FLOWTABLE_INFO_DESC,
    .function = flowtable_info_command_fn,
};

static clib_error_t *
flowtable_config_cpu(flowtable_main_t         *fm,
                     u32                       cpu_count,
                     flowtable_main_per_cpu_t *fmt)
{
    int i;
    flow_entry_t *f;
    clib_error_t *error = 0;

    /* init hashtable */
    pool_alloc(fmt->ht_lines, 2 * fm->flows_max);
    BV(clib_bihash_init)(&fmt->flows_ht, "flow hash table",
                         FM_NUM_BUCKETS, FM_MEMORY_SIZE);

    /* init timer wheel */
    fmt->time_index = ~0;
    for (i = 0; i < TIMER_MAX_LIFETIME; i++) {
        dlist_elt_t *timer_slot;
        pool_get(fmt->timers, timer_slot);

        u32 timer_slot_head_index = timer_slot - fmt->timers;
        clib_dlist_init(fmt->timers, timer_slot_head_index);
        vec_add1(fmt->timer_wheel, timer_slot_head_index);
    }

    /* fill flow entry cache */
    if (pthread_spin_lock(&fm->flows_lock) == 0) {
        for (i = 0; i < FLOW_CACHE_SZ; i++) {
            pool_get_aligned(fm->flows, f, CLIB_CACHE_LINE_BYTES);
            vec_add1(fmt->flow_cache, f - fm->flows);
        }
        fm->flows_cpt += FLOW_CACHE_SZ;

        pthread_spin_unlock(&fm->flows_lock);
    }

    fmt->clocks_per_second = fm->vlib_main->clib_time.clocks_per_second;

    /* Stats*/
    fmt->stat_l3 = NULL;
    fmt->stat_l4 = NULL;
    fmt->stat_l5 = NULL;
    fmt->stat_appli = NULL;
    clib_rwlock_init(&fmt->stat_lock);
    return error;
}

static clib_error_t *
flowtable_config(vlib_main_t *vm, unformat_input_t *input)
{
    u32 cpu_index;
    clib_error_t *error = 0;
    flowtable_main_t *fm = &flowtable_main;
    vlib_thread_main_t *tm = vlib_get_thread_main();
    int thread_count;

    /* Get configuration file parameters */
    while (unformat_check_input(input) != UNFORMAT_END_OF_INPUT) {
        /* max-flows */
        if (unformat(input, "max-flows %u", &fm->flows_max))
            ;
    }

    ft_print(vm, "[FT ] max flows: %d\n", fm->flows_max);

    /* Configure the flow pool */
    pool_init_fixed(fm->flows, fm->flows_max);
    pthread_spin_init(&fm->flows_lock, PTHREAD_PROCESS_PRIVATE);

    /* Configure per thread data structures */
    for (cpu_index = 0; cpu_index < tm->n_vlib_mains; cpu_index++) {
        error = flowtable_config_cpu(fm,
                                     tm->n_vlib_mains,
                                     &fm->per_cpu[cpu_index]);
        if (error) {
            ft_print(vm, "[FT ] Configuration failed for cpu %d\n", cpu_index);
            return error;
        }
    }

    thread_count = tm->n_threads;
    if (thread_count == 0) {
        thread_count = 1;
    }

    ft_print(vm, "[FT ] flowtable plugin initialized: %d threads\n", thread_count);

    return error;
}

static clib_error_t *
flowtable_init(vlib_main_t *vm)
{
    flowtable_main_t *fm = &flowtable_main;
    vlib_thread_main_t *tm = vlib_get_thread_main();

    fm->vlib_main = vm;
    fm->vnet_main = vnet_get_main();

    fm->flowtable_index = flowtable_input_node.index;

    /* By default, forward packets to ethernet-input */
    fm->next_node_index = FT_NEXT_ETHERNET_INPUT;

    /* By default, forwarding to DPI is enable */
    fm->dpi_enable = 1;

    /* ensures flow_info structure fits into vlib_buffer_t's opaque 1 field */
    ASSERT(sizeof(flow_data_t) <= 6 * sizeof(u32));

    /* Retrieve the address of service dpi functions  */
    fm->dpi_get_ixe_elements_fn = vlib_get_plugin_symbol("dpi_plugin.so",
                                                         "dpi_get_ixe_elements");

    /* Retrieve the Ixe elements  */
    if (!fm->dpi_get_ixe_elements_fn) {
        return clib_error_return(0,
                                 "[FT ] flowtable_init() ixe elements (engine, ...) retrieval failed\n");
    }

    fm->dpi_get_ixe_elements_fn(&(fm->engine),
                                &(fm->bundle),
                                &(fm->workers_table));

    /* init flow pool */
    fm->flows_max = FM_POOL_COUNT;
    fm->flows_cpt = 0;

    vec_validate(fm->per_cpu, tm->n_vlib_mains - 1);

    return NULL;
}

VLIB_INIT_FUNCTION(flowtable_init);
VLIB_CONFIG_FUNCTION(flowtable_config, "flowtable");

VNET_FEATURE_INIT(flowtable, static) = {
    .arc_name = "device-input",
    .node_name = "flowtable",
    .runs_before = VNET_FEATURES("dpi"),
};

/* *INDENT-OFF* */
VLIB_PLUGIN_REGISTER() = {
    .version = FLOWTABLE_PLUGIN_VER,
    .description = "Flowtable",
};
/* *INDENT-ON* */

/*
 * fd.io coding-style-patch-verification: ON
 *
 * Local Variables:
 * eval: (c-set-style "gnu")
 * End:
 */
