/*
  This file is a part of Qosmos ixEngine.

   Copyright  Qosmos 2022 - All rights reserved

  This computer program and all its components are protected by
  authors' rights and copyright law and by international treaties.
  Any representation, reproduction, distribution or modification
  of this program or any portion of it is forbidden without
  Qosmos explicit and written agreement and may result in severe
  civil and criminal penalties, and will be prosecuted
  to the maximum extent possible under the law.
*/

#ifndef __flowtable_h__
#define __flowtable_h__

#include <pthread.h>
#include <stdbool.h>
#include <vppinfra/error.h>
#include <vnet/vnet.h>
#include <vnet/ip/ip.h>
#include <vppinfra/bihash_8_8.h>
#include <vppinfra/dlist.h>
#include <vppinfra/pool.h>
#include <vppinfra/vec.h>

#include "flowdata.h"
#include "dpi/protodef.h"

#define ft_print(handle, ...) vlib_cli_output (handle, __VA_ARGS__)

#define foreach_flowtable_error                                \
    _(PACKETS,          "packets received")                    \
    _(FORWARDED,        "packets directly forwarded (DPI skipped)") \
    _(DPI,              "packets submitted to the DPI")        \
    _(HIT,              "packets with an existing flow")       \
    _(MISS,             "packets which created a new flow")    \
    _(UNHANDLED,        "Unhandled (non-ip) packets")          \
    _(TIMER_EXPIRE,     "flows that have expired")             \
    _(COLLISION,        "hashtable collisions")                \
    _(CREATION_FAILED,  "dpi flow creation error")             \
    _(FLOW_ALLOCATION,  "flow allocation error")               \
    _(FLOW_POOL_EXHAUSTED, "flow pool exhausted")

typedef enum {
#define _(sym, str) FLOWTABLE_ERROR_ ## sym,
    foreach_flowtable_error
#undef _
    FLOWTABLE_N_ERROR
} flowtable_error_t;

#define FLOWTABLE_COUNTER_INC(_vm, _sym, _val) \
    if (_val) \
        vlib_node_increment_counter(_vm, flowtable_input_node.index, \
                                    FLOWTABLE_ERROR_ ## _sym, _val)

typedef enum {
    FT_NEXT_ETHERNET_INPUT,
    FT_NEXT_DPI,
    FT_NEXT_N_NEXT
} flowtable_next_t;

/* Timers (in seconds) */
#define TIMER_DEFAULT_LIFETIME (60)
#define TIMER_MAX_LIFETIME (300)

/* Default max number of flows to expire during one run.
 * 256 is the max number of packets in a vector, so this is a minimum
 * if all packets create a flow. */
#define TIMER_MAX_EXPIRE (1 << 8)

/*
 * As advised in the thread below :
 * https://lists.fd.io/pipermail/vpp-dev/2016-October/002787.html
 * hashtable is configured to alloc (NUM_BUCKETS * CLIB_CACHE_LINE_BYTES) Bytes
 * with (flow_count / (BIHASH_KVP_PER_PAGE / 2)) Buckets
 */
#define FM_POOL_COUNT_LOG2 24
#define FM_POOL_COUNT (1 << FM_POOL_COUNT_LOG2)
#define FM_NUM_BUCKETS (1 << (FM_POOL_COUNT_LOG2 - (BIHASH_KVP_PER_PAGE / 2)))
#define FM_MEMORY_SIZE (FM_NUM_BUCKETS * CLIB_CACHE_LINE_BYTES * 6)

/* signatures */
struct ip6_sig {
    ip6_address_t src, dst;
    u8 proto;
    u16 port_src, port_dst;
}
__attribute__((packed));
struct ip4_sig {
    ip4_address_t src, dst;
    u8 proto;
    u16 port_src, port_dst;
}
__attribute__((packed));

typedef struct flow_signature {
    union {
        struct ip6_sig ip6;
        struct ip4_sig ip4;
        u8 data[0];  /* gcc will take the max */
    } s;
    u8 len;
} flow_signature_t;
#define flow_signature_is_ip4(s) (s->len == sizeof(struct ip4_sig))

typedef struct packet_signature {
    u32 data_offset;
    const void *l3hdr;
    const void *l4hdr;
} packet_signature_t;

/* dlist helpers */
#define dlist_is_empty(pool, head_index)                              \
    ({                                                                \
        dlist_elt_t * head = pool_elt_at_index((pool), (head_index)); \
        (head->next == (u32) ~0 || head->next == (head_index));       \
    })

typedef struct flow_entry {
    /* Required for pool_get_aligned  */
    CLIB_CACHE_LINE_ALIGN_MARK(cacheline0);

    /* flow signature */
    flow_signature_t sig;
    u16 tcp_state;
    u64 sig_hash;  /* used to delete hashtable entries */

    /* hashtable */
    u32 ht_line_index; /* index of the list head of the line in the hashtable */
    u32 ht_index; /* index in the hashtable line pool */

    /* timers */
    u32 expire;  /* in seconds */
    u16 lifetime;  /* in seconds */
    u32 timer_index;  /* index in the timer pool */

    /* stats - non-classified  */
    u16 pkt_cnt;
    u16 pkt_volume;

    /* the following union will be copied to vlib->opaque
     * - it MUST be less or equal CLIB_CACHE_LINE_BYTES
     * - it SHOULD be less than 24Bytes so that it can fit into
     * the plugins_data field */
    flow_data_t infos;
} flow_entry_t;

typedef struct {
    u32 id;
    u32 count;
    u64 volume;
} flowtable_stat_t;

typedef struct {
    CLIB_CACHE_LINE_ALIGN_MARK(cacheline0);

    /* hashtable */
    BVT(clib_bihash) flows_ht;
    dlist_elt_t *ht_lines;

    /* timers */
    dlist_elt_t *timers;
    u32 *timer_wheel;
    u32 time_index;

    /* flow cache
     * set cache size to 256 so that the worst node run fills the cache at most once */
#define FLOW_CACHE_SZ 256
    u32 *flow_cache;

    /* clock rate */
    f64 clocks_per_second;

    /* stats */
    clib_rwlock_t     stat_lock;
    flowtable_stat_t *stat_l3;
    flowtable_stat_t *stat_l4;
    flowtable_stat_t *stat_l5;
    flowtable_stat_t *stat_appli;
} flowtable_main_per_cpu_t;

typedef struct {
    /* flow entry pool */
    u32 flows_max;
    flow_entry_t *flows;
    pthread_spinlock_t flows_lock;
    u64 flows_cpt;

    /* per cpu */
    flowtable_main_per_cpu_t *per_cpu;

    /* flowtable node index */
    u32 flowtable_index;

    /* convenience */
    vlib_main_t *vlib_main;
    vnet_main_t *vnet_main;

    /* next-node of flowtable node, NOT pm node id */
    u32 next_node_index;

    /* API dynamically registered base ID. */
    u16 msg_id_base;

    /* configuration */
    int     dpi_enable;
    int     log_enable;
    int     stats_enable;

    /* dpi_nreg elements (allocated by the DPI module) */
    struct qmdpi_engine *engine;
    struct qmdpi_bundle *bundle;
    struct qmdpi_worker **workers_table;

    /* Point to dpi module functions */
    void (*dpi_get_ixe_elements_fn)(struct qmdpi_engine **engine,
                                    struct qmdpi_bundle **bundle,
                                    struct qmdpi_worker ***workers_table);
} flowtable_main_t;

extern flowtable_main_t flowtable_main;
extern vlib_node_registration_t flowtable_input_node;

/* Logs */
void flowtable_dump_flow(vlib_main_t *vm, flow_entry_t *flow);

/* Stats */
void flowtable_stats_reset(vlib_main_t *vm,
                           flowtable_main_t *fm);
void flowtable_stats_add(vlib_main_t *vm,
                         flowtable_main_t *fm,
                         flowtable_main_per_cpu_t *fmt,
                         flow_entry_t *flow,
                         vlib_buffer_t *b);

clib_error_t *flowtable_stats_show(vlib_main_t *vm,
                                   flowtable_main_t *fm);

/* API functions */
int
flowtable_enable_disable(flowtable_main_t *fm, u32 sw_if_index,
                         u8 enable_disable);

#endif  /* __flowtable_h__ */
