#error "This header should not be used. It is provided for documentation only."
/*
  This file is a part of Qosmos ixEngine.

  Copyright  Qosmos 2000-2019 - All rights reserved

  This computer program and all its components are protected by
  authors' rights and copyright law and by international treaties.
  Any representation, reproduction, distribution or modification
  of this program or any portion of it is forbidden without
  Qosmos explicit and written agreement and may result in severe
  civil and criminal penalties, and will be prosecuted
  to the maximum extent possible under the law.
*/
/**
 * @file ulayer_store.h
 * This header contains the prototypes of functions
 * that users can override in order to
 * use their own **Buffer Store** implementation,
 * it is not meant to be included.
 * @note This file is not meant to be included.
 */

/**
 * Type for buffer constructors.
 * @return 0 if successful, -1 otherwise.
 */
typedef int32_t (*ulayer_store_ctor_fn_t)(void *ptr, void *arg);

/**
 * Type for buffer destructors.
 */
typedef void (*ulayer_store_dtor_fn_t)(void *ptr);

/**
 * Type for buffer hash functions.
 */
typedef uint32_t (*ulayer_store_hash_fn_t)(const void *ptr);

/**
 * Type for buffer comparison functions.
 * @return 0 if buffers don't compare equal.
 */
typedef int32_t (*ulayer_store_cmp_fn_t)(const void *a, const void *b);

/**
 * Type for buffer debug functions.
 */
typedef int32_t (*ulayer_store_dbg_fn_t)(void *ptr, void *f,
        void (*printfn)(void *f, char const *fmt, ...), char const *prefix);

/**
 * Layer store structures.
 */
struct ulayer_store;
struct ulayer_store_lock;

/**
 * Perform basic initialization of the ulayer_store module.
 */
void ulayer_store_setup(void);

/**
 * do nothing, just for backward compatibility with old PB.
 */
void ulayer_store_setup_backward(void);


/**
 * Release global module resources.
 */
void ulayer_store_shutdown(void);

/**
 * do nothing, just for backward compatibility with old PB.
 */
void ulayer_store_shutdown_backward(void);


/**
 * Create a buffer store.
 *
 * Neither the name, nor the constructor, hash and comparison functions may be
 * NULL.
 * The buffer size must not be 0.
 * A maximum size of 0 makes the store use a default maximum size.
 *
 * The lifetime parameter is the default lifetime associated with new
 * buffers.
 * A lifetime of 0 means buffers have no timeout, and are invalidated only as
 * determined by the store management algorithm.
 * @note Lifetimes are expressed in seconds.
 */
int32_t ulayer_store_create(struct ulayer_store **storep,
                              const char *name, uint32_t buf_size,
                              uint32_t max_size, uint32_t lifetime,
                              ulayer_store_ctor_fn_t ctor_fn,
                              ulayer_store_dtor_fn_t dtor_fn,
                              ulayer_store_hash_fn_t hash_fn,
                              ulayer_store_cmp_fn_t cmp_fn,
                              ulayer_store_dbg_fn_t dbg_fn, uint32_t hash_size);
/**
 * Destroy a buffer store.
 * Any remaining buffer is invalidated.
 * @note This function does nothing if the given store is NULL.
 */
void ulayer_store_destroy(struct ulayer_store *store);

/**
 * Invalidate a buffer.
 * Future lookups will never return the invalidated buffer, which is scheduled
 * for destruction.
 * @attention The buffer must be locked.
 */
void ulayer_store_invalidate(struct ulayer_store *store, void *ptr);

/**
 * Report that a buffer has been recently used.
 *
 * The store buffer management algorithm can use this information to determine
 * access recency and frequency, and in turn which buffers are more likely
 * to be used in the future, so that relevant buffers are retained longer.
 *
 * @attention The buffer must be locked.
 */
void ulayer_store_reference(struct ulayer_store *store, void *ptr);

/**
 * Look up a buffer in a store.
 * If an expired buffer is found, it is automatically invalidated and not
 * returned.
 *
 * @return the found buffer with the buffer lock held, or NULL.
 */
void * ulayer_store_lookup(struct ulayer_store *store, const void *key,
                           struct ulayer_store_lock *lock);

/**
 * Look up and reference a buffer in a store and create it if not found.
 *
 * If the lookup is successful, a valid buffer is returned after being
 * referenced.
 * Otherwise, a new buffer is allocated and passed to the constructor along with
 * the arg parameter.
 * On successful construction, the new buffer is returned.
 * Returns NULL if both the lookup and the creation failed.
 *
 * @note The rationale for this function is to provide the ability to both
 * look-up and create buffers atomically.
 *
 * @attention Once creation is done and the lock is released, a buffer may
 * become invalid at any time.
 *
 * @return the new buffer with the buffer lock held, or NULL if both the lookup
 * and the creation failed.
 */
void * ulayer_store_lookup_create(struct ulayer_store *store, const void *key,
                                  void *arg, struct ulayer_store_lock *lock);

/**
 * Look up and reference a buffer in a store and create it if not found.
 * Initialize ttl and timeout with times values provided.
 *
 * If the lookup is successful, a valid buffer is returned after being
 * referenced.
 * Otherwise, a new buffer is allocated and passed to the constructor along
 * with the arg parameter.
 * On successful construction, the new buffer is returned.
 * Returns NULL if both the lookup and the creation failed.
 *
 * @note The rationale for this function is to provide the ability to both
 * look-up and create buffers atomically.
 *
 * @attention Once creation is done and the lock is released, a buffer may
 * become invalid at any time.
 *
 * @return the new buffer with the buffer lock held, or NULL if both the lookup
 * and the creation failed.
 */
void * ulayer_store_lookup_create_with_time(struct ulayer_store *store, const void *key,
                                  void *arg, struct ulayer_store_lock *lock, uint32_t lifetime, uint32_t current_time);
/**
 * Release a buffer lock.
 */
void ulayer_store_unlock(struct ulayer_store *store,
                         struct ulayer_store_lock *lock);

/**
 * Get the timeout of a store entry
 */
uint32_t ulayer_store_get_timeout(struct ulayer_store *_store, void *ptr);

/**
 * Report information about the internal state of a store.
 * If store is NULL, process all stores.
 * If the print function pointer is NULL, use a default print function.
 */
void ulayer_store_info(struct ulayer_store *store, void *f,
                       void (*print_fn)(void *f, const char *format, ...),
                       char const *prefix, int loglvl);

int32_t ulayer_store_ioctl_get(struct ulayer_store *stores[],
                                 uint32_t nr_stores, uint32_t code,
                                 void *ptr, uint32_t size);

int32_t ulayer_store_ioctl_set(struct ulayer_store *stores[],
                                 uint32_t nr_stores, uint32_t code,
                                 const void *ptr, uint32_t size);

/**
 * Set a custom lifetime for a buffer.
 * The buffer timeout is implicitely reloaded when calling this function.
 * Setting a lifetime of 0 removes the buffer timeout.
 * @attention The buffer must be locked.
 */
void ulayer_store_set_lifetime(struct ulayer_store *store, void *ptr,
                               uint32_t lifetime);

/**
 * Update the maximum number of buffers a store may contain.
 */
void ulayer_store_set_max_size(struct ulayer_store *store, uint32_t max_size);

/*
 * =============================================================================
 * BELOW THIS LINE: Legacy interface.
 */

/**
 * Legacy function:
 * The caller should prevent concurrent operations before calling this function.
 * @note this is a *Legacy Function*.
 */
void ulayer_store_foreach_unsafe(struct ulayer_store *store,
                                 void (*cb)(void *ptr, void *arg), void *arg);


/**
 * Legacy function: Invalidate all entries of a store.
 * @note this is a *Legacy Function*.
 */
void ulayer_store_foreach_invalidate(struct ulayer_store *store);

/**
 * Legacy function: Iterator on all entries of a layer store.
 * @note this is a *Legacy Function*.
 */
void ulayer_store_foreach(struct ulayer_store *store,
                          void (*cb)(void *ptr, void *arg), void *arg);

