#include <stdlib.h>
#include <stdio.h>
#include <errno.h>

#include <qmdpi.h>
#include <qmdpi_bundle_api.h>

#include "alloc.h"
#include "memstat.h"
#include "pcap_analyzer_getter.h"
#include "alloc_list.h"
#include <pthread.h>

#define MSHT_SZ 2048

struct ms_elt {
    void *ptr;
    LIST_ENTRY(ms_elt) next;
    size_t nr;
    size_t sz;
    size_t msz;
    unsigned int ctx;
    unsigned int sid;
};

struct ms_ht {
    LIST_HEAD(, ms_elt) *ms_elt;
    size_t sz;
};

static int _alloc_init;
static pthread_mutex_t hash_table_lock;
static struct ms_ht __ms;

static inline size_t msht_hash(struct ms_elt *e)
{
    return ((uintptr_t)e->ptr / sizeof(void *));
}

static void me_set(struct ms_elt *e, void *ptr, size_t nr, size_t sz,
                   unsigned int ctx, unsigned int sid, size_t memchunksz)
{
    e->ptr = ptr;
    e->ctx = ctx;
    e->sid = sid;
    e->nr = nr;
    e->sz = sz;
    e->msz = memchunksz;
}

static struct ms_elt *me_new(void *ptr, size_t nr, size_t sz, unsigned int ctx,
                             unsigned int sid, size_t memchunksize)
{
    struct ms_elt *e;

    e = malloc(sizeof(*e));
    if (e == NULL) {
        goto out;
    }

    me_set(e, ptr, nr, sz, ctx, sid, memchunksize);

out:
    return e;
}

static void me_free(struct ms_elt *me)
{
    free(me);
}

static inline size_t me_get_size(struct ms_elt *me)
{
    return me->sz;
}

static inline size_t me_get_nr(struct ms_elt *me)
{
    return me->nr;
}

static int msht_init(struct ms_ht *ht, size_t size)
{
    size_t i;

    ht->ms_elt = malloc(size * sizeof(*ht->ms_elt));

    if (ht->ms_elt == NULL) {
        return -ENOMEM;
    }

    for (i = 0; i < size; ++i) {
        LIST_INIT(&ht->ms_elt[i]);
    }

    ht->sz = size;
    return 0;
}

static void msht_exit(struct ms_ht *ht)
{
    struct ms_elt *e;
    size_t i;

    for (i = 0; i < ht->sz; ++i) {
        while (!LIST_EMPTY(&ht->ms_elt[i])) {
            e = LIST_FIRST(&ht->ms_elt[i]);
            LIST_REMOVE(e, next);
            me_free(e);
        }
    }
    free(ht->ms_elt);
}

static int msht_add(struct ms_ht *ht, struct ms_elt *e)
{
    struct ms_elt *p;
    size_t hash;

    hash = msht_hash(e);

    LIST_FOREACH(p, &ht->ms_elt[hash % ht->sz], next) {
        if (p->ptr == e->ptr) {
            return -EEXIST;
        }
    }
    LIST_INSERT_HEAD(&ht->ms_elt[hash % ht->sz], e, next);

    return 0;
}

static struct ms_elt *msht_find(struct ms_ht *ht, struct ms_elt *e)
{
    struct ms_elt *p;
    size_t hash;

    hash = msht_hash(e);

    LIST_FOREACH(p, &ht->ms_elt[hash % ht->sz], next) {
        if (p->ptr == e->ptr) {
            return p;
        }
    }

    return NULL;
}

static void msht_rm(struct ms_ht *ht, struct ms_elt *e)
{
    (void)ht;
    LIST_REMOVE(e, next);
}

static void *stat_malloc(uint32_t nr, uint32_t size, unsigned int ctx,
                         unsigned int sid)
{
    struct ms_elt *e;
    void *ptr = qm_malloc_default(nr, size, ctx, sid);

    if (ptr == NULL) {
        return NULL;
    }

    e = me_new(ptr, nr, size, ctx, sid, qm_memchunk_size_get_default(ptr));
    if (e == NULL) {
        return ptr;
    }

    /*lock in case of multi-threading*/
    pthread_mutex_lock(&hash_table_lock);

    msht_add(&__ms, e);
    memstat_add(nr, size, ctx, sid);

    pthread_mutex_unlock(&hash_table_lock);

    return ptr;
}

static void stat_free(void *ptr, unsigned int ctx, unsigned int sid)
{
    struct ms_elt *p;
    struct ms_elt e = {
        .ptr = ptr,
        .ctx = ctx,
        .sid = sid,
    };

    struct qmdpi_bundle *bundle = pcap_analyzer_bundle_get();
    if (_alloc_init && ptr) {
        const char *ctx_desc = qmdpi_malloc_ctx_desc_get_byid(bundle, ctx);
        const char *struct_desc = qmdpi_malloc_struct_desc_get_byid(bundle, ctx);

        /*lock in case of multi-threading*/
        pthread_mutex_lock(&hash_table_lock);

        p = msht_find(&__ms, &e);
        if (!p) {
            fprintf(stderr, "Freeing an invalid pointer %p ctx %s sid : %s\n",
                    ptr,
                    ctx_desc ? ctx_desc : "(null)",
                    struct_desc ? struct_desc : "(null)");
        } else {
            if (p->msz != qm_memchunk_size_get_default(ptr)) {
                fprintf(stderr, "qm_memchunk_size_get_default is buggy\n");
            }
            memstat_remove(p->nr, p->sz, ctx, sid);
            msht_rm(&__ms, p);
            me_free(p);
        }
        pthread_mutex_unlock(&hash_table_lock);
    }

    qm_free_default(ptr, ctx, sid);
}

static void *stat_realloc(void *ptr, uint32_t nr, uint32_t size,
                          unsigned int ctx, unsigned int sid)
{
    void *p = qm_realloc_default(ptr, nr, size, ctx, sid);
    struct ms_elt *pe;
    ssize_t osz = 0;
    ssize_t onr = 0;
    struct ms_elt e = {
        .ptr = ptr,
        .ctx = ctx,
        .sid = sid,
    };

    struct qmdpi_bundle *bundle = pcap_analyzer_bundle_get();
    const char *ctx_desc = qmdpi_malloc_ctx_desc_get_byid(bundle, ctx);
    const char *struct_desc = qmdpi_malloc_struct_desc_get_byid(bundle, ctx);


    if (p == NULL) {
        return NULL;
    }

    /*lock in case of multi-threading*/
    pthread_mutex_lock(&hash_table_lock);
    pe = msht_find(&__ms, &e);
    if (pe) {
        osz = me_get_size(pe);
        onr = me_get_nr(pe);
    }

    if (p != ptr) {
        if (ptr && pe == NULL) {
            fprintf(stderr, "Reallocating an invalid pointer %p ctx %s sid : %s\n",
                    ptr,
                    ctx_desc ? ctx_desc : "(null)",
                    struct_desc ? struct_desc : "(null)");
        } else if (pe) {
            msht_rm(&__ms, pe);
            me_free(pe);
        }

        pe = me_new(p, nr, size, ctx, sid, qm_memchunk_size_get_default(p));
        if (pe == NULL) {
            pthread_mutex_unlock(&hash_table_lock);
            return p;
        }

        msht_add(&__ms, pe);
    }

    me_set(pe, p, nr, size, ctx, sid, qm_memchunk_size_get_default(p));

    memstat_resize(nr, (ssize_t)size, onr, osz, ctx, sid);

    pthread_mutex_unlock(&hash_table_lock);

    return p;
}

int alloc_init(void)
{
    int ret;

    _alloc_init = 1;

    qmdpi_interface_set("malloc", (void *)stat_malloc);
    qmdpi_interface_set("free", stat_free);
    qmdpi_interface_set("realloc", (void *)stat_realloc);

    pthread_mutex_init(&hash_table_lock, NULL);

    ret = memstat_init();
    if (ret < 0) {
        return ret;
    }

    return msht_init(&__ms, MSHT_SZ);
}

void alloc_exit(void)
{
    msht_exit(&__ms);
    memstat_exit();
    pthread_mutex_destroy(&hash_table_lock);
    _alloc_init = 0;
}
