/*
  This file is a part of Qosmos ixEngine.

   Copyright  Qosmos 2017 - 2022 All rights reserved

  This computer program and all its components are protected by
  authors' rights and copyright law and by international treaties.
  Any representation, reproduction, distribution or modification
  of this program or any portion of it is forbidden without
  Qosmos explicit and written agreement and may result in severe
  civil and criminal penalties, and will be prosecuted
  to the maximum extent possible under the law.
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pcap.h>
#include <pthread.h>
#ifdef __FreeBSD__
#include <pthread_np.h>
#endif
#include <unistd.h>
#include <sched.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <inttypes.h>
#include <ctype.h>


#ifdef __FreeBSD__
typedef cpuset_t cpu_set_t;
#endif

/**
 * DPI include
 */
#include <qmdpi_const.h>
#include <qmdpi.h>
#include <qmdpi_bundle_api.h>

#include <packet_helper.h>
#include "parse_options.h"
#include "hooks.h"
#include "tune.h"
#include "alloc.h"

#define LOOP_HEADER_SZ 4
#define LLC_HEADER_SZ  2
#define PACKET_QUEUESZ (1 << 13)
#define PACKET_QUEUEMASK (PACKET_QUEUESZ - 1)

struct opt opt;

struct pa_dpi_stats {
    unsigned long processed;
    unsigned long errors;
    unsigned long dropped_ip_defrag_drop;
    unsigned long dropped_bounds_failure;
    unsigned long dropped_bounds_drop;
    unsigned long dropped_tcp_segment_dropped;
};

#define PI_DPI_STAT_INIT(s) do {                                           \
        (s)->processed = 0;                                                \
        (s)->errors = 0;                                                   \
        (s)->dropped_ip_defrag_drop = 0;                                   \
        (s)->dropped_bounds_failure = 0;                                   \
        (s)->dropped_bounds_drop = 0;                                      \
        (s)->dropped_tcp_segment_dropped = 0;                              \
    } while(/*CONSTCOND*/0)

/*
 * List of packets for a thread
 */
struct pa_thread {
    struct pa_dpi_stats st;
    int thread_id;
    int cpu_id;
    pthread_t handle;         /* thread handle */
    size_t read_index;
    size_t write_index;
    struct pa_pkt *packets[PACKET_QUEUESZ];
    struct qmdpi_worker *worker;
    pthread_mutex_t lock;
    uint64_t packet_number;
};

#define PI_THREAD_INIT(s) do {                  \
        (s)->thread_id = 0;                     \
        (s)->cpu_id    = -1;                    \
        (s)->read_index = 0;                    \
        (s)->write_index = 0;                   \
        PI_DPI_STAT_INIT(&(s)->st);             \
    } while(/*CONSTCOND*/0)

#define PACKET_INDEX(_value) ((_value) & PACKET_QUEUEMASK)

static char DIR_TYPE_CHAR[] = {
    [QMDPI_DIR_CTS]  = 'c',
    [QMDPI_DIR_STC]  = 's',
    [QMDPI_DIR_UNKNOWN] = '?'
};

static struct qmdpi_engine *engine;
static struct qmdpi_bundle *bundle;
static struct qmdpi_extralib *libqmmletc;

struct qmdpi_bundle *pcap_analyzer_bundle_get(void)
{
    return bundle;
}

struct qmdpi_engine *pcap_analyzer_engine_get(void)
{
    return engine;
}

/*
 * Array of packet buffer for all threads
 */
static struct pa_thread *threads;

static unsigned long pktdrop;

static char         **traces;
static unsigned int   tracesnr;

static int nr_cpus;

static int statistics_enabled;
static int flow_metrics_enabled;
static int classification_cache_enabled;

extern int hook_base_path_end;
extern int hook_base_final_path;
extern int hook_base_classified;
extern int hook_base_session_end;

struct pa_pkt {
    uint8_t *data;
    int32_t len;
    struct timeval timestamp;
    int32_t link_mode;
    int32_t thread_id;
    uint64_t packet_number;
};

/**
 * ------------------------
 * Packet functions
 * ------------------------
 */
/**
 * Alloc a new packet
 */
static inline struct pa_pkt *pa_pkt_alloc(uint32_t caplen)
{
    struct pa_pkt *packet;

    packet = (struct pa_pkt *)malloc(sizeof(*packet) + caplen);

    if (packet == NULL) {
        fprintf(stderr, "Can't malloc new packet\n");
        return NULL;
    }

    memset(packet, 0, sizeof(*packet));

    packet->len = caplen;
    packet->data = (unsigned char *)(packet + 1);

    return packet;
}

/**
 * Free a packet
 */
static inline void pa_pkt_free(struct pa_pkt *p)
{
    free(p);
}

/**
 * Initialize packet with data, remove_llc flags indicates if LLC header is
 * present
 */
static struct pa_pkt *
pa_pkt_build(struct pcap_pkthdr *phdr, const u_char *pdata, int link_mode,
             int remove_llc, int link_mode_loop)
{
    struct pa_pkt *packet;
    uint32_t caplen;

    if (link_mode_loop && phdr->caplen >= LOOP_HEADER_SZ) {
        uint32_t pf_mode = *(uint32_t *)(pdata);
        if (pf_mode == PF_INET) {
            link_mode = QMDPI_PROTO_IP;
        } else if (pf_mode == PF_INET6) {
            link_mode = QMDPI_PROTO_IP6;
        } else {
            pf_mode = ntohl(pf_mode);
            if (pf_mode == PF_INET) {
                link_mode = QMDPI_PROTO_IP;
            } else if (pf_mode == PF_INET6) {
                link_mode = QMDPI_PROTO_IP6;
            }
        }

        caplen = phdr->caplen - LOOP_HEADER_SZ;
        pdata += LOOP_HEADER_SZ;
    } else if (remove_llc && phdr->caplen >= LLC_HEADER_SZ) {
        caplen = phdr->caplen - LLC_HEADER_SZ;
        pdata += LLC_HEADER_SZ;
    } else {
        caplen = phdr->caplen;
    }

    packet = pa_pkt_alloc(caplen);
    if (packet == NULL) {
        return NULL;
    }

    packet->timestamp = phdr->ts;
    packet->link_mode = link_mode;
    packet->len = caplen;

    memcpy(packet->data, pdata, caplen);

    return packet;
}

/**
 * Enqueue packet in thread ring
 */
static inline void pa_pkt_queue(struct pa_thread *thread,
                                struct pa_pkt *packet)
{
    size_t next_index;

    if (packet) {
        packet->thread_id = thread->thread_id;
    }

    next_index = PACKET_INDEX(thread->write_index + 1);

    while (next_index == thread->read_index) {
        /**
         * Chances are that all thread queue are full. So going to sleep should
         * be ok, letting other thread a chance to process some packets
         */
        usleep(10);
    }

    thread->packets[thread->write_index] = packet;

    /**
     * This mutex lock is mainly used as a memory barrier to force reading cpu
     * to get the packet modification before the write_index one.
     */
    pthread_mutex_lock(&thread->lock);
    thread->write_index = next_index;
    pthread_mutex_unlock(&thread->lock);
}

/**
 * Dequeue packet in thread ring
 */
static inline struct pa_pkt *pa_pkt_dequeue(struct pa_thread *thread)
{
    struct pa_pkt *packet;

    pthread_mutex_lock(&thread->lock);
    while (thread->read_index == thread->write_index) {
        /**
         * Let other threads going
         */
        pthread_mutex_unlock(&thread->lock);
        sched_yield();
        pthread_mutex_lock(&thread->lock);
    }
    pthread_mutex_unlock(&thread->lock);

    packet = thread->packets[thread->read_index];

    /*
     * Mutex lock is only needed for write index. This is more used as a memory
     * barrier for packet coherency
     */
    thread->read_index = PACKET_INDEX(thread->read_index + 1);

    return packet;
}

/**
 * Send NULL packet to all thread to stop them
 */
static void pa_pkt_stop(void)
{
    int i;

    for (i = 0; i < nr_cpus; ++i) {
        pa_pkt_queue(&threads[i], NULL);
    }
}

/**
 * ------------------------
 * DPI functions
 * ------------------------
 */

static void pa_result_display(uint64_t packet_number,
                              struct qmdpi_worker *worker,
                              struct qmdpi_result *result)
{
    struct qmdpi_flow *flow = qmdpi_result_flow_get(result);

    struct qmdpi_engine *engine = qmdpi_worker_engine_get(worker);
    struct qmdpi_bundle *b = qmdpi_flow_bundle_get(flow);
    struct qmdpi_result_flags const *result_flags = qmdpi_result_flags_get(result);

    if (b == NULL) {
        b = qmdpi_bundle_get_active(engine);
    }

    if (opt.print_result) {
        if (QMDPI_RESULT_FLAGS_FLOW_CREATED(result_flags) ||
                QMDPI_RESULT_FLAGS_FLOW_EXPIRED(result_flags)) {
            printf("%d/%d %"PRIu64" f(%"PRIu64")/p/flow_created=%d,flow_expired=%d\n",
                   qmdpi_engine_id_get(engine),
                   qmdpi_worker_id_get(worker),
                   packet_number,
                   qmdpi_flow_id_get(flow),
                   QMDPI_RESULT_FLAGS_FLOW_CREATED(result_flags),
                   QMDPI_RESULT_FLAGS_FLOW_EXPIRED(result_flags));
        }

        if (QMDPI_RESULT_FLAGS_CLASSIFIED_STATE_CHANGED(result_flags) ||
                QMDPI_RESULT_FLAGS_CLASSIFIED_FINAL_STATE_CHANGED(result_flags) ||
                QMDPI_RESULT_FLAGS_OFFLOADED_STATE_CHANGED(result_flags) ||
                QMDPI_RESULT_FLAGS_PDU_DIR_CHANGED(result_flags) ||
                QMDPI_RESULT_FLAGS_PATH_CHANGED(result_flags)) {
            printf("%d/%d %"PRIu64" f(%"PRIu64")/p/classif_flags=classified_state_changed:%d,classified_final_state_changed:%d,offloaded_state_changed:%d,pdu_dir_changed:%d,classified_state:%d,classified_final_state:%d,offloaded_state:%d,pdu_dir_index:%d,pdu_dir:%d,cacheable_state:%d,path_changed:%d\n",
                   qmdpi_engine_id_get(engine),
                   qmdpi_worker_id_get(worker),
                   packet_number,
                   qmdpi_flow_id_get(flow),
                   QMDPI_RESULT_FLAGS_CLASSIFIED_STATE_CHANGED(result_flags),
                   QMDPI_RESULT_FLAGS_CLASSIFIED_FINAL_STATE_CHANGED(result_flags),
                   QMDPI_RESULT_FLAGS_OFFLOADED_STATE_CHANGED(result_flags),
                   QMDPI_RESULT_FLAGS_PDU_DIR_CHANGED(result_flags),
                   QMDPI_RESULT_FLAGS_CLASSIFIED_STATE(result_flags),
                   QMDPI_RESULT_FLAGS_CLASSIFIED_FINAL_STATE(result_flags),
                   QMDPI_RESULT_FLAGS_OFFLOADED_STATE(result_flags),
                   QMDPI_RESULT_FLAGS_PDU_DIR_INDEX(result_flags),
                   QMDPI_RESULT_FLAGS_PDU_DIR(result_flags),
                   0,
                   QMDPI_RESULT_FLAGS_PATH_CHANGED(result_flags));
        }

        /* print classification only if protocol path has changed */
        if (QMDPI_RESULT_FLAGS_PATH_CHANGED(result_flags)) {
            char buffer[1024];
            unsigned int i;
            int pos = 0;
            struct qmdpi_path *path = qmdpi_result_path_get(result);
            struct qmdpi_signature *signature;
            char const *name;
            if (path) {
                for (i = 0; i < path->qp_len; i++) {
                    signature = qmdpi_worker_signature_get_byid(worker, b, path->qp_value[i]);
                    if (signature != NULL) {
                        name = qmdpi_signature_name_get(signature);
                        pos += sprintf(buffer + pos, "%s%s", i > 0 ? "." : "",
                                       name);
                    } else {
                        pos += sprintf(buffer + pos, "%s%s", i > 0 ? "." : "",
                                       "(null)");
                    }
                }
                printf("%d/%d %"PRIu64" f(%"PRIu64")/p/path=%s\n",
                       qmdpi_engine_id_get(engine),
                       qmdpi_worker_id_get(worker),
                       packet_number,
                       qmdpi_flow_id_get(flow),
                       buffer);
            }
        }
    }
    /* print path_end only if protocol path has changed */
    if (QMDPI_RESULT_FLAGS_PATH_CHANGED(result_flags) && hook_base_path_end) {
        char buffer[1024];
        unsigned int i;
        int pos = 0;
        struct qmdpi_path *path = qmdpi_result_path_get(result);
        struct qmdpi_signature *signature;
        char const *name;
        if (path) {
            for (i = 0; i < path->qp_len; i++) {
                signature = qmdpi_worker_signature_get_byid(worker, b, path->qp_value[i]);
                if (signature != NULL) {
                    name = qmdpi_signature_name_get(signature);
                    pos += sprintf(buffer + pos, "%s%s", i > 0 ? "." : "",
                                   name);
                } else {
                    pos += sprintf(buffer + pos, "%s%s", i > 0 ? "." : "",
                                   "(null)");
                }
            }
            printf("%d/%d %"PRIu64" f(%"PRIu64")/p(%c)/base:path_end=%s\n",
                   qmdpi_engine_id_get(engine),
                   qmdpi_worker_id_get(worker),
                   packet_number,
                   qmdpi_flow_id_get(flow),
                   DIR_TYPE_CHAR[QMDPI_RESULT_FLAGS_PDU_DIR(result_flags)],
                   buffer);
        }
    }
    /* print final_path only if final classification has been reached */
    if (QMDPI_RESULT_FLAGS_CLASSIFIED_FINAL_STATE_CHANGED(result_flags) &&
            QMDPI_RESULT_FLAGS_CLASSIFIED_FINAL_STATE(result_flags) &&
            hook_base_final_path) {
        char buffer[1024];
        unsigned int i;
        int pos = 0;
        struct qmdpi_path *path = qmdpi_result_path_get(result);
        struct qmdpi_signature *signature;
        char const *name;
        if (path) {
            for (i = 0; i < path->qp_len; i++) {
                signature = qmdpi_worker_signature_get_byid(worker, b, path->qp_value[i]);
                if (signature != NULL) {
                    name = qmdpi_signature_name_get(signature);
                    pos += sprintf(buffer + pos, "%s%s", i > 0 ? "." : "",
                                   name);
                } else {
                    pos += sprintf(buffer + pos, "%s%s", i > 0 ? "." : "",
                                   "(null)");
                }
            }
            printf("%d/%d %"PRIu64" f(%"PRIu64")/p(%c)/base:final_path=%s\n",
                   qmdpi_engine_id_get(engine),
                   qmdpi_worker_id_get(worker),
                   packet_number,
                   qmdpi_flow_id_get(flow),
                   DIR_TYPE_CHAR[QMDPI_RESULT_FLAGS_PDU_DIR(result_flags)],
                   buffer);
        }
    }
    /*  print session_end only if flags flow_expired is set */
    if (hook_base_session_end == 1 &&
            QMDPI_RESULT_FLAGS_FLOW_EXPIRED(result_flags) == 1) {
        printf("%d/%d %"PRIu64" f(%"PRIu64")/p(%c)/base:session_end=%d\n",
               qmdpi_engine_id_get(engine),
               qmdpi_worker_id_get(worker),
               packet_number,
               qmdpi_flow_id_get(flow),
               DIR_TYPE_CHAR[QMDPI_RESULT_FLAGS_PDU_DIR(result_flags)],
               QMDPI_RESULT_FLAGS_FLOW_EXPIRED(result_flags));
    }
    /*  print classified value only if classified state changed */
    if (QMDPI_RESULT_FLAGS_CLASSIFIED_STATE_CHANGED(result_flags) &&
            hook_base_classified == 1) {
        printf("%d/%d %"PRIu64" f(%"PRIu64")/p(%c)/base:classified=%d\n",
               qmdpi_engine_id_get(engine),
               qmdpi_worker_id_get(worker),
               packet_number,
               qmdpi_flow_id_get(flow),
               DIR_TYPE_CHAR[QMDPI_RESULT_FLAGS_PDU_DIR(result_flags)],
               QMDPI_RESULT_FLAGS_CLASSIFIED_STATE(result_flags));
    }


    if (flow_metrics_enabled) {
        if (QMDPI_RESULT_FLAGS_FLOW_EXPIRED(result_flags)) {
            struct timeval start_time, last_packet_time, duration;
            qmdpi_flow_start_time_get(flow, &start_time);
            qmdpi_flow_last_packet_time_get(flow, &last_packet_time);
            qmdpi_flow_duration_get(flow, &duration);
            printf("%d/%d %"PRIu64" f(%"PRIu64")/p/packets=%"PRIu64",packets_cts=%"PRIu64",packets_stc=%"PRIu64",packets_processed=%"PRIu64",packets_processed_cts=%"PRIu64",packets_processed_stc=%"PRIu64",bytes=%"PRIu64",bytes_cts=%"PRIu64",bytes_stc=%"PRIu64",bytes_processed=%"PRIu64",bytes_processed_cts=%"PRIu64",bytes_processed_stc=%"PRIu64",start_time=%ld.%06ld,last_packet_time=%ld.%06ld,duration=%ld.%06ld\n",
                   qmdpi_engine_id_get(engine),
                   qmdpi_worker_id_get(worker),
                   packet_number,
                   qmdpi_flow_id_get(flow),
                   qmdpi_flow_packets_get(flow),
                   qmdpi_flow_packets_get_cts(flow),
                   qmdpi_flow_packets_get_stc(flow),
                   qmdpi_flow_packets_processed_get(flow),
                   qmdpi_flow_packets_processed_get_cts(flow),
                   qmdpi_flow_packets_processed_get_stc(flow),
                   qmdpi_flow_bytes_get(flow),
                   qmdpi_flow_bytes_get_cts(flow),
                   qmdpi_flow_bytes_get_stc(flow),
                   qmdpi_flow_bytes_processed_get(flow),
                   qmdpi_flow_bytes_processed_get_cts(flow),
                   qmdpi_flow_bytes_processed_get_stc(flow),
                   (long int)start_time.tv_sec, (long int)start_time.tv_usec,
                   (long int)last_packet_time.tv_sec, (long int)last_packet_time.tv_usec,
                   (long int)duration.tv_sec, (long int)duration.tv_usec);
        }
    }

    /* Only print cached path on 1st packet */
    if (classification_cache_enabled) {
        struct qmdpi_flow_info *flow_info = qmdpi_flow_info_get(flow);
        if (QMDPI_RESULT_FLAGS_FLOW_CREATED(result_flags) &&
                QMDPI_FLOW_INFO_CLASSIF_CACHED(flow_info)) {
            char buffer[1024];
            unsigned int i;
            int pos = 0;
            struct qmdpi_path *cached_path = qmdpi_result_cached_path_get(result);
            struct qmdpi_signature *signature;
            char const *name;
            if (cached_path) {
                for (i = 0; i < cached_path->qp_len; i++) {
                    signature = qmdpi_worker_signature_get_byid(worker, b,
                                                                cached_path->qp_value[i]);
                    if (signature != NULL) {
                        name = qmdpi_signature_name_get(signature);
                        pos += sprintf(buffer + pos, "%s%s", i > 0 ? "." : "",
                                       name);
                    } else {
                        pos += sprintf(buffer + pos, "%s%s", i > 0 ? "." : "",
                                       "(null)");
                    }
                }
                printf("%d/%d %"PRIu64" f(%"PRIu64")/p/cached_path=%s\n",
                       qmdpi_engine_id_get(engine),
                       qmdpi_worker_id_get(worker),
                       packet_number,
                       qmdpi_flow_id_get(flow),
                       buffer);
            }
        }
    }

    char const *data;
    int datalen;
    int proto_id;
    int layer_index;
    int attr_id;
    int flags;
    int ret;
    struct qmdpi_signature *signature;
    struct qmdpi_attr *attr;
    char const *sig_name;
    char const *attr_name;

    while (qmdpi_result_attr_getnext_with_index(result, &proto_id, &layer_index,
                                                &attr_id,
                                                &data, &datalen, &flags) == 0) {

        char buffer[4096 * 2 + 1];
        char flag_buffer[128];
        int i;
        if (!b) {
            printf("NO BUNDLE\n");
            break;
        }
        memset(buffer, 0, sizeof(buffer));
        if (attr_id == Q_MPA_TRANSACTION) {
            /* Print human-readable transactions */
            char const *trans_name;
            int trans_name_len = 0;
            qmdpi_data_transaction_get_name(bundle, *(uint32_t *)data,
                                            &trans_name, &trans_name_len);
            if (trans_name_len) {
                strncpy(buffer, trans_name, trans_name_len);
                buffer[trans_name_len] = 0;
            }
        } else {
            qmdpi_worker_data_attr_to_buffer(worker, bundle, buffer, sizeof(buffer),
                                             proto_id, attr_id, 0, data, datalen);
        }
        if (flags & QMDPI_ATTR_DIR_CTS) {
            flag_buffer[0] = 'c';
        } else {
            flag_buffer[0] = 's';
        }
        i = 1;
        flag_buffer[i] = 0;
        if (flags & QMDPI_ATTR_PARENT_START) {
            i += snprintf(flag_buffer + i, sizeof(flag_buffer) - i, ",start");
        }
        if (flags & QMDPI_ATTR_PARENT_END) {
            i += snprintf(flag_buffer + i, sizeof(flag_buffer) - i, ",end");
        }

        signature = qmdpi_worker_signature_get_byid(worker, b, proto_id);
        sig_name  = qmdpi_signature_name_get(signature);
        attr      = qmdpi_worker_attr_get_byid(worker, b, proto_id, attr_id);
        attr_name = qmdpi_attr_name_get(attr);

        printf("%d/%d %"PRIu64" f(%"PRIu64")/p(%s)/%s:%s=%s%s\n",
               qmdpi_engine_id_get(engine),
               qmdpi_worker_id_get(worker),
               packet_number,
               qmdpi_flow_id_get(flow),
               flag_buffer,
               sig_name,
               attr_name,
               datalen > (int)sizeof(buffer) ? "[truncated]" : "",
               buffer);

        if (opt.per_metadata_offloading) {
            if (layer_index) {
                ret = qmdpi_flow_offloading_attr_discard(worker, flow,
                                                         layer_index, proto_id, attr_id);
                if (ret < 0) {
                    fprintf(stderr, "Cannot discard metadata: %s, for %d:%d - %d\n",
                            qmdpi_error_get_string(NULL, ret), proto_id, attr_id, layer_index);
                }
            }
        }
    }
    if (opt.per_metadata_offloading) {
        ret = qmdpi_flow_offloaded_status_update(worker, flow);

        if (!QMDPI_RESULT_FLAGS_OFFLOADED_STATE(result_flags) && ret) {
            printf("%d/%d %"PRIu64" f(%"PRIu64")/p/classif_flags=classified_state_changed:%d,offloaded_state_changed:%d,pdu_dir_changed:%d,classified_state:%d,offloaded_state:%d,pdu_dir_index:%d,pdu_dir:%d,cacheable_state:%d,path_changed:%d\n",
                   qmdpi_engine_id_get(engine),
                   qmdpi_worker_id_get(worker),
                   packet_number,
                   qmdpi_flow_id_get(flow),
                   QMDPI_RESULT_FLAGS_CLASSIFIED_STATE_CHANGED(result_flags),
                   ret,
                   QMDPI_RESULT_FLAGS_PDU_DIR_CHANGED(result_flags),
                   QMDPI_RESULT_FLAGS_CLASSIFIED_STATE(result_flags),
                   ret,
                   QMDPI_RESULT_FLAGS_PDU_DIR_INDEX(result_flags),
                   QMDPI_RESULT_FLAGS_PDU_DIR(result_flags),
                   0,
                   QMDPI_RESULT_FLAGS_PATH_CHANGED(result_flags));
        }
    }
}

/**
 * Process packet by sending it to dpi process
 */
static void *pa_pkt_process(void *arg)
{
    struct pa_pkt *pkt;
    struct pa_thread *th;
    static time_t last_packet_ts = 0;

    th = (struct pa_thread *)arg;

    if (th->cpu_id >= 0) {
#if defined(__APPLE__) || defined(SUNCC)
        fprintf(stdout, "Cannot set thread affinity on this platform.\n");
#else
        cpu_set_t cpuset;
        int       s;

        CPU_ZERO(&cpuset);
        CPU_SET(th->cpu_id, &cpuset);

        s = pthread_setaffinity_np(pthread_self(), sizeof(cpuset), &cpuset);
        if (s != 0) {
            fprintf(stderr, "Cannot set affinity for thread %u: %s\n", th->thread_id,
                    strerror(s));
            return NULL;
        }

        fprintf(stdout, "Thread %i uses CPU %d\n", th->thread_id, th->cpu_id);
#endif
    }

    fprintf(stdout, "Thread %i %p started\n", th->thread_id,
            (void *) th->handle);

    while ((pkt = pa_pkt_dequeue(th)) != NULL) {
        int ret;

        th->packet_number = pkt->packet_number;

        if (qmdpi_worker_pdu_set(th->worker, pkt->data, pkt->len, &pkt->timestamp,
                                 pkt->link_mode, QMDPI_DIR_DEFAULT, 0) != 0) {
            ++th->st.errors;
            pa_pkt_free(pkt);
            continue;
        }
        do {
            struct qmdpi_result *result;
            ret = qmdpi_worker_process(th->worker, NULL, &result);
            if (ret < 0) {
                break;
            }

            if (opt.no_print == 0) {
                pa_result_display(pkt->packet_number, th->worker, result);
            }
        } while (ret == QMDPI_PROCESS_MORE);

        /**
         * Get DPI stats
         */
        if (ret < 0) {
            ++th->st.errors;
        } else {
            ++th->st.processed;
            switch (ret) {
                case QMDPI_PROCESS_IP_DEFRAG_DROP:
                    ++th->st.dropped_ip_defrag_drop;
                    break;
                case QMDPI_PROCESS_BOUNDS_FAILURE:
                    ++th->st.dropped_bounds_failure;
                    break;
                case QMDPI_PROCESS_BOUNDS_DROP:
                    ++th->st.dropped_bounds_drop;
                    break;
                case QMDPI_PROCESS_TCP_SEGMENT_DROPPED:
                    ++th->st.dropped_tcp_segment_dropped;
                    break;
            }
        }

        if (last_packet_ts < pkt->timestamp.tv_sec) {
            struct qmdpi_result *result;

            int nb_remaining = 100;
            do {
                ret = qmdpi_flow_expire_next(th->worker, &pkt->timestamp, &result);
                if (ret != 0) {
                    break;
                }

                nb_remaining --;
                if (opt.no_print == 0) {
                    pa_result_display(pkt->packet_number, th->worker, result);
                }
            } while (nb_remaining);
            if (nb_remaining) {
                last_packet_ts = pkt->timestamp.tv_sec;
            }
        }

        pa_pkt_free(pkt);
    }

    struct qmdpi_result *result;
    while (qmdpi_flow_expire_next(th->worker,
                                  NULL,
                                  &result) == 0) {
        if (opt.no_print == 0) {
            pa_result_display(0, th->worker, result);
        }
    }

    qmdpi_worker_destroy(th->worker);

    return NULL;
}

/**
 * Initialize the dpi engine
 */
static int pa_dpi_init(struct opt *opt)
{
    unsigned int i;
    int ret;

    char config_buffer[2048] = { };
    int pos = 0;

    for (i = 0; i < opt->cs.nb; i++) {
        pos += snprintf(config_buffer + pos,
                        sizeof(config_buffer) - pos,
                        "%s%s=%d",
                        pos > 0 ? ";" : "",
                        opt->cs.config[i].key,
                        opt->cs.config[i].value);
    }

    engine = qmdpi_engine_create(config_buffer);
    if (engine == NULL) {
        fprintf(stderr, "error initializing DPI engine: %s\n",
                qmdpi_error_get_string(NULL, qmdpi_error_get()));
        goto err_engine;
    }

    nr_cpus = qmdpi_engine_config_get(engine, "nb_workers");
    if (opt->cpu_mapping_id > 0 && (int)opt->cpu_mapping_id != nr_cpus) {
        fprintf(stderr,
                "Invalid cpu mapping configuration (%d threads, %u cpu mapping configurations)\n",
                nr_cpus, opt->cpu_mapping_id);
        goto err_engine;
    }

    bundle = qmdpi_bundle_create_from_file(engine, NULL);
    if (bundle == NULL) {
        fprintf(stderr, "Cannot load bundle: %s\n", qmdpi_error_get_string(NULL,
                                                                           qmdpi_error_get()));
        goto err_bundle;
    }

    qmdpi_bundle_activate(bundle);

    if (opt->load_libqmmletc) {
        libqmmletc = qmdpi_extralib_create_from_file(engine, QMDPI_EXTRALIB_LIBQMMLETC,
                                                     NULL);
        if (libqmmletc == NULL) {
            fprintf(stderr, "Cannot load extralib libqmmletc: %s\n",
                    qmdpi_error_get_string(NULL, qmdpi_error_get()));
            goto err_libqmmletc;
        }

        qmdpi_extralib_activate(libqmmletc);
    }

    if (opt->enabled_proto.nb == 0) {
        if ((ret = qmdpi_bundle_signature_enable_all(bundle)) < 0) {
            fprintf(stderr, "Failed to enable all protocols: %s\n",
                    qmdpi_error_get_string(bundle, ret));
            goto err_sig;
        }

        for (i = 0; i < opt->disabled_proto.nb; i++) {
            if ((ret = qmdpi_bundle_signature_disable(bundle,
                                                      opt->disabled_proto.proto[i])) < 0) {
                fprintf(stderr, "Failed to disable `%s' protocol: %s\n",
                        opt->disabled_proto.proto[i],
                        qmdpi_error_get_string(bundle, ret));
                goto err_sig;
            }
        }
    } else {
        for (i = 0; i < opt->enabled_proto.nb; i++) {
            if ((ret = qmdpi_bundle_signature_enable(bundle,
                                                     opt->enabled_proto.proto[i])) < 0) {
                fprintf(stderr, "Failed to enable `%s' protocol: %s\n",
                        opt->enabled_proto.proto[i],
                        qmdpi_error_get_string(bundle, ret));
                goto err_sig;
            }
        }
    }

    statistics_enabled = qmdpi_engine_config_get(engine, "monitoring_enable");

    flow_metrics_enabled = opt->flow_metrics_enable;

    classification_cache_enabled = qmdpi_engine_config_get(engine,
                                                           "classification_cache_enable");

    if (opt->hs.nb > 0) {
        pa_metadata_add(&opt->hs);
    }

    if (pa_tune_apply(&opt->ts) < 0) {
        goto err_tune;
    }

    if (statistics_enabled) {
        qmdpi_stats_init_start(engine);
    }

    if (opt->tdm_enable) {
        char *tdm_config = NULL;

        if (opt->tdm_config[0] != 0) {
            tdm_config = opt->tdm_config;
        }

        ret = qmdpi_report_init(engine, tdm_config);
        if (ret < 0) {
            fprintf(stderr, "Failed to init TDM: qmdpi_report_init.\n");
            if (ret == QMDPI_EPERM) {
                fprintf(stderr, " Permission denied (check license).\n");
            }
            goto err_tune;
        }
    }

    return 0;

err_tune:
err_sig:
    if (opt->load_libqmmletc) {
        qmdpi_extralib_destroy(libqmmletc);
    }

err_libqmmletc:
    qmdpi_bundle_destroy(bundle);

err_bundle:
    qmdpi_engine_destroy(engine);

err_engine:
    return -1;
}

/**
 * Cleanup the dpi engine
 */
static int pa_dpi_exit(struct opt *opt)
{
    if (statistics_enabled) {
        qmdpi_stats_dump(engine, stdout, (qmdpi_output_fn_t)fprintf);
        qmdpi_stats_exit(engine);
    }

    pa_metadata_del(&opt->hs);

    if (opt->tdm_enable) {
        qmdpi_report_cleanup(engine);
    }

    if (classification_cache_enabled) {
        qmdpi_classification_cache_reset(engine);
    }

    if (opt->load_libqmmletc) {
        qmdpi_extralib_destroy(libqmmletc);
    }
    qmdpi_bundle_destroy(bundle);
    qmdpi_engine_destroy(engine);
    qmdpi_license_destroy();
    return 0;
}


/**
 * ------------------------
 * Pcap functions
 * ------------------------
 */
/*
 * Open a pcap by its filename
 */
static pcap_t *pa_trace_open(const char *filename)
{
    pcap_t *pcap = NULL;
    char errbuf[PCAP_ERRBUF_SIZE];

    pcap = pcap_open_offline(filename, errbuf);

    if (pcap == NULL) {
        fprintf(stderr, "On trace %s, pcap_open: %s\n", filename, errbuf);
        return NULL;
    }
    fprintf(stdout, "Opening trace %s.\n", filename);

    return pcap;
}

/*
 * Close a pcap
 */
static void pa_trace_close(pcap_t *p)
{
    pcap_close(p);
}


/**
 * ------------------------
 * Thread functions
 * ------------------------
 */
/**
 * Launch all processing threads
 */
static int pa_thread_launch(void)
{
    int ret;
    int i;

    for (i = 0; i < nr_cpus; ++i) {
        threads[i].thread_id = i;
        threads[i].worker = qmdpi_worker_create(engine);
        pthread_mutex_init(&threads[i].lock, NULL);

        ret = pthread_create(&threads[i].handle, NULL, pa_pkt_process,
                             &threads[i]);
        if (ret != 0) {
            fprintf(stderr, "Starting threads error\n");
            return ret;
        }
    }

    if (statistics_enabled) {
        qmdpi_stats_init_end(engine);
    }

    return 0;
}

/**
 * Wait for all threads to terminate
 */
static inline void pa_thread_wait(void)
{
    int i;

    for (i = 0; i < nr_cpus; i++) {
        pthread_join(threads[i].handle, NULL);
        pthread_mutex_destroy(&threads[i].lock);
    }
}

/**
 * Dispatch packets for each thread (should be run by master thread)
 */
static int pa_pkt_dispatch(pcap_t *pcap, int nr_cpus)
{
    struct pcap_pkthdr *phdr;
    struct pa_pkt *packet;
    const u_char *pdata;
    int remove_llc = 0;
    int link_mode_loop = 0;
    int link_mode = QMDPI_PROTO_ETH;
    uint64_t packet_number = 0;

    switch (pcap_datalink(pcap)) {
        case DLT_EN10MB:
            link_mode = QMDPI_PROTO_ETH;
            break;
        case DLT_RAW:
        case 228: /* DLT_IPV4 - constant not defined for all platforms */
        case 18: /* Linux Classical-IP over ATM */
        case DLT_ATM_CLIP: /* (19) Linux Classical-IP over ATM */
        case 106: /* Linux Classical IP over ATM */
            link_mode = QMDPI_PROTO_IP;
            break;
        case DLT_LINUX_SLL:
            link_mode = QMDPI_PROTO_ETH;
            remove_llc = 1;
            break;
        case DLT_NULL:
        case DLT_LOOP:
            link_mode = QMDPI_PROTO_ETH;
            link_mode_loop = 1;
            break;
        case 229: /* DLT_IPV6 - constant not defined for all platforms */
            link_mode = QMDPI_PROTO_IP6;
            break;

        default:
            break;
    }

    while (pcap_next_ex(pcap, &phdr, &pdata) >= 0) {
        packet = pa_pkt_build(phdr, pdata, link_mode, remove_llc, link_mode_loop);
        if (packet == NULL) {
            ++pktdrop;
            continue;
        }
        uint32_t hashkey = qmdpi_packet_hashkey_get(pdata, phdr->caplen, link_mode);
        packet->packet_number = ++packet_number;
        pa_pkt_queue(&threads[hashkey % nr_cpus], packet);
    }

    return 0;
}



/**
 * ------------------------
 * Application functions
 * ------------------------
 */
/**
 * Process all pcap packets
 */
static int pa_run(void)
{
    pcap_t *pcap = NULL;
    int ret;
    size_t i;

    if (opt.tdm_enable) {
        if (qmdpi_report_start(engine)) {
            fprintf(stderr, "Failed to start tdm: qmdpi_report_start.\n");
        }
    }

    ret = pa_thread_launch();
    if (ret != 0) {
        return ret;
    }

    for (i = 0; i != tracesnr; ++i) {
        pcap = pa_trace_open(traces[i]);
        if (pcap == NULL) {
            continue;
        }

        ret = pa_pkt_dispatch(pcap, nr_cpus);
        if (ret != 0) {
            return ret;
        }

        pa_trace_close(pcap);
    }

    pa_pkt_stop();
    pa_thread_wait();

    if (opt.tdm_enable) {
        if (qmdpi_report_stop(engine)) {
            fprintf(stderr, "Failed to stop tdm: qmdpi_report_stop.\n");
        }

        printf("\nUNKNOWN\n");
        qmdpi_report_dump(engine, QMDPI_REPORT_TABLE_UNKNOWN, stdout,
                          (qmdpi_output_fn_t)fprintf);
        printf("\nPROTO_DISTRIB\n");
        qmdpi_report_dump(engine, QMDPI_REPORT_TABLE_PROTO_DISTRIB, stdout,
                          (qmdpi_output_fn_t)fprintf);
        printf("\nNEW\n");
        qmdpi_report_dump(engine, QMDPI_REPORT_TABLE_NEW, stdout,
                          (qmdpi_output_fn_t)fprintf);
        printf("\nCOMBINED\n");
        qmdpi_report_dump(engine, QMDPI_REPORT_TABLE_COMBINED, stdout,
                          (qmdpi_output_fn_t)fprintf);
        printf("\n");
    }

    return 0;
}

/**
 * Initialize the application
 */
static int pa_app_init(struct opt *opt)
{
    int i;

    threads = (struct pa_thread *)malloc(nr_cpus * sizeof(*threads));
    if (threads == NULL) {
        fprintf(stderr, "Can't malloc threads\n");
        return -1;
    }

    for (i = 0; i < nr_cpus; ++i) {
        PI_THREAD_INIT(&threads[i]);

        if (opt->cpu_mapping_id > 0) {
            threads[i].cpu_id = opt->cpu_mapping[i];
        }
    }

    return 0;
}

/**
 * Print general statistics
 */
static void pa_stats_print(void)
{
    int i;
    unsigned long dropped;
    fprintf(stdout, "\n"
            " --------------------\n"
            "| Packet Statistics  |\n"
            " --------------------\n\n");

    fprintf(stdout, "DPI:\n\n");
    for (i = 0; i < nr_cpus; ++i) {
        dropped = threads[i].st.dropped_ip_defrag_drop +
                  threads[i].st.dropped_bounds_failure +
                  threads[i].st.dropped_bounds_drop +
                  threads[i].st.dropped_tcp_segment_dropped;
        fprintf(stdout, " Core #%u: %lu processed (%lu dropped) / %lu errors\n", i,
                threads[i].st.processed,
                dropped,
                threads[i].st.errors);
        if (dropped != 0) {
            fprintf(stdout, "          (%lu dropped due to duplicated fragments,"
                    " %lu dropped due to unreadable L2-L4 data, %lu ignored due"
                    " to disabled IP defragmentation, %lu discarded by tcp reassembly)\n",
                    threads[i].st.dropped_ip_defrag_drop,
                    threads[i].st.dropped_bounds_failure,
                    threads[i].st.dropped_bounds_drop,
                    threads[i].st.dropped_tcp_segment_dropped);
        }
    }
    fprintf(stdout, "\n");
}

/**
 * Cleanup the application
 */
static int pa_app_exit(void)
{
    pa_stats_print();

    free(threads);
    return 0;
}

static int pa_simple_init(void)
{
    engine = qmdpi_engine_create("nb_flows=0;nb_workers=0");
    if (engine == NULL) {
        fprintf(stderr, "error initializing DPI engine: %s\n",
                qmdpi_error_get_string(NULL, qmdpi_error_get()));
        goto err;
    }

    bundle = qmdpi_bundle_create_from_file(engine, NULL);
    if (bundle == NULL) {
        fprintf(stderr, "Cannot load bundle\n");
        goto bundle_create_err;
    }

    qmdpi_bundle_activate(bundle);

    if (qmdpi_bundle_signature_enable_all(bundle) < 0) {
        fprintf(stderr, "Failed to enable all protocols\n");
        goto bundle_enable_err;
    }

    return 0;

bundle_enable_err:
    qmdpi_bundle_destroy(bundle);
bundle_create_err:
    qmdpi_engine_destroy(engine);
err:

    return -1;
}

static void pa_simple_exit(void)
{
    qmdpi_bundle_destroy(bundle);
    qmdpi_engine_destroy(engine);
}

static int pa_list_protocols_cb(struct qmdpi_bundle *bundle,
                                struct qmdpi_signature *signature,
                                void *arg)
{
    (void)bundle;
    (void)arg;

    printf("%s\n", qmdpi_signature_name_get(signature));

    return 0;
}

static int pa_list_protocols(void)
{
    if (pa_simple_init() < 0) {
        return 1;
    }

    qmdpi_bundle_signature_foreach(bundle, pa_list_protocols_cb, NULL);

    pa_simple_exit();

    return 0;
}

static int pa_list_attributes_cb(struct qmdpi_bundle *bundle,
                                 char const *signature_name,
                                 struct qmdpi_attr const *attr,
                                 void *arg)
{
    (void)bundle;
    (void)arg;

    printf("%s:%s\n", signature_name, qmdpi_attr_name_get(attr));

    return 0;
}

static int pa_list_attributes(char const *protocol)
{
    int ret;

    if (pa_simple_init() < 0) {

        return 1;
    }

    ret = qmdpi_bundle_attr_foreach(bundle, protocol, pa_list_attributes_cb, NULL);
    if (ret != 0) {
        fprintf(stderr, "Cannot find protocol %s\n", protocol);
    }

    pa_simple_exit();

    return ret;
}

static int pa_list_proto_tune_cb(struct qmdpi_bundle *bundle,
                                 char const *signature_name,
                                 struct qmdpi_tune *tune,
                                 void *arg)
{
    char const *tune_name = qmdpi_tune_name_get(tune);
    int val = qmdpi_tune_value_get(tune);

    (void)arg;
    (void)bundle;
    printf("%s:%s=%u\n", signature_name, tune_name, val);

    return 0;
}

static int pa_list_proto_for_proto_tune_cb(struct qmdpi_bundle *bundle,
                                           struct qmdpi_signature *signature,
                                           void *arg)
{
    (void)arg;

    qmdpi_bundle_tune_foreach(bundle, qmdpi_signature_name_get(signature),
                              pa_list_proto_tune_cb, (void *)(uintptr_t)signature);

    return 0;
}

static int pa_list_proto_tune(void)
{
    if (pa_simple_init() < 0) {
        return 1;
    }

    qmdpi_bundle_signature_foreach(bundle, pa_list_proto_for_proto_tune_cb, NULL);

    pa_simple_exit();

    return 0;
}

static int pa_list_config_values(void)
{
    engine = qmdpi_engine_create(NULL);
    if (engine == NULL) {
        fprintf(stderr, "error initializing DPI engine: %s\n",
                qmdpi_error_get_string(NULL, qmdpi_error_get()));
        return 1;
    }

    qmdpi_engine_config_dump(engine, stdout, (qmdpi_output_fn_t)fprintf);

    pa_simple_exit();

    return 0;
}

int main(int argc, char **argv)
{
    int ret;

    ret = pa_parse_param(argc, argv, &opt);
    if (ret == -1) {
        goto exit;
    }

    traces   = argv + ret;
    tracesnr = argc - ret;

    if (opt.memstat_enable) {
        ret = alloc_init();
        if (ret < 0) {
            goto exit;
        }
    }

    switch (opt.action) {
        case ACT_LIST_PROTO:
            exit(pa_list_protocols());
            break;
        case ACT_LIST_ATTR:
            exit(pa_list_attributes(opt.action_arg));
            break;
        case ACT_LIST_PROTO_TUNE:
            exit(pa_list_proto_tune());
            break;
        case ACT_LIST_CONFIG_VALUES:
            exit(pa_list_config_values());
            break;
    }

    ret = pa_dpi_init(&opt);
    if (ret != 0) {
        goto exit;
    }

    ret = pa_app_init(&opt);
    if (ret != 0) {
        goto exit;
    }

    if (nr_cpus) {
        pa_run();
    }

    pa_app_exit();

    if (opt.memstat_enable) {
        alloc_exit();
    }

    pa_dpi_exit(&opt);

exit:
    return ret == 0 ? 0 : 1;
}
