#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <math.h>
#include <qmdpi.h>
#include <qmdpi_bundle_api.h>
#include <string.h>

#include "memstat.h"
#include "pcap_analyzer_getter.h"
#include "alloc_list.h"

#define MSI_HTSZ 2048

struct msi_stat {
    LIST_ENTRY(msi_stat) next;
    unsigned int id;
    size_t nralloc;
    size_t nrbytes;
    ssize_t elmtsz;
    size_t elmtnr;
    ssize_t cursz;
    ssize_t maxsz;
    double average;
    double stddev;
    size_t nrtick;
    char fixed;
    char init;
};

struct msi_lookup_ht {
    LIST_HEAD(, msi_stat) ht[MSI_HTSZ];
};

static struct msi_lookup_ht __msil;
static struct msi_lookup_ht __cil;

static size_t msi_stat_hash(struct msi_stat *st)
{
    return st->id;
}

static struct msi_stat *msi_stat_new(unsigned int id, char init)
{
    struct msi_stat *st = malloc(sizeof(*st));
    if (st == NULL) {
        goto out;
    }

    st->id = id;
    st->nralloc = 0;
    st->nrbytes = 0;
    st->elmtsz = 0;
    st->elmtnr = 0;
    st->cursz = 0;
    st->maxsz = 0;
    st->fixed = 1;
    st->nrtick = 0;
    st->average = 0;
    st->init = init;

out:
    return st;
}

static void msi_stat_free(struct msi_stat *st)
{
    free(st);
}

static struct msi_stat *msi_stat_lookup(struct msi_lookup_ht *ht,
                                        struct msi_stat *st)
{
    size_t hash;
    struct msi_stat *s;

    hash = msi_stat_hash(st);

    LIST_FOREACH(s, &ht->ht[hash % MSI_HTSZ], next) {
        if (st->id == s->id && st->init == s->init) {
            return s;
        }
    }

    return NULL;
}

static void msi_stat_add(struct msi_lookup_ht *ht, struct msi_stat *st)
{
    size_t hash;
    hash = msi_stat_hash(st);

    LIST_INSERT_HEAD(&ht->ht[hash % MSI_HTSZ], st, next);
}

static void msi_stat_rm(struct msi_lookup_ht *ht, struct msi_stat *st)
{
    (void)ht;
    LIST_REMOVE(st, next);
}

int memstat_init(void)
{
    return 0;
}

void memstat_dump(void)
{
    size_t i;
    struct msi_stat *st;

    struct qmdpi_bundle *bundle = pcap_analyzer_bundle_get();

    if (bundle == NULL) {
        return;
    }

    for (i = 0; i < MSI_HTSZ; ++i) {
        while (!LIST_EMPTY(&__msil.ht[i])) {
            st = LIST_FIRST(&__msil.ht[i]);
            const char *name = qmdpi_malloc_struct_desc_get_byid(bundle, st->id);
            if (name == NULL) {
                printf("(null),%u,%u,%zu,%zu,%zu\n",
                       st->id, st->init,
                       st->nralloc, st->nrbytes, st->maxsz);

            } else {
                printf("%s,%u,%u,%zu,%zu,%zu\n",
                       name, st->id, st->init,
                       st->nralloc, st->nrbytes, st->maxsz);
            }
            msi_stat_rm(&__msil, st);
            msi_stat_free(st);
        }
    }

    for (i = 0; i < MSI_HTSZ; ++i) {
        while (!LIST_EMPTY(&__cil.ht[i])) {
            st = LIST_FIRST(&__cil.ht[i]);
            msi_stat_rm(&__cil, st);
            msi_stat_free(st);
        }
    }
}

void memstat_exit(void)
{
    memstat_dump();
}

#define ULAYER_NS_INIT (255 << 24) /* XXX compat.h */

int memstat_remove(ssize_t nr, ssize_t sz, unsigned int ctx, unsigned int id)
{
    struct msi_stat *st;
    struct msi_stat lookup;

    memset(&lookup, 0, sizeof(lookup));
    lookup.id = id;
    if ((int)ctx == ULAYER_NS_INIT) {
        lookup.init = 1;
    }

    st = msi_stat_lookup(&__msil, &lookup);
    if (st == NULL) {
        return -EINVAL;
    }

    if (st->cursz < nr * sz) {
        return -EINVAL;
    }

    st->cursz = (ssize_t)st->cursz - (nr * sz);

    return 0;
}

int memstat_add(ssize_t nr, ssize_t sz, unsigned int ctx, unsigned int id)
{
    struct msi_stat *st;
    struct msi_stat lookup;

    memset(&lookup, 0, sizeof(lookup));
    lookup.id = id;
    if ((int)ctx == ULAYER_NS_INIT) {
        lookup.init = 1;
    }

    st = msi_stat_lookup(&__msil, &lookup);
    if (st == NULL) {
        st = msi_stat_new(id, lookup.init);
        msi_stat_add(&__msil, st);
    }

    if (st == NULL) {
        return -ENOMEM;
    }

    ++st->nralloc;
    st->nrbytes = (ssize_t)st->nrbytes + (sz * nr);

    st->cursz = (ssize_t)st->cursz + (sz * nr);
    if (st->cursz > st->maxsz) {
        st->maxsz = st->cursz;
    }

    if (st->elmtsz == 0) {
        st->elmtsz = sz;
    }

    if (st->elmtnr == 0) {
        st->elmtnr = nr;
    }

    if (st->elmtsz != sz || st->elmtnr == 0) {
        st->fixed = 0;
    }

    lookup.id = ctx;
    lookup.init = 0;
    st = msi_stat_lookup(&__cil, &lookup);
    if (st == NULL) {
        st = msi_stat_new(ctx, 0);
        msi_stat_add(&__cil, st);
    }

    if (st == NULL) {
        return -ENOMEM;
    }

    ++st->nralloc;
    st->nrbytes = (ssize_t)st->nrbytes + (sz * nr);

    return 0;
}

int memstat_resize(ssize_t nr, ssize_t sz, size_t oldnr, size_t oldsz,
                   unsigned int ctx, unsigned int id)
{
    struct msi_stat *st;
    struct msi_stat lookup;

    memset(&lookup, 0, sizeof(lookup));
    lookup.id = id;
    if ((int)ctx == ULAYER_NS_INIT) {
        lookup.init = 1;
    }

    st = msi_stat_lookup(&__msil, &lookup);
    if (st == NULL) {
        st = msi_stat_new(id, lookup.init);
        msi_stat_add(&__msil, st);
    }

    if (st == NULL) {
        return -ENOMEM;
    }

    ++st->nralloc;
    st->nrbytes = (ssize_t)st->nrbytes + ((sz * nr) - (oldnr * oldsz));

    st->cursz = (ssize_t)st->cursz + ((sz * nr) - (oldnr * oldsz));
    if (st->cursz > st->maxsz) {
        st->maxsz = st->cursz;
    }

    if (st->elmtsz == 0) {
        st->elmtsz = sz;
    }

    if (st->elmtnr == 0) {
        st->elmtnr = nr;
    }

    if (st->elmtsz != sz || st->elmtnr == 0) {
        st->fixed = 0;
    }

    lookup.id = ctx;
    lookup.init = 0;
    st = msi_stat_lookup(&__cil, &lookup);
    if (st == NULL) {
        st = msi_stat_new(ctx, 0);
        msi_stat_add(&__cil, st);
    }

    if (st == NULL) {
        return -ENOMEM;
    }

    ++st->nralloc;
    st->nrbytes = (ssize_t)st->nrbytes + ((sz * nr) - (oldnr * oldsz));

    return 0;
}

int memstat_average(void)
{
    size_t i;
    struct msi_stat *st;

    for (i = 0; i < MSI_HTSZ; ++i) {
        LIST_FOREACH(st, &__msil.ht[i], next) {
            ++st->nrtick;
            st->average = st->average - (st->average / st->nrtick) +
                          ((double)(st->cursz) / st->nrtick);
            st->stddev = st->stddev - (st->stddev / st->nrtick) +
                         ((double)(st->cursz * st->cursz) / st->nrtick);
        }
    }
    return 0;
}
