/*
*   This file is a part of Qosmos ixEngine.
*   Copyright  Qosmos 2000-2016 - All rights reserved
*/
#include <getopt.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <qmdpi.h>

#include "parse_options.h"

/**
 * Print  usage
 *
 * @param name : program name
 */
void pa_usage(char *name)
{
    printf("Usage:\n\t%s [options] <pcap>\n\n",
           name);
    printf("Options:\n");
    printf("\t-q or --no-print                   Quiet mode\n");
    printf("\t--config <key>=<value>             Set engine configuration value\n");
    printf("\t--hook <proto>:<attr>              Register a specific attribute\n");
    printf("\t--print-flow-result                Print flow creation and classification callback information\n");
    printf("\t--tune <proto>:<param>=<val>       Configure protocol/signature parameters\n");
    printf("\t--enable-per-metadata-offloading   Enable flow offloading after registered attributes have been extracted once\n");
    printf("\t--cpu-mapping <value>,<value>,...  Set cpu mapping for each created instance\n");
    printf("\t--disable-protocol <protocol>      Disable a specific protocol\n");
    printf("\t--enable-protocol <protocol>       Enable a specific protocol\n");
    printf("\t--list-protocols                   List enabled protocols\n");
    printf("\t--list-attributes <protocol>       List protocol's attributes\n");
    printf("\t--list-proto-tunes                 List proto-tunes\n");
    printf("\t--list-config-values               List engine configuration values\n");
    printf("\t--memstat                          Enable statistics on allocations\n");
    printf("\t--tdm-enable                       Enable Traffic Distribution Module (TDM)\n");
    printf("\t--tdm-config <config>              Set TDM configuration value\n");
    printf("\t--load-libqmmletc                  Load libqmmletc, needed if the attributes base:ml_etc (Q_BASE_ML_ETC) or base:ml_etc_score (Q_BASE_ML_ETC_SCORE) are registered\n");
}

/**
 * Parse arguments passed to the application and fill an opt structure
 *
 * @param argc : number of arguments
 * @param argv : argument array
 * @param opt : structure filled with appropriate information extracted from
 * the arguments.
 * @return the number of parsed arguments
 */
int pa_parse_param(int argc, char *argv[], struct opt *opt)
{
    static struct option opts[] = {
        {"memstat", 0, 0, 0},
        {"no-print", 0, 0, 'q'},
        {"config", 1, 0, 0},
        {"hook", 1, 0, 0},
        {"print-flow-result", 0, 0, 0},
        {"tune", 1, 0, 0},
        {"cpu-mapping", 1, 0, 0},
        {"disable-protocol", 1, 0, 0},
        {"enable-protocol", 1, 0, 0},
        {"list-protocols", 0, 0, 0},
        {"list-attributes", 1, 0, 0},
        {"list-proto-tunes", 0, 0, 0},
        {"enable-per-metadata-offloading", 0, 0, 0},
        {"list-config-values", 0, 0, 0},
        {"tdm-enable", 0, 0, 0},
        {"tdm-config", 1, 0, 0},
        {"load-libqmmletc", 0, 0, 0},
        {0, 0, 0, 0},
    };

    int c, opti;
    unsigned int i;
    int thread_count_set = 0;
    int nb_flows_set = 0;

    OPT_INIT(opt);

    if (argc < 2) {
        goto usage;
    }

    while ((c = getopt_long(argc, argv, "q", opts, &opti)) != -1) {
        switch (c) {
            case 'q':
                opt->no_print = 1;
                break;
            case 0 :
                if (strcmp(opts[opti].name, "config") ==  0) {
                    char *start;
                    char *saveptr = NULL;
                    char *token;

                    for (start = optarg; (token = strtok_r(start, ";", &saveptr)) != NULL;
                            start = NULL) {
                        char *key = strtok(token, "=");
                        char *value = strtok(NULL, ";");
                        if (key == NULL || value == NULL) {
                            printf("Unable to parse configuration value `%s'\n", optarg);
                            goto usage;
                        }
                        CONFIG_STORE_ADD(&opt->cs, key, atoi(value));
                        if (strcmp(key, "fm_flow_metrics_enable") == 0) {
                            if (atoi(value)) {
                                opt->flow_metrics_enable = 1;
                            } else {
                                opt->flow_metrics_enable = 0;
                            }
                        }
                    }
                } else if (strcmp(opts[opti].name, "hook") ==  0) {
                    char *start;
                    char *saveptr = NULL;
                    char *token;

                    for (start = optarg; (token = strtok_r(start, ",", &saveptr)) != NULL;
                            start = NULL) {
                        char *layer = strtok(token, ":");
                        char *attr = strtok(NULL, ",");
                        if ((layer == NULL) || (attr == NULL)) {
                            goto usage;
                        }
                        HOOK_STORE_ADD(&opt->hs, layer, attr);
                    }
                } else if (strcmp(opts[opti].name, "tune") == 0) {
                    char *layer = strtok(optarg, ":");
                    char *attr = strtok(NULL, "=");
                    char *value = strtok(NULL, "=");
                    if (layer == NULL || attr == NULL || value == NULL) {
                        goto usage;
                    }
                    TUNE_STORE_ADD(&opt->ts, layer, attr, atoi(value));
                } else if (strcmp(opts[opti].name, "print-flow-result") == 0) {
                    opt->print_result = 1;
                } else if (strcmp(opts[opti].name, "cpu-mapping") == 0) {

                    char *mapping = strtok(optarg, ",");
                    do {
                        opt->cpu_mapping[opt->cpu_mapping_id++] = atoi(mapping);

                        mapping = strtok(NULL, ",");
                    } while (mapping != NULL);

                } else if (strcmp(opts[opti].name, "disable-protocol") == 0) {
                    PROTO_LIST_ADD(&opt->disabled_proto, optarg);
                } else if (strcmp(opts[opti].name, "enable-protocol") == 0) {
                    PROTO_LIST_ADD(&opt->enabled_proto, optarg);
                } else if (strcmp(opts[opti].name, "list-protocols") == 0) {
                    opt->action = ACT_LIST_PROTO;
                } else if (strcmp(opts[opti].name, "list-attributes") == 0) {
                    opt->action = ACT_LIST_ATTR;
                    opt->action_arg = optarg;
                } else if (strcmp(opts[opti].name, "list-proto-tunes") == 0) {
                    opt->action = ACT_LIST_PROTO_TUNE;
                } else if (strcmp(opts[opti].name, "enable-per-metadata-offloading") == 0) {
                    opt->per_metadata_offloading = 1;
                } else if (strcmp(opts[opti].name, "list-config-values") == 0) {
                    opt->action = ACT_LIST_CONFIG_VALUES;
                } else if (strcmp(opts[opti].name, "memstat") == 0) {
                    opt->memstat_enable = 1;
                } else if (strcmp(opts[opti].name, "tdm-enable") == 0) {
                    opt->tdm_enable = 1;
                    /* to enable tdm, we need to enable flow_metrics */
                    CONFIG_STORE_ADD(&opt->cs, "fm_flow_metrics_enable", 1);
                } else if (strcmp(opts[opti].name, "tdm-config") == 0) {
                    strcpy(opt->tdm_config, optarg);
                } else if (strcmp(opts[opti].name, "load-libqmmletc") == 0) {
                    opt->load_libqmmletc = 1;
                }

                break;
            default :
                goto usage;
        }
    }

    for (i = 0; i < opt->cs.nb; i++) {
        if (strcmp(opt->cs.config[i].key, "nb_workers") == 0) {
            thread_count_set = 1;
        }
        if (strcmp(opt->cs.config[i].key, "nb_flows") == 0) {
            nb_flows_set = 1;
        }
    }
    if (!thread_count_set) {
        CONFIG_STORE_ADD(&opt->cs, "nb_workers", 1);
    }
    if (!nb_flows_set) {
        CONFIG_STORE_ADD(&opt->cs, "nb_flows", 200000);
    }

    return optind;

usage :
    pa_usage(argv[0]);
    return -1;
}
