/*
*   This file is a part of Qosmos ixEngine.
*   Copyright  Qosmos 2000-2016 - All rights reserved
*/
#ifndef __PARSE_OPTIONS__
#define __PARSE_OPTIONS__

#include <stdint.h>
#include "tune.h"
#include "hooks.h"

#define MAX_PROTO   64

/**
 * Structure that keeps a proto name list
 */
struct proto_list {
    /**
     * The proto name list
     */
    char *proto[MAX_PROTO];
    /**
     * Number of elements in the list
     */
    size_t nb;
};

#define PROTO_LIST_INIT(pl)     ((pl)->nb = 0)
#define PROTO_LIST_OVERFLOW(pl) ((pl)->nb >= MAX_PROTO)
#define PROTO_LIST_GET(pl, i)   ((pl)->proto[i])

#define PROTO_LIST_ADD(pl, p) do {                                      \
        if(!PROTO_LIST_OVERFLOW(pl)) {                                  \
            ((pl)->proto[(pl)->nb++] = p);                              \
        } else {                                                        \
            fprintf(stderr, "Overflow: Cannot add proto %s\n", p);      \
        }                                                               \
    } while(0)

/**
 * Structure that keeps configuration value
 */
struct config {
    char *key;
    int value;
};

#define CONFIG_SET(c, k, v) do {                                        \
        (c)->key = k;                                                       \
        (c)->value = v;                                                     \
    } while(0)

#define MAX_CONFIG   64

/**
 * Structure that keeps configuration value list
 */
struct config_store {
    struct config config[MAX_CONFIG];
    size_t nb;
    size_t nb_md;
};

#define CONFIG_STORE_INIT(cs) do {                                      \
        (cs)->nb = 0;                                                       \
        (cs)->nb_md = 0;                                                    \
    } while(0)

#define CONFIG_STORE_ADD(hc, k, v) do {                                 \
        if((hc)->nb == MAX_CONFIG) {                                        \
            fprintf(stderr,                                             \
                    "Max config reached; cannot add config %s=%d",      \
                    k, v);                                              \
        } else {                                                            \
            CONFIG_SET(&((hc)->config[(hc)->nb]), k, v);                    \
            ++(hc)->nb;                                                     \
            if (strncmp(k, "md_", 3) == 0) {                            \
                ++(hc)->nb_md;                                              \
            }                                                               \
        }                                                                   \
    } while(0)

enum {
    ACT_PROCESS = 0,
    ACT_LIST_PROTO,
    ACT_LIST_ATTR,
    ACT_LIST_PROTO_TUNE,
    ACT_LIST_CONFIG_VALUES,
};

/**
 * Structure that keeps globals options
 */
struct opt {
    /**
     * List of protocols to disable
     */
    struct proto_list disabled_proto;
    /**
     * List of protocols to enable
     */
    struct proto_list enabled_proto;
    /**
    * List of engine configuration parameters
    */
    struct config_store cs;
    /**
     * List of protocols' attributes to hook
     */
    struct hook_store hs;
    /**
     * List of protocols' tune
     */
    struct tune_store ts;

    uint8_t print_result;

    uint32_t cpu_mapping[128];
    uint32_t cpu_mapping_id;

    /**
     * Action function identifier to execute
     */
    int32_t  action;
    /**
     * The user defined argument to pass to the action function
     */
    char  *action_arg;

    int no_print: 1;

    int per_metadata_offloading: 1;

    int memstat_enable: 1;

    int tdm_enable: 1;

    int flow_metrics_enable: 1;

    int load_libqmmletc: 1;

    char tdm_config[1024];
};

#define OPT_INIT(o)                             \
    do {                                        \
        (o)->no_print = 0;                      \
        CONFIG_STORE_INIT(&(o)->cs);            \
        HOOK_STORE_INIT(&(o)->hs);              \
        TUNE_STORE_INIT(&(o)->ts);              \
        PROTO_LIST_INIT(&(o)->disabled_proto);  \
        PROTO_LIST_INIT(&(o)->enabled_proto);   \
        (o)->print_result = 0;                  \
        (o)->cpu_mapping_id = 0;                \
        (o)->action = ACT_PROCESS;              \
        (o)->action_arg = NULL;                 \
        (o)->per_metadata_offloading = 0;       \
        (o)->memstat_enable = 0;                \
        (o)->tdm_enable = 0;                    \
        (o)->flow_metrics_enable = 0;           \
        (o)->load_libqmmletc = 0;               \
        (o)->tdm_config[0] = 0;                 \
    } while(0)

void pa_usage(char *name);

int pa_parse_param(int argc, char *argv[], struct opt *opt);

#endif
