/*
*   This file is a part of Qosmos ixEngine.
*   Copyright  Qosmos 2000-2017 - All rights reserved
*/
#include <string.h>
#include <stdlib.h>

#include "fasthash.h"
#include "dns.h"

void
dns_ht_init(struct dns_ht *ht)
{
    ht->ht_size = 0;
    memset(ht->ht, 0, DNS_HT_SIZE * sizeof(struct dns_ht_entry *));
}

static inline void
dns_ht_entry_init(struct dns_ht_entry *entry,
                  unsigned int hostlen, uint32_t addr)
{
    entry->hostname = malloc(hostlen);
    entry->dest_addr = addr;
    entry->next = NULL;
}

void
dns_ht_clean(struct dns_ht *ht)
{
    int ht_index;

    for (ht_index = 0; ht_index < DNS_HT_SIZE; ht_index++) {
        while (ht->ht[ht_index] != NULL) {
            struct dns_ht_entry *entry = ht->ht[ht_index];
            ht->ht[ht_index] = entry->next;
            free(entry->hostname);
            free(entry);
        }
    }

    ht->ht_size = 0;
}

static inline int
dns_ht_hash(uint32_t addr)
{
    return pcap_logger_fasthash_32((uint8_t *)&addr) % DNS_HT_SIZE;
}

struct dns_ht_entry *
dns_ht_entry_fetch(struct dns_ht *ht, uint32_t addr)
{
    struct dns_ht_entry *entry = ht->ht[dns_ht_hash(addr)];

    while (entry != NULL && entry->dest_addr != addr) {
        entry = entry->next;
    }

    return entry;
}

struct dns_ht_entry *
dns_ht_entry_insert(struct dns_ht *ht, uint32_t addr,
                    char *hostname, unsigned int hostlen)
{
    struct dns_ht_entry **entry = &(ht->ht[dns_ht_hash(addr)]);

    /* find bucket for ht entry */
    while (*entry != NULL && (*entry)->dest_addr != addr) {
        entry = &((*entry)->next);
    }

    /* new attribute/value entry */
    if (*entry == NULL) {
        *entry = malloc(sizeof(struct dns_ht_entry));
        if (*entry == NULL) {
            return NULL;
        }
        dns_ht_entry_init(*entry, hostlen + 1, addr);
        ht->ht_size++;

    } else if ((*entry)->hostname != NULL &&
               strlen((*entry)->hostname) != hostlen) {
        free((*entry)->hostname);
        (*entry)->hostname = malloc(hostlen + 1);
    }

    /* set/update the hostname value for this IP */
    if ((*entry)->hostname != NULL) {
        strncpy((*entry)->hostname, hostname, hostlen);
        (*entry)->hostname[hostlen] = '\0';
    }

    return *entry;
}

