/*
*   This file is a part of Qosmos ixEngine.
*   Copyright  Qosmos 2000-2017 - All rights reserved
*/
#include <stdlib.h>
#include <string.h>

#include <dpi/protodef.h>

#include "exclusion.h"
#include "config.h"
#include "data_format.h"

/* Exclusion patterns list */
static struct attribute_excl_list EXCLS;

int
excls_attribute_is_excluded(int proto_id, int attr_id,
                            const uint8_t *value, unsigned int value_len)
{
    unsigned int pattern_len;
    int excl_index = 0;
    char *pattern;

    if (value_len == 0) {
        return 0;
    }

    while (excl_index < EXCLS.excl_len) {
        pattern = EXCLS.excl[excl_index].pattern;
        pattern_len = EXCLS.excl[excl_index].pattern_len;

        if ((EXCLS.excl[excl_index].proto_id == proto_id) &&
                (EXCLS.excl[excl_index].attr_id == attr_id)) {

            if (pattern[0] != '*' &&
                    pattern[pattern_len - 1] != '*') {
                /* exact matching */
                if (pattern_len == value_len &&
                        !memcmp(pattern, value, value_len)) {
                    return 1;
                }

            } else if (pattern[pattern_len - 1] == '*') {
                if (pattern[0] != '*') {
                    /* prefix matching */
                    if (data_str_start_with((char *)value, value_len,
                                            pattern, pattern_len - 1)) {
                        return 1;
                    }
                } else if (pattern_len > 1) {
                    /* word search */
                    if (data_str_find((char *)value, value_len,
                                      pattern + 1, pattern_len - 2)) {
                        return 1;
                    }
                }

                /* suffix matching */
            } else if (data_str_end_with((char *)value, value_len,
                                         pattern + 1, pattern_len - 1)) {
                return 1;
            }
        }
        excl_index++;
    }

    return 0;
}

void
excls_setup_line(char *line, int line_count)
{
    const char token[2] = { DATA_CFG_SEP, '\0' };
    char *tok = strtok(line, token);

    /* field 1 is "proto" */
    if (tok == NULL) {
        fprintf(stderr, "\r[ERR] config:[exclude]line %d: syntax error\n", line_count);
        return;
    }
    EXCLS.excl[EXCLS.excl_len].proto_id = data_signame_get_byname(tok);
    if (EXCLS.excl[EXCLS.excl_len].proto_id <= 0) {
        fprintf(stderr, "\r[ERR] config:[exclude]line %d: unknown protocol: %s\n",
                line_count, tok);
        return;
    }

    /* field 2 is "attr" */
    if (!(tok = strtok(NULL, token))) {
        fprintf(stderr, "\r[ERR] config:[exclude]line %d: syntax error\n", line_count);
        return;
    }
    EXCLS.excl[EXCLS.excl_len].attr_id = data_attrname_get_byname(
                                             EXCLS.excl[EXCLS.excl_len].proto_id, tok);
    if (EXCLS.excl[EXCLS.excl_len].attr_id <= 0) {
        fprintf(stderr, "\r[ERR] config:[exclude]line %d: unknown attribute: %s\n",
                line_count, tok);
        return;
    }

    /* field 3 is "pattern" */
    if (!(tok = strtok(NULL, token))) {
        fprintf(stderr, "\r[ERR] config:[exclude]line %d: syntax error\n", line_count);
        return;
    }
    if (strlen(tok) == 1 && tok[0] == '*') {
        fprintf(stderr, "\r[ERR] config:[exclude]line %d: syntax error\n", line_count);
        return;
    }

    strcpy(EXCLS.excl[EXCLS.excl_len].pattern, tok);
    EXCLS.excl[EXCLS.excl_len].pattern_len = strlen(tok);
    EXCLS.excl_len++;
}

void
excls_setup(void)
{
    EXCLS.excl_len = 0;
    memset(EXCLS.excl, 0, EXCL_MAX_CNT * sizeof(attribute_excl_t));
}
