/*
*   This file is a part of Qosmos ixEngine.
*   Copyright  Qosmos 2000-2017 - All rights reserved
*/
#ifndef __FLOWS__
#define __FLOWS__

#define FLOW_MAX_PKT_CNT 5
#define FLOW_HT_SIZE 4096
#define L4_BUFFER_LEN   50
#define PATH_LEN 20

#include <dpi/protodef.h>
#include "attributes.h"

/**
 * Statistics, unknwon L4-payload and Qosmos ixEngine attributes
 * related to a unique flow.
 */
struct flow_item {
    int64_t first_pkt_number;   /* packet number of the first injected packet */
    uint32_t flow_id;           /* unique flow id for the packets of this flow */

    uint32_t top_proto_id;      /* proto ID of the top layer from flow path */
    uint32_t l4_proto_id;       /* transport layer proto ID */
    int32_t l3_addr_clt;        /* IPv4 src address - used for DNS caching */
    int32_t l3_addr_srv;        /* IPv4 dest address - used for DNS caching */

    int flow_dir;               /* flow direction */

    attribute_list_t *attributes; /* relevant Qosmos ixEngine attributes */
    uint32_t         *path;       /* classification path */

    uint8_t  payload_buffer_cts[L4_BUFFER_LEN]; /* first bytes of CTS stream */
    uint8_t  payload_buffer_stc[L4_BUFFER_LEN]; /* first bytes of STC stream */
    uint64_t vol_pld_cts;                       /* CTS stream total volume */
    uint64_t vol_pld_stc;                       /* STC stream total volume */
    uint32_t cnt_pld_cts;                       /* count of non-empty PDUs CTS */
    uint32_t cnt_pld_stc;                       /* count of non-empty PDUs STC */
};

struct flow_statistics {
    struct flow_item *flow;
    uint64_t vol_pkt_cts;       /* total raw packets volume */
    uint64_t vol_pkt_stc;       /* total raw packets volume */
    uint32_t duration;          /* duration from the first packet to the last */
    uint32_t first_pkt_ts;      /* timestamp -> used for duration computation */
    uint32_t packet_count_cts;  /* client-to-server injected packets count */
    uint32_t packet_count_stc;  /* server-to-client injected packets count */
    int      established: 1;    /* no handshake (TCP) seen */
};

/**
 * Flow item allocation
 */
static inline void
flow_item_init(struct flow_item *flow, int flow_id)
{
    flow->flow_id = flow_id;
    flow->flow_dir = QMDPI_DIR_UNKNOWN;
    flow->first_pkt_number = -1;
    flow->vol_pld_cts = 0;
    flow->vol_pld_stc = 0;
    flow->cnt_pld_cts = 0;
    flow->cnt_pld_stc = 0;
    flow->top_proto_id = 0;
    flow->l4_proto_id = Q_PROTO_UNKNOWN;
    flow->l3_addr_srv = -1;
    flow->l3_addr_clt = -1;
    flow->attributes = NULL;
    flow->path = NULL;
}

/**
 * Flow item cleaning
 */
static inline void
flow_item_clean(struct flow_item *flow)
{
    attribute_list_t *attribute = flow->attributes;
    while (attribute != NULL) {
        attribute_list_t *rem = attribute;
        attribute = rem->next;
        attribute_list_item_clean(&(rem->item));
        free(rem);
    }
    free(flow->path);
    flow->path = NULL;
}

/**
 * Add an attribue <item> reference into the attribute item list of a <flow>.
 *
 * Returns: 1 if the attribute reference was successfully added to the flow,
 *          0 if it failed.
 */
static inline int
flow_attr_list_item_add(struct flow_item *flow, struct attribute_item *item)
{
    return attribute_list_item_add(&flow->attributes, item);
}

#endif
