/*
*   This file is a part of Qosmos ixEngine.
*   Copyright  Qosmos 2000-2017 - All rights reserved
*/
#include <stdlib.h>
#include <string.h>
#include <qmdpi.h>

#include "parse_options.h"
#include "data_format.h"
#include "hooks.h"

extern struct qmdpi_bundle *bundle;

void
hooks_setup_line(char *line, int line_count,
                 struct hook_store *hs)
{
    unsigned int line_len = strlen(line);

    struct attribute *attr_desc = NULL;
    char  proto[128] = "";
    char  attr[128] = "";
    char *sep = NULL;
    int   protoid = 0;
    int   attrid = 0;

    memset(proto, 0, 128);
    memset(attr, 0, 128);

    if ((sep = strchr(line, DATA_CFG_SEP)) == NULL) {
        fprintf(stderr, "\r[ERR] config:[attributes]line %d: syntax error\n",
                line_count);
        return;
    }

    strncpy(proto, line, sep - line);
    strncpy(attr, sep + 1, line_len - strlen(proto) - 1);
    protoid = data_signame_get_byname(proto);
    attrid = data_attrname_get_byname(protoid, attr);

    if (protoid <= 0 || attrid <= 0) {
        fprintf(stderr,
                "\r[ERR] config:[attributes]line %d: unknown to this version of csma: %s:%s\n",
                line_count, proto, attr);
        return;
    }

    /* getting the attribute definition */
    attr_desc = attribute_def_get_by_ids(protoid, attrid);
    if (attr_desc == NULL) {
        fprintf(stderr,
                "\r[ERR] config:[attributes]line %d: not supported by this version of csma: %s:%s\n",
                line_count, proto, attr);
        return;
    }

    /* registering the attribute in the ixE */
    if (protoid == Q_PROTO_TCP && attrid == Q_MPA_STREAM) {
        logs_param_tcp_stream_set(1);
    } else if (protoid == Q_PROTO_UDP && attrid == Q_MPA_STREAM) {
        logs_param_udp_stream_set(1);
    } else if (protoid == Q_PROTO_IP && attrid == Q_IP_RESOLV_NAME) {
        HOOK_STORE_ADD(hs, "dns", "query");
        HOOK_STORE_ADD(hs, "dns", "host_addr");
    } else {
        HOOK_STORE_ADD(hs, proto, attr);
    }

    /* activating the attribute */
    attr_desc->activated = 1;
}

int
hooks_apply(struct hook_store *hs)
{
    char *layer;
    char *attr;
    struct hook h;
    unsigned int i;

    for (i = 0; i < hs->nb; ++i) {
        h = HOOK_STORE_GET(hs, i);
        layer = HOOK_LAYER(&h);
        attr = HOOK_ATTR(&h);

        int layer_id = data_signame_get_byname(layer);
        if (layer_id <= 0) {
            continue;
        }

        if (data_attrname_get_byname(layer_id, attr) <= 0) {
            continue;
        }

        qmdpi_bundle_attr_register(bundle, layer, attr);
    }

    return 0;
}

void
hooks_remove(struct hook_store *hs)
{
    char *layer;
    char *attr;
    struct hook h;
    unsigned int i;

    for (i = 0; i < hs->nb; ++i) {
        h = HOOK_STORE_GET(hs, i);
        layer = HOOK_LAYER(&h);
        attr = HOOK_ATTR(&h);

        int layer_id = data_signame_get_byname(layer);
        if (layer_id <= 0) {
            continue;
        }

        if (data_attrname_get_byname(layer_id, attr) <= 0) {
            continue;
        }

        qmdpi_bundle_attr_unregister(bundle, layer, attr);
    }
}

