/*
*   This file is a part of Qosmos ixEngine.
*   Copyright  Qosmos 2000-2017 - All rights reserved
*/
#ifndef __HOOKS__
#define __HOOKS__

#include "logs.h"

#define HOOK_VALUE_MAX_LEN 128
#define MAX_HOOKS   64

/**
 * Structure that define a protocol's attribute hook
 */
struct hook {
    char layer[HOOK_VALUE_MAX_LEN];
    char attr[HOOK_VALUE_MAX_LEN];
};

#define HOOK_SET_LAYER(h, l) (strcpy((h)->layer, l))
#define HOOK_SET_ATTR(h, a) (strcpy((h)->attr, a))
#define HOOK_SET(h, l, a) do {                                              \
        HOOK_SET_LAYER(h, l);                                               \
        HOOK_SET_ATTR(h, a);                                                \
    } while(0);
#define HOOK_LAYER(h) ((h)->layer)
#define HOOK_ATTR(h) ((h)->attr)

/**
 * Structure that keeps a list of protocols' attribute hooks
 */
struct hook_store {
    /**
     * List of hooks
     */
    struct hook hook[MAX_HOOKS];
    /**
     * Number of hook in the list
     */
    size_t nb;
};

#define HOOK_STORE_INIT(hs) ((hs)->nb = 0)
#define HOOK_STORE_ADD(hs, l, a) do {                                       \
        if((hs)->nb == MAX_HOOKS) {                                         \
            fprintf(stderr,                                                 \
                    "Max hook reached; cannot hook %s:%s", l, a);       \
        } else {                                                            \
            HOOK_SET(&((hs)->hook[(hs)->nb]), l, a);                        \
            ++(hs)->nb;                                                     \
        }                                                                   \
    } while(0);
#define HOOK_STORE_GET(hs, i) ((hs)->hook[i])

/**
 * Read attributes hooks file description and populate hook store.
 */
void hooks_setup_line(char *line, int line_count, struct hook_store *hs);

/**
 * Function that registers all layer hooks in a hook store.
 */
int hooks_apply(struct hook_store *hs);

/**
 * Function that unregisters all layer hooks in a hook store.
 */
void hooks_remove(struct hook_store *hs);

#endif
