/*
*   This file is a part of Qosmos ixEngine.
*   Copyright  Qosmos 2000-2017 - All rights reserved
*/
#ifndef __LOGS__
#define __LOGS__

#include <qmdpi.h>

#include "flows.h"
#include "attributes.h"

#define LOG_FILENAME "./output.csv"
#define LOG_VERSION_CSV "4"

extern struct flow_ht FLOW_HT;
extern struct attribute_ht ATTRIBUTE_HT;

typedef enum {
    LOGS_SPLIT_NO,
    LOGS_SPLIT_SIZE,
    LOGS_SPLIT_TIME
} logs_split_policy_t;

/**
 * Init/cleanup functions.
 */
void logs_init(void);
void logs_cleanup(void);

/**
 * This function sets the CSV LOG split <policy> and <threshold>.
 * If set, the LOG writing rotates when the given <threshold> is reached.
 *
 * Default behavior: there's no file split.
 */
void logs_param_split_policy_set(logs_split_policy_t p, uint64_t threshold);

/**
 * This function sets if the CSV output should contain the
 * TCP UNKNOWN L4 payload column or not.
 *
 * When this function's <parameter> is set to '0', the payload
 * saving and logging is disabled.
 *
 * Default behavior: the column is included in the CSV (1).
 */
void logs_param_tcp_stream_set(int parameter);

/**
 * This function sets if the CSV output should contain the
 * UDP UNKNOWN L4 payload column or not.
 *
 * When this function's <parameter> is set to '0', the payload
 * saving and logging is disabled.
 *
 * Default behavior: the column is included in the CSV (1).
 */
void logs_param_udp_stream_set(int parameter);

/**
 * This function sets if the CSV output should contain protocol names
 * (like "tcp" or "udp") instead of protocol IDs (like 205 or 216).
 *
 * When this function's <parameter> is set to '1', protocol names
 * are written.
 *
 * Default behavior: protocol IDs (integer) are used (0).
 */
void logs_param_with_proto_names(int parameter);

/**
 * This function sets if the CSV output should contain an additional column
 * will the full protocol path information of the flow.
 *
 * When this function's <parameter> is set to '1', protocol path is added
 */
void logs_param_with_path(int parameter);

/**
 * This function writes the <flow> information dump to the CSV <filename>,
 * with LOG rotation mechanism.
 *
 * The function can be called with 3 parameter configurations:
 *
 * (1) If <filename> is NOT NULL and <flow> is NULL, then a NEW CSV LOG file
 *     is opened for writing, and the CSV header line is written to it.
 *     This function MUST be called at first with this parameter configuration.
 *
 * (2) If <filename> is NULL and <flow> is NULL, then the CSV file is closed.
 *
 * (3) If <filename> is NULL and <flow> is NOT NULL, then:
 *
 *     >> if a CSV file is currently opened, then a new <flow>
 *        dump line is appended to this current CSV file.
 *
 *     >> else (CSV file is closed), a LOG rotation is performed.
 *        The previous <filename> LOG file is moved to <filename>-YYYYMMDD-NNN
 *        Then <filename> is opened again for writing, and:
 *        - the CSV header line is written to it,
 *        - the <flow> dump line is appended to it.
 *
 *        Notes: YYYYMMDD is today's date,
 *               NNN is an index, incremented while the date does not change.
 *
 * LOG rotation example (filename="mylog"):
 * > mylog (current LOG file)
 * > mylog-20161129-000 (oldest archive)
 * > mylog-20161129-001
 * > mylog-20161129-002
 * > mylog-20161130-000
 * > mylog-20161130-001 (newest archive)
 *
 * Returns: -1 if the file opening failed,
 *          the number of bytes (>=0) written to the current LOG file.
 */
int logs_file_write(const char *filename, struct flow_statistics *flow);

/**
 * New attribute from the Qosmos ixEngine.
 *
 * This function allocates the new attribute in the attributes HT (if new),
 * registers a new DNS entry in the DNS Caching HT (if any),
 * and adds a pointer to this attribute item in the attribute list
 * of the <flow>.
 *
 * This function also performs attribute value filtering based on the
 * attributes exclusion file.
 *
 * Returns:  1 if the attribute was successfully added,
 *           0 if the attribute was filtered,
 *          -1 if attribute storage failed.
 */
int logs_new_attribute(struct flow_item *flow, int layer_id, int attr_id,
                       const uint8_t *value, uint32_t value_len);

/**
 * New packet from the Qosmos ixEngine.
 *
 * This function updates packet metrics of the <flow>, stores each stream
 * direction first bytes (transport layer payload), and stores IP addresses
 * (used by DNS caching).
 *
 * Returns:  1 if <flow> context was changed,
 *           0 if <flow> is unchanged,
 *          -1 if the <flow> allocation failed.
 */
int logs_new_packet(struct flow_item **flow, uint64_t pkt_num,
                    uint32_t cnx_type, uint32_t cnx_way,
                    uint8_t *payload, uint32_t payload_len,
                    uint32_t ip_src, uint32_t ip_dst);

/**
 * The classification <path> of a <flow> has changed.
 *
 * This function updates the "top proto" of the <flow>.
 */
void logs_new_classification_path(struct flow_item *flow,
                                  struct qmdpi_path *path);

/**
 * Flow data finalization and cleaning.
 *
 * SHOULD be called after all packets all the flow have been captured.
 */
void logs_flow_finalize(struct flow_statistics *stats);

#endif
