/*
*   This file is a part of Qosmos ixEngine.
*   Copyright  Qosmos 2000-2020 - All rights reserved
*/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include <pcap.h>
#include <signal.h>

#include <qmdpi.h>
#include <qmdpi_bundle_api.h>

#include "logs.h"
#include "config.h"
#include "data_format.h"
#include "parse_options.h"
#include <packet_helper.h>

static struct qmdpi_engine *engine;
static struct qmdpi_worker *worker;
struct qmdpi_bundle *bundle;

/* libpcap */
static pcap_t  *pcap_fd_in = NULL;
static int      pcap_file_index = 0;
static uint64_t pcap_pkt_number = 0;
static int      pcap_remove_lcc = 0;

/* juniper ethernet capture */
#define JUNIPER_FRAME "MGC"

/* packet link-layer mode */
int link_mode = QMDPI_PROTO_ETH;

/* initialize Qosmos ixEngine */
static void
pcap_logger_ixe_init(struct opt *opt)
{
    char config_str[256];
    int ret;

    snprintf(config_str, 256,
             "injection_mode=packet;"       \
             "nb_workers=1;"                \
             "nb_flows=%d;"                 \
             "fm_flow_metrics_enable=1;"    \
             "fm_tcp_reass_enable=1;"       \
             "fm_ip_defrag_enable=1;",
             OPT_MCNX(opt));

    /* create engine instance */
    engine = qmdpi_engine_create(config_str);
    if (engine == NULL) {
        fprintf(stderr, "\r[ERR] Error initializing DPI engine: %s\n",
                qmdpi_error_get_string(NULL, qmdpi_error_get()));
        goto err_engine;
    }

    /* create worker instance */
    worker = qmdpi_worker_create(engine);
    if (worker == NULL) {
        fprintf(stderr, "\r[ERR] Cannot create worker instance: %s\n",
                qmdpi_error_get_string(NULL, qmdpi_error_get()));
        goto err_worker;
    }

    /* create default bundle instance */
    bundle = qmdpi_bundle_create_from_file(engine, NULL);
    if (bundle == NULL) {
        fprintf(stderr, "\r[ERR] Cannot load bundle: %s\n",
                qmdpi_error_get_string(NULL, qmdpi_error_get()));
        goto err_bundle;
    }

    /* enable all signatures on bundle */
    ret = qmdpi_bundle_signature_enable_all(bundle);
    if (ret < 0) {
        fprintf(stderr, "\r[ERR] Error enabling all protocols: %s\n",
                qmdpi_error_get_string(bundle, ret));
        goto err_sig;
    }

    /* activate bundle */
    ret = qmdpi_bundle_activate(bundle);
    if (ret < 0) {
        fprintf(stderr, "\r[ERR] Cannot activate bundle: %s\n",
                qmdpi_error_get_string(bundle, ret));
        goto err_sig;

    }

    /* disabling legacy HTTP Fast Cases */
    qmdpi_bundle_tune_set(bundle, "http", "disable_fast_cases", 1);

    return;

err_sig:
    qmdpi_bundle_destroy(bundle);
err_bundle:
    qmdpi_worker_destroy(worker);
err_worker:
    qmdpi_engine_destroy(engine);
err_engine:
    exit(-1);
}

/* pcap link mode detection */
static inline void
pcap_logger_pcap_datalink(pcap_t *pcap)
{
    switch (pcap_datalink(pcap)) {
        case DLT_LINUX_SLL:
            pcap_remove_lcc = 1;
            break;
        case DLT_ATM_CLIP:
        case DLT_RAW:
        case 18:
        case 106:
            link_mode = QMDPI_PROTO_IP;
            break;
        default:
            ;
    }
}

/* open an ethernet interface for packet reading */
static inline pcap_t *
pcap_logger_pcap_open_live(const char *net_if)
{
    char errbuf[PCAP_ERRBUF_SIZE];
    pcap_if_t *devices = NULL;
    pcap_t *pcap = NULL;

    /* check for a default interface */
    if (net_if == NULL) {
        if (pcap_findalldevs(&devices, errbuf) != 0) {
            fprintf(stderr, "\r[ERR] Couldn't find default device: %s\n", errbuf);
            devices = NULL;
            goto end;
        } else if (devices == NULL) {
            fprintf(stderr, "\r[ERR] Couldn't find default device\n");
            goto end;
        }

        net_if = devices[0].name;
        fprintf(stdout, "[MSG] Opening interface %s\n", net_if);
    }

    /* open interface */
    pcap = pcap_open_live(net_if, 65535, 1, 0, errbuf);
    if (pcap == NULL) {
        fprintf(stderr, "\r[ERR] libpcap: pcap_open(%s): %s\n", net_if, errbuf);
        goto end;
    }

    /* datalink detection */
    pcap_logger_pcap_datalink(pcap);

end:
    if (devices != NULL) {
        pcap_freealldevs(devices);
    }
    return pcap;
}

/* open a PCAP file for packet reading */
static inline pcap_t *
pcap_logger_pcap_open(char const filename[], long *file_size)
{
    char    errbuf[PCAP_ERRBUF_SIZE];
    pcap_t *fd;

    /* open file for size reading */
    FILE *file = fopen(filename, "r");
    if (file == NULL) {
        fprintf(stderr, "\r[ERR] Cannot open file: %s\n", filename);
        return NULL;
    }

    if (fseek(file, 0, SEEK_END) == 0) {
        *file_size = ftell(file);
    }
    fclose(file);

    /* open file with libpcap */
    fd = pcap_open_offline(filename, errbuf);
    if (fd == 0) {
        fprintf(stderr, "\r[ERR] Cannot open PCAP: %s: %s\n", filename, errbuf);
        return NULL;
    }

    /* datalink detection */
    pcap_logger_pcap_datalink(fd);

    return fd;
}

/* update the classification path of a given flow */
static inline void
pcap_logger_path_update(struct qmdpi_flow *f, struct qmdpi_result *result)
{
    struct flow_item *fi = (struct flow_item *)qmdpi_flow_user_handle_get(f);
    struct qmdpi_path *path = qmdpi_result_path_get(result);

    if (fi != NULL) {
        logs_new_classification_path(fi, path);
    }
}

/* process Qosmos ixEngine results for a given flow */
static inline void
pcap_logger_result_expire(struct qmdpi_result *result)
{
    struct flow_statistics stats;
    struct qmdpi_flow *flow;
    struct timeval duration;
    struct timeval start;

    if ((flow = qmdpi_result_flow_get(result)) == NULL) {
        return;
    }

    /* get user handle */
    struct flow_item *fi = (struct flow_item *)qmdpi_flow_user_handle_get(flow);
    if (fi == NULL) {
        return;
    }

    /* flow information */
    struct qmdpi_flow_info *info = qmdpi_flow_info_get(flow);

    /* final classification path storage */
    pcap_logger_path_update(flow, result);

    qmdpi_flow_start_time_get(flow, &start);
    qmdpi_flow_duration_get(flow, &duration);

    stats.flow = fi;
    stats.established = 0;
    stats.duration = duration.tv_sec;
    stats.first_pkt_ts = start.tv_sec;
    stats.packet_count_cts = qmdpi_flow_packets_get_cts(flow);
    stats.packet_count_stc = qmdpi_flow_packets_get_stc(flow);
    stats.vol_pkt_cts = qmdpi_flow_bytes_get_cts(flow);
    stats.vol_pkt_stc = qmdpi_flow_bytes_get_stc(flow);
    if (QMDPI_FLOW_INFO_TCP_HANDSHAKE_SEEN(info) == 0) {
        stats.established = 1;
    }

    logs_flow_finalize(&stats);

    qmdpi_flow_user_handle_set(flow, NULL);
}

/* process Qosmos ixEngine results for a given flow */
static inline void
pcap_logger_result_get(struct qmdpi_worker *worker, struct qmdpi_result *result)
{
    struct qmdpi_result_flags const *result_flags = qmdpi_result_flags_get(result);
    const char *attr_value = NULL;
    uint8_t *stream_data = NULL;
    struct qm_ip_hdr *ip_hdr = NULL;
    struct qmdpi_flow *flow;
    struct qmdpi_path *path;
    int proto_id = 0;
    int attr_id = 0;
    int attr_len = 0;
    int attr_flags = 0;
    int stream_data_len = 0;
    int index = 0;
    int ret = 0;
    void *header;

    if ((flow = qmdpi_result_flow_get(result)) == NULL) {
        return;
    }
    if ((path = qmdpi_result_path_get(result)) == NULL) {
        return;
    }

    /* get user handle */
    struct flow_item *fi = (struct flow_item *)qmdpi_flow_user_handle_get(flow);

    /* on first packet: get addresses from innermost IPv4 layer */
    if (fi == NULL) {
        while ((proto_id = qmdpi_worker_pdu_ntuple_get_proto(worker, index)) > 0) {
            if (proto_id == Q_PROTO_IP) {
                header = qmdpi_worker_pdu_ntuple_get_header(worker, index);
                ip_hdr = (struct qm_ip_hdr *)header;
            }
            index++;
        }
    }

    /* get stream data pointer and length */
    qmdpi_worker_stream_get(worker, (void *)&stream_data, &stream_data_len);

    /* populate logs with packet data */
    ret = logs_new_packet(&fi, pcap_pkt_number,
                          QMDPI_RESULT_FLAGS_PDU_DIR(result_flags),
                          QMDPI_RESULT_FLAGS_PDU_DIR_INDEX(result_flags),
                          stream_data, stream_data_len,
                          (ip_hdr) ? ip_hdr->saddr : 0,
                          (ip_hdr) ? ip_hdr->daddr : 0);
    if (ret == -1) {
        return;
    }


    /* attributes storage */
    while (qmdpi_result_attr_getnext(result, &proto_id, &attr_id,
                                     &attr_value, &attr_len, &attr_flags) == 0) {
        logs_new_attribute(fi, proto_id, attr_id,
                           (uint8_t *)attr_value, attr_len);
    }

    /* store the flow_item in the user handle */
    if (ret) {
        qmdpi_flow_user_handle_set(flow, fi);
    }
}

struct pcap_file_env {
    long total;
    long read;
    int pcap_nb;
    int pcap_total;
};

/* process a libpcap packet */
static void
pcap_logger_loop(uint8_t *user, const struct pcap_pkthdr *phdr,
                 const uint8_t *pdata)
{
    struct pcap_file_env *env = (struct pcap_file_env *) user;
    struct qmdpi_result *result;
    int nb_remaining = 100;
    int offset = 0;
    int ret = 0;

    pcap_pkt_number++;

    /* progression indication */
    if (env != NULL) {
        env->read = env->read + phdr->len + 16;
        if (pcap_pkt_number % 100000 == 0) {
            fprintf(stdout, "\r[MSG] Processing pcap %d/%d %2ld%%",
                    env->pcap_nb,
                    env->pcap_total,
                    (env->read * 100) / env->total);
            fflush(stdout);
        }
    }

    /* apply LCC header offset if needed */
    offset = (pcap_remove_lcc && (phdr->caplen >= 2)) ? 2 : 0;

    /* detect JUNIPER encapsulation */
    if (phdr->caplen > sizeof(JUNIPER_FRAME) &&
            !strncmp((char *) pdata, JUNIPER_FRAME, sizeof(JUNIPER_FRAME) - 1)) {
        offset += sizeof(JUNIPER_FRAME);
    }

    /* set PDU information to be processed by the worker */
    if (qmdpi_worker_pdu_set(worker, pdata + offset, phdr->caplen - offset,
                             &phdr->ts, link_mode, QMDPI_DIR_DEFAULT, 0)) {
        return;
    }

    /* process packet with worker and provide DPI result */
    do {
        ret = qmdpi_worker_process(worker, NULL, &result);
        if (ret >= 0) {
            pcap_logger_result_get(worker, result);
        }
    } while (ret == QMDPI_PROCESS_MORE);

    /* destroy expired flows and provide DPI result for each of them */
    do {
        if (qmdpi_flow_expire_next(worker, &phdr->ts, &result) != 0) {
            break;
        }
        pcap_logger_result_expire(result);
    } while (nb_remaining--);
}

/* user interrupt during packet capture */
static void
pcap_logger_terminate(int signal)
{
    (void)signal;
    /* stop the capture */
    pcap_file_index = -1;
    if (pcap_fd_in) {
        pcap_breakloop(pcap_fd_in);
    }
}

/* expire all flows after one pcap reading is complete */
static inline void
pcap_logger_expire_all(void)
{
    struct qmdpi_result *result;

    while (qmdpi_flow_expire_next(worker, NULL, &result) == 0) {
        pcap_logger_result_expire(result);
    }
}

/* capture packets from interface */
static int
pcap_logger_process_live_packets(char **argv)
{
    fprintf(stdout, "\r[MSG] Capturing on %s\n",
            argv[0] ? argv[0] : "default interface");

    /* open the ethernet interface */
    pcap_fd_in = pcap_logger_pcap_open_live(argv[0]);
    if (pcap_fd_in == NULL) {
        return -1;
    }

    /* packet capture loop is interrupted with SIGINT */
    fprintf(stdout, "<stop capture with CTRL-C>");
    fflush(stdout);
    pcap_loop(pcap_fd_in, -1, pcap_logger_loop, NULL);
    fprintf(stdout, "\n");
    pcap_close(pcap_fd_in);

    pcap_fd_in = NULL;
    pcap_logger_expire_all();

    return 0;
}

/* capture packets from file/interface */
static int
pcap_logger_process_packets(int argc, char **argv, struct opt *opt)
{
    struct pcap_file_env env;
    int cnt = 0;
    int ret = 0;

    /* setup Qosmos ixEngine attributes extraction */
    ret = hooks_apply(&OPT_HOOK_STORE(opt));
    if (ret != 0) {
        goto process_exit;
    }

    /* packet capture loop is interrupted with SIGINT */
    signal(SIGINT, pcap_logger_terminate);

    if (OPT_LIVE_CAPTURE(opt)) {
        ret = pcap_logger_process_live_packets(argv);
    } else {
        /* iterate over several pcap files */
        pcap_file_index = 0;
        while (pcap_file_index < argc) {
            fprintf(stdout, "\r[MSG] Processing pcap %d/%d ...", pcap_file_index + 1, argc);
            fflush(stdout);
            pcap_fd_in = pcap_logger_pcap_open(argv[pcap_file_index], &(env.total));
            if (pcap_fd_in != NULL) {
                env.pcap_nb = pcap_file_index + 1;
                env.pcap_total = argc;
                env.read = 20;
                pcap_loop(pcap_fd_in, -1, pcap_logger_loop, (uint8_t *)&env);
                pcap_close(pcap_fd_in);
                pcap_fd_in = NULL;
                pcap_logger_expire_all();
            }

            /* CTRL-C stop */
            if (pcap_file_index++ == -1) {
                break;
            }

            cnt++;
        }
        fprintf(stdout, "\r[MSG] Finished processing %d/%d pcap(s)\n", cnt, argc);
    }

process_exit:
    hooks_remove(&OPT_HOOK_STORE(opt));
    signal(SIGINT, SIG_IGN);

    return ret;
}

static void
pcap_logger_exit(void)
{
    pcap_logger_expire_all();
    logs_cleanup();

    qmdpi_bundle_destroy(bundle);
    qmdpi_worker_destroy(worker);
    qmdpi_engine_destroy(engine);
    qmdpi_license_destroy();
}

int main(int argc, char **argv)
{
    struct opt opt;
    int nr_attr = 0;

    OPT_INIT(&opt);

    signal(SIGINT, SIG_IGN);

    /* parse CLI parameters */
    nr_attr = parse_param(argc, argv, &opt);
    argc -= nr_attr;
    argv += nr_attr;

    /* initialize Qosmos ixEngine */
    pcap_logger_ixe_init(&opt);

    /* initialize DNS hash table and exclusion list */
    logs_init();

    /* read configuration from file */
    config_parse(&opt);

    /* write CSV file header */
    if (logs_file_write(OPT_LOG_FILENAME(&opt), NULL) == -1) {
        pcap_logger_exit();
        return 1;
    }

    /* process packets using libpcap */
    if (pcap_logger_process_packets(argc, argv, &opt) != 0 ||
            pcap_pkt_number == 0) {
        fprintf(stdout, "[MSG] No packet captured\n");
    }

    /* close CSV file */
    if (logs_file_write(NULL, NULL) != -1) {
        fprintf(stdout, "[MSG] CSV dump complete\n");
    }

    pcap_logger_exit();
    return 0;
}

