// Copyright 2020 Nubeva, Inc. All rights reserved.
#ifndef __NUDEC_H__
#define __NUDEC_H__

#ifdef __cplusplus
extern "C" {
#endif

#include <sys/time.h>

#define TLS_TYPE_1_2 0x0303
#define TLS_TYPE_1_3 0x0304

    //process record responses
enum nu_process_record_response_e {
    RECORD_NO_ERROR,                // No error.  Either nothing to decrypt (header) or need more data to decrypt
    RECORD_DECRYPTED,               // No error and decrypted data available
    RECORD_HANDSHAKE_DECRYPTED,     // No error and handshake data decrypted
    RECORD_ALERT_DECRYPTED,         // No error and alert data decrypted
    RECORD_TYPE_NOT_SUPPORTED,      // An unknown record type tried to be decrypted
    RECORD_LICENSE_EXPIRED,         // Nubeva License expired
    RECORD_NULL_INPUT_DATA,         // Input data pointer is null
    RECORD_NULL_OUTPUT_DATA,        // Output data pointer is null
    RECORD_NOT_ENCRYPTED,           // Input data is not encrypted data
    RECORD_NEED_DESEGMENTATION,     // TLS Record needs desegmentation
    RECORD_THREAD_NOT_INITIALISED,  // Required thread data has not been initialised
    RECORD_NO_PACKET_INFO,          // nupacket info pointer is null
    RECORD_TYPE_NOT_ALLOWED,        // An TLS record type was seen that isn't allowed for this TLS type
    RECORD_INVALID_SESSION_ID,      // The TLS Session ID presented does not conform to specification
    RECORD_NO_SESSION_INFO,         // No session info
    RECORD_NO_RECORD,               // Unable to find a TLS record present
    RECORD_SESSION_NOT_INITIALISED, // Session data information not initialised
    RECORD_INVALID_CLIENT_RANDOM,   // Invalid client random
    RECORD_NO_KEY_RETURNED,         // No key returned for session
    RECORD_AEAD_FAILURE,            // AEAD Failed decryption, probably bad key
    RECORD_INCORRECT_BLOCKSIZE,     // Incoming data does not match required cipher block size
    RECORD_INCORRECT_MACSIZE,       // Incoming data does not match required MAC size
    RECORD_DECRYPT_ERROR,           // Wrong decryption key
    RECORD_INCORRECT_PADDING,       // Wrong padding size
    RECORD_MAC_FAILURE,             // Incorrect MAC
    RECORD_MAC_UNABLE_TO_DECOMPRESS, // Error decompressing
    RECORD_ZERO_LENGTH_HANDSHAKE,    // zero-length Handshake fragments are forbidden by RFC 8446
    RECORD_UNKNOWN_CIPHER,           // Cipher parameters are incorrect or unsupported
    RECORD_INSUFFICIENT_DATA,        // Not enough input data to parse into explicit nonce, ciphertext and/or tag
    RECORD_UNABLE_TO_PARSE,          // Unable to parse input to retreive decryption parameters
    RECORD_ERROR_SETTING_IV,         // Unable to set the IV from input data
    RECORD_AEAD_AUTHENTICATION_FAIL,  // Failure authenticating data
    RECORD_TAG_FAILURE,              // Incorrect TAG
    RECORD_ERROR_CONTACTSUPPORT      // unknown error, contact support with core dump if available
};

typedef enum nu_process_record_response_e nu_process_record_response;

extern const char* nu_process_record_response_as_string(nu_process_record_response nuerr);

// initialisation responses
enum nu_processor_init_response_e {
    INIT_NO_ERROR,        // no error
    INIT_INVALID_INPUT,   // Input data is invalid, likely null pointers
    INIT_LICENSE_INVALID,  // Nubeva License invalid
};
typedef enum nu_processor_init_response_e nu_processor_init_response;

extern const char* nu_init_response_as_string(nu_processor_init_response nuerr);


// Nubeva Keys
struct nu_key_s {
    unsigned int key_type;            // TLS1.2 or TLS1.3
    unsigned int error;               // Reserved for future use
    char client_random[32];           // Client Random value
    char master_key[48];              // Master Key value if TLS1.2
    char exporter_secret[64];         // Exporter Secret (TLS1.3 only)
    char early_traffic_secret[64];    // Early Traffic Secret (TLS1.3 only)
    char client_handshake_secret[64]; // Client Handshake Secret (TLS1.3 only)
    char server_handshake_secret[64]; // Server Handshake Secret (TLS1.3 only)
    char client_traffic_secret0[64];  // Client Traffic Secret (TLS1.3 only)
    char server_traffic_secret0[64];  // Server Traffic Secret (TLS1.3 only)
    unsigned long long int hash_len;  // The length of the TLS1.3 secrets
};
typedef struct nu_key_s nu_key;


// per packet information

struct nu_packet_info_s {
    struct timeval abs_ts;            // Packet time stamp
    struct timeval rel_ts;            // Relative timestamp
    unsigned char *l2_dst;            // L2 start
    unsigned char *net_src;           // IP dst
    unsigned char *net_dst;           // IP src
    unsigned char *src;               // IP src
    unsigned char *dst;               // IP dst
    unsigned short srcport;           // TCP or UDP src port
    unsigned short destport;          // TCP or UDP dest port

    /* mtu */
    unsigned int mss;

    // client IP and port...needed to send FIN/RST
    unsigned char *clientaddress;
    unsigned int clientport;
    unsigned int framenum;            // frame #, starts from 1, not 0.  For debugging only.
    
    unsigned char pad[3];
};

typedef struct nu_packet_info_s nu_packet_info;



////////////////////
// Thread initialisation and cleanup
////////////////////

// This is the function definition for what is going to be called in order to translate a client random into a master key
typedef nu_key* (*nu_get_master_key_fptr)(unsigned char client_random[32]);

// This is the function call to free the memory associated with the key
typedef void (*nu_clean_key_fptr)(nu_key* key);


struct nu_processor_options_s {
    nu_get_master_key_fptr get_key_f;                    // function to use to map client random to nu_key
    nu_clean_key_fptr clean_key_f;                       // function to use to free key memory
};

typedef struct nu_processor_options_s nu_processor_options;

// input:   processor options as defined above
// output:  nuerr - an initResponse enum
// returns: nubeva thread local data handle (nutldh)
extern void*
nu_record_processor_init(nu_processor_options *popts, nu_processor_init_response *nuerr);

// input: nubeva thread local data handle (nutldh)
extern void
nu_record_processor_cleanup(void *nutldh);

// Handle lost frames.  We can continue decrypting in most cases.
extern void
nu_lost_client_frame(void *nush);

extern void
nu_lost_server_frame(void *nush);

///////////////////
// Session initialisation and cleanup
///////////////////

// input:
//     nutldh - the nubeva thread local data handle that is returned from the nu_record processor_init call
//     nupacket - information about the packet
// returns:
//     nush - the new nubeva session handle
extern void *
nu_new_session(void *nutldh);

// Cleans-up resources
// input:  nush - the nubeva session handle
extern void
nu_session_cleanup(void *nush);


//////////////////////
// Record processing
//////////////////////

// Parses all TLS data
// input:
//   - nush - the nubeva session handle
//   - nupacket - the packet information
//   - indata - the encrypted data
//   - len - the length of data input
// output:
//   - outdata - where to put the decrypted data (can be the same as indata)
//   - len - the length of the decrypted data (can be less than incoming if not a full record)
// returns:
//   - process record response enum

extern nu_process_record_response
nu_record_processor(void *nush, nu_packet_info *pinfo, const unsigned char *indata, unsigned char *outdata, unsigned int *len);


extern char *
nu_version();

#ifdef __cplusplus
}
#endif

#endif /* __NUDEC_H__ */
