/*====================================================================================
    EVS Codec 3GPP TS26.443 Jan 17, 2017. Version 12.8.0 / 13.4.0
  ====================================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <assert.h>
#include "options.h"
#include "prot.h"
#include "cnst.h"
#include "rom_com.h"
#include "stat_enc.h"
#include "stat_dec.h"
#include "mime.h"


/*------------------------------------------------------------------------------------------*
 * Global variables
 *------------------------------------------------------------------------------------------*/
long frame = 0;                 /* Counter of frames */

/*-------------------------------------------------------------------*
 * decoder_selectCodec_evs()
 *
 *
 *-------------------------------------------------------------------*/

static void decoder_selectCodec_evs(
    Decoder_State *st,            /* i/o: decoder state structure                */
    const long total_brate,    /* i  : total bitrate                          */
    const short bit0           /* i  : first bit                              */
)
{
    /* set the AMR-WB IO flag */
    if( total_brate == SID_1k75 ||
            total_brate == ACELP_6k60  || total_brate == ACELP_8k85  || total_brate == ACELP_12k65 ||
            total_brate == ACELP_14k25 || total_brate == ACELP_15k85 || total_brate == ACELP_18k25 ||
            total_brate == ACELP_19k85 || total_brate == ACELP_23k05 || total_brate == ACELP_23k85 )
    {
        st->Opt_AMR_WB = 1;
    }
    else if ( total_brate != FRAME_NO_DATA )
    {
        st->Opt_AMR_WB = 0;
    }

    if ( st->Opt_AMR_WB )
    {
        st->codec_mode = MODE1;
    }
    else
    {
        switch ( total_brate )
        {
        case 0:
            st->codec_mode = st->last_codec_mode;
            break;
        case 2400:
            st->codec_mode = st->last_codec_mode;
            break;
        case 2800:
            st->codec_mode = MODE1;
            break;
        case 7200:
            st->codec_mode = MODE1;
            break;
        case 8000:
            st->codec_mode = MODE1;
            break;
        case 9600:
            st->codec_mode = MODE2;
            break;
        case 13200:
            st->codec_mode = MODE1;
            break;
        case 16400:
            st->codec_mode = MODE2;
            break;
        case 24400:
            st->codec_mode = MODE2;
            break;
        case 32000:
            st->codec_mode = MODE1;
            break;
        case 48000:
            st->codec_mode = MODE2;
            break;
        case 64000:
            st->codec_mode = MODE1;
            break;
        case 96000:
            st->codec_mode = MODE2;
            break;
        case 128000:
            st->codec_mode = MODE2;
            break;
        default :   /* validate that total_brate (derived from RTP packet or a file header) is one of the defined bit rates  */
            st->codec_mode = st->last_codec_mode;
            st->bfi = 1;
            break;
        }
    }

    if( st->ini_frame == 0 )
    {
        if( st->codec_mode == -1 )
        {
            st->codec_mode = MODE1;
        }
        st->last_codec_mode = st->codec_mode;
    }

    /* set SID/CNG type */
    if( total_brate == SID_2k40 )
    {
        if( bit0 == G192_BIN0 )
        {
            st->cng_type = LP_CNG;

            /* force MODE1 when selecting LP_CNG */
            st->codec_mode = MODE1;
        }
        else
        {
            st->cng_type = FD_CNG;
            if ( st->last_codec_mode == MODE2 && st->last_total_brate == ACELP_13k20 )
            {
                st->codec_mode = MODE1;
            }
        }
        st->last_cng_type = st->cng_type;     /* CNG type switching at the first correctly received SID frame */
    }


    return;
}
/*-------------------------------------------------------------------*
* unpack_bit()
*
* unpack a bit from packed octet
*-------------------------------------------------------------------*/
static Word16 unpack_bit(
    UWord8 **pt,         /* i/o: pointer to octet array from which bit will be read */
    UWord8 *mask         /* i/o: mask to indicate the bit in the octet */
)
{
    Word16 bit;

    bit = (**pt & *mask) != 0;
    *mask >>= 1;
    if (*mask == 0)
    {
        *mask = 0x80;
        (*pt)++;
    }

    return bit;
}

/*-------------------------------------------------------------------*
* rate2AMRWB_IOmode()
*
* lookup AMRWB IO mode
*-------------------------------------------------------------------*/

static Word16 rate2AMRWB_IOmode(
    Word32 rate                    /* i: bit rate */
)
{
    switch ( rate )
    {
    /* EVS AMR-WB IO modes */
    case SID_1k75      :
        return AMRWB_IO_SID;
    case ACELP_6k60    :
        return AMRWB_IO_6600;
    case ACELP_8k85    :
        return AMRWB_IO_8850;
    case ACELP_12k65   :
        return AMRWB_IO_1265;
    case ACELP_14k25   :
        return AMRWB_IO_1425;
    case ACELP_15k85   :
        return AMRWB_IO_1585;
    case ACELP_18k25   :
        return AMRWB_IO_1825;
    case ACELP_19k85   :
        return AMRWB_IO_1985;
    case ACELP_23k05   :
        return AMRWB_IO_2305;
    case ACELP_23k85   :
        return AMRWB_IO_2385;
    default:
        return -1;
    }
}

/*-------------------------------------------------------------------*
* rate2EVSmode()
*
* lookup EVS mode
*-------------------------------------------------------------------*/
static Word16 rate2EVSmode(
    Word32 rate                    /* i: bit rate */
)
{
    switch ( rate )
    {
    /* EVS Primary modes */
    case FRAME_NO_DATA :
        return NO_DATA;
    case SID_2k40      :
        return PRIMARY_SID;
    case PPP_NELP_2k80 :
        return PRIMARY_2800;
    case ACELP_7k20    :
        return PRIMARY_7200;
    case ACELP_8k00    :
        return PRIMARY_8000;
    case ACELP_9k60    :
        return PRIMARY_9600;
    case ACELP_13k20   :
        return PRIMARY_13200;
    case ACELP_16k40   :
        return PRIMARY_16400;
    case ACELP_24k40   :
        return PRIMARY_24400;
    case ACELP_32k     :
        return PRIMARY_32000;
    case ACELP_48k     :
        return PRIMARY_48000;
    case ACELP_64k     :
        return PRIMARY_64000;
    case HQ_96k        :
        return PRIMARY_96000;
    case HQ_128k       :
        return PRIMARY_128000;
    default            :
        return rate2AMRWB_IOmode(rate);
    }
}

static void decision_matrix_core_dec(
    Decoder_State *st                 /* i/o: decoder state structure                   */
)
{
    short start_idx;
    long ind;
    short nBits;

    assert(st->bfi != 1);

    st->core = -1;
    st->bwidth = -1;

    if ( st->total_brate == FRAME_NO_DATA || st->total_brate == SID_2k40 )
    {
        st->core = ACELP_CORE;
    }
    /* SC-VBR */
    else if ( st->total_brate == PPP_NELP_2k80 )
    {
        st->core = ACELP_CORE;
        return;
    }

    /*---------------------------------------------------------------------*
     * ACELP/HQ core selection
     *---------------------------------------------------------------------*/

    if ( st->total_brate < ACELP_24k40 )
    {
        st->core = ACELP_CORE;
    }
    else if ( st->total_brate >= ACELP_24k40 && st->total_brate <= ACELP_64k )
    {
        /* read the ACELP/HQ core selection bit */
        st->core = ((short) get_next_indice( st, 1 ))*HQ_CORE;
    }
    else
    {
        st->core = HQ_CORE;
    }

    /*-----------------------------------------------------------------*
     * Read ACELP signalling bits from the bitstream
     *-----------------------------------------------------------------*/

    if ( st->core == ACELP_CORE )
    {
        /* find the section in the ACELP signalling table corresponding to bitrate */
        start_idx = 0;
        while ( acelp_sig_tbl[start_idx] != st->total_brate )
        {
            start_idx++;
        }

        /* skip the bitrate */
        start_idx += 1;

        /* retrieve the number of bits */
        nBits = (short) acelp_sig_tbl[start_idx++];

        /* retrieve the signalling indice */
        ind = acelp_sig_tbl[start_idx + get_next_indice( st, nBits )];
        st->bwidth = (ind >> 3) & 0x7;

        /* convert signalling indice into signalling information */
        if ( (ind & 0x7) == LR_MDCT )
        {
            st->core = HQ_CORE;
        }
    }

    /*-----------------------------------------------------------------*
     * Read HQ signalling bits from the bitstream
     * Set HQ core type
     *-----------------------------------------------------------------*/

    if ( st->core == HQ_CORE )
    {
        /* read the HQ/TCX core switching flag */
        if ( get_next_indice( st, 1 ) )
        {
            st->core = TCX_20_CORE;
        }

        /* For TCX: read/set band-width (needed for different I/O sampling rate support) */
        if( st->core == TCX_20_CORE && st->total_brate > ACELP_16k40 )
        {
            ind = get_next_indice( st, 2 );

            if( ind == 0 )
            {
                st->bwidth = NB;
            }
            else if( ind == 1 )
            {
                st->bwidth = WB;
            }
            else if( ind == 2 )
            {
                st->bwidth = SWB;
            }
            else
            {
                st->bwidth = FB;
            }
        }
    }

    return;
}

static void mdct_switching_dec(
    Decoder_State *st           /* i/o: decoder state structure     */
)
{
    if (st->Opt_AMR_WB)
    {
        return;
    }

    if (st->total_brate == ACELP_13k20 || st->total_brate == ACELP_32k)
    {
        st->mdct_sw_enable = MODE1;
    }
    else if (ACELP_16k40 <= st->total_brate && st->total_brate <= ACELP_24k40)
    {
        st->mdct_sw_enable = MODE2;
    }

    if ( st->codec_mode == MODE1 && st->mdct_sw_enable == MODE1 )
    {
        /* Read ahead core mode signaling */
        short next_bit_pos_save = st->next_bit_pos;
        short core_save         = st->core;
        short bwidth_save       = st->bwidth;

        decision_matrix_core_dec(st); /* sets st->core */

        if (st->core == TCX_20_CORE)
        {
            /* Trigger TCX */
            st->codec_mode = MODE2;
            st->mdct_sw = MODE1;
        }
        else
        {
            /* Rewind bitstream */
            st->next_bit_pos = next_bit_pos_save;
            if (st->bfi)
            {
                st->core   = core_save;
                st->bwidth = bwidth_save;
            }
        }
    }
    else if (st->codec_mode == MODE2 && st->mdct_sw_enable == MODE2)
    {
        /* Read ahead core mode signaling */
        short next_bit_pos_save = st->next_bit_pos;
        short core_save         = st->core;
        short bwidth_save       = st->bwidth;

        dec_prm_core(st); /* sets st->core */

        if (st->core == HQ_CORE)
        {
            /* Trigger HQ_CORE */
            st->codec_mode = MODE1;
            st->mdct_sw = MODE2;
        }
        else
        {
            /* Rewind bitstream */
            st->next_bit_pos = next_bit_pos_save;
            if (st->bfi)
            {
                st->core   = core_save;
            }

            /* always reset bwidth, to not interfere with BER logic */
            st->bwidth = bwidth_save;
        }
    }

    return;
}

/* Read the payload and decode
 * decode_evs()
 * Decoder_State *st,         in/out
 * const char* payload,       in
 * int length,                in
 *  short* decodedData        out
 * )
 */
int decode_evs(Decoder_State *st,  const char* payload, int length, short* decodedData )
{

    short output_frame, dec_delay, zero_pad;
    short quietMode = 0;
    short noDelayCmp = 0;
    float output[L_FRAME48k];           /* 'float' buffer for output synthesis */

    srand( (unsigned int) time(0) );

    /* output frame length */
    output_frame = (short)(st->output_Fs / 50);

    /* calculate the delay compensation to have the decoded signal aligned with the original input signal */
    /* the number of first output samples will be reduced by this amount */
    dec_delay = NS2SA(st->output_Fs, get_delay(DEC, st->output_Fs));

    zero_pad = dec_delay;

    /*------------------------------------------------------------------------------------------*
     * Loop for every packet (frame) of bitstream data
     * - Read the bitstream packet
     * - Run the decoder
     *------------------------------------------------------------------------------------------*/

    read_indices_evs(st, payload, length);

    /* run the main decoding routine */
    if ( st->codec_mode == MODE1 )
    {
        if ( st->Opt_AMR_WB )
        {
            amr_wb_dec( st, output );
        }
        else
        {
            evs_dec( st, output, FRAMEMODE_NORMAL );
        }
    }
    else
    {
        if( !st->bfi )
        {
            evs_dec( st, output, FRAMEMODE_NORMAL );
        }
        else
        {
            evs_dec( st, output, FRAMEMODE_MISSING );
        }
    }

    /* convert 'float' output data to 'short' */
    syn_output( output, output_frame, decodedData );

    /* increase the counter of initialization frames */
    if( st->ini_frame < MAX_FRAME_COUNTER )
    {
        st->ini_frame++;
    }

    frame++;

    return 0;
}

/* Set the sampling rate for decoder */

void io_ini_dec_evs(
    Word32 Sr,
	Decoder_State *st
)
{

    print_disclaimer( stderr );

    st->writeFECoffset = 0;

    /*-----------------------------------------------------------------*
     * Initialization
     *-----------------------------------------------------------------*/

    st->codec_mode = 0; /* unknown before first frame */
    st->Opt_AMR_WB = 0;
    st->Opt_VOIP = 0;

    st->amrwb_rfc4867_flag = -1;


    st->output_Fs = Sr * 1000;

    if( st->output_Fs != 8000 && st->output_Fs != 16000 && st->output_Fs != 32000 && st->output_Fs != 48000 )
    {
        fprintf(stderr, "Error: %d kHz is not a supported sampling rate\n\n", Sr );
        //usage_dec();
    }

    return;
}

/* read the buffer and store it in bit_stream */

Word16 read_indices_evs(                /* o  : 1 = reading OK, 0 = problem            */
    Decoder_State *st,                   /* i/o: decoder state structure                */
    UWord8 *buffer,                      /* i  : bitstream buffer                       */
    Word32 length                        /* i  : length of the buffer                   */
	)
{
    Word16 k, isAMRWB_IOmode, cmi, core_mode = -1, qbit,sti, i;
    UWord8 header;
    UWord16 hdr_full_fmt = FALSE;
    UWord8 pFrame[(MAX_BITS_PER_FRAME + 7) >> 3];
    UWord8 mask= 0x80, *pt_pFrame=buffer;
    UWord16 *bit_stream_ptr;
    Word16 num_bits;
    Word32 total_brate;
    Word16 curr_ft_good_sp;
    Word16 amrwb_sid_first, sid_upd_bad, sid_update;
    Word16 speech_bad, speech_lost;
    Word16 no_data;
    Word16 num_bytes_read;

    st->BER_detect = 0;
    st->bfi = 0;
    st->mdct_sw_enable = 0;
    st->mdct_sw = 0;
    reset_indices_dec( st );

    /* read the FT Header field from the bitstream */
    /* read the FT header */

    /* init local RXDTX flags */
    curr_ft_good_sp = 0;
    speech_lost = 0;
    speech_bad = 0;

    sid_update = 0;
    sid_upd_bad = 0;
    sti = -1;
    amrwb_sid_first = 0;  /* derived from sti  SID_FIRST indicator in AMRWB payload */
    no_data = 0;
    isAMRWB_IOmode = 0;

    st->bfi = 0;

    /* Calculate the total bit rate in kbps  from the payload length */
    total_brate = length*8*50;

    /* Check if the bitrate is compatible with compact format */
    switch(total_brate)
    {
    case SID_1k75:
    case ACELP_6k60:
    case ACELP_8k85:
    case ACELP_12k65:
    case ACELP_14k25:
    case ACELP_15k85:
    case ACELP_18k25:
    case ACELP_19k85:
    case ACELP_23k05:
    case ACELP_23k85:
    	isAMRWB_IOmode = 1;
    	core_mode = rate2EVSmode(total_brate);
    	break;
    case SID_2k40:
    case ACELP_7k20:
    case ACELP_8k00:
    case ACELP_9k60:
    case ACELP_13k20:
    case ACELP_16k40:
    case ACELP_24k40:
    case ACELP_32k:
    case ACELP_48k:
    case ACELP_64k:
    case HQ_96k:
    case HQ_128k:
    	isAMRWB_IOmode = 0;
    	core_mode = rate2EVSmode(total_brate);
    	break;
    default:
    	hdr_full_fmt = TRUE;
    	break;
    }

    if(TRUE == hdr_full_fmt)
    {
    	/*0 1 2 3 4 5 6 7   MS-bit ---> LS-bit
    	+-+-+-+-+-+-+-+-+
    	|H|F|E|x| brate |
    	+-+-+-+-+-+-+-+-+
    	where :
    	"E|x|  brate "  is the 6 bit "FT" -field
    	x is unused    if E=0, (should be 0 )
    	x is the q-bit if E=1, q==1(good), Q==0(bad, maybe bit errors in payload )
    	H,F  always   0 in RTP format.
    	*/
    	isAMRWB_IOmode = (pt_pFrame[0] & 0x20) > 0;   /* get EVS mode-from header */ /*    b2   */
    	core_mode      = (pt_pFrame[0] & 0x0F);        /* b4,b5,b6,b7 */

    	if( isAMRWB_IOmode )
    	{
    	    qbit = (pt_pFrame[0] & 0x10) > 0;      /* get Q bit,    valid for IO rates */ /* b3 */
    	    total_brate = AMRWB_IOmode2rate[core_mode];
    	}
    	else
    	{
    	    qbit = 1;  /* assume good q_bit for the unused EVS-mode bit,    complete ToC validity checked later */
    	    total_brate = PRIMARYmode2rate[ core_mode ];
    	}
        st->bfi = !qbit;
    }

    /* set up RX-DTX-handler input */
    if(   core_mode == 14  )
    {
        /* SP_LOST */
        speech_lost=1;
    }
    if ( core_mode  == 15)
    {
        /*  NO_DATA unsent CNG frame OR  any frame marked or injected  as no_data  by e.g a signaling layer or dejitter buffer */
        no_data=1;
    }
    num_bits = (Word16)(total_brate/50);
    if( total_brate < 0 )
    {
        /* validate that total_brate (derived from RTP packet or a file header) is one of the defined bit rates  */
        fprintf(stderr, "\n  Error illegal total bit rate (= %d)  \n",     total_brate );
        num_bits = -1;
    }

    /* unpack speech data */
    bit_stream_ptr = st->bit_stream;

    for( k=0; k<num_bits; k++ )
    {
        if( isAMRWB_IOmode )
        {
            st->bit_stream[sort_ptr[core_mode][k]] = unpack_bit(&pt_pFrame,&mask);
            bit_stream_ptr++;
        }
        else
        {
            *bit_stream_ptr++ = unpack_bit(&pt_pFrame,&mask);
        }
    }
    /* unpack auxiliary bits */
    /* Note: these cmi bits are unpacked for demo purposes;  they are actually not needed  */
    if( isAMRWB_IOmode && total_brate == SID_1k75 )
    {
        sti = unpack_bit(&pt_pFrame,&mask);
        cmi  = unpack_bit(&pt_pFrame,&mask) << 3;
        cmi |= unpack_bit(&pt_pFrame,&mask) << 2;
        cmi |= unpack_bit(&pt_pFrame,&mask) << 1;
        cmi |= unpack_bit(&pt_pFrame,&mask);

        if( sti == 0 )
        {
            total_brate = 0;     /* signal received SID_FIRST as a good frame with no bits */
            for(k=0; k<35; k++)
            {
                st->bfi  |= st->bit_stream[k] ; /* partity check of 35 zeroes,  any single 1 gives BFI */
            }
        }

    }
    /*add two zero bytes for arithmetic coder flush*/
    for( k=0; k< 2*8; ++k )
    {
        *bit_stream_ptr++ = 0;
    }
    /* RX_DTX handler */

    /* keep st->CNG , st_bfi and total_brate  updated  for proper synthesis in DTX and FER  */
    if( total_brate > SID_2k40 )
    {
        if( st->bfi == 0 )   /* so  far derived from q bit in AMRWB/AMRWBIO cases   */
        {
           curr_ft_good_sp = 1;
        }
    }

    /* handle q_bit and  lost_sp  clash ,  assume worst case  */
    if( speech_lost != 0 )  /*  overrides  a good q_bit */
    {
        curr_ft_good_sp = 0;
        st->bfi      = 1;     /* override  qbit */
    }
    /* now_bfi_fx has been set based on q_bit and ToC fields */

    /* SID_UPDATE check */
    if( total_brate == SID_1k75 || total_brate == SID_2k40 )
    {
        if( st->bfi == 0 )
        {
            /* typically from q bit  */
            sid_update = 1;
        }
        else
        {
            sid_upd_bad = 1;  /* may happen in saving from e.g. a CS-connection */
        }
    }

    if( isAMRWB_IOmode && total_brate == 0 && sti == 0 )
    {
        if( st->bfi )
        {
            sid_upd_bad = 1;          /*  corrupt sid_first, signaled as bad sid  */
        }
        else
        {
            amrwb_sid_first =  1;     /* 1-sti  */
        }
    }
    if( total_brate > SID_2k40 && st->bfi )  /* typically from q bit  */
    {
        speech_bad = 1;    /* initial assumption,   CNG synt state decides what to actually do */
    }
    /* all frame types decoded */

    /*    update CNG synthesis state */
    /*    Decoder can only  enter CNG-synthesis  for  CNG frame types (sid_upd,  sid_bad, sid_first) */
    if( st->CNG != 0 )
    {
        /* We were in CNG synthesis  */
        if( curr_ft_good_sp != 0  )
        {
            /* only a good speech frame makes decoder leave CNG synthesis */
            st->CNG = 0;
        }
    }
    else
    {
        /*   We were in SPEECH synthesis  */
        /*   only a received SID frame can make the decoder enter into CNG synthesis  */
        if( amrwb_sid_first || sid_update || sid_upd_bad )
        {
            st->CNG = 1;
        }
    }
    /* Now modify bfi flag for the  decoder's  SPEECH/CNG synthesis logic  */
    /*   in SPEECH synthesis, make sure to activate speech plc for a received no_data frame,
         bo_data frames may be injected by the network or by the dejitter buffer   */
    /*   modify bfi_flag to stay/move to the into the correct decoder PLC section  */
    if ( (st->CNG == 0)  &&  ( no_data != 0 )  )
    {
        /*  treat no_data received in speech synthesis as  SP_LOST frames, SPEECH PLC code will now become active */
        st->bfi = 1;
        /* total_brate= 0;    always zero for no_data */
    }

    /* in CNG  */
    /* handle bad speech frame(and bad sid frame) in the decoders CNG synthesis settings pair (total_brate, bfi)  */
    if( ( st->CNG != 0 && ( speech_bad || speech_lost || no_data ))  || /* SP_BAD or SPEECH_LOST)   --> stay in CNG */
            sid_upd_bad )                                                 /* SID_UPD_BAD               --> start/stay  CNG   */
    {
        st->bfi     = 0;   /* mark as good to not start speech PLC */
        total_brate = 0;   /* this zeroing needed  for  speech_bad, sid_bad frames */
    }

    /*  handle available AMRWB/AMRWBIO MIME header ToC rate-info at startup   */
    if( st->bfi == 0 )
    {
        /* select MODE1 or MODE2 in  MIME */
        decoder_selectCodec_evs( st, total_brate, *st->bit_stream ? G192_BIN1 : G192_BIN0);

        /* a change of the total bitrate should not be known to the decoder, if the received frame was truly lost */
        st->total_brate  = total_brate;
        mdct_switching_dec(st);
    }
    /* else{ bfi stay in past synthesis mode(SP,CNG) } */
    return 1;
}
