/**************************************************************************
 *
 *   File Name:  e_homing.c
 *
 *   Purpose:
 *      This file contains the following functions:
 *
 *      encoder_homing_frame_test()  checks if a frame of input samples
 *                                   matches the Encoder Homing Frame pattern.
 *
 *      encoder_reset()              called by reset_enc() to reset all
 *                                   the state variables for the encoder.
 *
 *      reset_enc()                  calls functions to reset the state
 *                                   variables for the encoder and VAD, and
 *                                   for the transmit DTX and Comfort Noise.
 *
 **************************************************************************/

#include "typedef.h"
#include "cnst.h"
#include "vad.h"
#include "dtx.h"
#include "codec.h"
#include "sig_proc.h"
#include "e_homing.h"

/***************************************************************************
 *
 *   FUNCTION NAME:  encoder_homing_frame_test
 *
 *   PURPOSE:
 *      Checks if a frame of input samples matches the Encoder Homing Frame
 *      pattern, which is 0x0008 for all 160 samples in the frame.
 *
 *   INPUT:
 *      input_frame[]    one frame of speech samples
 *
 *   OUTPUT:
 *      None
 *
 *   RETURN:
 *      0       input frame does not match the Encoder Homing Frame pattern.
 *      1       input frame matches the Encoder Homing Frame pattern.
 *
 **************************************************************************/

Word16 encoder_homing_frame_test (Word16 input_frame[])
{
    Word16 i, j;

    for (i = 0; i < L_FRAME; i++)
    {
        j = input_frame[i] ^ EHF_MASK;

        if (j)
            break;
    }

    return !j;
}

/***************************************************************************
 *
 *   FUNCTION NAME:  encoder_reset
 *
 *   PURPOSE:
 *      resets all of the state variables for the encoder
 *
 *   INPUT:
 *      None
 *
 *   OUTPUT:
 *      None
 *
 *   RETURN:
 *      None
 *
 **************************************************************************/

void encoder_reset (void)
{
    /* External declarations for encoder variables which need to be reset */

    /* Variables defined in levinson.c */
    /* ------------------------------- */
    extern Word16 old_A[M + 1]; /* Last A(z) for case of unstable filter */

    /* Variables defined in q_gains.c */
    /* ------------------------------- */
    /* Memories of gain quantization: */
    extern Word16 past_qua_en[4], pred[4];

    /* Variables defined in q_plsf_5.c */
    /* ------------------------------- */
    /* Past quantized prediction error */
    extern Word16 past_r2_q[M];

    Word16 i;

    /* reset all the encoder state variables */
    /* ------------------------------------- */

    /* Variables in cod_12k2.c: */
    Init_Coder_12k2 ();

    /* Variables in levinson.c: */
    old_A[0] = 4096;            /* Last A(z) for case of unstable filter */
    for (i = 1; i < M + 1; i++)
    {
        old_A[i] = 0;
    }

    /* Variables in pre_proc.c: */
    Init_Pre_Process ();

    /* Variables in q_gains.c: */
    for (i = 0; i < 4; i++)
    {
        past_qua_en[i] = -2381; /* past quantized energies */
    }

    pred[0] = 44;               /* MA prediction coeff */
    pred[1] = 37;               /* MA prediction coeff */
    pred[2] = 22;               /* MA prediction coeff */
    pred[3] = 12;               /* MA prediction coeff */

    /* Variables in q_plsf_5.c: */
    for (i = 0; i < M; i++)
    {
        past_r2_q[i] = 0;       /* Past quantized prediction error */
    }

    return;
}

/***************************************************************************
 *
 *   FUNCTION NAME:  reset_enc
 *
 *   PURPOSE:
 *      resets all of the state variables for the encoder and VAD, and for
 *      the transmit DTX and Comfort Noise.
 *
 *   INPUT:
 *      None
 *
 *   OUTPUT:
 *      None
 *
 *   RETURN:
 *      None
 *
 **************************************************************************/

void reset_enc (void)
{
    encoder_reset (); /* reset all the state variables in the speech encoder*/
    vad_reset ();     /* reset all the VAD state variables */
    reset_tx_dtx ();  /* reset all the transmit DTX and CN variables */

    return;
}
