/**************************************************************************
 *
 *   File Name:  gsm_hr.c
 *
 *   Purpose:
 *
 *     This file contains the main routine for the GSM Half Rate speech
 *     codec.  The code for the speech coder, including the VAD, DTX,
 *     Comfort Noise, bad frame handling (error concealment), and codec
 *     homing functions is in the files listed below:
 *
 *     gsm_hr.c      globdefs.c      homing.c      host.c
 *     mathdp31.c    mathhalf.c      sp_dec.c      sp_enc.c
 *     sp_frm.c      sp_rom.c        sp_sfrm.c     vad.c
 *     dtx.c         err_conc.c
 *
 *     This particular file only has the level 0, main(), and all level
 *     1, main()'s daughter functions, in it.
 *
 *     Details on how to run gsm_hr are in the readme.txt file.
 *
 *     Below is a listing of all the functions appearing in the file.
 *     The ordering is hierarchical.
 *
 *     main() "gsm_hr", main program.
 *       encode() - encodes a speech file, outputs an encoded parameter file
 *       decode() - decodes a parameter file, outputs a decoded speech file
 *
 **************************************************************************/

/*_________________________________________________________________________
 |                                                                         |
 |                              Include Files                              |
 |_________________________________________________________________________|
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "gsm_hr.h"
#include "homing.h"
#include "host.h"
#include "sp_dec.h"
#include "sp_enc.h"
#include "typedefs.h"
#include "nlbis.h"

#if 0
/****************************************************************************
 *
 *   PROGRAM NAME:  gsm_hr
 *
 *   PURPOSE:
 *      gsm_hr - main program for the GSM Half-Rate speech coder.
 *      Allows running one of the following operational modes:
 *         0. Encoder only with speech input / encoded parameter data output.
 *         1. Decoder only with speech parameter data input / speech output.
 *
 *   INPUT:
 *      inputFileName     0. speech file in 8 kHz, pcm format.
 *                        1. speech parameter file in decoder input format.
 *      <number of frames> - number of frames to be processed (optional).
 *      <dtx|nodtx>        - switch to enable/disable the DTX functions.
 *
 *   OUTPUT:
 *      outputFileName    0. encoded paramter output file.
 *                        1. synthesized speech file in 8 kHz, pcm format.
 *
 *   RETURN:
 *      none
 *
 *   REFERENCES: Sub-clause 4.0 of GSM Recomendation 06.20
 *
 *   KEYWORDS: main, gsm_hr, speech codec, top level
 *
 ***************************************************************************/
int    main(int argc, char *argv[])
{

/*_________________________________________________________________________
 |                                                                         |
 |                            Local Constants                              |
 |_________________________________________________________________________|
*/

#define  DEFAULT_NUMFRAMES  32766

/*_________________________________________________________________________
 |                                                                         |
 |                            Automatic Variables                          |
 |_________________________________________________________________________|
*/

  int    iDoneFrm,
         iMaxNumFrms,
         option,
         i;

  FILE  *pfileInFile,
        *pfileOutFile;

/*_________________________________________________________________________
 |                                                                         |
 |                              Executable Code                            |
 |_________________________________________________________________________|
*/

  /* check command line arguments */
  /* ---------------------------- */

  iMaxNumFrms = DEFAULT_NUMFRAMES;
  GsmHr_giDTXon = 1;

  /* check command line arguments */
  /* ---------------------------- */

  switch (argc)
  {
    case 4:

      break;

    case 5:
    case 6:

      for (i = 4; i < argc; i++)
      {
        if (!strcmp(argv[i], "dtx"))
          GsmHr_giDTXon = 1;
        else if (!strcmp(argv[i], "nodtx"))
          GsmHr_giDTXon = 0;
        else if ((iMaxNumFrms = atoi(argv[i])) <= 0)
        {
          printf("invalid number of frames or wrong DTX switch, %s", argv[i]);
          exit(1);
        }
      }
      break;

    default:

      printf("\n\nUsage:\n\n");
      printf("gsm_hr option inputFileName outputFileName ");
      printf("<number of frames> <dtx|nodtx>\n");
      printf(" or\n");
      printf("gsm_hr option inputFileName outputFileName ");
      printf("<dtx|nodtx> <number of frames>\n");
      printf("\nOptions: ");
      printf("enc    ");
      puts("inputFileName: speech --> outputFileName: encoder output");
      printf("         ");
      printf("dec    ");
      puts("inputFileName: decoder input --> outputFileName: speech");
      printf("\n");
      exit(1);
  }

  /* check encoding and decoding options */
  /* ----------------------------------- */

  if (!strcmp(argv[1], "enc"))
    option = 0;
  else if (!strcmp(argv[1], "dec"))
    option = 1;
  else
  {
    printf("error in option selection\n");
    printf(" Your entry : %s \n", argv[1]);
    printf("\n\nUsage:\n\n");
    printf("gsm_hr option inputFileName outputFileName ");
    printf("<number of frames> <dtx|nodtx>\n");
    printf(" or\n");
    printf("gsm_hr option inputFileName outputFileName ");
    printf("<dtx|nodtx> <number of frames>\n");
    printf("\nOptions: ");
    printf("enc    ");
    puts("inputFileName: speech --> outputFileName: encoder output");
    printf("         ");
    printf("dec    ");
    puts("inputFileName: decoder input --> outputFileName: speech");
    printf("\n");
    exit(1);
  }


  /* open the input and output files */
  /* ------------------------------- */

#ifdef VAX
  pfileInFile = fopen(argv[2], "rb", "mrs=2", "rfm=fix", "ctx=stm");
  pfileOutFile = fopen(argv[3], "wb", "mrs=2", "rfm=fix", "ctx=stm");
#else
  pfileInFile = fopen(argv[2], "rb");
  pfileOutFile = fopen(argv[3], "wb");
#endif

  if (pfileInFile == NULL)
  {
    printf("error: can not open file %s\n", argv[2]);
    exit(1);
  }
  if (pfileOutFile == NULL)
  {
    printf("error: can not open file %s\n", argv[3]);
    exit(1);
  }

  puts("\n\n");
  puts("   ****************************************");
  puts("   *                                      *");
  puts("   *      GSM Half-Rate Speech Coder      *");
  puts("   *                                      *");
  puts("   *                                      *");
  printf("   *        %s            *\n", VERSION);
  printf("   *        %s            *\n", DATE);
  puts("   *                                      *");
  puts("   ****************************************");
  puts("\n");


  printf("option:       ");

  switch (option)
  {
    case 0:
      puts("enc (speech encoder)");
      break;
    case 1:
      puts("dec (speech decoder)");
      break;
    default:
      puts("invalid option");
      exit(1);
  }

  if (GsmHr_giDTXon)
    printf("DTX mode:     enabled\n");
  else
    printf("DTX mode:     disabled\n");

  printf("input file:   %s\n", argv[2]);
  printf("output file:  %s\n\n", argv[3]);


  switch (option)
  {
    case 0:                            /* encode */

      /* start the encoder, VAD, and transmit DTX in the home state */
      /* ---------------------------------------------------------- */

      resetEnc();

      /* encode:  analyze 8 kHz speech and output encoded parameter file */
      /* --------------------------------------------------------------- */

      for (GsmHr_giFrmCnt = 1, iDoneFrm = 0;
           !iDoneFrm && GsmHr_giFrmCnt <= iMaxNumFrms;
           GsmHr_giFrmCnt++)
      {

#ifndef SILENT
        printf("encoding frame %d \r", GsmHr_giFrmCnt);
#endif

        if (encode(pfileInFile, pfileOutFile))
          iDoneFrm = 1;
      }

      if (iDoneFrm)
        GsmHr_giFrmCnt--;

      printf(" %d speech frames encoded\n", GsmHr_giFrmCnt - 1);
      break;

    case 1:                            /* decode */

      /* start the decoder and receive DTX in the home state */
      /* --------------------------------------------------- */

      resetDec();

      /* decode:  synthesize speech */
      /* -------------------------- */

      for (GsmHr_giFrmCnt = 1, iDoneFrm = 0;
           !iDoneFrm && GsmHr_giFrmCnt <= iMaxNumFrms;
           GsmHr_giFrmCnt++)
      {

#ifndef SILENT
        printf("decoding frame %d \r", GsmHr_giFrmCnt);
#endif

        if (decode(pfileInFile, pfileOutFile))
          iDoneFrm = 1;
      }

      if (iDoneFrm)
        GsmHr_giFrmCnt--;

      printf(" %d speech frames decoded\n", GsmHr_giFrmCnt - 1);
      break;
  }

  fclose(pfileInFile);
  fclose(pfileOutFile);

  return (0);
}

/**************************************************************************
 *
 *   FUNCTION NAME:  decode
 *
 *   PURPOSE:
 *      Reads in one frame of speech parameters and outputs one frame of
 *      synthesized speech.  Resets the decoder to the home state if the
 *      Decoder Homing Frame pattern is detected.
 *
 *   INPUT:
 *      pfileDec        speech parameter input file.
 *
 *   OUTPUT:
 *      pfileSpeechOut  synthesized speech file
 *
 *   RETURN:
 *      0               successfully synthesized a complete frame of speech
 *      1               failed to synthesize a complete frame of speech
 *
 *   REFERENCES: Sub-clause 4.2 of GSM Recomendation 06.20
 *
 *   KEYWORDS:
 *      pfileDec, pfileSpeechOut
 **************************************************************************/

int    decode(FILE *pfileDec, FILE *pfileSpeechOut)
{

/*_________________________________________________________________________
 |                                                                         |
 |                            Local Constants                              |
 |_________________________________________________________________________|
*/

/* These constants define the number of consecutive */
/* parameters that decoderHomingFrameTest checks.   */
/* -------------------------------------------------*/

#define  WHOLE_FRAME  18
#define  TO_FIRST_SUBFRAME  9

/*_________________________________________________________________________
 |                                                                         |
 |                            Static Variables                             |
 |_________________________________________________________________________|
*/
  static Shortword pswSpeechPara[22],
         pswDecodedSpeechFrame[F_LEN];

  static int reset_flag_decoder_old = 1;

/*_________________________________________________________________________
 |                                                                         |
 |                            Automatic Variables                          |
 |_________________________________________________________________________|
*/
  int    i,
         reset_flag_decoder;

/*_________________________________________________________________________
 |                                                                         |
 |                              Executable Code                            |
 |_________________________________________________________________________|
*/

  if (readDecfile(pfileDec, pswSpeechPara))
    return (1);

  if(reset_flag_decoder_old)
    reset_flag_decoder=decoderHomingFrameTest(pswSpeechPara,
					      TO_FIRST_SUBFRAME);
  else
    reset_flag_decoder=0;

  if (reset_flag_decoder && reset_flag_decoder_old)
  {
    /* force the output to be the encoder homing frame pattern */

    for (i = 0; i < F_LEN; i++)
      pswDecodedSpeechFrame[i] = EHF_MASK;
  }
  else
  {
    speechDecoder(pswSpeechPara, pswDecodedSpeechFrame);
  }

  speechDecoderHostInterface(pswDecodedSpeechFrame, pfileSpeechOut);

  if(!reset_flag_decoder_old)
    reset_flag_decoder=decoderHomingFrameTest(pswSpeechPara, WHOLE_FRAME);

  if (reset_flag_decoder)
    resetDec();                        /* bring the decoder and receive DTX
                                        * to the home state */

  reset_flag_decoder_old = reset_flag_decoder;

  return (0);
}

/**************************************************************************
 *
 *   FUNCTION NAME:  encode
 *
 *   PURPOSE:
 *      Reads in one frame of speech samples and outputs one frame of
 *      speech parameters.  Resets the encoder to the home state if the
 *      Encoder Homing Frame pattern is detected.
 *
 *   INPUT:
 *      pfileSpeechIn   speech file
 *
 *   OUTPUT:
 *      pfileEnc        speech, encoded paramater data
 *
 *   RETURN:
 *      0                successfully wrote a complete frame of data
 *      1                failed to write a complete frame of data
 *
 *   REFERENCES: Sub-clause 4.1 of GSM Recomendation 06.20
 *
 *   KEYWORDS:
 *      pfileSpeechIn, pfileEnc
 **************************************************************************/

int    encode(FILE *pfileSpeechIn, FILE *pfileEnc)
{

/*_________________________________________________________________________
 |                                                                         |
 |                            Static Variables                             |
 |_________________________________________________________________________|
*/
  static Shortword pswSpeechPara[20];
  static Shortword pswSpeechBuff[F_LEN];

/*_________________________________________________________________________
 |                                                                         |
 |                            Automatic Variables                          |
 |_________________________________________________________________________|
*/
  int    iNumRead,
         reset_flag;

/*_________________________________________________________________________
 |                                                                         |
 |                              Executable Code                            |
 |_________________________________________________________________________|
*/

  iNumRead = hostEncoderInterface(pfileSpeechIn, F_LEN, &pswSpeechBuff[0]);

  if (iNumRead < F_LEN)
    return (1);

  reset_flag = encoderHomingFrameTest(&pswSpeechBuff[0]);

  speechEncoder(&pswSpeechBuff[0], pswSpeechPara);

  if (writeEncfile(pswSpeechPara, pfileEnc) != 20)
    return (1);

  if (reset_flag)
    resetEnc();                        /* Bring the encoder, VAD, and DTX to
                                        * the home state */

  return (0);
}
#endif

#define GSMHR_CODEC_FRAME_BITS 112
#define GSMHR_CODEC_FRAME_BYTES 14
#define GSMHR_DECODER_INPUT_BUF_BYTES 22


//extern Shortword swSidDetection(Shortword pswParameters[], Shortword pswErrorFlag[]);
extern void RandomParameters(Shortword pswParameters[]);

static Longword prepareDecoderInput( unsigned char frametype, unsigned char* codecdata, Shortword *swOutPara);



__attribute__ ((visibility ("default"))) int apiGSMHRInit()
{
		/* start the encoder, VAD, and transmit DTX in the home state */
		resetEnc();
}

__attribute__ ((visibility ("default"))) int apiGSMHRdecode(const unsigned char* rtpPayload, unsigned int payloadLength, short *pcmbuf)
{
#define  WHOLE_FRAME  18
#define  TO_FIRST_SUBFRAME  9

  short pswSpeechPara[22];
  short *pswDecodedSpeechFrame;

  static int reset_flag_decoder_old = 1;

  int    i, tocIndex,
         reset_flag_decoder;

  unsigned char *pCodecData;
  int isLastToc;

  /*go through toc list to get the number of codec frames*/
  unsigned char *pToc = rtpPayload;
  unsigned int numToc = 0;
  while(numToc < payloadLength)
  {
    unsigned tocFollowed = *(pToc+numToc)&0x80;
    numToc++;

    if(tocFollowed == 0) break; //This is the last toc
  }

  /*revisit all toc and codec data*/
  pCodecData = pToc + numToc; //Pointer to the first codec data
  pswDecodedSpeechFrame = pcmbuf;

  for (tocIndex = 0; tocIndex < numToc; tocIndex++)
  {
    unsigned char frametype = *(pToc + tocIndex)&0x70;
    if((frametype == 2) || (frametype == 0))//SID or speech frame
    {
      if(pCodecData+14 > rtpPayload+payloadLength) break;//Insufficient bytes.

      prepareDecoderInput(frametype, pCodecData, pswSpeechPara);

      if(reset_flag_decoder_old)
        reset_flag_decoder=decoderHomingFrameTest(pswSpeechPara, TO_FIRST_SUBFRAME);
      else
        reset_flag_decoder=0;
    
      if (reset_flag_decoder && reset_flag_decoder_old)
      {
        /* force the output to be the encoder homing frame pattern */
    
        for (i = 0; i < F_LEN; i++)
          pswDecodedSpeechFrame[i] = EHF_MASK;
      }
      else
      {
        speechDecoder(pswSpeechPara, pswDecodedSpeechFrame);
      }
    
      if(!reset_flag_decoder_old)
        reset_flag_decoder=decoderHomingFrameTest(pswSpeechPara, WHOLE_FRAME);
    
      if (reset_flag_decoder)
        resetDec();                        /* bring the decoder and receive DTX to the home state */
    
      reset_flag_decoder_old = reset_flag_decoder;

      pswDecodedSpeechFrame += 160; //move to next place for another 20 ms pcm output
      pCodecData += 14;    //move 112 bits to next codec frame
    }
    else if(frametype == 7)//no data, no output
    {
      for (i = 0; i < F_LEN; i++)
        pswDecodedSpeechFrame[i] = 0;
    }
    else //unknown frame type. unable to proceed.
    {
      break;
    }
  }

  return (0);
}


/*-------------------------------------------------------------------------------------------
 *   INPUT DATA FORMAT:
 *
 *      The format/content of the input parameters is the so called
 *      bit alloc format.
 *
 *     voiced mode bit alloc format:
 *     -----------------------------
 *     index    number of bits  parameter name
 *     0        5               R0
 *     1        11              k1Tok3
 *     2        9               k4Tok6
 *     3        8               k7Tok10
 *     4        1               softInterpolation
 *     5        2               voicingDecision
 * 
 *     6        8               frameLag
 *     7        9               code_1
 *     8        5               gsp0_1
 *     9        4               deltaLag_2
 *     10       9               code_2
 *     11       5               gsp0_2
 *     12       4               deltaLag_3
 *     13       9               code_3
 *     14       5               gsp0_3
 *     15       4               deltaLag_4
 *     16       9               code_4
 *     17       5               gsp0_4
 *
 *     18       1               BFI
 *     19       1               UFI
 *     20       2               SID
 *     21       1               TAF
 *
 *
 *     unvoiced mode bit alloc format:
 *     -------------------------------
 *
 *     index    number of bits  parameter name
 *     0        5               R0
 *     1        11              k1Tok3
 *     2        9               k4Tok6
 *     3        8               k7Tok10
 *     4        1               softInterpolation
 *     5        2               voicingDecision
 *
 *     6        7               code1_1
 *     7        7               code2_1
 *     8        5               gsp0_1
 *     9        7               code1_2
 *     10       7               code2_2
 *     11       5               gsp0_2
 *     12       7               code1_3
 *     13       7               code2_3
 *     14       5               gsp0_3
 *     15       7               code1_4
 *     16       7               code2_4
 *     17       5               gsp0_4
 *
 *     18       1               BFI
 *     19       1               UFI
 *     20       2               SID
 *     21       1               TAF
 *
 *
 *   REFERENCES: Sub_Clause 4.2 of GSM Recomendation 06.20
 *-----------------------------------------------------------------------------------------*/

static Longword prepareDecoderInput( unsigned char frametype, unsigned char* codecdata, Shortword *swOutPara)
{

#define SPEECH      1
#define CNIFIRSTSID 2
#define CNICONT     3
#define VALIDSID    11
#define GOODSPEECH  33

    NlBis bis;

    static Shortword swDecoMode = {SPEECH};
    static Shortword swTAFCnt = {1};
    Shortword i, swFrameType;
    //Shortword pswErrorFlag[3];
 
    NlBis_InitFromBuf(&bis, codecdata, GSMHR_CODEC_FRAME_BITS, 0, 0);

    /* Copy input parameters to output parameters (error free transmission) */
    swOutPara[0] = NlBis_FieldValue(&bis, 5);
    NlBis_AdvanceB(&bis, 5);

    swOutPara[1] = NlBis_FieldValue(&bis, 11);
    NlBis_AdvanceB(&bis, 11);

    swOutPara[2] = NlBis_FieldValue(&bis, 9);
    NlBis_AdvanceB(&bis, 9);

    swOutPara[3] = NlBis_FieldValue(&bis, 8);
    NlBis_AdvanceB(&bis, 8);

    swOutPara[4] = NlBis_FieldValue(&bis, 1);
    NlBis_AdvanceB(&bis, 1);

    swOutPara[5] = NlBis_FieldValue(&bis, 2);
    NlBis_AdvanceB(&bis, 2);

    if(swOutPara[5] == 0) //unvoiced speech (MODE=0)
    {
        swOutPara[6] = NlBis_FieldValue(&bis, 7);
        NlBis_AdvanceB(&bis, 7);
        swOutPara[7] = NlBis_FieldValue(&bis, 7);
        NlBis_AdvanceB(&bis, 7);
        swOutPara[8] = NlBis_FieldValue(&bis, 5);
        NlBis_AdvanceB(&bis, 5);

        swOutPara[9] = NlBis_FieldValue(&bis, 7);
        NlBis_AdvanceB(&bis, 7);
        swOutPara[10] = NlBis_FieldValue(&bis, 7);
        NlBis_AdvanceB(&bis, 7);
        swOutPara[11] = NlBis_FieldValue(&bis, 5);
        NlBis_AdvanceB(&bis, 5);

        swOutPara[12] = NlBis_FieldValue(&bis, 7);
        NlBis_AdvanceB(&bis, 7);
        swOutPara[13] = NlBis_FieldValue(&bis, 7);
        NlBis_AdvanceB(&bis, 7);
        swOutPara[14] = NlBis_FieldValue(&bis, 5);
        NlBis_AdvanceB(&bis, 5);

        swOutPara[15] = NlBis_FieldValue(&bis, 7);
        NlBis_AdvanceB(&bis, 7);
        swOutPara[16] = NlBis_FieldValue(&bis, 7);
        NlBis_AdvanceB(&bis, 7);
        swOutPara[17] = NlBis_FieldValue(&bis, 5);
        NlBis_AdvanceB(&bis, 5);
    }
    else //voiced speech
    {
        swOutPara[6] = NlBis_FieldValue(&bis, 8);
        NlBis_AdvanceB(&bis, 8);
        swOutPara[7] = NlBis_FieldValue(&bis, 9);
        NlBis_AdvanceB(&bis, 9);
        swOutPara[8] = NlBis_FieldValue(&bis, 5);
        NlBis_AdvanceB(&bis, 5);

        swOutPara[9] = NlBis_FieldValue(&bis, 4);
        NlBis_AdvanceB(&bis, 4);
        swOutPara[10] = NlBis_FieldValue(&bis, 9);
        NlBis_AdvanceB(&bis, 9);
        swOutPara[11] = NlBis_FieldValue(&bis, 5);
        NlBis_AdvanceB(&bis, 5);

        swOutPara[12] = NlBis_FieldValue(&bis, 4);
        NlBis_AdvanceB(&bis, 4);
        swOutPara[13] = NlBis_FieldValue(&bis, 9);
        NlBis_AdvanceB(&bis, 9);
        swOutPara[14] = NlBis_FieldValue(&bis, 5);
        NlBis_AdvanceB(&bis, 5);

        swOutPara[15] = NlBis_FieldValue(&bis, 4);
        NlBis_AdvanceB(&bis, 4);
        swOutPara[16] = NlBis_FieldValue(&bis, 9);
        NlBis_AdvanceB(&bis, 9);
        swOutPara[17] = NlBis_FieldValue(&bis, 5);
        NlBis_AdvanceB(&bis, 5);
    }


    /* Set channel status flags (error free transmission) */
    /* -------------------------------------------------- */
    swOutPara[18] = 0;     /* BFI flag */
    swOutPara[19] = 0;     /* UFI flag */

    /* Evaluate TAF flag */
    /* ----------------- */
    if (swTAFCnt == 0) swOutPara[21] = 1;
    else               swOutPara[21] = 0;
    swTAFCnt = (swTAFCnt + 1) % 12;
   
    /* Evaluate SID flag */
    /* ----------------- */
    //pswErrorFlag[0] = 0;   /* BFI flag */
    //pswErrorFlag[1] = 0;   /* UFI flag */
    //pswErrorFlag[2] = 0;   /* BCI flag */
    //swOutPara[20] = swSidDetection(swOutPara, pswErrorFlag);
    swOutPara[20] = frametype;

    if      ( swOutPara[20] == 2) swFrameType = VALIDSID;
    else if ( swOutPara[20] == 0) swFrameType = GOODSPEECH;
    else {
       RandomParameters(swOutPara);
       /* Set flags such, that an "unusable frame" is produced */
       swOutPara[18] = 1;     /* BFI flag */
       swOutPara[19] = 1;     /* UFI flag */
       swOutPara[20] = 0;     /* SID flag */
       return( 0 );
    }

    /* Update of decoder state */
    /* ----------------------- */
    if (swDecoMode == SPEECH) {
       if      (swFrameType == VALIDSID)   swDecoMode = CNIFIRSTSID;
       else if (swFrameType == GOODSPEECH) swDecoMode = SPEECH;
    }
    else {  /* comfort noise insertion mode */
       if      (swFrameType == VALIDSID)   swDecoMode = CNICONT;
       else if (swFrameType == GOODSPEECH) swDecoMode = SPEECH;
    }


    /* Replace parameters by random data if in CNICONT-mode and TAF=0 */
    /* -------------------------------------------------------------- */
    if ((swDecoMode == CNICONT) && (swOutPara[21] == 0)){
       RandomParameters(swOutPara);
       /* Set flags such, that an "unusable frame" is produced */
       swOutPara[18] = 1;     /* BFI flag */
       swOutPara[19] = 1;     /* UFI flag */
       swOutPara[20] = 0;     /* SID flag */
    }    

    return( 0 );
}


