/**************************************************************************
 *
 *   File Name:  sp_rom.c
 *
 *   Purpose:  Contains all globally available constant (ROM) data
 *      used by the speech codec: quantization tables, codebooks, etc.
 *      Some ROM is defined within the file where it is used.  In those
 *      cases, the scope of the ROM variables is limited to the file
 *
 **************************************************************************/
/*_________________________________________________________________________
 |                                                                         |
 |                              Include Files                              |
 |_________________________________________________________________________|
*/
#include "mathhalf.h"
#include "sp_rom.h"

/*_________________________________________________________________________
 |                                                                         |
 |                               Global Data                               |
 |                  (scope is global to the entire program)                |
 |_________________________________________________________________________|
*/

/* gsp0 vector quantizer */
/*-----------------------*/

/*   THE GSP0 ERROR TERM VECTORS FOR  5 BITS FOLLOW
  THE GSP0 TABLE WAS READ FROM THE FILES gsp0_1_*.qnt_97CQ_4
  ET VALUES WRITTEN WERE SCALED BY  4  */

ShortwordRom pppsrGsp0
       [GSP0_NUM_OF_TABLES][GSP0_NUM][GSP0_VECTOR_SIZE] =
{
  {

    {33, 34, 0, 0, 0},                 /* Table 1, Centroid #  1 */
    {58, 63, 0, 0, 0},                 /* Table 1, Centroid #  2 */
    {85, 89, 0, 0, 0},                 /* Table 1, Centroid #  3 */
    {139, 149, -1, -1, -1},            /* Table 1, Centroid #  4 */
    {233, 235, -3, -2, -2},            /* Table 1, Centroid #  5 */
    {313, 329, -6, -3, -3},            /* Table 1, Centroid #  6 */
    {440, 456, -12, -6, -6},           /* Table 1, Centroid #  7 */
    {703, 530, -23, -15, -9},          /* Table 1, Centroid #  8 */
    {892, 767, -42, -24, -18},         /* Table 1, Centroid #  9 */
    {963, 1329, -78, -28, -54},        /* Table 1, Centroid # 10 */
    {1447, 1165, -103, -64, -41},      /* Table 1, Centroid # 11 */
    {1894, 1362, -157, -109, -57},     /* Table 1, Centroid # 12 */
    {1639, 2194, -219, -82, -147},     /* Table 1, Centroid # 13 */
    {2120, 3211, -415, -137, -315},    /* Table 1, Centroid # 14 */
    {2994, 2147, -392, -274, -141},    /* Table 1, Centroid # 15 */
    {4132, 2566, -647, -521, -201},    /* Table 1, Centroid # 16 */
    {3040, 4009, -744, -282, -490},    /* Table 1, Centroid # 17 */
    {4012, 5316, -1302, -491, -862},   /* Table 1, Centroid # 18 */
    {5356, 3752, -1226, -875, -430},   /* Table 1, Centroid # 19 */
    {7068, 4287, -1850, -1525, -561},  /* Table 1, Centroid # 20 */
    {4927, 7072, -2127, -741, -1526},  /* Table 1, Centroid # 21 */
    {6996, 7304, -3118, -1493, -1628}, /* Table 1, Centroid # 22 */
    {9047, 6154, -3398, -2498, -1156}, /* Table 1, Centroid # 23 */
    {11747, 6009, -4309, -4211, -1102},/* Table 1, Centroid # 24 */
    {5144, 10368, -3255, -808, -3280}, /* Table 1, Centroid # 25 */
    {7677, 12151, -5693, -1798, -4506},/* Table 1, Centroid # 26 */
    {8796, 9286, -4986, -2361, -2632}, /* Table 1, Centroid # 27 */
    {11323, 9802, -6774, -3912, -2932},/* Table 1, Centroid # 28 */
    {10555, 15059, -9701, -3400, -6920},        /* Table 1, Centroid # 29 */
    {17566, 15641, -16769, -9417, -7465},       /* Table 1, Centroid # 30 */
    {13934, 9966, -8476, -5925, -3031},/* Table 1, Centroid # 31 */
    {17646, 8051, -8672, -9503, -1978},/* Table 1, Centroid # 32 */
  },
  {
    {65, 62, 0, 0, 0},                 /* Table 2, Centroid #  1 */
    {199, 174, -2, -1, -1},            /* Table 2, Centroid #  2 */
    {526, 613, -20, -8, -11},          /* Table 2, Centroid #  3 */
    {705, 1191, -51, -15, -43},        /* Table 2, Centroid #  4 */
    {1695, 1067, -110, -88, -35},      /* Table 2, Centroid #  5 */
    {2176, 1897, -252, -144, -110},    /* Table 2, Centroid #  6 */
    {848, 3038, -157, -22, -282},      /* Table 2, Centroid #  7 */
    {2105, 3693, -474, -135, -416},    /* Table 2, Centroid #  8 */
    {3483, 1605, -341, -370, -79},     /* Table 2, Centroid #  9 */
    {3552, 3236, -702, -385, -320},    /* Table 2, Centroid # 10 */
    {4869, 2028, -603, -723, -126},    /* Table 2, Centroid # 11 */
    {6296, 2220, -853, -1210, -150},   /* Table 2, Centroid # 12 */
    {1425, 5682, -494, -62, -985},     /* Table 2, Centroid # 13 */
    {3538, 5556, -1200, -382, -942},   /* Table 2, Centroid # 14 */
    {1468, 8528, -764, -66, -2219},    /* Table 2, Centroid # 15 */
    {4805, 8004, -2347, -705, -1955},  /* Table 2, Centroid # 16 */
    {5189, 4143, -1312, -822, -524},   /* Table 2, Centroid # 17 */
    {6409, 5573, -2180, -1254, -948},  /* Table 2, Centroid # 18 */
    {7493, 2955, -1351, -1713, -266},  /* Table 2, Centroid # 19 */
    {8902, 2344, -1274, -2418, -168},  /* Table 2, Centroid # 20 */
    {8212, 5266, -2639, -2058, -846},  /* Table 2, Centroid # 21 */
    {10190, 5396, -3356, -3169, -889}, /* Table 2, Centroid # 22 */
    {10556, 3007, -1938, -3401, -276}, /* Table 2, Centroid # 23 */
    {12517, 2934, -2242, -4781, -263}, /* Table 2, Centroid # 24 */
    {5383, 11357, -3731, -884, -3936}, /* Table 2, Centroid # 25 */
    {9432, 9038, -5203, -2715, -2493}, /* Table 2, Centroid # 26 */
    {7993, 17273, -8426, -1950, -9105},/* Table 2, Centroid # 27 */
    {16236, 12894, -12778, -8045, -5073},       /* Table 2, Centroid # 28 */
    {12832, 6896, -5401, -5025, -1451},/* Table 2, Centroid # 29 */
    {14842, 4024, -3645, -6723, -494}, /* Table 2, Centroid # 30 */
    {18013, 7332, -8062, -9902, -1641},/* Table 2, Centroid # 31 */
    {19019, 3737, -4338, -11039, -426},/* Table 2, Centroid # 32 */
  },
  {
    {1004, 731, -45, -31, -16},        /* Table 3, Centroid #  1 */
    {1412, 1843, -159, -61, -104},     /* Table 3, Centroid #  2 */
    {2700, 1489, -245, -222, -68},     /* Table 3, Centroid #  3 */
    {2785, 2657, -452, -237, -215},    /* Table 3, Centroid #  4 */
    {4054, 1682, -416, -502, -86},     /* Table 3, Centroid #  5 */
    {4499, 2664, -732, -618, -217},    /* Table 3, Centroid #  6 */
    {5155, 1429, -450, -811, -62},     /* Table 3, Centroid #  7 */
    {6006, 1561, -572, -1101, -74},    /* Table 3, Centroid #  8 */
    {3421, 4218, -881, -357, -543},    /* Table 3, Centroid #  9 */
    {4313, 5518, -1452, -568, -929},   /* Table 3, Centroid # 10 */
    {5540, 3384, -1144, -937, -350},   /* Table 3, Centroid # 11 */
    {6832, 3531, -1472, -1424, -381},  /* Table 3, Centroid # 12 */
    {6382, 2333, -909, -1243, -166},   /* Table 3, Centroid # 13 */
    {7417, 1574, -713, -1679, -76},    /* Table 3, Centroid # 14 */
    {8037, 2688, -1319, -1971, -221},  /* Table 3, Centroid # 15 */
    {9014, 1637, -900, -2479, -82},    /* Table 3, Centroid # 16 */
    {6141, 6173, -2314, -1151, -1163}, /* Table 3, Centroid # 17 */
    {7883, 4573, -2201, -1896, -638},  /* Table 3, Centroid # 18 */
    {8608, 7132, -3747, -2261, -1552}, /* Table 3, Centroid # 19 */
    {10098, 4698, -2896, -3112, -674}, /* Table 3, Centroid # 20 */
    {9036, 3633, -2004, -2492, -403},  /* Table 3, Centroid # 21 */
    {9953, 2317, -1408, -3023, -164},  /* Table 3, Centroid # 22 */
    {11108, 3228, -2188, -3766, -318}, /* Table 3, Centroid # 23 */
    {11570, 1843, -1302, -4085, -104}, /* Table 3, Centroid # 24 */
    {11706, 6610, -4723, -4182, -1333},/* Table 3, Centroid # 25 */
    {13128, 4692, -3760, -5260, -672}, /* Table 3, Centroid # 26 */
    {12725, 2951, -2292, -4942, -266}, /* Table 3, Centroid # 27 */
    {14028, 2325, -1990, -6005, -165}, /* Table 3, Centroid # 28 */
    {16126, 7702, -7580, -7936, -1810},/* Table 3, Centroid # 29 */
    {15583, 4486, -4267, -7411, -614}, /* Table 3, Centroid # 30 */
    {16137, 2609, -2569, -7947, -208}, /* Table 3, Centroid # 31 */
    {19264, 3907, -4593, -11325, -466},/* Table 3, Centroid # 32 */
  },
  {
    {1900, 1716, -199, -110, -90},     /* Table 4, Centroid #  1 */
    {2946, 979, -176, -265, -29},      /* Table 4, Centroid #  2 */
    {3683, 842, -189, -414, -22},      /* Table 4, Centroid #  3 */
    {4395, 1025, -275, -589, -32},     /* Table 4, Centroid #  4 */
    {3540, 2364, -511, -382, -171},    /* Table 4, Centroid #  5 */
    {5001, 1944, -594, -763, -115},    /* Table 4, Centroid #  6 */
    {5189, 852, -270, -822, -22},      /* Table 4, Centroid #  7 */
    {5862, 1050, -376, -1049, -34},    /* Table 4, Centroid #  8 */
    {5221, 2984, -951, -832, -272},    /* Table 4, Centroid #  9 */
    {6189, 3184, -1203, -1169, -309},  /* Table 4, Centroid # 10 */
    {6408, 1263, -494, -1253, -49},    /* Table 4, Centroid # 11 */
    {7032, 1044, -448, -1509, -33},    /* Table 4, Centroid # 12 */
    {6965, 2270, -965, -1481, -157},   /* Table 4, Centroid # 13 */
    {7875, 2276, -1094, -1893, -158},  /* Table 4, Centroid # 14 */
    {7691, 1181, -554, -1805, -43},    /* Table 4, Centroid # 15 */
    {8365, 944, -482, -2135, -27},     /* Table 4, Centroid # 16 */
    {7657, 4252, -1987, -1789, -552},  /* Table 4, Centroid # 17 */
    {8878, 2876, -1559, -2405, -253},  /* Table 4, Centroid # 18 */
    {8733, 1813, -966, -2328, -100},   /* Table 4, Centroid # 19 */
    {9279, 1167, -661, -2628, -42},    /* Table 4, Centroid # 20 */
    {9894, 3465, -2092, -2988, -366},  /* Table 4, Centroid # 21 */
    {10615, 2340, -1516, -3439, -167}, /* Table 4, Centroid # 22 */
    {10057, 1444, -886, -3087, -64},   /* Table 4, Centroid # 23 */
    {10975, 1209, -810, -3676, -45},   /* Table 4, Centroid # 24 */
    {11358, 4510, -3127, -3937, -621}, /* Table 4, Centroid # 25 */
    {12248, 2748, -2054, -4578, -230}, /* Table 4, Centroid # 26 */
    {11824, 1559, -1125, -4267, -74},  /* Table 4, Centroid # 27 */
    {12838, 1441, -1129, -5030, -63},  /* Table 4, Centroid # 28 */
    {13808, 3978, -3353, -5819, -483}, /* Table 4, Centroid # 29 */
    {14250, 1928, -1677, -6197, -113}, /* Table 4, Centroid # 30 */
    {16642, 3962, -4025, -8452, -479}, /* Table 4, Centroid # 31 */
    {16641, 1762, -1790, -8451, -95},  /* Table 4, Centroid # 32 */
  }
};


/* unvoiced code vectors */
/*-----------------------*/

ShortwordRom pppsrUvCodeVec
       [UVCODEVEC_NUM_OF_CODE_BOOKS][UVCODEVEC_NUM_OF_CODE_BITS][S_LEN] =
{
  {
    {
      12, 11, -49, -53, 9,             /* File# 1, Vec# 1,  1- 5 */
      -45, -19, -59, -88, 43,          /* File# 1, Vec# 1,  6-10 */
      2, -15, 17, -18, 30,             /* File# 1, Vec# 1, 11-15 */
      71, 19, 22, 30, 1,               /* File# 1, Vec# 1, 16-20 */
      38, 1, -17, -22, -34,            /* File# 1, Vec# 1, 21-25 */
      17, -21, -37, -45, -19,          /* File# 1, Vec# 1, 26-30 */
      23, 5, 40, 40, 14,               /* File# 1, Vec# 1, 31-35 */
      25, 21, 2, -12, 24               /* File# 1, Vec# 1, 36-40 */
    },
    {
      4, -31, 50, -3, -36,             /* File# 1, Vec# 2,  1- 5 */
      22, 28, 3, -18, -5,              /* File# 1, Vec# 2,  6-10 */
      0, 7, 77, 7, -54,                /* File# 1, Vec# 2, 11-15 */
      42, -29, -48, -11, -3,           /* File# 1, Vec# 2, 16-20 */
      -18, -34, -4, -47, -61,          /* File# 1, Vec# 2, 21-25 */
      1, -26, -4, 13, -46,             /* File# 1, Vec# 2, 26-30 */
      7, -21, -24, 36, 22,             /* File# 1, Vec# 2, 31-35 */
      59, 38, 34, 62, 12               /* File# 1, Vec# 2, 36-40 */
    },
    {
      -5, -17, -4, 7, 21,              /* File# 1, Vec# 3,  1- 5 */
      3, -60, -29, 8, 25,              /* File# 1, Vec# 3,  6-10 */
      58, 22, 23, 69, -25,             /* File# 1, Vec# 3, 11-15 */
      -4, 15, -43, -32, -20,           /* File# 1, Vec# 3, 16-20 */
      26, -27, 27, 20, -5,             /* File# 1, Vec# 3, 21-25 */
      -23, -31, 20, 39, -4,            /* File# 1, Vec# 3, 26-30 */
      -69, 46, 52, -2, -12,            /* File# 1, Vec# 3, 31-35 */
      17, 24, -22, -14, -6             /* File# 1, Vec# 3, 36-40 */
    },
    {
      -28, 28, 51, 24, -5,             /* File# 1, Vec# 4,  1- 5 */
      16, 8, -27, 24, 19,              /* File# 1, Vec# 4,  6-10 */
      -61, 26, 15, -46, 17,            /* File# 1, Vec# 4, 11-15 */
      -17, -22, 22, 48, 31,            /* File# 1, Vec# 4, 16-20 */
      36, 13, 20, 53, -36,             /* File# 1, Vec# 4, 21-25 */
      -43, -29, -20, 3, -77,           /* File# 1, Vec# 4, 26-30 */
      -40, 30, -12, -3, -24,           /* File# 1, Vec# 4, 31-35 */
      -15, 45, -13, 1, 25              /* File# 1, Vec# 4, 36-40 */
    },
    {
      -29, -5, -47, -65, -6,           /* File# 1, Vec# 5,  1- 5 */
      81, -13, 21, 45, 16,             /* File# 1, Vec# 5,  6-10 */
      30, 29, 76, -12, -14,            /* File# 1, Vec# 5, 11-15 */
      27, -24, -4, 44, -14,            /* File# 1, Vec# 5, 16-20 */
      -43, 17, -39, -19, -9,           /* File# 1, Vec# 5, 21-25 */
      -17, 13, -9, 35, 30,             /* File# 1, Vec# 5, 26-30 */
      -20, 11, 10, 12, 20,             /* File# 1, Vec# 5, 31-35 */
      -53, -55, 2, -21, 6              /* File# 1, Vec# 5, 36-40 */
    },
    {
      37, -22, -30, 35, -36,           /* File# 1, Vec# 6,  1- 5 */
      -17, -18, 15, 5, -53,            /* File# 1, Vec# 6,  6-10 */
      -18, 34, 61, 21, 58,             /* File# 1, Vec# 6, 11-15 */
      27, 40, 52, -5, -3,              /* File# 1, Vec# 6, 16-20 */
      24, 6, 14, -8, -43,              /* File# 1, Vec# 6, 21-25 */
      11, -48, -56, -8, 26,            /* File# 1, Vec# 6, 26-30 */
      -35, -27, 21, -51, -65,          /* File# 1, Vec# 6, 31-35 */
      7, 11, 5, 3, -17                 /* File# 1, Vec# 6, 36-40 */
    },
    {
      -135, 34, -2, -72, -42,          /* File# 1, Vec# 7,  1- 5 */
      21, 40, 7, 26, -25,              /* File# 1, Vec# 7,  6-10 */
      -11, 3, 8, 22, 11,               /* File# 1, Vec# 7, 11-15 */
      24, 22, 22, -19, 16,             /* File# 1, Vec# 7, 16-20 */
      34, 52, 48, -27, 21,             /* File# 1, Vec# 7, 21-25 */
      38, -1, 2, -5, -39,              /* File# 1, Vec# 7, 26-30 */
      -12, 8, 10, -12, -14,            /* File# 1, Vec# 7, 31-35 */
      16, -8, -20, -18, -25            /* File# 1, Vec# 7, 36-40 */
    }
  },
  {
    {
      -60, 5, 8, 52, 56,               /* File# 2, Vec# 1,  1- 5 */
      -8, -44, 0, -28, -45,            /* File# 2, Vec# 1,  6-10 */
      -36, -9, 12, 25, 54,             /* File# 2, Vec# 1, 11-15 */
      17, -59, 7, 35, -20,             /* File# 2, Vec# 1, 16-20 */
      34, -11, -31, -36, 14,           /* File# 2, Vec# 1, 21-25 */
      -5, -46, 60, 27, -4,             /* File# 2, Vec# 1, 26-30 */
      28, 0, 16, 28, -8,               /* File# 2, Vec# 1, 31-35 */
      -2, -54, 1, 16, -42              /* File# 2, Vec# 1, 36-40 */
    },
    {
      -25, -29, 13, -18, 37,           /* File# 2, Vec# 2,  1- 5 */
      -7, -60, -13, -3, -11,           /* File# 2, Vec# 2,  6-10 */
      -24, 20, 8, -54, -18,            /* File# 2, Vec# 2, 11-15 */
      4, -24, -21, 14, 67,             /* File# 2, Vec# 2, 16-20 */
      52, -9, 43, 13, -9,              /* File# 2, Vec# 2, 21-25 */
      52, 5, -16, 33, 65,              /* File# 2, Vec# 2, 26-30 */
      17, -14, -14, -55, 66,           /* File# 2, Vec# 2, 31-35 */
      46, -17, -13, -15, -11           /* File# 2, Vec# 2, 36-40 */
    },
    {
      -53, -22, -4, 20, 24,            /* File# 2, Vec# 3,  1- 5 */
      -60, 18, 75, -7, -2,             /* File# 2, Vec# 3,  6-10 */
      22, -7, -30, -49, 5,             /* File# 2, Vec# 3, 11-15 */
      26, -6, -76, 19, 18,             /* File# 2, Vec# 3, 16-20 */
      -13, 23, -18, -13, 7,            /* File# 2, Vec# 3, 21-25 */
      -7, -35, -57, 10, 2,             /* File# 2, Vec# 3, 26-30 */
      -41, 38, 32, -1, -4,             /* File# 2, Vec# 3, 31-35 */
      -38, 8, 26, 40, 18               /* File# 2, Vec# 3, 36-40 */
    },
    {
      -9, 5, 1, 24, 46,                /* File# 2, Vec# 4,  1- 5 */
      -22, -21, 22, 69, 60,            /* File# 2, Vec# 4,  6-10 */
      -55, -42, 27, -15, -17,          /* File# 2, Vec# 4, 11-15 */
      -4, 48, 51, -12, 12,             /* File# 2, Vec# 4, 16-20 */
      -47, 10, 41, -67, -14,           /* File# 2, Vec# 4, 21-25 */
      -41, -8, 21, -25, 14,            /* File# 2, Vec# 4, 26-30 */
      1, -5, 27, 21, 41,               /* File# 2, Vec# 4, 31-35 */
      38, 7, 24, -4, -18               /* File# 2, Vec# 4, 36-40 */
    },
    {
      33, -39, 51, -27, 34,            /* File# 2, Vec# 5,  1- 5 */
      -26, -79, 38, -23, -12,          /* File# 2, Vec# 5,  6-10 */
      26, 29, 20, -19, -28,            /* File# 2, Vec# 5, 11-15 */
      -26, 2, 59, 5, -30,              /* File# 2, Vec# 5, 16-20 */
      -6, 44, 29, -37, 37,             /* File# 2, Vec# 5, 21-25 */
      38, 18, -2, -29, -61,            /* File# 2, Vec# 5, 26-30 */
      -41, -20, -49, 23, -14,          /* File# 2, Vec# 5, 31-35 */
      -26, -8, -20, 10, 29             /* File# 2, Vec# 5, 36-40 */
    },
    {
      72, 92, 7, 14, 19,               /* File# 2, Vec# 6,  1- 5 */
      -11, 34, 23, -8, 0,              /* File# 2, Vec# 6,  6-10 */
      -27, -12, 11, -15, -32,          /* File# 2, Vec# 6, 11-15 */
      17, 10, 13, -64, -5,             /* File# 2, Vec# 6, 16-20 */
      35, 23, -21, -35, -21,           /* File# 2, Vec# 6, 21-25 */
      -11, -15, -29, 61, -21,          /* File# 2, Vec# 6, 26-30 */
      -10, 71, -20, 2, 36,             /* File# 2, Vec# 6, 31-35 */
      -6, -52, -40, -11, -6            /* File# 2, Vec# 6, 36-40 */
    },
    {
      -43, -45, 61, 38, 37,            /* File# 2, Vec# 7,  1- 5 */
      8, 55, -4, -8, 67,               /* File# 2, Vec# 7,  6-10 */
      25, 13, 16, 11, 0,               /* File# 2, Vec# 7, 11-15 */
      24, 31, 29, -1, -57,             /* File# 2, Vec# 7, 16-20 */
      44, 5, -66, 3, -10,              /* File# 2, Vec# 7, 21-25 */
      -24, 24, -27, -24, 3,            /* File# 2, Vec# 7, 26-30 */
      -36, -3, -26, -55, 24,           /* File# 2, Vec# 7, 31-35 */
      24, -43, 0, 5, -35               /* File# 2, Vec# 7, 36-40 */
    }
  }
};


/* voiced code vectors */
/*---------------------*/

ShortwordRom pppsrVcdCodeVec
       [VCDCODEVEC_NUM_OF_CODE_BOOKS][VCDCODEVEC_NUM_OF_CODE_BITS][S_LEN] =
{
  {
    {
      24, 13, -55, -51, -8,            /* File# 1, Vec# 1,  1- 5 */
      -58, 2, -75, -70, 22,            /* File# 1, Vec# 1,  6-10 */
      -4, -21, 19, -21, 23,            /* File# 1, Vec# 1, 11-15 */
      54, 16, 10, 32, 1,               /* File# 1, Vec# 1, 16-20 */
      30, 3, -15, -7, -42,             /* File# 1, Vec# 1, 21-25 */
      -2, -22, -36, -20, -27,          /* File# 1, Vec# 1, 26-30 */
      12, 21, 46, 35, 31,              /* File# 1, Vec# 1, 31-35 */
      27, 3, 16, -9, 4                 /* File# 1, Vec# 1, 36-40 */
    },
    {
      -10, -53, 46, 10, -28,           /* File# 1, Vec# 2,  1- 5 */
      18, 19, 22, -17, -11,            /* File# 1, Vec# 2,  6-10 */
      6, 0, 58, 7, -37,                /* File# 1, Vec# 2, 11-15 */
      32, -24, -49, -7, -10,           /* File# 1, Vec# 2, 16-20 */
      -25, -32, -3, -48, -41,          /* File# 1, Vec# 2, 21-25 */
      -6, -24, -18, 18, -43,           /* File# 1, Vec# 2, 26-30 */
      9, -18, -15, 36, 28,             /* File# 1, Vec# 2, 31-35 */
      74, 58, 47, 49, 21               /* File# 1, Vec# 2, 36-40 */
    },
    {
      -25, -1, 8, 20, 13,              /* File# 1, Vec# 3,  1- 5 */
      -34, -52, -35, 23, 18,           /* File# 1, Vec# 3,  6-10 */
      70, 44, 25, 76, -20,             /* File# 1, Vec# 3, 11-15 */
      1, 6, -49, -18, -43,             /* File# 1, Vec# 3, 16-20 */
      27, -29, 32, 36, 3,              /* File# 1, Vec# 3, 21-25 */
      -6, -34, 11, 37, -13,            /* File# 1, Vec# 3, 26-30 */
      -50, 58, 32, 0, 12,              /* File# 1, Vec# 3, 31-35 */
      16, 16, -6, -20, -3              /* File# 1, Vec# 3, 36-40 */
    },
    {
      -50, 38, 47, 43, 9,              /* File# 1, Vec# 4,  1- 5 */
      31, 1, -23, 25, 21,              /* File# 1, Vec# 4,  6-10 */
      -38, 20, 7, -55, 7,              /* File# 1, Vec# 4, 11-15 */
      -30, -8, 23, 44, 28,             /* File# 1, Vec# 4, 16-20 */
      44, 19, 26, 49, -32,             /* File# 1, Vec# 4, 21-25 */
      -28, -51, -30, -11, -65,         /* File# 1, Vec# 4, 26-30 */
      -44, 40, -9, 22, -1,             /* File# 1, Vec# 4, 31-35 */
      -6, 44, -3, 7, 13                /* File# 1, Vec# 4, 36-40 */
    },
    {
      -93, -25, -46, -35, 11,          /* File# 1, Vec# 5,  1- 5 */
      69, -8, 37, 45, 21,              /* File# 1, Vec# 5,  6-10 */
      42, 23, 64, -25, -10,            /* File# 1, Vec# 5, 11-15 */
      16, -19, 8, 41, -7,              /* File# 1, Vec# 5, 16-20 */
      -32, 10, -45, -17, -20,          /* File# 1, Vec# 5, 21-25 */
      -35, 25, -6, 16, 46,             /* File# 1, Vec# 5, 26-30 */
      -1, 14, 10, 6, 4,                /* File# 1, Vec# 5, 31-35 */
      -43, -29, -9, -14, -7            /* File# 1, Vec# 5, 36-40 */
    },
    {
      76, -13, -23, 33, -35,           /* File# 1, Vec# 6,  1- 5 */
      5, -19, 7, 18, -59,              /* File# 1, Vec# 6,  6-10 */
      -15, 40, 76, 36, 57,             /* File# 1, Vec# 6, 11-15 */
      29, 38, 58, 2, 0,                /* File# 1, Vec# 6, 16-20 */
      16, 6, 9, -14, -25,              /* File# 1, Vec# 6, 21-25 */
      9, -49, -39, -5, 17,             /* File# 1, Vec# 6, 26-30 */
      -33, -44, 9, -43, -32,           /* File# 1, Vec# 6, 31-35 */
      -1, 13, 1, 9, -6                 /* File# 1, Vec# 6, 36-40 */
    },
    {
      -196, 16, -22, -35, -34,         /* File# 1, Vec# 7,  1- 5 */
      20, 30, 10, 19, -22,             /* File# 1, Vec# 7,  6-10 */
      2, -13, -4, 22, 14,              /* File# 1, Vec# 7, 11-15 */
      22, 20, 15, -15, 22,             /* File# 1, Vec# 7, 16-20 */
      29, 40, 43, -14, 30,             /* File# 1, Vec# 7, 21-25 */
      36, -8, -10, -19, -42,           /* File# 1, Vec# 7, 26-30 */
      -12, -15, -9, -12, -12,          /* File# 1, Vec# 7, 31-35 */
      0, -9, -5, -11, -7               /* File# 1, Vec# 7, 36-40 */
    },
    {
      -77, 48, 24, 65, 45,             /* File# 1, Vec# 8,  1- 5 */
      -3, -38, 15, -29, -44,           /* File# 1, Vec# 8,  6-10 */
      -47, 6, 2, 4, 63,                /* File# 1, Vec# 8, 11-15 */
      27, -26, 27, 47, -13,            /* File# 1, Vec# 8, 16-20 */
      17, -16, -23, -31, -3,           /* File# 1, Vec# 8, 21-25 */
      -10, -33, 46, 18, 15,            /* File# 1, Vec# 8, 26-30 */
      43, 9, 25, 24, -26,              /* File# 1, Vec# 8, 31-35 */
      -10, -46, -27, -13, -37          /* File# 1, Vec# 8, 36-40 */
    },
    {
      -24, -56, 30, -9, 27,            /* File# 1, Vec# 9,  1- 5 */
      -13, -56, -1, -24, -24,          /* File# 1, Vec# 9,  6-10 */
      -38, -1, 10, -57, -21,           /* File# 1, Vec# 9, 11-15 */
      -1, -34, -12, 29, 55,            /* File# 1, Vec# 9, 16-20 */
      42, -1, 44, 33, 2,               /* File# 1, Vec# 9, 21-25 */
      54, 11, -9, 55, 73,              /* File# 1, Vec# 9, 26-30 */
      13, -2, 3, -46, 52,              /* File# 1, Vec# 9, 31-35 */
      27, 1, 4, -4, -17                /* File# 1, Vec# 9, 36-40 */
    }
  }
};


/* vector quantizer tables */
/*-------------------------*/

ShortwordRom psrQuant1
       [QUANT1_NUM_OF_WORDS] =
{
  -5980,                               /* Vector    0, coef 1(232), 2(164) */
  25056,                               /* Vector    0, coef 3( 97) Vector 1,
                                        * Coef 1(224) */
  -22691,                              /* Vector    1, coef 2(167), 3( 93) */
  -7724,                               /* Vector    2, coef 1(225), 2(212) */
  27102,                               /* Vector    2, coef 3(105) Vector 3,
                                        * Coef 1(222) */
  -18323,                              /* Vector    3, coef 2(184), 3(109) */
  -8275,                               /* Vector    4, coef 1(223), 2(173) */
  18136,                               /* Vector    4, coef 3( 70) Vector 5,
                                        * Coef 1(216) */
  -13700,                              /* Vector    5, coef 2(202), 3(124) */
  -11073,                              /* Vector    6, coef 1(212), 2(191) */
  -27189,                              /* Vector    6, coef 3(149) Vector 7,
                                        * Coef 1(203) */
  -21101,                              /* Vector    7, coef 2(173), 3(147) */
  -15434,                              /* Vector    8, coef 1(195), 2(182) */
  -31279,                              /* Vector    8, coef 3(133) Vector 9,
                                        * Coef 1(209) */
  -22689,                              /* Vector    9, coef 2(167), 3( 95) */
  -12363,                              /* Vector   10, coef 1(207), 2(181) */
  21448,                               /* Vector   10, coef 3( 83) Vector 11,
                                        * Coef 1(200) */
  -21397,                              /* Vector   11, coef 2(172), 3(107) */
  -12572,                              /* Vector   12, coef 1(206), 2(228) */
  -22070,                              /* Vector   12, coef 3(169) Vector 13,
                                        * Coef 1(202) */
  -5974,                               /* Vector   13, coef 2(232), 3(170) */
  -14112,                              /* Vector   14, coef 1(200), 2(224) */
  -25915,                              /* Vector   14, coef 3(154) Vector 15,
                                        * Coef 1(197) */
  -6003,                               /* Vector   15, coef 2(232), 3(141) */
  -13613,                              /* Vector   16, coef 1(202), 2(211) */
  -29240,                              /* Vector   16, coef 3(141) Vector 17,
                                        * Coef 1(200) */
  -10112,                              /* Vector   17, coef 2(216), 3(128) */
  -15142,                              /* Vector   18, coef 1(196), 2(218) */
  31938,                               /* Vector   18, coef 3(124) Vector 19,
                                        * Coef 1(194) */
  -12410,                              /* Vector   19, coef 2(207), 3(134) */
  -13114,                              /* Vector   20, coef 1(204), 2(198) */
  30406,                               /* Vector   20, coef 3(118) Vector 21,
                                        * Coef 1(198) */
  -15487,                              /* Vector   21, coef 2(195), 3(129) */
  -16186,                              /* Vector   22, coef 1(192), 2(198) */
  -31809,                              /* Vector   22, coef 3(131) Vector 23,
                                        * Coef 1(191) */
  -17285,                              /* Vector   23, coef 2(188), 3(123) */
  -15419,                              /* Vector   24, coef 1(195), 2(197) */
  28093,                               /* Vector   24, coef 3(109) Vector 25,
                                        * Coef 1(189) */
  -19864,                              /* Vector   25, coef 2(178), 3(104) */
  -15396,                              /* Vector   26, coef 1(195), 2(220) */
  -24635,                              /* Vector   26, coef 3(159) Vector 27,
                                        * Coef 1(197) */
  -12651,                              /* Vector   27, coef 2(206), 3(149) */
  -16421,                              /* Vector   28, coef 1(191), 2(219) */
  -21825,                              /* Vector   28, coef 3(170) Vector 29,
                                        * Coef 1(191) */
  -7791,                               /* Vector   29, coef 2(225), 3(145) */
  -16956,                              /* Vector   30, coef 1(189), 2(196) */
  23237,                               /* Vector   30, coef 3( 90) Vector 31,
                                        * Coef 1(197) */
  -25725,                              /* Vector   31, coef 2(155), 3(131) */
  -17473,                              /* Vector   32, coef 1(187), 2(191) */
  29630,                               /* Vector   32, coef 3(115) Vector 33,
                                        * Coef 1(190) */
  -9829,                               /* Vector   33, coef 2(217), 3(155) */
  -16935,                              /* Vector   34, coef 1(189), 2(217) */
  -29000,                              /* Vector   34, coef 3(142) Vector 35,
                                        * Coef 1(184) */
  -7789,                               /* Vector   35, coef 2(225), 3(147) */
  -18216,                              /* Vector   36, coef 1(184), 2(216) */
  -27975,                              /* Vector   36, coef 3(146) Vector 37,
                                        * Coef 1(185) */
  -5732,                               /* Vector   37, coef 2(233), 3(156) */
  -18214,                              /* Vector   38, coef 1(184), 2(218) */
  -24388,                              /* Vector   38, coef 3(160) Vector 39,
                                        * Coef 1(188) */
  -12122,                              /* Vector   39, coef 2(208), 3(166) */
  -17207,                              /* Vector   40, coef 1(188), 2(201) */
  -25675,                              /* Vector   40, coef 3(155) Vector 41,
                                        * Coef 1(181) */
  -9054,                               /* Vector   41, coef 2(220), 3(162) */
  -19743,                              /* Vector   42, coef 1(178), 2(225) */
  -23118,                              /* Vector   42, coef 3(165) Vector 43,
                                        * Coef 1(178) */
  -8809,                               /* Vector   43, coef 2(221), 3(151) */
  -20512,                              /* Vector   44, coef 1(175), 2(224) */
  -24644,                              /* Vector   44, coef 3(159) Vector 45,
                                        * Coef 1(188) */
  -12400,                              /* Vector   45, coef 2(207), 3(144) */
  -17712,                              /* Vector   46, coef 1(186), 2(208) */
  -30792,                              /* Vector   46, coef 3(135) Vector 47,
                                        * Coef 1(184) */
  -15224,                              /* Vector   47, coef 2(196), 3(136) */
  -19258,                              /* Vector   48, coef 1(180), 2(198) */
  -31564,                              /* Vector   48, coef 3(132) Vector 49,
                                        * Coef 1(180) */
  -9845,                               /* Vector   49, coef 2(217), 3(139) */
  -20263,                              /* Vector   50, coef 1(176), 2(217) */
  -29518,                              /* Vector   50, coef 3(140) Vector 51,
                                        * Coef 1(178) */
  -12929,                              /* Vector   51, coef 2(205), 3(127) */
  -20531,                              /* Vector   52, coef 1(175), 2(205) */
  -31814,                              /* Vector   52, coef 3(131) Vector 53,
                                        * Coef 1(186) */
  -9355,                               /* Vector   53, coef 2(219), 3(117) */
  -18480,                              /* Vector   54, coef 1(183), 2(208) */
  30131,                               /* Vector   54, coef 3(117) Vector 55,
                                        * Coef 1(179) */
  -10383,                              /* Vector   55, coef 2(215), 3(113) */
  -17459,                              /* Vector   56, coef 1(187), 2(205) */
  -18246,                              /* Vector   56, coef 3(184) Vector 57,
                                        * Coef 1(186) */
  -14416,                              /* Vector   57, coef 2(199), 3(176) */
  -19502,                              /* Vector   58, coef 1(179), 2(210) */
  -21581,                              /* Vector   58, coef 3(171) Vector 59,
                                        * Coef 1(179) */
  -13909,                              /* Vector   59, coef 2(201), 3(171) */
  -18994,                              /* Vector   60, coef 1(181), 2(206) */
  -24905,                              /* Vector   60, coef 3(158) Vector 61,
                                        * Coef 1(183) */
  -14443,                              /* Vector   61, coef 2(199), 3(149) */
  -19515,                              /* Vector   62, coef 1(179), 2(197) */
  -26437,                              /* Vector   62, coef 3(152) Vector 63,
                                        * Coef 1(187) */
  -19835,                              /* Vector   63, coef 2(178), 3(133) */
  -20543,                              /* Vector   64, coef 1(175), 2(193) */
  30890,                               /* Vector   64, coef 3(120) Vector 65,
                                        * Coef 1(170) */
  -15250,                              /* Vector   65, coef 2(196), 3(110) */
  -20031,                              /* Vector   66, coef 1(177), 2(193) */
  -27219,                              /* Vector   66, coef 3(149) Vector 67,
                                        * Coef 1(173) */
  -9822,                               /* Vector   67, coef 2(217), 3(162) */
  -21295,                              /* Vector   68, coef 1(172), 2(209) */
  -23893,                              /* Vector   68, coef 3(162) Vector 69,
                                        * Coef 1(171) */
  -12373,                              /* Vector   69, coef 2(207), 3(171) */
  -22574,                              /* Vector   70, coef 1(167), 2(210) */
  -21337,                              /* Vector   70, coef 3(172) Vector 71,
                                        * Coef 1(167) */
  -12133,                              /* Vector   71, coef 2(208), 3(155) */
  -23855,                              /* Vector   72, coef 1(162), 2(209) */
  -19800,                              /* Vector   72, coef 3(178) Vector 73,
                                        * Coef 1(168) */
  -15203,                              /* Vector   73, coef 2(196), 3(157) */
  -23104,                              /* Vector   74, coef 1(165), 2(192) */
  -26452,                              /* Vector   74, coef 3(152) Vector 75,
                                        * Coef 1(172) */
  -11884,                              /* Vector   75, coef 2(209), 3(148) */
  -21560,                              /* Vector   76, coef 1(171), 2(200) */
  -28751,                              /* Vector   76, coef 3(143) Vector 77,
                                        * Coef 1(177) */
  -17781,                              /* Vector   77, coef 2(186), 3(139) */
  -21315,                              /* Vector   78, coef 1(172), 2(189) */
  -30038,                              /* Vector   78, coef 3(138) Vector 79,
                                        * Coef 1(170) */
  -12415,                              /* Vector   79, coef 2(207), 3(129) */
  -22588,                              /* Vector   80, coef 1(167), 2(196) */
  32420,                               /* Vector   80, coef 3(126) Vector 81,
                                        * Coef 1(164) */
  -13177,                              /* Vector   81, coef 2(204), 3(135) */
  -24123,                              /* Vector   82, coef 1(161), 2(197) */
  -31056,                              /* Vector   82, coef 3(134) Vector 83,
                                        * Coef 1(176) */
  -21620,                              /* Vector   83, coef 2(171), 3(140) */
  -22607,                              /* Vector   84, coef 1(167), 2(177) */
  -27221,                              /* Vector   84, coef 3(149) Vector 85,
                                        * Coef 1(171) */
  -23137,                              /* Vector   85, coef 2(165), 3(159) */
  -22092,                              /* Vector   86, coef 1(169), 2(180) */
  -30810,                              /* Vector   86, coef 3(135) Vector 87,
                                        * Coef 1(166) */
  -18816,                              /* Vector   87, coef 2(182), 3(128) */
  -19531,                              /* Vector   88, coef 1(179), 2(181) */
  -32602,                              /* Vector   88, coef 3(128) Vector 89,
                                        * Coef 1(166) */
  -8046,                               /* Vector   89, coef 2(224), 3(146) */
  -24094,                              /* Vector   90, coef 1(161), 2(226) */
  -28254,                              /* Vector   90, coef 3(145) Vector 91,
                                        * Coef 1(162) */
  -11878,                              /* Vector   91, coef 2(209), 3(154) */
  -24873,                              /* Vector   92, coef 1(158), 2(215) */
  -24931,                              /* Vector   92, coef 3(158) Vector 93,
                                        * Coef 1(157) */
  -8315,                               /* Vector   93, coef 2(223), 3(133) */
  -22318,                              /* Vector   94, coef 1(168), 2(210) */
  27555,                               /* Vector   94, coef 3(107) Vector 95,
                                        * Coef 1(163) */
  -13201,                              /* Vector   95, coef 2(204), 3(111) */
  -18764,                              /* Vector   96, coef 1(182), 2(180) */
  28848,                               /* Vector   96, coef 3(112) Vector 97,
                                        * Coef 1(176) */
  -13975,                              /* Vector   97, coef 2(201), 3(105) */
  -21579,                              /* Vector   98, coef 1(171), 2(181) */
  29348,                               /* Vector   98, coef 3(114) Vector 99,
                                        * Coef 1(164) */
  -20371,                              /* Vector   99, coef 2(176), 3(109) */
  -23379,                              /* Vector  100, coef 1(164), 2(173) */
  -31583,                              /* Vector  100, coef 3(132) Vector
                                        * 101, Coef 1(161) */
  -21381,                              /* Vector  101, coef 2(172), 3(123) */
  -18777,                              /* Vector  102, coef 1(182), 2(167) */
  -32080,                              /* Vector  102, coef 3(130) Vector
                                        * 103, Coef 1(176) */
  -21382,                              /* Vector  103, coef 2(172), 3(122) */
  -17754,                              /* Vector  104, coef 1(186), 2(166) */
  30898,                               /* Vector  104, coef 3(120) Vector
                                        * 105, Coef 1(178) */
  -23697,                              /* Vector  105, coef 2(163), 3(111) */
  -20839,                              /* Vector  106, coef 1(174), 2(153) */
  26034,                               /* Vector  106, coef 3(101) Vector
                                        * 107, Coef 1(178) */
  -22436,                              /* Vector  107, coef 2(168), 3( 92) */
  -19020,                              /* Vector  108, coef 1(181), 2(180) */
  20657,                               /* Vector  108, coef 3( 80) Vector
                                        * 109, Coef 1(177) */
  -19125,                              /* Vector  109, coef 2(181), 3( 75) */
  -19525,                              /* Vector  110, coef 1(179), 2(187) */
  27311,                               /* Vector  110, coef 3(106) Vector
                                        * 111, Coef 1(175) */
  -20381,                              /* Vector  111, coef 2(176), 3( 99) */
  -22618,                              /* Vector  112, coef 1(167), 2(166) */
  27812,                               /* Vector  112, coef 3(108) Vector
                                        * 113, Coef 1(164) */
  -25246,                              /* Vector  113, coef 2(157), 3( 98) */
  -21598,                              /* Vector  114, coef 1(171), 2(162) */
  -30806,                              /* Vector  114, coef 3(135) Vector
                                        * 115, Coef 1(170) */
  -27260,                              /* Vector  115, coef 2(149), 3(132) */
  -22109,                              /* Vector  116, coef 1(169), 2(163) */
  31139,                               /* Vector  116, coef 3(121) Vector
                                        * 117, Coef 1(163) */
  -25736,                              /* Vector  117, coef 2(155), 3(120) */
  -24671,                              /* Vector  118, coef 1(159), 2(161) */
  -32356,                              /* Vector  118, coef 3(129) Vector
                                        * 119, Coef 1(156) */
  -24717,                              /* Vector  119, coef 2(159), 3(115) */
  -21870,                              /* Vector  120, coef 1(170), 2(146) */
  29597,                               /* Vector  120, coef 3(115) Vector
                                        * 121, Coef 1(157) */
  -28308,                              /* Vector  121, coef 2(145), 3(108) */
  -24695,                              /* Vector  122, coef 1(159), 2(137) */
  -31576,                              /* Vector  122, coef 3(132) Vector
                                        * 123, Coef 1(168) */
  -19364,                              /* Vector  123, coef 2(180), 3( 92) */
  -25438,                              /* Vector  124, coef 1(156), 2(162) */
  25250,                               /* Vector  124, coef 3( 98) Vector
                                        * 125, Coef 1(162) */
  -18316,                              /* Vector  125, coef 2(184), 3(116) */
  -25172,                              /* Vector  126, coef 1(157), 2(172) */
  28575,                               /* Vector  126, coef 3(111) Vector
                                        * 127, Coef 1(159) */
  -19868,                              /* Vector  127, coef 2(178), 3(100) */
  -9361,                               /* Vector  128, coef 1(219), 2(111) */
  16344,                               /* Vector  128, coef 3( 63) Vector
                                        * 129, Coef 1(216) */
  -25510,                              /* Vector  129, coef 2(156), 3( 90) */
  -11889,                              /* Vector  130, coef 1(209), 2(143) */
  20151,                               /* Vector  130, coef 3( 78) Vector
                                        * 131, Coef 1(183) */
  -26768,                              /* Vector  131, coef 2(151), 3(112) */
  -23158,                              /* Vector  132, coef 1(165), 2(138) */
  26316,                               /* Vector  132, coef 3(102) Vector
                                        * 133, Coef 1(204) */
  32120,                               /* Vector  133, coef 2(125), 3(120) */
  -17047,                              /* Vector  134, coef 1(189), 2(105) */
  28596,                               /* Vector  134, coef 3(111) Vector
                                        * 135, Coef 1(180) */
  22908,                               /* Vector  135, coef 2( 89), 3(124) */
  -18617,                              /* Vector  136, coef 1(183), 2( 71) */
  30431,                               /* Vector  136, coef 3(118) Vector
                                        * 137, Coef 1(223) */
  29029,                               /* Vector  137, coef 2(113), 3(101) */
  -13988,                              /* Vector  138, coef 1(201), 2( 92) */
  22966,                               /* Vector  138, coef 3( 89) Vector
                                        * 139, Coef 1(182) */
  28514,                               /* Vector  139, coef 2(111), 3( 98) */
  -20898,                              /* Vector  140, coef 1(174), 2( 94) */
  23493,                               /* Vector  140, coef 3( 91) Vector
                                        * 141, Coef 1(197) */
  -28552,                              /* Vector  141, coef 2(144), 3(120) */
  -14958,                              /* Vector  142, coef 1(197), 2(146) */
  25786,                               /* Vector  142, coef 3(100) Vector
                                        * 143, Coef 1(186) */
  -31898,                              /* Vector  143, coef 2(131), 3(102) */
  -14725,                              /* Vector  144, coef 1(198), 2(123) */
  23205,                               /* Vector  144, coef 3( 90) Vector
                                        * 145, Coef 1(165) */
  -31843,                              /* Vector  145, coef 2(131), 3(157) */
  -21397,                              /* Vector  146, coef 1(172), 2(107) */
  -23115,                              /* Vector  146, coef 3(165) Vector
                                        * 147, Coef 1(181) */
  30855,                               /* Vector  147, coef 2(120), 3(135) */
  -24461,                              /* Vector  148, coef 1(160), 2(115) */
  -32612,                              /* Vector  148, coef 3(128) Vector
                                        * 149, Coef 1(156) */
  -32647,                              /* Vector  149, coef 2(128), 3(121) */
  -29316,                              /* Vector  150, coef 1(141), 2(124) */
  -31330,                              /* Vector  150, coef 3(133) Vector
                                        * 151, Coef 1(158) */
  25740,                               /* Vector  151, coef 2(100), 3(140) */
  -28069,                              /* Vector  152, coef 1(146), 2( 91) */
  -31863,                              /* Vector  152, coef 3(131) Vector
                                        * 153, Coef 1(137) */
  27512,                               /* Vector  153, coef 2(107), 3(120) */
  -20354,                              /* Vector  154, coef 1(176), 2(126) */
  31142,                               /* Vector  154, coef 3(121) Vector
                                        * 155, Coef 1(166) */
  29296,                               /* Vector  155, coef 2(114), 3(112) */
  -20606,                              /* Vector  156, coef 1(175), 2(130) */
  25763,                               /* Vector  156, coef 3(100) Vector
                                        * 157, Coef 1(163) */
  29794,                               /* Vector  157, coef 2(116), 3( 98) */
  -26781,                              /* Vector  158, coef 1(151), 2( 99) */
  29340,                               /* Vector  158, coef 3(114) Vector
                                        * 159, Coef 1(156) */
  23144,                               /* Vector  159, coef 2( 90), 3(104) */
  -16214,                              /* Vector  160, coef 1(192), 2(170) */
  27334,                               /* Vector  160, coef 3(106) Vector
                                        * 161, Coef 1(198) */
  -28846,                              /* Vector  161, coef 2(143), 3( 82) */
  -16251,                              /* Vector  162, coef 1(192), 2(133) */
  18092,                               /* Vector  162, coef 3( 70) Vector
                                        * 163, Coef 1(172) */
  27212,                               /* Vector  163, coef 2(106), 3( 76) */
  31572,                               /* Vector  164, coef 1(123), 2( 84) */
  28050,                               /* Vector  164, coef 3(109) Vector
                                        * 165, Coef 1(146) */
  25184,                               /* Vector  165, coef 2( 98), 3( 96) */
  -26783,                              /* Vector  166, coef 1(151), 2( 97) */
  21102,                               /* Vector  166, coef 3( 82) Vector
                                        * 167, Coef 1(110) */
  17765,                               /* Vector  167, coef 2( 69), 3(101) */
  28473,                               /* Vector  168, coef 1(111), 2( 57) */
  26502,                               /* Vector  168, coef 3(103) Vector
                                        * 169, Coef 1(134) */
  18264,                               /* Vector  169, coef 2( 71), 3( 88) */
  -31417,                              /* Vector  170, coef 1(133), 2( 71) */
  17578,                               /* Vector  170, coef 3( 68) Vector
                                        * 171, Coef 1(170) */
  20558,                               /* Vector  171, coef 2( 80), 3( 78) */
  -16280,                              /* Vector  172, coef 1(192), 2(104) */
  14727,                               /* Vector  172, coef 3( 57) Vector
                                        * 173, Coef 1(135) */
  11105,                               /* Vector  173, coef 2( 43), 3( 97) */
  -25554,                              /* Vector  174, coef 1(156), 2( 46) */
  17859,                               /* Vector  174, coef 3( 69) Vector
                                        * 175, Coef 1(195) */
  -18870,                              /* Vector  175, coef 2(182), 3( 74) */
  -17242,                              /* Vector  176, coef 1(188), 2(166) */
  21943,                               /* Vector  176, coef 3( 85) Vector
                                        * 177, Coef 1(183) */
  -26531,                              /* Vector  177, coef 2(152), 3( 93) */
  -20848,                              /* Vector  178, coef 1(174), 2(144) */
  21677,                               /* Vector  178, coef 3( 84) Vector
                                        * 179, Coef 1(173) */
  32333,                               /* Vector  179, coef 2(126), 3( 77) */
  -20610,                              /* Vector  180, coef 1(175), 2(126) */
  15288,                               /* Vector  180, coef 3( 59) Vector
                                        * 181, Coef 1(184) */
  -24003,                              /* Vector  181, coef 2(162), 3( 61) */
  -21876,                              /* Vector  182, coef 1(170), 2(140) */
  14235,                               /* Vector  182, coef 3( 55) Vector
                                        * 183, Coef 1(155) */
  31579,                               /* Vector  183, coef 2(123), 3( 91) */
  -24198,                              /* Vector  184, coef 1(161), 2(122) */
  20891,                               /* Vector  184, coef 3( 81) Vector
                                        * 185, Coef 1(155) */
  30017,                               /* Vector  185, coef 2(117), 3( 65) */
  -30876,                              /* Vector  186, coef 1(135), 2(100) */
  13949,                               /* Vector  186, coef 3( 54) Vector
                                        * 187, Coef 1(125) */
  25416,                               /* Vector  187, coef 2( 99), 3( 72) */
  22097,                               /* Vector  188, coef 1( 86), 2( 81) */
  18554,                               /* Vector  188, coef 3( 72) Vector
                                        * 189, Coef 1(122) */
  22866,                               /* Vector  189, coef 2( 89), 3( 82) */
  31829,                               /* Vector  190, coef 1(124), 2( 85) */
  18009,                               /* Vector  190, coef 3( 70) Vector
                                        * 191, Coef 1( 89) */
  10361,                               /* Vector  191, coef 2( 40), 3(121) */
  -31366,                              /* Vector  192, coef 1(133), 2(122) */
  30579,                               /* Vector  192, coef 3(119) Vector
                                        * 193, Coef 1(115) */
  21641,                               /* Vector  193, coef 2( 84), 3(137) */
  31080,                               /* Vector  194, coef 1(121), 2(104) */
  28527,                               /* Vector  194, coef 3(111) Vector
                                        * 195, Coef 1(111) */
  22390,                               /* Vector  195, coef 2( 87), 3(118) */
  25710,                               /* Vector  196, coef 1(100), 2(110) */
  -30132,                              /* Vector  196, coef 3(138) Vector
                                        * 197, Coef 1( 76) */
  26486,                               /* Vector  197, coef 2(103), 3(118) */
  18280,                               /* Vector  198, coef 1( 71), 2(104) */
  25914,                               /* Vector  198, coef 3(101) Vector
                                        * 199, Coef 1( 58) */
  25195,                               /* Vector  199, coef 2( 98), 3(107) */
  21862,                               /* Vector  200, coef 1( 85), 2(102) */
  26705,                               /* Vector  200, coef 3(104) Vector
                                        * 201, Coef 1( 81) */
  25186,                               /* Vector  201, coef 2( 98), 3( 98) */
  18018,                               /* Vector  202, coef 1( 70), 2( 98) */
  23104,                               /* Vector  202, coef 3( 90) Vector
                                        * 203, Coef 1( 64) */
  22366,                               /* Vector  203, coef 2( 87), 3( 94) */
  19808,                               /* Vector  204, coef 1( 77), 2( 96) */
  -31928,                              /* Vector  204, coef 3(131) Vector
                                        * 205, Coef 1( 72) */
  21635,                               /* Vector  205, coef 2( 84), 3(131) */
  15195,                               /* Vector  206, coef 1( 59), 2( 91) */
  30781,                               /* Vector  206, coef 3(120) Vector
                                        * 207, Coef 1( 61) */
  20852,                               /* Vector  207, coef 2( 81), 3(116) */
  22109,                               /* Vector  208, coef 1( 86), 2( 93) */
  28754,                               /* Vector  208, coef 3(112) Vector
                                        * 209, Coef 1( 82) */
  22631,                               /* Vector  209, coef 2( 88), 3(103) */
  18510,                               /* Vector  210, coef 1( 72), 2( 78) */
  26704,                               /* Vector  210, coef 3(104) Vector
                                        * 211, Coef 1( 80) */
  19037,                               /* Vector  211, coef 2( 74), 3( 93) */
  -31637,                              /* Vector  212, coef 1(132), 2(107) */
  25971,                               /* Vector  212, coef 3(101) Vector
                                        * 213, Coef 1(115) */
  24156,                               /* Vector  213, coef 2( 94), 3( 92) */
  25180,                               /* Vector  214, coef 1( 98), 2( 92) */
  22104,                               /* Vector  214, coef 3( 86) Vector
                                        * 215, Coef 1( 88) */
  21587,                               /* Vector  215, coef 2( 84), 3( 83) */
  29812,                               /* Vector  216, coef 1(116), 2(116) */
  31338,                               /* Vector  216, coef 3(122) Vector
                                        * 217, Coef 1(106) */
  26998,                               /* Vector  217, coef 2(105), 3(118) */
  31609,                               /* Vector  218, coef 1(123), 2(121) */
  26732,                               /* Vector  218, coef 3(104) Vector
                                        * 219, Coef 1(108) */
  28006,                               /* Vector  219, coef 2(109), 3(102) */
  25192,                               /* Vector  220, coef 1( 98), 2(104) */
  22624,                               /* Vector  220, coef 3( 88) Vector
                                        * 221, Coef 1( 96) */
  24669,                               /* Vector  221, coef 2( 96), 3( 93) */
  22109,                               /* Vector  222, coef 1( 86), 2( 93) */
  21064,                               /* Vector  222, coef 3( 82) Vector
                                        * 223, Coef 1( 72) */
  15988,                               /* Vector  223, coef 2( 62), 3(116) */
  -26486,                              /* Vector  224, coef 1(152), 2(138) */
  24715,                               /* Vector  224, coef 3( 96) Vector
                                        * 225, Coef 1(139) */
  -31895,                              /* Vector  225, coef 2(131), 3(105) */
  -28549,                              /* Vector  226, coef 1(144), 2(123) */
  25732,                               /* Vector  226, coef 3(100) Vector
                                        * 227, Coef 1(132) */
  30810,                               /* Vector  227, coef 2(120), 3( 90) */
  -23917,                              /* Vector  228, coef 1(162), 2(147) */
  21395,                               /* Vector  228, coef 3( 83) Vector
                                        * 229, Coef 1(147) */
  -32690,                              /* Vector  229, coef 2(128), 3( 78) */
  -24437,                              /* Vector  230, coef 1(160), 2(139) */
  16777,                               /* Vector  230, coef 3( 65) Vector
                                        * 231, Coef 1(137) */
  28733,                               /* Vector  231, coef 2(112), 3( 61) */
  31610,                               /* Vector  232, coef 1(123), 2(122) */
  21354,                               /* Vector  232, coef 3( 83) Vector
                                        * 233, Coef 1(106) */
  27217,                               /* Vector  233, coef 2(106), 3( 81) */
  27500,                               /* Vector  234, coef 1(107), 2(108) */
  18562,                               /* Vector  234, coef 3( 72) Vector
                                        * 235, Coef 1(130) */
  29517,                               /* Vector  235, coef 2(115), 3( 77) */
  27748,                               /* Vector  236, coef 1(108), 2(100) */
  19572,                               /* Vector  236, coef 3( 76) Vector
                                        * 237, Coef 1(116) */
  28226,                               /* Vector  237, coef 2(110), 3( 66) */
  30824,                               /* Vector  238, coef 1(120), 2(104) */
  16812,                               /* Vector  238, coef 3( 65) Vector
                                        * 239, Coef 1(172) */
  -20397,                              /* Vector  239, coef 2(176), 3( 83) */
  -21056,                              /* Vector  240, coef 1(173), 2(192) */
  19107,                               /* Vector  240, coef 3( 74) Vector
                                        * 241, Coef 1(163) */
  -20396,                              /* Vector  241, coef 2(176), 3( 84) */
  -26473,                              /* Vector  242, coef 1(152), 2(151) */
  22428,                               /* Vector  242, coef 3( 87) Vector
                                        * 243, Coef 1(156) */
  -26036,                              /* Vector  243, coef 2(154), 3( 76) */
  -27243,                              /* Vector  244, coef 1(149), 2(149) */
  18583,                               /* Vector  244, coef 3( 72) Vector
                                        * 245, Coef 1(151) */
  -30398,                              /* Vector  245, coef 2(137), 3( 66) */
  -24418,                              /* Vector  246, coef 1(160), 2(158) */
  13967,                               /* Vector  246, coef 3( 54) Vector
                                        * 247, Coef 1(143) */
  -31173,                              /* Vector  247, coef 2(134), 3( 59) */
  -31366,                              /* Vector  248, coef 1(133), 2(122) */
  13472,                               /* Vector  248, coef 3( 52) Vector
                                        * 249, Coef 1(160) */
  -22478,                              /* Vector  249, coef 2(168), 3( 50) */
  -24400,                              /* Vector  250, coef 1(160), 2(176) */
  11155,                               /* Vector  250, coef 3( 43) Vector
                                        * 251, Coef 1(147) */
  -26308,                              /* Vector  251, coef 2(153), 3( 60) */
  -26977,                              /* Vector  252, coef 1(150), 2(159) */
  12937,                               /* Vector  252, coef 3( 50) Vector
                                        * 253, Coef 1(137) */
  -29876,                              /* Vector  253, coef 2(139), 3( 76) */
  -32644,                              /* Vector  254, coef 1(128), 2(124) */
  18051,                               /* Vector  254, coef 3( 70) Vector
                                        * 255, Coef 1(131) */
  32567,                               /* Vector  255, coef 2(127), 3( 55) */
  -23642,                              /* Vector  256, coef 1(163), 2(166) */
  -28516,                              /* Vector  256, coef 3(144) Vector
                                        * 257, Coef 1(156) */
  -11895,                              /* Vector  257, coef 2(209), 3(137) */
  -26147,                              /* Vector  258, coef 1(153), 2(221) */
  -29801,                              /* Vector  258, coef 3(139) Vector
                                        * 259, Coef 1(151) */
  -11645,                              /* Vector  259, coef 2(210), 3(131) */
  -24632,                              /* Vector  260, coef 1(159), 2(200) */
  -23393,                              /* Vector  260, coef 3(164) Vector
                                        * 261, Coef 1(159) */
  -14952,                              /* Vector  261, coef 2(197), 3(152) */
  -26161,                              /* Vector  262, coef 1(153), 2(207) */
  -25449,                              /* Vector  262, coef 3(156) Vector
                                        * 263, Coef 1(151) */
  -13672,                              /* Vector  263, coef 2(202), 3(152) */
  -24391,                              /* Vector  264, coef 1(160), 2(185) */
  -25445,                              /* Vector  264, coef 3(156) Vector
                                        * 265, Coef 1(155) */
  -16989,                              /* Vector  265, coef 2(189), 3(163) */
  -26692,                              /* Vector  266, coef 1(151), 2(188) */
  -25444,                              /* Vector  266, coef 3(156) Vector
                                        * 267, Coef 1(156) */
  -20575,                              /* Vector  267, coef 2(175), 3(161) */
  -23875,                              /* Vector  268, coef 1(162), 2(189) */
  -28512,                              /* Vector  268, coef 3(144) Vector
                                        * 269, Coef 1(160) */
  -18552,                              /* Vector  269, coef 2(183), 3(136) */
  -25661,                              /* Vector  270, coef 1(155), 2(195) */
  -29285,                              /* Vector  270, coef 3(141) Vector
                                        * 271, Coef 1(155) */
  -18043,                              /* Vector  271, coef 2(185), 3(133) */
  -25673,                              /* Vector  272, coef 1(155), 2(183) */
  -28518,                              /* Vector  272, coef 3(144) Vector
                                        * 273, Coef 1(154) */
  -21105,                              /* Vector  273, coef 2(173), 3(143) */
  -25682,                              /* Vector  274, coef 1(155), 2(174) */
  -31586,                              /* Vector  274, coef 3(132) Vector
                                        * 275, Coef 1(158) */
  -15491,                              /* Vector  275, coef 2(195), 3(125) */
  -25918,                              /* Vector  276, coef 1(154), 2(194) */
  31387,                               /* Vector  276, coef 3(122) Vector
                                        * 277, Coef 1(155) */
  -18569,                              /* Vector  277, coef 2(183), 3(119) */
  -27201,                              /* Vector  278, coef 1(149), 2(191) */
  30613,                               /* Vector  278, coef 3(119) Vector
                                        * 279, Coef 1(149) */
  -8817,                               /* Vector  279, coef 2(221), 3(143) */
  -27441,                              /* Vector  280, coef 1(148), 2(207) */
  -29549,                              /* Vector  280, coef 3(140) Vector
                                        * 281, Coef 1(147) */
  -13669,                              /* Vector  281, coef 2(202), 3(155) */
  -26941,                              /* Vector  282, coef 1(150), 2(195) */
  -29293,                              /* Vector  282, coef 3(141) Vector
                                        * 283, Coef 1(147) */
  -18543,                              /* Vector  283, coef 2(183), 3(145) */
  -27465,                              /* Vector  284, coef 1(148), 2(183) */
  -31080,                              /* Vector  284, coef 3(134) Vector
                                        * 285, Coef 1(152) */
  -24667,                              /* Vector  285, coef 2(159), 3(165) */
  -26156,                              /* Vector  286, coef 1(153), 2(212) */
  27796,                               /* Vector  286, coef 3(108) Vector
                                        * 287, Coef 1(148) */
  -14481,                              /* Vector  287, coef 2(199), 3(111) */
  -28202,                              /* Vector  288, coef 1(145), 2(214) */
  -29298,                              /* Vector  288, coef 3(141) Vector
                                        * 289, Coef 1(142) */
  -11636,                              /* Vector  289, coef 2(210), 3(140) */
  -28460,                              /* Vector  290, coef 1(144), 2(212) */
  -23409,                              /* Vector  290, coef 3(164) Vector
                                        * 291, Coef 1(143) */
  -10342,                              /* Vector  291, coef 2(215), 3(154) */
  -28472,                              /* Vector  292, coef 1(144), 2(200) */
  -27254,                              /* Vector  292, coef 3(149) Vector
                                        * 293, Coef 1(138) */
  -8540,                               /* Vector  293, coef 2(222), 3(164) */
  -29996,                              /* Vector  294, coef 1(138), 2(212) */
  -27514,                              /* Vector  294, coef 3(148) Vector
                                        * 295, Coef 1(134) */
  -9582,                               /* Vector  295, coef 2(218), 3(146) */
  -30774,                              /* Vector  296, coef 1(135), 2(202) */
  -29553,                              /* Vector  296, coef 3(140) Vector
                                        * 297, Coef 1(143) */
  -15458,                              /* Vector  297, coef 2(195), 3(158) */
  -29247,                              /* Vector  298, coef 1(141), 2(193) */
  -28278,                              /* Vector  298, coef 3(145) Vector
                                        * 299, Coef 1(138) */
  -16226,                              /* Vector  299, coef 2(192), 3(158) */
  -30789,                              /* Vector  300, coef 1(135), 2(187) */
  -26479,                              /* Vector  300, coef 3(152) Vector
                                        * 301, Coef 1(145) */
  -12675,                              /* Vector  301, coef 2(206), 3(125) */
  -28729,                              /* Vector  302, coef 1(143), 2(199) */
  32653,                               /* Vector  302, coef 3(127) Vector
                                        * 303, Coef 1(141) */
  -9353,                               /* Vector  303, coef 2(219), 3(119) */
  -29998,                              /* Vector  304, coef 1(138), 2(210) */
  31885,                               /* Vector  304, coef 3(124) Vector
                                        * 305, Coef 1(141) */
  -16766,                              /* Vector  305, coef 2(190), 3(130) */
  -29250,                              /* Vector  306, coef 1(141), 2(190) */
  29832,                               /* Vector  306, coef 3(116) Vector
                                        * 307, Coef 1(136) */
  -16776,                              /* Vector  307, coef 2(190), 3(120) */
  -30022,                              /* Vector  308, coef 1(138), 2(186) */
  -30331,                              /* Vector  308, coef 3(137) Vector
                                        * 309, Coef 1(133) */
  -18037,                              /* Vector  309, coef 2(185), 3(139) */
  -27969,                              /* Vector  310, coef 1(146), 2(191) */
  -30575,                              /* Vector  310, coef 3(136) Vector
                                        * 311, Coef 1(145) */
  -20333,                              /* Vector  311, coef 2(176), 3(147) */
  -28493,                              /* Vector  312, coef 1(144), 2(179) */
  -31599,                              /* Vector  312, coef 3(132) Vector
                                        * 313, Coef 1(145) */
  -9888,                               /* Vector  313, coef 2(217), 3( 96) */
  -28727,                              /* Vector  314, coef 1(143), 2(201) */
  26753,                               /* Vector  314, coef 3(104) Vector
                                        * 315, Coef 1(129) */
  -9071,                               /* Vector  315, coef 2(220), 3(145) */
  -32052,                              /* Vector  316, coef 1(130), 2(204) */
  -29814,                              /* Vector  316, coef 3(139) Vector
                                        * 317, Coef 1(138) */
  -21334,                              /* Vector  317, coef 2(172), 3(170) */
  -30769,                              /* Vector  318, coef 1(135), 2(207) */
  30339,                               /* Vector  318, coef 3(118) Vector
                                        * 319, Coef 1(131) */
  -12935,                              /* Vector  319, coef 2(205), 3(121) */
  -26214,                              /* Vector  320, coef 1(153), 2(154) */
  -32620,                              /* Vector  320, coef 3(128) Vector
                                        * 321, Coef 1(148) */
  -26256,                              /* Vector  321, coef 2(153), 3(112) */
  -28025,                              /* Vector  322, coef 1(146), 2(135) */
  29334,                               /* Vector  322, coef 3(114) Vector
                                        * 323, Coef 1(150) */
  -21884,                              /* Vector  323, coef 2(170), 3(132) */
  -27981,                              /* Vector  324, coef 1(146), 2(179) */
  30596,                               /* Vector  324, coef 3(119) Vector
                                        * 325, Coef 1(132) */
  -19082,                              /* Vector  325, coef 2(181), 3(118) */
  -30804,                              /* Vector  326, coef 1(135), 2(172) */
  -28542,                              /* Vector  326, coef 3(144) Vector
                                        * 327, Coef 1(130) */
  -22133,                              /* Vector  327, coef 2(169), 3(139) */
  -32336,                              /* Vector  328, coef 1(129), 2(176) */
  -32112,                              /* Vector  328, coef 3(130) Vector
                                        * 329, Coef 1(144) */
  -21369,                              /* Vector  329, coef 2(172), 3(135) */
  -29781,                              /* Vector  330, coef 1(139), 2(171) */
  -31336,                              /* Vector  330, coef 3(133) Vector
                                        * 331, Coef 1(152) */
  -24432,                              /* Vector  331, coef 2(160), 3(144) */
  -28262,                              /* Vector  332, coef 1(145), 2(154) */
  -28533,                              /* Vector  332, coef 3(144) Vector
                                        * 333, Coef 1(139) */
  -25466,                              /* Vector  333, coef 2(156), 3(134) */
  -28277,                              /* Vector  334, coef 1(145), 2(139) */
  -28282,                              /* Vector  334, coef 3(145) Vector
                                        * 335, Coef 1(134) */
  -31086,                              /* Vector  335, coef 2(134), 3(146) */
  -32104,                              /* Vector  336, coef 1(130), 2(152) */
  -30313,                              /* Vector  336, coef 3(137) Vector
                                        * 337, Coef 1(151) */
  -21895,                              /* Vector  337, coef 2(170), 3(121) */
  -27739,                              /* Vector  338, coef 1(147), 2(165) */
  31118,                               /* Vector  338, coef 3(121) Vector
                                        * 339, Coef 1(142) */
  -23173,                              /* Vector  339, coef 2(165), 3(123) */
  -29533,                              /* Vector  340, coef 1(140), 2(163) */
  29073,                               /* Vector  340, coef 3(113) Vector
                                        * 341, Coef 1(145) */
  -27011,                              /* Vector  341, coef 2(150), 3(125) */
  -29806,                              /* Vector  342, coef 1(139), 2(146) */
  30603,                               /* Vector  342, coef 3(119) Vector
                                        * 343, Coef 1(139) */
  -20099,                              /* Vector  343, coef 2(177), 3(125) */
  -30548,                              /* Vector  344, coef 1(136), 2(172) */
  31110,                               /* Vector  344, coef 3(121) Vector
                                        * 345, Coef 1(134) */
  -23679,                              /* Vector  345, coef 2(163), 3(129) */
  -32084,                              /* Vector  346, coef 1(130), 2(172) */
  31868,                               /* Vector  346, coef 3(124) Vector
                                        * 347, Coef 1(124) */
  -22394,                              /* Vector  347, coef 2(168), 3(134) */
  -31067,                              /* Vector  348, coef 1(134), 2(165) */
  29063,                               /* Vector  348, coef 3(113) Vector
                                        * 349, Coef 1(135) */
  -19860,                              /* Vector  349, coef 2(178), 3(108) */
  -31068,                              /* Vector  350, coef 1(134), 2(164) */
  -23941,                              /* Vector  350, coef 3(162) Vector
                                        * 351, Coef 1(123) */
  -27734,                              /* Vector  351, coef 2(147), 3(170) */
  32175,                               /* Vector  352, coef 1(125), 2(175) */
  31357,                               /* Vector  352, coef 3(122) Vector
                                        * 353, Coef 1(125) */
  -27518,                              /* Vector  353, coef 2(148), 3(130) */
  -31593,                              /* Vector  354, coef 1(132), 2(151) */
  30592,                               /* Vector  354, coef 3(119) Vector
                                        * 355, Coef 1(128) */
  -27536,                              /* Vector  355, coef 2(148), 3(112) */
  32647,                               /* Vector  356, coef 1(127), 2(135) */
  32636,                               /* Vector  356, coef 3(127) Vector
                                        * 357, Coef 1(124) */
  -31627,                              /* Vector  357, coef 2(132), 3(117) */
  29315,                               /* Vector  358, coef 1(114), 2(131) */
  29057,                               /* Vector  358, coef 3(113) Vector
                                        * 359, Coef 1(129) */
  -25220,                              /* Vector  359, coef 2(157), 3(124) */
  -32341,                              /* Vector  360, coef 1(129), 2(171) */
  29566,                               /* Vector  360, coef 3(115) Vector
                                        * 361, Coef 1(126) */
  -24202,                              /* Vector  361, coef 2(161), 3(118) */
  31391,                               /* Vector  362, coef 1(122), 2(159) */
  29047,                               /* Vector  362, coef 3(113) Vector
                                        * 363, Coef 1(119) */
  -23934,                              /* Vector  363, coef 2(162), 3(130) */
  29856,                               /* Vector  364, coef 1(116), 2(160) */
  29807,                               /* Vector  364, coef 3(116) Vector
                                        * 365, Coef 1(111) */
  -24721,                              /* Vector  365, coef 2(159), 3(111) */
  28055,                               /* Vector  366, coef 1(109), 2(151) */
  30849,                               /* Vector  366, coef 3(120) Vector
                                        * 367, Coef 1(129) */
  -23959,                              /* Vector  367, coef 2(162), 3(105) */
  -32612,                              /* Vector  368, coef 1(128), 2(156) */
  24696,                               /* Vector  368, coef 3( 96) Vector
                                        * 369, Coef 1(120) */
  -27016,                              /* Vector  369, coef 2(150), 3(120) */
  30607,                               /* Vector  370, coef 1(119), 2(143) */
  29301,                               /* Vector  370, coef 3(114) Vector
                                        * 371, Coef 1(117) */
  -26770,                              /* Vector  371, coef 2(151), 3(110) */
  29072,                               /* Vector  372, coef 1(113), 2(144) */
  27002,                               /* Vector  372, coef 3(105) Vector
                                        * 373, Coef 1(122) */
  -25242,                              /* Vector  373, coef 2(157), 3(102) */
  27536,                               /* Vector  374, coef 1(107), 2(144) */
  25980,                               /* Vector  374, coef 3(101) Vector
                                        * 375, Coef 1(124) */
  -22418,                              /* Vector  375, coef 2(168), 3(110) */
  29084,                               /* Vector  376, coef 1(113), 2(156) */
  25975,                               /* Vector  376, coef 3(101) Vector
                                        * 377, Coef 1(119) */
  -27246,                              /* Vector  377, coef 2(149), 3(146) */
  28818,                               /* Vector  378, coef 1(112), 2(146) */
  -29326,                              /* Vector  378, coef 3(141) Vector
                                        * 379, Coef 1(114) */
  -28032,                              /* Vector  379, coef 2(146), 3(128) */
  28299,                               /* Vector  380, coef 1(110), 2(139) */
  31855,                               /* Vector  380, coef 3(124) Vector
                                        * 381, Coef 1(111) */
  -32370,                              /* Vector  381, coef 2(129), 3(142) */
  -32117,                              /* Vector  382, coef 1(130), 2(139) */
  25978,                               /* Vector  382, coef 3(101) Vector
                                        * 383, Coef 1(122) */
  -29080,                              /* Vector  383, coef 2(142), 3(104) */
  27029,                               /* Vector  384, coef 1(105), 2(149) */
  29539,                               /* Vector  384, coef 3(115) Vector
                                        * 385, Coef 1( 99) */
  -29600,                              /* Vector  385, coef 2(140), 3( 96) */
  27813,                               /* Vector  386, coef 1(108), 2(165) */
  30313,                               /* Vector  386, coef 3(118) Vector
                                        * 387, Coef 1(105) */
  -23931,                              /* Vector  387, coef 2(162), 3(133) */
  25753,                               /* Vector  388, coef 1(100), 2(153) */
  32099,                               /* Vector  388, coef 3(125) Vector
                                        * 389, Coef 1( 99) */
  -21640,                              /* Vector  389, coef 2(171), 3(120) */
  24739,                               /* Vector  390, coef 1( 96), 2(163) */
  31066,                               /* Vector  390, coef 3(121) Vector
                                        * 391, Coef 1( 90) */
  -25483,                              /* Vector  391, coef 2(156), 3(117) */
  25507,                               /* Vector  392, coef 1( 99), 2(163) */
  28510,                               /* Vector  392, coef 3(111) Vector
                                        * 393, Coef 1( 94) */
  -25492,                              /* Vector  393, coef 2(156), 3(108) */
  23194,                               /* Vector  394, coef 1( 90), 2(154) */
  27240,                               /* Vector  394, coef 3(106) Vector
                                        * 395, Coef 1(104) */
  -24972,                              /* Vector  395, coef 2(158), 3(116) */
  26525,                               /* Vector  396, coef 1(103), 2(157) */
  27752,                               /* Vector  396, coef 3(108) Vector
                                        * 397, Coef 1(104) */
  -26520,                              /* Vector  397, coef 2(152), 3(104) */
  25234,                               /* Vector  398, coef 1( 98), 2(146) */
  26465,                               /* Vector  398, coef 3(103) Vector
                                        * 399, Coef 1( 97) */
  -27277,                              /* Vector  399, coef 2(149), 3(115) */
  23440,                               /* Vector  400, coef 1( 91), 2(144) */
  28507,                               /* Vector  400, coef 3(111) Vector
                                        * 401, Coef 1( 91) */
  -28569,                              /* Vector  401, coef 2(144), 3(103) */
  26529,                               /* Vector  402, coef 1(103), 2(161) */
  25695,                               /* Vector  402, coef 3(100) Vector
                                        * 403, Coef 1( 95) */
  -25503,                              /* Vector  403, coef 2(156), 3( 97) */
  25245,                               /* Vector  404, coef 1( 98), 2(157) */
  22617,                               /* Vector  404, coef 3( 88) Vector
                                        * 405, Coef 1( 89) */
  -23698,                              /* Vector  405, coef 2(163), 3(110) */
  28322,                               /* Vector  406, coef 1(110), 2(162) */
  26474,                               /* Vector  406, coef 3(103) Vector
                                        * 407, Coef 1(106) */
  -24994,                              /* Vector  407, coef 2(158), 3( 94) */
  26264,                               /* Vector  408, coef 1(102), 2(152) */
  22626,                               /* Vector  408, coef 3( 88) Vector
                                        * 409, Coef 1( 98) */
  -27810,                              /* Vector  409, coef 2(147), 3( 94) */
  28073,                               /* Vector  410, coef 1(109), 2(169) */
  25450,                               /* Vector  410, coef 3( 99) Vector
                                        * 411, Coef 1(106) */
  -22163,                              /* Vector  411, coef 2(169), 3(109) */
  26023,                               /* Vector  412, coef 1(101), 2(167) */
  25700,                               /* Vector  412, coef 3(100) Vector
                                        * 413, Coef 1(100) */
  -20887,                              /* Vector  413, coef 2(174), 3(105) */
  23959,                               /* Vector  414, coef 1( 93), 2(151) */
  32341,                               /* Vector  414, coef 3(126) Vector
                                        * 415, Coef 1( 85) */
  -28302,                              /* Vector  415, coef 2(145), 3(114) */
  21910,                               /* Vector  416, coef 1( 85), 2(150) */
  26710,                               /* Vector  416, coef 3(104) Vector
                                        * 417, Coef 1( 86) */
  -29597,                              /* Vector  417, coef 2(140), 3( 99) */
  23189,                               /* Vector  418, coef 1( 90), 2(149) */
  21591,                               /* Vector  418, coef 3( 84) Vector
                                        * 419, Coef 1( 87) */
  -27552,                              /* Vector  419, coef 2(148), 3( 96) */
  21392,                               /* Vector  420, coef 1( 83), 2(144) */
  23377,                               /* Vector  420, coef 3( 91) Vector
                                        * 421, Coef 1( 81) */
  -29612,                              /* Vector  421, coef 2(140), 3( 84) */
  21143,                               /* Vector  422, coef 1( 82), 2(151) */
  28748,                               /* Vector  422, coef 3(112) Vector
                                        * 423, Coef 1( 76) */
  -27542,                              /* Vector  423, coef 2(148), 3(106) */
  19094,                               /* Vector  424, coef 1( 74), 2(150) */
  25156,                               /* Vector  424, coef 3( 98) Vector
                                        * 425, Coef 1( 68) */
  -28322,                              /* Vector  425, coef 2(145), 3( 94) */
  19088,                               /* Vector  426, coef 1( 74), 2(144) */
  30278,                               /* Vector  426, coef 3(118) Vector
                                        * 427, Coef 1( 70) */
  -28307,                              /* Vector  427, coef 2(145), 3(109) */
  16780,                               /* Vector  428, coef 1( 65), 2(140) */
  28993,                               /* Vector  428, coef 3(113) Vector
                                        * 429, Coef 1( 65) */
  -29338,                              /* Vector  429, coef 2(141), 3(102) */
  15752,                               /* Vector  430, coef 1( 61), 2(136) */
  25164,                               /* Vector  430, coef 3( 98) Vector
                                        * 431, Coef 1( 76) */
  -28829,                              /* Vector  431, coef 2(143), 3( 99) */
  18570,                               /* Vector  432, coef 1( 72), 2(138) */
  23371,                               /* Vector  432, coef 3( 91) Vector
                                        * 433, Coef 1( 75) */
  -28073,                              /* Vector  433, coef 2(146), 3( 87) */
  16779,                               /* Vector  434, coef 1( 65), 2(139) */
  21308,                               /* Vector  434, coef 3( 83) Vector
                                        * 435, Coef 1( 60) */
  -31653,                              /* Vector  435, coef 2(132), 3( 91) */
  15492,                               /* Vector  436, coef 1( 60), 2(132) */
  21315,                               /* Vector  436, coef 3( 83) Vector
                                        * 437, Coef 1( 67) */
  -32431,                              /* Vector  437, coef 2(129), 3( 81) */
  23695,                               /* Vector  438, coef 1( 92), 2(143) */
  22614,                               /* Vector  438, coef 3( 88) Vector
                                        * 439, Coef 1( 86) */
  -30628,                              /* Vector  439, coef 2(136), 3( 92) */
  20869,                               /* Vector  440, coef 1( 81), 2(133) */
  22349,                               /* Vector  440, coef 3( 87) Vector
                                        * 441, Coef 1( 77) */
  -29841,                              /* Vector  441, coef 2(139), 3(111) */
  19846,                               /* Vector  442, coef 1( 77), 2(134) */
  25670,                               /* Vector  442, coef 3(100) Vector
                                        * 443, Coef 1( 70) */
  -31125,                              /* Vector  443, coef 2(134), 3(107) */
  17284,                               /* Vector  444, coef 1( 67), 2(132) */
  25673,                               /* Vector  444, coef 3(100) Vector
                                        * 445, Coef 1( 73) */
  -32416,                              /* Vector  445, coef 2(129), 3( 96) */
  17789,                               /* Vector  446, coef 1( 69), 2(125) */
  23353,                               /* Vector  446, coef 3( 91) Vector
                                        * 447, Coef 1( 57) */
  32090,                               /* Vector  447, coef 2(125), 3( 90) */
  31131,                               /* Vector  448, coef 1(121), 2(155) */
  23923,                               /* Vector  448, coef 3( 93) Vector
                                        * 449, Coef 1(115) */
  -26530,                              /* Vector  449, coef 2(152), 3( 94) */
  28820,                               /* Vector  450, coef 1(112), 2(148) */
  22667,                               /* Vector  450, coef 3( 88) Vector
                                        * 451, Coef 1(139) */
  -13724,                              /* Vector  451, coef 2(202), 3(100) */
  -30527,                              /* Vector  452, coef 1(136), 2(193) */
  24451,                               /* Vector  452, coef 3( 95) Vector
                                        * 453, Coef 1(131) */
  -21399,                              /* Vector  453, coef 2(172), 3(105) */
  -31816,                              /* Vector  454, coef 1(131), 2(184) */
  26240,                               /* Vector  454, coef 3(102) Vector
                                        * 455, Coef 1(128) */
  -21408,                              /* Vector  455, coef 2(172), 3( 96) */
  -30770,                              /* Vector  456, coef 1(135), 2(206) */
  23938,                               /* Vector  456, coef 3( 93) Vector
                                        * 457, Coef 1(130) */
  -16805,                              /* Vector  457, coef 2(190), 3( 91) */
  -30780,                              /* Vector  458, coef 1(135), 2(196) */
  19333,                               /* Vector  458, coef 3( 75) Vector
                                        * 459, Coef 1(133) */
  -14522,                              /* Vector  459, coef 2(199), 3( 70) */
  32705,                               /* Vector  460, coef 1(127), 2(193) */
  21879,                               /* Vector  460, coef 3( 85) Vector
                                        * 461, Coef 1(119) */
  -20397,                              /* Vector  461, coef 2(176), 3( 83) */
  31414,                               /* Vector  462, coef 1(122), 2(182) */
  18296,                               /* Vector  462, coef 3( 71) Vector
                                        * 463, Coef 1(120) */
  -23705,                              /* Vector  463, coef 2(163), 3(103) */
  31664,                               /* Vector  464, coef 1(123), 2(176) */
  24952,                               /* Vector  464, coef 3( 97) Vector
                                        * 465, Coef 1(120) */
  -22441,                              /* Vector  465, coef 2(168), 3( 87) */
  28058,                               /* Vector  466, coef 1(109), 2(154) */
  24434,                               /* Vector  466, coef 3( 95) Vector
                                        * 467, Coef 1(114) */
  -22690,                              /* Vector  467, coef 2(167), 3( 94) */
  29096,                               /* Vector  468, coef 1(113), 2(168) */
  21869,                               /* Vector  468, coef 3( 85) Vector
                                        * 469, Coef 1(109) */
  -26028,                              /* Vector  469, coef 2(154), 3( 84) */
  -32339,                              /* Vector  470, coef 1(129), 2(173) */
  19328,                               /* Vector  470, coef 3( 75) Vector
                                        * 471, Coef 1(128) */
  -17298,                              /* Vector  471, coef 2(188), 3(110) */
  31922,                               /* Vector  472, coef 1(124), 2(178) */
  28794,                               /* Vector  472, coef 3(112) Vector
                                        * 473, Coef 1(122) */
  -19094,                              /* Vector  473, coef 2(181), 3(106) */
  30384,                               /* Vector  474, coef 1(118), 2(176) */
  25473,                               /* Vector  474, coef 3( 99) Vector
                                        * 475, Coef 1(129) */
  -12961,                              /* Vector  475, coef 2(205), 3( 95) */
  -30286,                              /* Vector  476, coef 1(137), 2(178) */
  23174,                               /* Vector  476, coef 3( 90) Vector
                                        * 477, Coef 1(134) */
  -19115,                              /* Vector  477, coef 2(181), 3( 85) */
  32679,                               /* Vector  478, coef 1(127), 2(167) */
  22649,                               /* Vector  478, coef 3( 88) Vector
                                        * 479, Coef 1(121) */
  -24493,                              /* Vector  479, coef 2(160), 3( 83) */
  17023,                               /* Vector  480, coef 1( 66), 2(127) */
  17465,                               /* Vector  480, coef 3( 68) Vector
                                        * 481, Coef 1( 57) */
  30533,                               /* Vector  481, coef 2(119), 3( 69) */
  27029,                               /* Vector  482, coef 1(105), 2(149) */
  19550,                               /* Vector  482, coef 3( 76) Vector
                                        * 483, Coef 1( 94) */
  -28339,                              /* Vector  483, coef 2(145), 3( 77) */
  22152,                               /* Vector  484, coef 1( 86), 2(136) */
  19040,                               /* Vector  484, coef 3( 74) Vector
                                        * 485, Coef 1( 96) */
  -29868,                              /* Vector  485, coef 2(139), 3( 84) */
  22916,                               /* Vector  486, coef 1( 89), 2(132) */
  20550,                               /* Vector  486, coef 3( 80) Vector
                                        * 487, Coef 1( 70) */
  31821,                               /* Vector  487, coef 2(124), 3( 77) */
  25481,                               /* Vector  488, coef 1( 99), 2(137) */
  18772,                               /* Vector  488, coef 3( 73) Vector
                                        * 489, Coef 1( 84) */
  32327,                               /* Vector  489, coef 2(126), 3( 71) */
  20866,                               /* Vector  490, coef 1( 81), 2(130) */
  16727,                               /* Vector  490, coef 3( 65) Vector
                                        * 491, Coef 1( 87) */
  -31687,                              /* Vector  491, coef 2(132), 3( 57) */
  27019,                               /* Vector  492, coef 1(105), 2(139) */
  16227,                               /* Vector  492, coef 3( 63) Vector
                                        * 493, Coef 1( 99) */
  -30664,                              /* Vector  493, coef 2(136), 3( 56) */
  21370,                               /* Vector  494, coef 1( 83), 2(122) */
  16220,                               /* Vector  494, coef 3( 63) Vector
                                        * 495, Coef 1( 92) */
  32307,                               /* Vector  495, coef 2(126), 3( 51) */
  31126,                               /* Vector  496, coef 1(121), 2(150) */
  20338,                               /* Vector  496, coef 3( 79) Vector
                                        * 497, Coef 1(114) */
  -27828,                              /* Vector  497, coef 2(147), 3( 76) */
  28556,                               /* Vector  498, coef 1(111), 2(140) */
  17018,                               /* Vector  498, coef 3( 66) Vector
                                        * 499, Coef 1(122) */
  -26311,                              /* Vector  499, coef 2(153), 3( 57) */
  31896,                               /* Vector  500, coef 1(124), 2(152) */
  11387,                               /* Vector  500, coef 3( 44) Vector
                                        * 501, Coef 1(123) */
  -22971,                              /* Vector  501, coef 2(166), 3( 69) */
  28310,                               /* Vector  502, coef 1(110), 2(150) */
  18033,                               /* Vector  502, coef 3( 70) Vector
                                        * 503, Coef 1(113) */
  -26820,                              /* Vector  503, coef 2(151), 3( 60) */
  -29504,                              /* Vector  504, coef 1(140), 2(192) */
  16255,                               /* Vector  504, coef 3( 63) Vector
                                        * 505, Coef 1(127) */
  -22736,                              /* Vector  505, coef 2(167), 3( 48) */
  -29236,                              /* Vector  506, coef 1(141), 2(204) */
  14455,                               /* Vector  506, coef 3( 56) Vector
                                        * 507, Coef 1(119) */
  -23758,                              /* Vector  507, coef 2(163), 3( 50) */
  29346,                               /* Vector  508, coef 1(114), 2(162) */
  17762,                               /* Vector  508, coef 3( 69) Vector
                                        * 509, Coef 1( 98) */
  -28097,                              /* Vector  509, coef 2(146), 3( 63) */
  25490,                               /* Vector  510, coef 1( 99), 2(146) */
  13430,                               /* Vector  510, coef 3( 52) Vector
                                        * 511, Coef 1(118) */
  -21968,                              /* Vector  511, coef 2(170), 3( 48) */
  -31811,                              /* Vector  512, coef 1(131), 2(189) */
  -32385,                              /* Vector  512, coef 3(129) Vector
                                        * 513, Coef 1(127) */
  -17794,                              /* Vector  513, coef 2(186), 3(126) */
  32473,                               /* Vector  514, coef 1(126), 2(217) */
  -26494,                              /* Vector  514, coef 3(152) Vector
                                        * 515, Coef 1(130) */
  -13400,                              /* Vector  515, coef 2(203), 3(168) */
  32208,                               /* Vector  516, coef 1(125), 2(208) */
  -22914,                              /* Vector  516, coef 3(166) Vector
                                        * 517, Coef 1(126) */
  -13686,                              /* Vector  517, coef 2(202), 3(138) */
  31947,                               /* Vector  518, coef 1(124), 2(203) */
  -26754,                              /* Vector  518, coef 3(151) Vector
                                        * 519, Coef 1(126) */
  -17256,                              /* Vector  519, coef 2(188), 3(152) */
  31447,                               /* Vector  520, coef 1(122), 2(215) */
  -28297,                              /* Vector  520, coef 3(145) Vector
                                        * 521, Coef 1(119) */
  -10357,                              /* Vector  521, coef 2(215), 3(139) */
  30671,                               /* Vector  522, coef 1(119), 2(207) */
  -23178,                              /* Vector  522, coef 3(165) Vector
                                        * 523, Coef 1(118) */
  -13674,                              /* Vector  523, coef 2(202), 3(150) */
  31683,                               /* Vector  524, coef 1(123), 2(195) */
  -30857,                              /* Vector  524, coef 3(135) Vector
                                        * 525, Coef 1(119) */
  -14202,                              /* Vector  525, coef 2(200), 3(134) */
  30908,                               /* Vector  526, coef 1(120), 2(188) */
  -21898,                              /* Vector  526, coef 3(170) Vector
                                        * 527, Coef 1(118) */
  -19031,                              /* Vector  527, coef 2(181), 3(169) */
  31665,                               /* Vector  528, coef 1(123), 2(177) */
  -26243,                              /* Vector  528, coef 3(153) Vector
                                        * 529, Coef 1(125) */
  -23655,                              /* Vector  529, coef 2(163), 3(153) */
  30125,                               /* Vector  530, coef 1(117), 2(173) */
  -28288,                              /* Vector  530, coef 3(145) Vector
                                        * 531, Coef 1(128) */
  -18292,                              /* Vector  531, coef 2(184), 3(140) */
  31667,                               /* Vector  532, coef 1(123), 2(179) */
  -29830,                              /* Vector  532, coef 3(139) Vector
                                        * 533, Coef 1(122) */
  -18302,                              /* Vector  533, coef 2(184), 3(130) */
  29882,                               /* Vector  534, coef 1(116), 2(186) */
  -29568,                              /* Vector  534, coef 3(140) Vector
                                        * 535, Coef 1(128) */
  -9601,                               /* Vector  535, coef 2(218), 3(127) */
  32215,                               /* Vector  536, coef 1(125), 2(215) */
  30846,                               /* Vector  536, coef 3(120) Vector
                                        * 537, Coef 1(126) */
  -15244,                              /* Vector  537, coef 2(196), 3(116) */
  30676,                               /* Vector  538, coef 1(119), 2(212) */
  31095,                               /* Vector  538, coef 3(121) Vector
                                        * 539, Coef 1(119) */
  -9370,                               /* Vector  539, coef 2(219), 3(102) */
  29405,                               /* Vector  540, coef 1(114), 2(221) */
  -31632,                              /* Vector  540, coef 3(132) Vector
                                        * 541, Coef 1(112) */
  -10869,                              /* Vector  541, coef 2(213), 3(139) */
  29129,                               /* Vector  542, coef 1(113), 2(201) */
  -28304,                              /* Vector  542, coef 3(145) Vector
                                        * 543, Coef 1(112) */
  -15715,                              /* Vector  543, coef 2(194), 3(157) */
  30635,                               /* Vector  544, coef 1(119), 2(171) */
  30578,                               /* Vector  544, coef 3(119) Vector
                                        * 545, Coef 1(114) */
  -22917,                              /* Vector  545, coef 2(166), 3(123) */
  29364,                               /* Vector  546, coef 1(114), 2(180) */
  31600,                               /* Vector  546, coef 3(123) Vector
                                        * 547, Coef 1(112) */
  -20877,                              /* Vector  547, coef 2(174), 3(115) */
  27819,                               /* Vector  548, coef 1(108), 2(171) */
  31854,                               /* Vector  548, coef 3(124) Vector
                                        * 549, Coef 1(110) */
  -23672,                              /* Vector  549, coef 2(163), 3(136) */
  26288,                               /* Vector  550, coef 1(102), 2(176) */
  31860,                               /* Vector  550, coef 3(124) Vector
                                        * 551, Coef 1(116) */
  -22420,                              /* Vector  551, coef 2(168), 3(108) */
  29893,                               /* Vector  552, coef 1(116), 2(197) */
  32369,                               /* Vector  552, coef 3(126) Vector
                                        * 553, Coef 1(113) */
  -16253,                              /* Vector  553, coef 2(192), 3(131) */
  29350,                               /* Vector  554, coef 1(114), 2(166) */
  -28042,                              /* Vector  554, coef 3(146) Vector
                                        * 555, Coef 1(118) */
  -20095,                              /* Vector  555, coef 2(177), 3(129) */
  28854,                               /* Vector  556, coef 1(112), 2(182) */
  -29329,                              /* Vector  556, coef 3(141) Vector
                                        * 557, Coef 1(111) */
  -19324,                              /* Vector  557, coef 2(180), 3(132) */
  27571,                               /* Vector  558, coef 1(107), 2(179) */
  32379,                               /* Vector  558, coef 3(126) Vector
                                        * 559, Coef 1(123) */
  -14999,                              /* Vector  559, coef 2(197), 3(105) */
  31172,                               /* Vector  560, coef 1(121), 2(196) */
  30839,                               /* Vector  560, coef 3(120) Vector
                                        * 561, Coef 1(119) */
  -17801,                              /* Vector  561, coef 2(186), 3(119) */
  30650,                               /* Vector  562, coef 1(119), 2(186) */
  26739,                               /* Vector  562, coef 3(104) Vector
                                        * 563, Coef 1(115) */
  -18066,                              /* Vector  563, coef 2(185), 3(110) */
  28849,                               /* Vector  564, coef 1(112), 2(177) */
  26484,                               /* Vector  564, coef 3(103) Vector
                                        * 565, Coef 1(116) */
  -15510,                              /* Vector  565, coef 2(195), 3(106) */
  28861,                               /* Vector  566, coef 1(112), 2(189) */
  25200,                               /* Vector  566, coef 3( 98) Vector
                                        * 567, Coef 1(112) */
  -14226,                              /* Vector  567, coef 2(200), 3(110) */
  28342,                               /* Vector  568, coef 1(110), 2(182) */
  28009,                               /* Vector  568, coef 3(109) Vector
                                        * 569, Coef 1(105) */
  -19604,                              /* Vector  569, coef 2(179), 3(108) */
  29111,                               /* Vector  570, coef 1(113), 2(183) */
  23406,                               /* Vector  570, coef 3( 91) Vector
                                        * 571, Coef 1(110) */
  -16004,                              /* Vector  571, coef 2(193), 3(124) */
  27581,                               /* Vector  572, coef 1(107), 2(189) */
  30056,                               /* Vector  572, coef 3(117) Vector
                                        * 573, Coef 1(104) */
  -16512,                              /* Vector  573, coef 2(191), 3(128) */
  27847,                               /* Vector  574, coef 1(108), 2(199) */
  -28053,                              /* Vector  574, coef 3(146) Vector
                                        * 575, Coef 1(107) */
  -15987,                              /* Vector  575, coef 2(193), 3(141) */
  27844,                               /* Vector  576, coef 1(108), 2(196) */
  27243,                               /* Vector  576, coef 3(106) Vector
                                        * 577, Coef 1(107) */
  -16796,                              /* Vector  577, coef 2(190), 3(100) */
  29143,                               /* Vector  578, coef 1(113), 2(215) */
  29805,                               /* Vector  578, coef 3(116) Vector
                                        * 579, Coef 1(109) */
  -12174,                              /* Vector  579, coef 2(208), 3(114) */
  28127,                               /* Vector  580, coef 1(109), 2(223) */
  32106,                               /* Vector  580, coef 3(125) Vector
                                        * 581, Coef 1(106) */
  -7554,                               /* Vector  581, coef 2(226), 3(126) */
  27343,                               /* Vector  582, coef 1(106), 2(207) */
  -32666,                              /* Vector  582, coef 3(128) Vector
                                        * 583, Coef 1(102) */
  -10880,                              /* Vector  583, coef 2(213), 3(128) */
  27608,                               /* Vector  584, coef 1(107), 2(216) */
  28520,                               /* Vector  584, coef 3(111) Vector
                                        * 585, Coef 1(104) */
  -11418,                              /* Vector  585, coef 2(211), 3(102) */
  25817,                               /* Vector  586, coef 1(100), 2(217) */
  29794,                               /* Vector  586, coef 3(116) Vector
                                        * 587, Coef 1( 98) */
  -13449,                              /* Vector  587, coef 2(203), 3(119) */
  26307,                               /* Vector  588, coef 1(102), 2(195) */
  29026,                               /* Vector  588, coef 3(113) Vector
                                        * 589, Coef 1( 98) */
  -16018,                              /* Vector  589, coef 2(193), 3(110) */
  25037,                               /* Vector  590, coef 1( 97), 2(205) */
  26464,                               /* Vector  590, coef 3(103) Vector
                                        * 591, Coef 1( 96) */
  -13989,                              /* Vector  591, coef 2(201), 3( 91) */
  26042,                               /* Vector  592, coef 1(101), 2(186) */
  31076,                               /* Vector  592, coef 3(121) Vector
                                        * 593, Coef 1(100) */
  -18831,                              /* Vector  593, coef 2(182), 3(113) */
  24752,                               /* Vector  594, coef 1( 96), 2(176) */
  27997,                               /* Vector  594, coef 3(109) Vector
                                        * 595, Coef 1( 93) */
  -19848,                              /* Vector  595, coef 2(178), 3(120) */
  24516,                               /* Vector  596, coef 1( 95), 2(196) */
  30811,                               /* Vector  596, coef 3(120) Vector
                                        * 597, Coef 1( 91) */
  -16780,                              /* Vector  597, coef 2(190), 3(116) */
  23489,                               /* Vector  598, coef 1( 91), 2(193) */
  27227,                               /* Vector  598, coef 3(106) Vector
                                        * 599, Coef 1( 91) */
  -20629,                              /* Vector  599, coef 2(175), 3(107) */
  25541,                               /* Vector  600, coef 1( 99), 2(197) */
  22384,                               /* Vector  600, coef 3( 87) Vector
                                        * 601, Coef 1(112) */
  -13481,                              /* Vector  601, coef 2(203), 3( 87) */
  27338,                               /* Vector  602, coef 1(106), 2(202) */
  22632,                               /* Vector  602, coef 3( 88) Vector
                                        * 603, Coef 1(104) */
  -15513,                              /* Vector  603, coef 2(195), 3(103) */
  25529,                               /* Vector  604, coef 1( 99), 2(185) */
  24415,                               /* Vector  604, coef 3( 95) Vector
                                        * 605, Coef 1( 95) */
  -17561,                              /* Vector  605, coef 2(187), 3(103) */
  23990,                               /* Vector  606, coef 1( 93), 2(182) */
  24679,                               /* Vector  606, coef 3( 96) Vector
                                        * 607, Coef 1(103) */
  -11631,                              /* Vector  607, coef 2(210), 3(145) */
  23486,                               /* Vector  608, coef 1( 91), 2(190) */
  22363,                               /* Vector  608, coef 3( 87) Vector
                                        * 609, Coef 1( 91) */
  -18859,                              /* Vector  609, coef 2(182), 3( 85) */
  23222,                               /* Vector  610, coef 1( 90), 2(182) */
  19798,                               /* Vector  610, coef 3( 77) Vector
                                        * 611, Coef 1( 86) */
  -18849,                              /* Vector  611, coef 2(182), 3( 95) */
  20402,                               /* Vector  612, coef 1( 79), 2(178) */
  24399,                               /* Vector  612, coef 3( 95) Vector
                                        * 613, Coef 1( 79) */
  -19625,                              /* Vector  613, coef 2(179), 3( 87) */
  20145,                               /* Vector  614, coef 1( 78), 2(177) */
  20303,                               /* Vector  614, coef 3( 79) Vector
                                        * 615, Coef 1( 79) */
  -21672,                              /* Vector  615, coef 2(171), 3( 88) */
  18597,                               /* Vector  616, coef 1( 72), 2(165) */
  21831,                               /* Vector  616, coef 3( 85) Vector
                                        * 617, Coef 1( 71) */
  -21677,                              /* Vector  617, coef 2(171), 3( 83) */
  16806,                               /* Vector  618, coef 1( 65), 2(166) */
  20033,                               /* Vector  618, coef 3( 78) Vector
                                        * 619, Coef 1( 65) */
  -23983,                              /* Vector  619, coef 2(162), 3( 81) */
  16799,                               /* Vector  620, coef 1( 65), 2(159) */
  19260,                               /* Vector  620, coef 3( 75) Vector
                                        * 621, Coef 1( 60) */
  -24252,                              /* Vector  621, coef 2(161), 3( 68) */
  26822,                               /* Vector  622, coef 1(104), 2(198) */
  18537,                               /* Vector  622, coef 3( 72) Vector
                                        * 623, Coef 1(105) */
  -16066,                              /* Vector  623, coef 2(193), 3( 62) */
  25272,                               /* Vector  624, coef 1( 98), 2(184) */
  13654,                               /* Vector  624, coef 3( 53) Vector
                                        * 625, Coef 1( 86) */
  -21694,                              /* Vector  625, coef 2(171), 3( 66) */
  19876,                               /* Vector  626, coef 1( 77), 2(164) */
  13925,                               /* Vector  626, coef 3( 54) Vector
                                        * 627, Coef 1(101) */
  -12727,                              /* Vector  627, coef 2(206), 3( 73) */
  25286,                               /* Vector  628, coef 1( 98), 2(198) */
  15702,                               /* Vector  628, coef 3( 61) Vector
                                        * 629, Coef 1( 86) */
  -18368,                              /* Vector  629, coef 2(184), 3( 64) */
  21431,                               /* Vector  630, coef 1( 83), 2(183) */
  13649,                               /* Vector  630, coef 3( 53) Vector
                                        * 631, Coef 1( 81) */
  -20919,                              /* Vector  631, coef 2(174), 3( 73) */
  19110,                               /* Vector  632, coef 1( 74), 2(166) */
  17240,                               /* Vector  632, coef 3( 67) Vector
                                        * 633, Coef 1( 88) */
  -19397,                              /* Vector  633, coef 2(180), 3( 59) */
  22452,                               /* Vector  634, coef 1( 87), 2(180) */
  11853,                               /* Vector  634, coef 3( 46) Vector
                                        * 635, Coef 1( 77) */
  -23470,                              /* Vector  635, coef 2(164), 3( 82) */
  17821,                               /* Vector  636, coef 1( 69), 2(157) */
  20289,                               /* Vector  636, coef 3( 79) Vector
                                        * 637, Coef 1( 65) */
  -27070,                              /* Vector  637, coef 2(150), 3( 66) */
  19357,                               /* Vector  638, coef 1( 75), 2(157) */
  9303,                                /* Vector  638, coef 3( 36) Vector
                                        * 639, Coef 1( 87) */
  -16823,                              /* Vector  639, coef 2(190), 3( 73) */
  20395,                               /* Vector  640, coef 1( 79), 2(171) */
  25164,                               /* Vector  640, coef 3( 98) Vector
                                        * 641, Coef 1( 76) */
  -19611,                              /* Vector  641, coef 2(179), 3(101) */
  18603,                               /* Vector  642, coef 1( 72), 2(171) */
  25411,                               /* Vector  642, coef 3( 99) Vector
                                        * 643, Coef 1( 67) */
  -21667,                              /* Vector  643, coef 2(171), 3( 93) */
  20646,                               /* Vector  644, coef 1( 80), 2(166) */
  27466,                               /* Vector  644, coef 3(107) Vector
                                        * 645, Coef 1( 74) */
  -24216,                              /* Vector  645, coef 2(161), 3(104) */
  19361,                               /* Vector  646, coef 1( 75), 2(161) */
  24902,                               /* Vector  646, coef 3( 97) Vector
                                        * 647, Coef 1( 70) */
  -25252,                              /* Vector  647, coef 2(157), 3( 92) */
  18340,                               /* Vector  648, coef 1( 71), 2(164) */
  24896,                               /* Vector  648, coef 3( 97) Vector
                                        * 649, Coef 1( 64) */
  -24226,                              /* Vector  649, coef 2(161), 3( 94) */
  15771,                               /* Vector  650, coef 1( 61), 2(155) */
  24893,                               /* Vector  650, coef 3( 97) Vector
                                        * 651, Coef 1( 61) */
  -22427,                              /* Vector  651, coef 2(168), 3(101) */
  15778,                               /* Vector  652, coef 1( 61), 2(162) */
  26680,                               /* Vector  652, coef 3(104) Vector
                                        * 653, Coef 1( 56) */
  -24476,                              /* Vector  653, coef 2(160), 3(100) */
  13733,                               /* Vector  654, coef 1( 53), 2(165) */
  25906,                               /* Vector  654, coef 3(101) Vector
                                        * 655, Coef 1( 50) */
  -25499,                              /* Vector  655, coef 2(156), 3(101) */
  15266,                               /* Vector  656, coef 1( 59), 2(162) */
  23348,                               /* Vector  656, coef 3( 91) Vector
                                        * 657, Coef 1( 52) */
  -25252,                              /* Vector  657, coef 2(157), 3( 92) */
  13981,                               /* Vector  658, coef 1( 54), 2(157) */
  21566,                               /* Vector  658, coef 3( 84) Vector
                                        * 659, Coef 1( 62) */
  -27789,                              /* Vector  659, coef 2(147), 3(115) */
  14997,                               /* Vector  660, coef 1( 58), 2(149) */
  28485,                               /* Vector  660, coef 3(111) Vector
                                        * 661, Coef 1( 69) */
  -26521,                              /* Vector  661, coef 2(152), 3(103) */
  16541,                               /* Vector  662, coef 1( 64), 2(157) */
  22337,                               /* Vector  662, coef 3( 87) Vector
                                        * 663, Coef 1( 65) */
  -26250,                              /* Vector  663, coef 2(153), 3(118) */
  17820,                               /* Vector  664, coef 1( 69), 2(156) */
  28482,                               /* Vector  664, coef 3(111) Vector
                                        * 665, Coef 1( 66) */
  -24211,                              /* Vector  665, coef 2(161), 3(109) */
  16284,                               /* Vector  666, coef 1( 63), 2(156) */
  26940,                               /* Vector  666, coef 3(105) Vector
                                        * 667, Coef 1( 60) */
  -24205,                              /* Vector  667, coef 2(161), 3(115) */
  14239,                               /* Vector  668, coef 1( 55), 2(159) */
  28984,                               /* Vector  668, coef 3(113) Vector
                                        * 669, Coef 1( 56) */
  -25747,                              /* Vector  669, coef 2(155), 3(109) */
  12697,                               /* Vector  670, coef 1( 49), 2(153) */
  28226,                               /* Vector  670, coef 3(110) Vector
                                        * 671, Coef 1( 66) */
  -22669,                              /* Vector  671, coef 2(167), 3(115) */
  13704,                               /* Vector  672, coef 1( 53), 2(136) */
  25391,                               /* Vector  672, coef 3( 99) Vector
                                        * 673, Coef 1( 47) */
  -31650,                              /* Vector  673, coef 2(132), 3( 94) */
  12160,                               /* Vector  674, coef 1( 47), 2(128) */
  22573,                               /* Vector  674, coef 3( 88) Vector
                                        * 675, Coef 1( 45) */
  32082,                               /* Vector  675, coef 2(125), 3( 82) */
  12950,                               /* Vector  676, coef 1( 50), 2(150) */
  18229,                               /* Vector  676, coef 3( 71) Vector
                                        * 677, Coef 1( 53) */
  -26530,                              /* Vector  677, coef 2(152), 3( 94) */
  11929,                               /* Vector  678, coef 1( 46), 2(153) */
  20777,                               /* Vector  678, coef 3( 81) Vector
                                        * 679, Coef 1( 41) */
  -26531,                              /* Vector  679, coef 2(152), 3( 93) */
  10900,                               /* Vector  680, coef 1( 42), 2(148) */
  22559,                               /* Vector  680, coef 3( 88) Vector
                                        * 681, Coef 1( 31) */
  -28589,                              /* Vector  681, coef 2(144), 3( 83) */
  14991,                               /* Vector  682, coef 1( 58), 2(143) */
  27442,                               /* Vector  682, coef 3(107) Vector
                                        * 683, Coef 1( 50) */
  -29848,                              /* Vector  683, coef 2(139), 3(104) */
  13974,                               /* Vector  684, coef 1( 54), 2(150) */
  26670,                               /* Vector  684, coef 3(104) Vector
                                        * 685, Coef 1( 46) */
  -27547,                              /* Vector  685, coef 2(148), 3(101) */
  10382,                               /* Vector  686, coef 1( 40), 2(142) */
  26942,                               /* Vector  686, coef 3(105) Vector
                                        * 687, Coef 1( 62) */
  -27546,                              /* Vector  687, coef 2(148), 3(102) */
  16277,                               /* Vector  688, coef 1( 63), 2(149) */
  24380,                               /* Vector  688, coef 3( 95) Vector
                                        * 689, Coef 1( 60) */
  -27560,                              /* Vector  689, coef 2(148), 3( 88) */
  13719,                               /* Vector  690, coef 1( 53), 2(151) */
  21564,                               /* Vector  690, coef 3( 84) Vector
                                        * 691, Coef 1( 60) */
  -27314,                              /* Vector  691, coef 2(149), 3( 78) */
  12176,                               /* Vector  692, coef 1( 47), 2(144) */
  20274,                               /* Vector  692, coef 3( 79) Vector
                                        * 693, Coef 1( 50) */
  -28832,                              /* Vector  693, coef 2(143), 3( 96) */
  10637,                               /* Vector  694, coef 1( 41), 2(141) */
  23336,                               /* Vector  694, coef 3( 91) Vector
                                        * 695, Coef 1( 40) */
  -30620,                              /* Vector  695, coef 2(136), 3(100) */
  9092,                                /* Vector  696, coef 1( 35), 2(132) */
  23612,                               /* Vector  696, coef 3( 92) Vector
                                        * 697, Coef 1( 60) */
  -28840,                              /* Vector  697, coef 2(143), 3( 88) */
  13193,                               /* Vector  698, coef 1( 51), 2(137) */
  22575,                               /* Vector  698, coef 3( 88) Vector
                                        * 699, Coef 1( 47) */
  -29615,                              /* Vector  699, coef 2(140), 3( 81) */
  12425,                               /* Vector  700, coef 1( 48), 2(137) */
  18241,                               /* Vector  700, coef 3( 71) Vector
                                        * 701, Coef 1( 65) */
  -28085,                              /* Vector  701, coef 2(146), 3( 75) */
  9620,                                /* Vector  702, coef 1( 37), 2(148) */
  25888,                               /* Vector  702, coef 3(101) Vector
                                        * 703, Coef 1( 32) */
  -29084,                              /* Vector  703, coef 2(142), 3(100) */
  21665,                               /* Vector  704, coef 1( 84), 2(161) */
  27983,                               /* Vector  704, coef 3(109) Vector
                                        * 705, Coef 1( 79) */
  -25753,                              /* Vector  705, coef 2(155), 3(103) */
  20371,                               /* Vector  706, coef 1( 79), 2(147) */
  20585,                               /* Vector  706, coef 3( 80) Vector
                                        * 707, Coef 1(105) */
  -18849,                              /* Vector  707, coef 2(182), 3( 95) */
  26542,                               /* Vector  708, coef 1(103), 2(174) */
  23399,                               /* Vector  708, coef 3( 91) Vector
                                        * 709, Coef 1(103) */
  -21419,                              /* Vector  709, coef 2(172), 3( 85) */
  22191,                               /* Vector  710, coef 1( 86), 2(175) */
  25194,                               /* Vector  710, coef 3( 98) Vector
                                        * 711, Coef 1(106) */
  -16561,                              /* Vector  711, coef 2(191), 3( 79) */
  23973,                               /* Vector  712, coef 1( 93), 2(165) */
  26209,                               /* Vector  712, coef 3(102) Vector
                                        * 713, Coef 1( 97) */
  -21666,                              /* Vector  713, coef 2(171), 3( 94) */
  23204,                               /* Vector  714, coef 1( 90), 2(164) */
  22619,                               /* Vector  714, coef 3( 88) Vector
                                        * 715, Coef 1( 91) */
  -21151,                              /* Vector  715, coef 2(173), 3( 97) */
  21925,                               /* Vector  716, coef 1( 85), 2(165) */
  24917,                               /* Vector  716, coef 3( 97) Vector
                                        * 717, Coef 1( 85) */
  -21670,                              /* Vector  717, coef 2(171), 3( 90) */
  21156,                               /* Vector  718, coef 1( 82), 2(164) */
  22357,                               /* Vector  718, coef 3( 87) Vector
                                        * 719, Coef 1( 85) */
  -25503,                              /* Vector  719, coef 2(156), 3( 97) */
  21400,                               /* Vector  720, coef 1( 83), 2(152) */
  23628,                               /* Vector  720, coef 3( 92) Vector
                                        * 721, Coef 1( 76) */
  -25509,                              /* Vector  721, coef 2(156), 3( 91) */
  18325,                               /* Vector  722, coef 1( 71), 2(149) */
  21855,                               /* Vector  722, coef 3( 85) Vector
                                        * 723, Coef 1( 95) */
  -20394,                              /* Vector  723, coef 2(176), 3( 86) */
  23214,                               /* Vector  724, coef 1( 90), 2(174) */
  20566,                               /* Vector  724, coef 3( 80) Vector
                                        * 725, Coef 1( 86) */
  -23472,                              /* Vector  725, coef 2(164), 3( 80) */
  21154,                               /* Vector  726, coef 1( 82), 2(162) */
  18250,                               /* Vector  726, coef 3( 71) Vector
                                        * 727, Coef 1( 74) */
  -25014,                              /* Vector  727, coef 2(158), 3( 74) */
  19096,                               /* Vector  728, coef 1( 74), 2(152) */
  20063,                               /* Vector  728, coef 3( 78) Vector
                                        * 729, Coef 1( 95) */
  -21696,                              /* Vector  729, coef 2(171), 3( 64) */
  22951,                               /* Vector  730, coef 1( 89), 2(167) */
  13416,                               /* Vector  730, coef 3( 52) Vector
                                        * 731, Coef 1(104) */
  -19376,                              /* Vector  731, coef 2(180), 3( 80) */
  26031,                               /* Vector  732, coef 1(101), 2(175) */
  18016,                               /* Vector  732, coef 3( 70) Vector
                                        * 733, Coef 1( 96) */
  -23469,                              /* Vector  733, coef 2(164), 3( 83) */
  22427,                               /* Vector  734, coef 1( 87), 2(155) */
  21846,                               /* Vector  734, coef 3( 85) Vector
                                        * 735, Coef 1( 86) */
  -25010,                              /* Vector  735, coef 2(158), 3( 78) */
  20618,                               /* Vector  736, coef 1( 80), 2(138) */
  19531,                               /* Vector  736, coef 3( 76) Vector
                                        * 737, Coef 1( 75) */
  -29111,                              /* Vector  737, coef 2(142), 3( 73) */
  16263,                               /* Vector  738, coef 1( 63), 2(135) */
  19262,                               /* Vector  738, coef 3( 75) Vector
                                        * 739, Coef 1( 62) */
  -31168,                              /* Vector  739, coef 2(134), 3( 64) */
  16255,                               /* Vector  740, coef 1( 63), 2(127) */
  19316,                               /* Vector  740, coef 3( 75) Vector
                                        * 741, Coef 1(116) */
  -20925,                              /* Vector  741, coef 2(174), 3( 67) */
  27561,                               /* Vector  742, coef 1(107), 2(169) */
  22598,                               /* Vector  742, coef 3( 88) Vector
                                        * 743, Coef 1( 70) */
  -28348,                              /* Vector  743, coef 2(145), 3( 68) */
  16012,                               /* Vector  744, coef 1( 62), 2(140) */
  15188,                               /* Vector  744, coef 3( 59) Vector
                                        * 745, Coef 1( 84) */
  -24771,                              /* Vector  745, coef 2(159), 3( 61) */
  31683,                               /* Vector  746, coef 1(123), 2(195) */
  21876,                               /* Vector  746, coef 3( 85) Vector
                                        * 747, Coef 1(116) */
  -18865,                              /* Vector  747, coef 2(182), 3( 79) */
  29364,                               /* Vector  748, coef 1(114), 2(180) */
  18294,                               /* Vector  748, coef 3( 71) Vector
                                        * 749, Coef 1(118) */
  -13998,                              /* Vector  749, coef 2(201), 3( 82) */
  29629,                               /* Vector  750, coef 1(115), 2(189) */
  18004,                               /* Vector  750, coef 3( 70) Vector
                                        * 751, Coef 1( 84) */
  -26040,                              /* Vector  751, coef 2(154), 3( 72) */
  25252,                               /* Vector  752, coef 1( 98), 2(164) */
  17491,                               /* Vector  752, coef 3( 68) Vector
                                        * 753, Coef 1( 83) */
  -26818,                              /* Vector  753, coef 2(151), 3( 62) */
  26022,                               /* Vector  754, coef 1(101), 2(166) */
  14432,                               /* Vector  754, coef 3( 56) Vector
                                        * 755, Coef 1( 96) */
  -23246,                              /* Vector  755, coef 2(165), 3( 50) */
  26528,                               /* Vector  756, coef 1(103), 2(160) */
  21095,                               /* Vector  756, coef 3( 82) Vector
                                        * 757, Coef 1(103) */
  -25781,                              /* Vector  757, coef 2(155), 3( 75) */
  26270,                               /* Vector  758, coef 1(102), 2(158) */
  16743,                               /* Vector  758, coef 3( 65) Vector
                                        * 759, Coef 1(103) */
  -25543,                              /* Vector  759, coef 2(156), 3( 57) */
  24219,                               /* Vector  760, coef 1( 94), 2(155) */
  19543,                               /* Vector  760, coef 3( 76) Vector
                                        * 761, Coef 1( 87) */
  -27836,                              /* Vector  761, coef 2(147), 3( 68) */
  20875,                               /* Vector  762, coef 1( 81), 2(139) */
  16718,                               /* Vector  762, coef 3( 65) Vector
                                        * 763, Coef 1( 78) */
  -30154,                              /* Vector  763, coef 2(138), 3( 54) */
  30390,                               /* Vector  764, coef 1(118), 2(182) */
  14443,                               /* Vector  764, coef 3( 56) Vector
                                        * 765, Coef 1(107) */
  -23761,                              /* Vector  765, coef 2(163), 3( 47) */
  21651,                               /* Vector  766, coef 1( 84), 2(147) */
  13423,                               /* Vector  766, coef 3( 52) Vector
                                        * 767, Coef 1(111) */
  -20949,                              /* Vector  767, coef 2(174), 3( 43) */
  18842,                               /* Vector  768, coef 1( 73), 2(154) */
  -24767,                              /* Vector  768, coef 3(159) Vector
                                        * 769, Coef 1( 65) */
  -24168,                              /* Vector  769, coef 2(161), 3(152) */
  18065,                               /* Vector  770, coef 1( 70), 2(145) */
  -30909,                              /* Vector  770, coef 3(135) Vector
                                        * 771, Coef 1( 67) */
  -27738,                              /* Vector  771, coef 2(147), 3(166) */
  15262,                               /* Vector  772, coef 1( 59), 2(158) */
  -24002,                              /* Vector  772, coef 3(162) Vector
                                        * 773, Coef 1( 62) */
  -30549,                              /* Vector  773, coef 2(136), 3(171) */
  15494,                               /* Vector  774, coef 1( 60), 2(134) */
  -24513,                              /* Vector  774, coef 3(160) Vector
                                        * 775, Coef 1( 63) */
  -26988,                              /* Vector  775, coef 2(150), 3(148) */
  15247,                               /* Vector  776, coef 1( 59), 2(143) */
  -27850,                              /* Vector  776, coef 3(147) Vector
                                        * 777, Coef 1( 54) */
  -26474,                              /* Vector  777, coef 2(152), 3(150) */
  13459,                               /* Vector  778, coef 1( 52), 2(147) */
  -29375,                              /* Vector  778, coef 3(141) Vector
                                        * 779, Coef 1( 65) */
  -26235,                              /* Vector  779, coef 2(153), 3(133) */
  15257,                               /* Vector  780, coef 1( 59), 2(153) */
  -30153,                              /* Vector  780, coef 3(138) Vector
                                        * 781, Coef 1( 55) */
  -26747,                              /* Vector  781, coef 2(151), 3(133) */
  17042,                               /* Vector  782, coef 1( 66), 2(146) */
  32572,                               /* Vector  782, coef 3(127) Vector
                                        * 783, Coef 1( 60) */
  -28292,                              /* Vector  783, coef 2(145), 3(124) */
  15497,                               /* Vector  784, coef 1( 60), 2(137) */
  31799,                               /* Vector  784, coef 3(124) Vector
                                        * 785, Coef 1( 55) */
  -30089,                              /* Vector  785, coef 2(138), 3(119) */
  15755,                               /* Vector  786, coef 1( 61), 2(139) */
  -29896,                              /* Vector  786, coef 3(139) Vector
                                        * 787, Coef 1( 56) */
  -30074,                              /* Vector  787, coef 2(138), 3(134) */
  13197,                               /* Vector  788, coef 1( 51), 2(141) */
  -31944,                              /* Vector  788, coef 3(131) Vector
                                        * 789, Coef 1( 56) */
  -24449,                              /* Vector  789, coef 2(160), 3(127) */
  13978,                               /* Vector  790, coef 1( 54), 2(154) */
  31537,                               /* Vector  790, coef 3(123) Vector
                                        * 791, Coef 1( 49) */
  -25723,                              /* Vector  791, coef 2(155), 3(133) */
  12170,                               /* Vector  792, coef 1( 47), 2(138) */
  -27319,                              /* Vector  792, coef 3(149) Vector
                                        * 793, Coef 1( 73) */
  -30326,                              /* Vector  793, coef 2(137), 3(138) */
  17031,                               /* Vector  794, coef 1( 66), 2(135) */
  -32709,                              /* Vector  794, coef 3(128) Vector
                                        * 795, Coef 1( 59) */
  32645,                               /* Vector  795, coef 2(127), 3(133) */
  15772,                               /* Vector  796, coef 1( 61), 2(156) */
  32055,                               /* Vector  796, coef 3(125) Vector
                                        * 797, Coef 1( 55) */
  -24178,                              /* Vector  797, coef 2(161), 3(142) */
  12444,                               /* Vector  798, coef 1( 48), 2(156) */
  -28114,                              /* Vector  798, coef 3(146) Vector
                                        * 799, Coef 1( 46) */
  -27250,                              /* Vector  799, coef 2(149), 3(142) */
  11664,                               /* Vector  800, coef 1( 45), 2(144) */
  28200,                               /* Vector  800, coef 3(110) Vector
                                        * 801, Coef 1( 40) */
  -30867,                              /* Vector  801, coef 2(135), 3(109) */
  11659,                               /* Vector  802, coef 1( 45), 2(139) */
  -32203,                              /* Vector  802, coef 3(130) Vector
                                        * 803, Coef 1( 53) */
  -28038,                              /* Vector  803, coef 2(146), 3(122) */
  12687,                               /* Vector  804, coef 1( 49), 2(143) */
  29998,                               /* Vector  804, coef 3(117) Vector
                                        * 805, Coef 1( 46) */
  -30598,                              /* Vector  805, coef 2(136), 3(122) */
  10884,                               /* Vector  806, coef 1( 42), 2(132) */
  30253,                               /* Vector  806, coef 3(118) Vector
                                        * 807, Coef 1( 45) */
  -26495,                              /* Vector  807, coef 2(152), 3(129) */
  11929,                               /* Vector  808, coef 1( 46), 2(153) */
  30501,                               /* Vector  808, coef 3(119) Vector
                                        * 809, Coef 1( 37) */
  -26251,                              /* Vector  809, coef 2(153), 3(117) */
  10399,                               /* Vector  810, coef 1( 40), 2(159) */
  32290,                               /* Vector  810, coef 3(126) Vector
                                        * 811, Coef 1( 34) */
  -25730,                              /* Vector  811, coef 2(155), 3(126) */
  7323,                                /* Vector  812, coef 1( 28), 2(155) */
  32027,                               /* Vector  812, coef 3(125) Vector
                                        * 813, Coef 1( 27) */
  -25737,                              /* Vector  813, coef 2(155), 3(119) */
  5525,                                /* Vector  814, coef 1( 21), 2(149) */
  30503,                               /* Vector  814, coef 3(119) Vector
                                        * 815, Coef 1( 39) */
  -30327,                              /* Vector  815, coef 2(137), 3(137) */
  9358,                                /* Vector  816, coef 1( 36), 2(142) */
  -31969,                              /* Vector  816, coef 3(131) Vector
                                        * 817, Coef 1( 31) */
  -30586,                              /* Vector  817, coef 2(136), 3(134) */
  6032,                                /* Vector  818, coef 1( 23), 2(144) */
  -31211,                              /* Vector  818, coef 3(134) Vector
                                        * 819, Coef 1( 21) */
  -31357,                              /* Vector  819, coef 2(133), 3(131) */
  10643,                               /* Vector  820, coef 1( 41), 2(147) */
  31782,                               /* Vector  820, coef 3(124) Vector
                                        * 821, Coef 1( 38) */
  -28300,                              /* Vector  821, coef 2(145), 3(116) */
  8594,                                /* Vector  822, coef 1( 33), 2(146) */
  31515,                               /* Vector  822, coef 3(123) Vector
                                        * 823, Coef 1( 27) */
  -28806,                              /* Vector  823, coef 2(143), 3(122) */
  9352,                                /* Vector  824, coef 1( 36), 2(136) */
  31004,                               /* Vector  824, coef 3(121) Vector
                                        * 825, Coef 1( 28) */
  -32135,                              /* Vector  825, coef 2(130), 3(121) */
  8587,                                /* Vector  826, coef 1( 33), 2(139) */
  28955,                               /* Vector  826, coef 3(113) Vector
                                        * 827, Coef 1( 27) */
  -29841,                              /* Vector  827, coef 2(139), 3(111) */
  9622,                                /* Vector  828, coef 1( 37), 2(150) */
  27930,                               /* Vector  828, coef 3(109) Vector
                                        * 829, Coef 1( 26) */
  -26001,                              /* Vector  829, coef 2(154), 3(111) */
  7571,                                /* Vector  830, coef 1( 29), 2(147) */
  28185,                               /* Vector  830, coef 3(110) Vector
                                        * 831, Coef 1( 25) */
  -27289,                              /* Vector  831, coef 2(149), 3(103) */
  30040,                               /* Vector  832, coef 1(117), 2( 88) */
  -25811,                              /* Vector  832, coef 3(155) Vector
                                        * 833, Coef 1( 45) */
  -30325,                              /* Vector  833, coef 2(137), 3(139) */
  13704,                               /* Vector  834, coef 1( 53), 2(136) */
  -25802,                              /* Vector  834, coef 3(155) Vector
                                        * 835, Coef 1( 54) */
  -32624,                              /* Vector  835, coef 2(128), 3(144) */
  11898,                               /* Vector  836, coef 1( 46), 2(122) */
  -29142,                              /* Vector  836, coef 3(142) Vector
                                        * 837, Coef 1( 42) */
  -32105,                              /* Vector  837, coef 2(130), 3(151) */
  8830,                                /* Vector  838, coef 1( 34), 2(126) */
  -28640,                              /* Vector  838, coef 3(144) Vector
                                        * 839, Coef 1( 32) */
  31642,                               /* Vector  839, coef 2(123), 3(154) */
  6017,                                /* Vector  840, coef 1( 23), 2(129) */
  -28122,                              /* Vector  840, coef 3(146) Vector
                                        * 841, Coef 1( 38) */
  28819,                               /* Vector  841, coef 2(112), 3(147) */
  8553,                                /* Vector  842, coef 1( 33), 2(105) */
  -29403,                              /* Vector  842, coef 3(141) Vector
                                        * 843, Coef 1( 37) */
  29575,                               /* Vector  843, coef 2(115), 3(135) */
  7031,                                /* Vector  844, coef 1( 27), 2(119) */
  -31161,                              /* Vector  844, coef 3(134) Vector
                                        * 845, Coef 1( 71) */
  30613,                               /* Vector  845, coef 2(119), 3(149) */
  16753,                               /* Vector  846, coef 1( 65), 2(113) */
  -29896,                              /* Vector  846, coef 3(139) Vector
                                        * 847, Coef 1( 56) */
  29330,                               /* Vector  847, coef 2(114), 3(146) */
  12651,                               /* Vector  848, coef 1( 49), 2(107) */
  -29655,                              /* Vector  848, coef 3(140) Vector
                                        * 849, Coef 1( 41) */
  27520,                               /* Vector  849, coef 2(107), 3(128) */
  12131,                               /* Vector  850, coef 1( 47), 2( 99) */
  -31960,                              /* Vector  850, coef 3(131) Vector
                                        * 851, Coef 1( 40) */
  23935,                               /* Vector  851, coef 2( 93), 3(127) */
  16482,                               /* Vector  852, coef 1( 64), 2( 98) */
  -23246,                              /* Vector  852, coef 3(165) Vector
                                        * 853, Coef 1( 50) */
  25238,                               /* Vector  853, coef 2( 98), 3(150) */
  11868,                               /* Vector  854, coef 1( 46), 2( 92) */
  -29657,                              /* Vector  854, coef 3(140) Vector
                                        * 855, Coef 1( 39) */
  21392,                               /* Vector  855, coef 2( 83), 3(144) */
  15943,                               /* Vector  856, coef 1( 62), 2( 71) */
  -30403,                              /* Vector  856, coef 3(137) Vector
                                        * 857, Coef 1( 61) */
  13715,                               /* Vector  857, coef 2( 53), 3(147) */
  10111,                               /* Vector  858, coef 1( 39), 2(127) */
  -30940,                              /* Vector  858, coef 3(135) Vector
                                        * 859, Coef 1( 36) */
  31872,                               /* Vector  859, coef 2(124), 3(128) */
  10368,                               /* Vector  860, coef 1( 40), 2(128) */
  -24027,                              /* Vector  860, coef 3(162) Vector
                                        * 861, Coef 1( 37) */
  28322,                               /* Vector  861, coef 2(110), 3(162) */
  7786,                                /* Vector  862, coef 1( 30), 2(106) */
  -26076,                              /* Vector  862, coef 3(154) Vector
                                        * 863, Coef 1( 36) */
  22435,                               /* Vector  863, coef 2( 87), 3(163) */
  16750,                               /* Vector  864, coef 1( 65), 2(110) */
  27965,                               /* Vector  864, coef 3(109) Vector
                                        * 865, Coef 1( 61) */
  -31122,                              /* Vector  865, coef 2(134), 3(110) */
  13449,                               /* Vector  866, coef 1( 52), 2(137) */
  28977,                               /* Vector  866, coef 3(113) Vector
                                        * 867, Coef 1( 49) */
  -31891,                              /* Vector  867, coef 2(131), 3(109) */
  8576,                                /* Vector  868, coef 1( 33), 2(128) */
  27437,                               /* Vector  868, coef 3(107) Vector
                                        * 869, Coef 1( 45) */
  32616,                               /* Vector  869, coef 2(127), 3(104) */
  11133,                               /* Vector  870, coef 1( 43), 2(125) */
  24355,                               /* Vector  870, coef 3( 95) Vector
                                        * 871, Coef 1( 35) */
  30310,                               /* Vector  871, coef 2(118), 3(102) */
  9846,                                /* Vector  872, coef 1( 38), 2(118) */
  23608,                               /* Vector  872, coef 3( 92) Vector
                                        * 873, Coef 1( 56) */
  27263,                               /* Vector  873, coef 2(106), 3(127) */
  15486,                               /* Vector  874, coef 1( 60), 2(126) */
  31288,                               /* Vector  874, coef 3(122) Vector
                                        * 875, Coef 1( 56) */
  31861,                               /* Vector  875, coef 2(124), 3(117) */
  13440,                               /* Vector  876, coef 1( 52), 2(128) */
  -32721,                              /* Vector  876, coef 3(128) Vector
                                        * 877, Coef 1( 47) */
  31362,                               /* Vector  877, coef 2(122), 3(130) */
  11123,                               /* Vector  878, coef 1( 43), 2(115) */
  32294,                               /* Vector  878, coef 3(126) Vector
                                        * 879, Coef 1( 38) */
  31605,                               /* Vector  879, coef 2(123), 3(117) */
  8306,                                /* Vector  880, coef 1( 32), 2(114) */
  29759,                               /* Vector  880, coef 3(116) Vector
                                        * 881, Coef 1( 63) */
  29820,                               /* Vector  881, coef 2(116), 3(124) */
  14197,                               /* Vector  882, coef 1( 55), 2(117) */
  31023,                               /* Vector  882, coef 3(121) Vector
                                        * 883, Coef 1( 47) */
  31349,                               /* Vector  883, coef 2(122), 3(117) */
  12663,                               /* Vector  884, coef 1( 49), 2(119) */
  27957,                               /* Vector  884, coef 3(109) Vector
                                        * 885, Coef 1( 53) */
  28020,                               /* Vector  885, coef 2(109), 3(116) */
  11372,                               /* Vector  886, coef 1( 44), 2(108) */
  29239,                               /* Vector  886, coef 3(114) Vector
                                        * 887, Coef 1( 55) */
  25717,                               /* Vector  887, coef 2(100), 3(117) */
  10083,                               /* Vector  888, coef 1( 39), 2( 99) */
  29500,                               /* Vector  888, coef 3(115) Vector
                                        * 889, Coef 1( 60) */
  30061,                               /* Vector  889, coef 2(117), 3(109) */
  13940,                               /* Vector  890, coef 1( 54), 2(116) */
  26417,                               /* Vector  890, coef 3(103) Vector
                                        * 891, Coef 1( 49) */
  27752,                               /* Vector  891, coef 2(108), 3(104) */
  10090,                               /* Vector  892, coef 1( 39), 2(106) */
  25402,                               /* Vector  892, coef 3( 99) Vector
                                        * 893, Coef 1( 58) */
  32618,                               /* Vector  893, coef 2(127), 3(106) */
  14204,                               /* Vector  894, coef 1( 55), 2(124) */
  25411,                               /* Vector  894, coef 3( 99) Vector
                                        * 895, Coef 1( 67) */
  31853,                               /* Vector  895, coef 2(124), 3(109) */
  21613,                               /* Vector  896, coef 1( 84), 2(109) */
  -31144,                              /* Vector  896, coef 3(134) Vector
                                        * 897, Coef 1( 88) */
  28024,                               /* Vector  897, coef 2(109), 3(120) */
  20343,                               /* Vector  898, coef 1( 79), 2(119) */
  30278,                               /* Vector  898, coef 3(118) Vector
                                        * 899, Coef 1( 70) */
  30321,                               /* Vector  899, coef 2(118), 3(113) */
  17773,                               /* Vector  900, coef 1( 69), 2(109) */
  29804,                               /* Vector  900, coef 3(116) Vector
                                        * 901, Coef 1(108) */
  31340,                               /* Vector  901, coef 2(122), 3(108) */
  19579,                               /* Vector  902, coef 1( 76), 2(123) */
  -31932,                              /* Vector  902, coef 3(131) Vector
                                        * 903, Coef 1( 68) */
  31617,                               /* Vector  903, coef 2(123), 3(129) */
  18823,                               /* Vector  904, coef 1( 73), 2(135) */
  30787,                               /* Vector  904, coef 3(120) Vector
                                        * 905, Coef 1( 67) */
  -32140,                              /* Vector  905, coef 2(130), 3(116) */
  25981,                               /* Vector  906, coef 1(101), 2(125) */
  -30359,                              /* Vector  906, coef 3(137) Vector
                                        * 907, Coef 1(105) */
  32380,                               /* Vector  907, coef 2(126), 3(124) */
  23933,                               /* Vector  908, coef 1( 93), 2(125) */
  31073,                               /* Vector  908, coef 3(121) Vector
                                        * 909, Coef 1( 97) */
  -28793,                              /* Vector  909, coef 2(143), 3(135) */
  25228,                               /* Vector  910, coef 1( 98), 2(140) */
  31833,                               /* Vector  910, coef 3(124) Vector
                                        * 911, Coef 1( 89) */
  -30087,                              /* Vector  911, coef 2(138), 3(121) */
  21648,                               /* Vector  912, coef 1( 84), 2(144) */
  32335,                               /* Vector  912, coef 3(126) Vector
                                        * 913, Coef 1( 79) */
  -30080,                              /* Vector  913, coef 2(138), 3(128) */
  21635,                               /* Vector  914, coef 1( 84), 2(131) */
  32588,                               /* Vector  914, coef 3(127) Vector
                                        * 915, Coef 1( 76) */
  32631,                               /* Vector  915, coef 2(127), 3(119) */
  20872,                               /* Vector  916, coef 1( 81), 2(136) */
  27468,                               /* Vector  916, coef 3(107) Vector
                                        * 917, Coef 1( 76) */
  -32658,                              /* Vector  917, coef 2(128), 3(110) */
  19836,                               /* Vector  918, coef 1( 77), 2(124) */
  26728,                               /* Vector  918, coef 3(104) Vector
                                        * 919, Coef 1(104) */
  -29580,                              /* Vector  919, coef 2(140), 3(116) */
  25740,                               /* Vector  920, coef 1(100), 2(140) */
  27752,                               /* Vector  920, coef 3(108) Vector
                                        * 921, Coef 1(104) */
  -31636,                              /* Vector  921, coef 2(132), 3(108) */
  23686,                               /* Vector  922, coef 1( 92), 2(134) */
  28759,                               /* Vector  922, coef 3(112) Vector
                                        * 923, Coef 1( 87) */
  -32659,                              /* Vector  923, coef 2(128), 3(109) */
  22661,                               /* Vector  924, coef 1( 88), 2(133) */
  25693,                               /* Vector  924, coef 3(100) Vector
                                        * 925, Coef 1( 93) */
  30062,                               /* Vector  925, coef 2(117), 3(110) */
  21620,                               /* Vector  926, coef 1( 84), 2(116) */
  27475,                               /* Vector  926, coef 3(107) Vector
                                        * 927, Coef 1( 83) */
  -29807,                              /* Vector  927, coef 2(139), 3(145) */
  18543,                               /* Vector  928, coef 1( 72), 2(111) */
  26676,                               /* Vector  928, coef 3(104) Vector
                                        * 929, Coef 1( 52) */
  25953,                               /* Vector  929, coef 2(101), 3( 97) */
  25967,                               /* Vector  930, coef 1(101), 2(111) */
  23656,                               /* Vector  930, coef 3( 92) Vector
                                        * 931, Coef 1(104) */
  -29606,                              /* Vector  931, coef 2(140), 3( 90) */
  26505,                               /* Vector  932, coef 1(103), 2(137) */
  21581,                               /* Vector  932, coef 3( 84) Vector
                                        * 933, Coef 1( 77) */
  32596,                               /* Vector  933, coef 2(127), 3( 84) */
  16756,                               /* Vector  934, coef 1( 65), 2(116) */
  24122,                               /* Vector  934, coef 3( 94) Vector
                                        * 935, Coef 1( 58) */
  28762,                               /* Vector  935, coef 2(112), 3( 90) */
  11631,                               /* Vector  936, coef 1( 45), 2(111) */
  21046,                               /* Vector  936, coef 3( 82) Vector
                                        * 937, Coef 1( 54) */
  26451,                               /* Vector  937, coef 2(103), 3( 83) */
  18297,                               /* Vector  938, coef 1( 71), 2(121) */
  25141,                               /* Vector  938, coef 3( 98) Vector
                                        * 939, Coef 1( 53) */
  30294,                               /* Vector  939, coef 2(118), 3( 86) */
  25726,                               /* Vector  940, coef 1(100), 2(126) */
  26194,                               /* Vector  940, coef 3(102) Vector
                                        * 941, Coef 1( 82) */
  32606,                               /* Vector  941, coef 2(127), 3( 94) */
  25476,                               /* Vector  942, coef 1( 99), 2(132) */
  24413,                               /* Vector  942, coef 3( 95) Vector
                                        * 943, Coef 1( 93) */
  31837,                               /* Vector  943, coef 2(124), 3( 93) */
  20855,                               /* Vector  944, coef 1( 81), 2(119) */
  25167,                               /* Vector  944, coef 3( 98) Vector
                                        * 945, Coef 1( 79) */
  29529,                               /* Vector  945, coef 2(115), 3( 89) */
  22654,                               /* Vector  946, coef 1( 88), 2(126) */
  22618,                               /* Vector  946, coef 3( 88) Vector
                                        * 947, Coef 1( 90) */
  31824,                               /* Vector  947, coef 2(124), 3( 80) */
  19318,                               /* Vector  948, coef 1( 75), 2(118) */
  21059,                               /* Vector  948, coef 3( 82) Vector
                                        * 949, Coef 1( 67) */
  29512,                               /* Vector  949, coef 2(115), 3( 72) */
  24434,                               /* Vector  950, coef 1( 95), 2(114) */
  25684,                               /* Vector  950, coef 3(100) Vector
                                        * 951, Coef 1( 84) */
  28000,                               /* Vector  951, coef 2(109), 3( 96) */
  19312,                               /* Vector  952, coef 1( 75), 2(112) */
  24908,                               /* Vector  952, coef 3( 97) Vector
                                        * 953, Coef 1( 76) */
  27737,                               /* Vector  953, coef 2(108), 3( 89) */
  22381,                               /* Vector  954, coef 1( 87), 2(109) */
  22368,                               /* Vector  954, coef 3( 87) Vector
                                        * 955, Coef 1( 96) */
  30551,                               /* Vector  955, coef 2(119), 3( 87) */
  22899,                               /* Vector  956, coef 1( 89), 2(115) */
  20849,                               /* Vector  956, coef 3( 81) Vector
                                        * 957, Coef 1(113) */
  -30367,                              /* Vector  957, coef 2(137), 3( 97) */
  27261,                               /* Vector  958, coef 1(106), 2(125) */
  23401,                               /* Vector  958, coef 3( 91) Vector
                                        * 959, Coef 1(105) */
  -31918,                              /* Vector  959, coef 2(131), 3( 82) */
  -26696,                              /* Vector  960, coef 1(151), 2(184) */
  28560,                               /* Vector  960, coef 3(111) Vector
                                        * 961, Coef 1(144) */
  -18578,                              /* Vector  961, coef 2(183), 3(110) */
  -29263,                              /* Vector  962, coef 1(141), 2(177) */
  27285,                               /* Vector  962, coef 3(106) Vector
                                        * 963, Coef 1(149) */
  -23188,                              /* Vector  963, coef 2(165), 3(108) */
  -28516,                              /* Vector  964, coef 1(144), 2(156) */
  26259,                               /* Vector  964, coef 3(102) Vector
                                        * 965, Coef 1(147) */
  -21659,                              /* Vector  965, coef 2(171), 3(101) */
  -29279,                              /* Vector  966, coef 1(141), 2(161) */
  24198,                               /* Vector  966, coef 3( 94) Vector
                                        * 967, Coef 1(134) */
  -24981,                              /* Vector  967, coef 2(158), 3(107) */
  -30036,                              /* Vector  968, coef 1(138), 2(172) */
  17832,                               /* Vector  968, coef 3( 69) Vector
                                        * 969, Coef 1(168) */
  -11945,                              /* Vector  969, coef 2(209), 3( 87) */
  -24895,                              /* Vector  970, coef 1(158), 2(193) */
  26522,                               /* Vector  970, coef 3(103) Vector
                                        * 971, Coef 1(154) */
  -20130,                              /* Vector  971, coef 2(177), 3( 94) */
  -24913,                              /* Vector  972, coef 1(158), 2(175) */
  19614,                               /* Vector  972, coef 3( 76) Vector
                                        * 973, Coef 1(158) */
  -17591,                              /* Vector  973, coef 2(187), 3( 73) */
  -27229,                              /* Vector  974, coef 1(149), 2(163) */
  21392,                               /* Vector  974, coef 3( 83) Vector
                                        * 975, Coef 1(144) */
  -25527,                              /* Vector  975, coef 2(156), 3( 73) */
  -25151,                              /* Vector  976, coef 1(157), 2(193) */
  18073,                               /* Vector  976, coef 3( 70) Vector
                                        * 977, Coef 1(153) */
  -16305,                              /* Vector  977, coef 2(192), 3( 79) */
  -28506,                              /* Vector  978, coef 1(144), 2(166) */
  19087,                               /* Vector  978, coef 3( 74) Vector
                                        * 979, Coef 1(143) */
  -21432,                              /* Vector  979, coef 2(172), 3( 72) */
  -24877,                              /* Vector  980, coef 1(158), 2(211) */
  27802,                               /* Vector  980, coef 3(108) Vector
                                        * 981, Coef 1(154) */
  -14497,                              /* Vector  981, coef 2(199), 3( 95) */
  -27205,                              /* Vector  982, coef 1(149), 2(187) */
  23954,                               /* Vector  982, coef 3( 93) Vector
                                        * 983, Coef 1(146) */
  -16543,                              /* Vector  983, coef 2(191), 3( 97) */
  -27966,                              /* Vector  984, coef 1(146), 2(194) */
  21132,                               /* Vector  984, coef 3( 82) Vector
                                        * 985, Coef 1(140) */
  -18360,                              /* Vector  985, coef 2(184), 3( 72) */
  -29012,                              /* Vector  986, coef 1(142), 2(172) */
  23689,                               /* Vector  986, coef 3( 92) Vector
                                        * 987, Coef 1(137) */
  -22954,                              /* Vector  987, coef 2(166), 3( 86) */
  -32355,                              /* Vector  988, coef 1(129), 2(157) */
  21642,                               /* Vector  988, coef 3( 84) Vector
                                        * 989, Coef 1(138) */
  -27540,                              /* Vector  989, coef 2(148), 3(108) */
  -31080,                              /* Vector  990, coef 1(134), 2(152) */
  23694,                               /* Vector  990, coef 3( 92) Vector
                                        * 991, Coef 1(142) */
  -27555,                              /* Vector  991, coef 2(148), 3( 93) */
  28018,                               /* Vector  992, coef 1(109), 2(114) */
  20080,                               /* Vector  992, coef 3( 78) Vector
                                        * 993, Coef 1(112) */
  -29880,                              /* Vector  993, coef 2(139), 3( 72) */
  -32615,                              /* Vector  994, coef 1(128), 2(153) */
  15446,                               /* Vector  994, coef 3( 60) Vector
                                        * 995, Coef 1( 86) */
  26962,                               /* Vector  995, coef 2(105), 3( 82) */
  24947,                               /* Vector  996, coef 1( 97), 2(115) */
  19797,                               /* Vector  996, coef 3( 77) Vector
                                        * 997, Coef 1( 85) */
  27464,                               /* Vector  997, coef 2(107), 3( 72) */
  27769,                               /* Vector  998, coef 1(108), 2(121) */
  18795,                               /* Vector  998, coef 3( 73) Vector
                                        * 999, Coef 1(107) */
  31040,                               /* Vector  999, coef 2(121), 3( 64) */
  25711,                               /* Vector 1000, coef 1(100), 2(111) */
  18028,                               /* Vector 1000, coef 3( 70) Vector
                                        * 1001, Coef 1(108) */
  28991,                               /* Vector 1001, coef 2(113), 3( 63) */
  -32096,                              /* Vector 1002, coef 1(130), 2(160) */
  19580,                               /* Vector 1002, coef 3( 76) Vector
                                        * 1003, Coef 1(124) */
  -29351,                              /* Vector 1003, coef 2(141), 3( 89) */
  29568,                               /* Vector 1004, coef 1(115), 2(128) */
  24696,                               /* Vector 1004, coef 3( 96) Vector
                                        * 1005, Coef 1(120) */
  -28071,                              /* Vector 1005, coef 2(146), 3( 89) */
  29834,                               /* Vector 1006, coef 1(116), 2(138) */
  21871,                               /* Vector 1006, coef 3( 85) Vector
                                        * 1007, Coef 1(111) */
  -32181,                              /* Vector 1007, coef 2(130), 3( 75) */
  23931,                               /* Vector 1008, coef 1( 93), 2(123) */
  18782,                               /* Vector 1008, coef 3( 73) Vector
                                        * 1009, Coef 1( 94) */
  30789,                               /* Vector 1009, coef 2(120), 3( 69) */
  27266,                               /* Vector 1010, coef 1(106), 2(130) */
  15441,                               /* Vector 1010, coef 3( 60) Vector
                                        * 1011, Coef 1( 81) */
  28216,                               /* Vector 1011, coef 2(110), 3( 56) */
  -32124,                              /* Vector 1012, coef 1(130), 2(132) */
  23681,                               /* Vector 1012, coef 3( 92) Vector
                                        * 1013, Coef 1(129) */
  -30638,                              /* Vector 1013, coef 2(136), 3( 82) */
  31363,                               /* Vector 1014, coef 1(122), 2(131) */
  19846,                               /* Vector 1014, coef 3( 77) Vector
                                        * 1015, Coef 1(134) */
  -30140,                              /* Vector 1015, coef 2(138), 3( 68) */
  -32366,                              /* Vector 1016, coef 1(129), 2(146) */
  19322,                               /* Vector 1016, coef 3( 75) Vector
                                        * 1017, Coef 1(122) */
  -29117,                              /* Vector 1017, coef 2(142), 3( 67) */
  -32626,                              /* Vector 1018, coef 1(128), 2(142) */
  15236,                               /* Vector 1018, coef 3( 59) Vector
                                        * 1019, Coef 1(132) */
  -26058,                              /* Vector 1019, coef 2(154), 3( 54) */
  -30574,                              /* Vector 1020, coef 1(136), 2(146) */
  16251,                               /* Vector 1020, coef 3( 63) Vector
                                        * 1021, Coef 1(123) */
  -32712,                              /* Vector 1021, coef 2(128), 3( 56) */
  -30057,                              /* Vector 1022, coef 1(138), 2(151) */
  13200,                               /* Vector 1022, coef 3( 51) Vector
                                        * 1023, Coef 1(144) */
  -25818,                              /* Vector 1023, coef 2(155), 3( 38) */
  26266,                               /* Vector 1024, coef 1(102), 2(154) */
  -30353,                              /* Vector 1024, coef 3(137) Vector
                                        * 1025, Coef 1(111) */
  -19040,                              /* Vector 1025, coef 2(181), 3(160) */
  26812,                               /* Vector 1026, coef 1(104), 2(188) */
  -25241,                              /* Vector 1026, coef 3(157) Vector
                                        * 1027, Coef 1(103) */
  -17776,                              /* Vector 1027, coef 2(186), 3(144) */
  26291,                               /* Vector 1028, coef 1(102), 2(179) */
  -29854,                              /* Vector 1028, coef 3(139) Vector
                                        * 1029, Coef 1( 98) */
  -13175,                              /* Vector 1029, coef 2(204), 3(137) */
  25025,                               /* Vector 1030, coef 1( 97), 2(193) */
  -24480,                              /* Vector 1030, coef 3(160) Vector
                                        * 1031, Coef 1( 96) */
  -17773,                              /* Vector 1031, coef 2(186), 3(147) */
  25277,                               /* Vector 1032, coef 1( 98), 2(189) */
  -32161,                              /* Vector 1032, coef 3(130) Vector
                                        * 1033, Coef 1( 95) */
  -18302,                              /* Vector 1033, coef 2(184), 3(130) */
  23491,                               /* Vector 1034, coef 1( 91), 2(195) */
  -30119,                              /* Vector 1034, coef 3(138) Vector
                                        * 1035, Coef 1( 89) */
  -18302,                              /* Vector 1035, coef 2(184), 3(130) */
  28319,                               /* Vector 1036, coef 1(110), 2(159) */
  -23961,                              /* Vector 1036, coef 3(162) Vector
                                        * 1037, Coef 1(103) */
  -21862,                              /* Vector 1037, coef 2(170), 3(154) */
  25250,                               /* Vector 1038, coef 1( 98), 2(162) */
  -26279,                              /* Vector 1038, coef 3(153) Vector
                                        * 1039, Coef 1( 89) */
  -15459,                              /* Vector 1039, coef 2(195), 3(157) */
  22711,                               /* Vector 1040, coef 1( 88), 2(183) */
  -22446,                              /* Vector 1040, coef 3(168) Vector
                                        * 1041, Coef 1( 82) */
  -13414,                              /* Vector 1041, coef 2(203), 3(154) */
  21688,                               /* Vector 1042, coef 1( 84), 2(184) */
  -26529,                              /* Vector 1042, coef 3(152) Vector
                                        * 1043, Coef 1( 95) */
  -24663,                              /* Vector 1043, coef 2(159), 3(169) */
  23450,                               /* Vector 1044, coef 1( 91), 2(154) */
  -26276,                              /* Vector 1044, coef 3(153) Vector
                                        * 1045, Coef 1( 92) */
  -30558,                              /* Vector 1045, coef 2(136), 3(162) */
  20654,                               /* Vector 1046, coef 1( 80), 2(174) */
  -25762,                              /* Vector 1046, coef 3(155) Vector
                                        * 1047, Coef 1( 94) */
  -21106,                              /* Vector 1047, coef 2(173), 3(142) */
  23204,                               /* Vector 1048, coef 1( 90), 2(164) */
  -29093,                              /* Vector 1048, coef 3(142) Vector
                                        * 1049, Coef 1( 91) */
  -21374,                              /* Vector 1049, coef 2(172), 3(130) */
  21686,                               /* Vector 1050, coef 1( 84), 2(182) */
  -30115,                              /* Vector 1050, coef 3(138) Vector
                                        * 1051, Coef 1( 93) */
  -11127,                              /* Vector 1051, coef 2(212), 3(137) */
  22746,                               /* Vector 1052, coef 1( 88), 2(218) */
  -27814,                              /* Vector 1052, coef 3(147) Vector
                                        * 1053, Coef 1( 90) */
  -12679,                              /* Vector 1053, coef 2(206), 3(121) */
  22210,                               /* Vector 1054, coef 1( 86), 2(194) */
  -32174,                              /* Vector 1054, coef 3(130) Vector
                                        * 1055, Coef 1( 82) */
  -13684,                              /* Vector 1055, coef 2(202), 3(140) */
  19354,                               /* Vector 1056, coef 1( 75), 2(154) */
  28505,                               /* Vector 1056, coef 3(111) Vector
                                        * 1057, Coef 1( 89) */
  -21386,                              /* Vector 1057, coef 2(172), 3(118) */
  21933,                               /* Vector 1058, coef 1( 85), 2(173) */
  27725,                               /* Vector 1058, coef 3(108) Vector
                                        * 1059, Coef 1( 77) */
  -19857,                              /* Vector 1059, coef 2(178), 3(111) */
  19627,                               /* Vector 1060, coef 1( 76), 2(171) */
  31048,                               /* Vector 1060, coef 3(121) Vector
                                        * 1061, Coef 1( 72) */
  -21904,                              /* Vector 1061, coef 2(170), 3(112) */
  22164,                               /* Vector 1062, coef 1( 86), 2(148) */
  -29361,                              /* Vector 1062, coef 3(141) Vector
                                        * 1063, Coef 1( 79) */
  -27771,                              /* Vector 1063, coef 2(147), 3(133) */
  18839,                               /* Vector 1064, coef 1( 73), 2(151) */
  -30141,                              /* Vector 1064, coef 3(138) Vector
                                        * 1065, Coef 1( 67) */
  -24438,                              /* Vector 1065, coef 2(160), 3(138) */
  20133,                               /* Vector 1066, coef 1( 78), 2(165) */
  -27055,                              /* Vector 1066, coef 3(150) Vector
                                        * 1067, Coef 1( 81) */
  -23414,                              /* Vector 1067, coef 2(164), 3(138) */
  19105,                               /* Vector 1068, coef 1( 74), 2(161) */
  -29611,                              /* Vector 1068, coef 3(140) Vector
                                        * 1069, Coef 1( 85) */
  -22140,                              /* Vector 1069, coef 2(169), 3(132) */
  20658,                               /* Vector 1070, coef 1( 80), 2(178) */
  -31661,                              /* Vector 1070, coef 3(132) Vector
                                        * 1071, Coef 1( 83) */
  -20873,                              /* Vector 1071, coef 2(174), 3(119) */
  20646,                               /* Vector 1072, coef 1( 80), 2(166) */
  30297,                               /* Vector 1072, coef 3(118) Vector
                                        * 1073, Coef 1( 89) */
  -23938,                              /* Vector 1073, coef 2(162), 3(126) */
  21658,                               /* Vector 1074, coef 1( 84), 2(154) */
  32595,                               /* Vector 1074, coef 3(127) Vector
                                        * 1075, Coef 1( 83) */
  -24714,                              /* Vector 1075, coef 2(159), 3(118) */
  19864,                               /* Vector 1076, coef 1( 77), 2(152) */
  31054,                               /* Vector 1076, coef 3(121) Vector
                                        * 1077, Coef 1( 78) */
  -24191,                              /* Vector 1077, coef 2(161), 3(129) */
  18846,                               /* Vector 1078, coef 1( 73), 2(158) */
  32070,                               /* Vector 1078, coef 3(125) Vector
                                        * 1079, Coef 1( 70) */
  -25988,                              /* Vector 1079, coef 2(154), 3(124) */
  19373,                               /* Vector 1080, coef 1( 75), 2(173) */
  -29625,                              /* Vector 1080, coef 3(140) Vector
                                        * 1081, Coef 1( 71) */
  -19576,                              /* Vector 1081, coef 2(179), 3(136) */
  18346,                               /* Vector 1082, coef 1( 71), 2(170) */
  -30651,                              /* Vector 1082, coef 3(136) Vector
                                        * 1083, Coef 1( 69) */
  -22911,                              /* Vector 1083, coef 2(166), 3(129) */
  16803,                               /* Vector 1084, coef 1( 65), 2(163) */
  31563,                               /* Vector 1084, coef 3(123) Vector
                                        * 1085, Coef 1( 75) */
  -23951,                              /* Vector 1085, coef 2(162), 3(113) */
  19638,                               /* Vector 1086, coef 1( 76), 2(182) */
  31812,                               /* Vector 1086, coef 3(124) Vector
                                        * 1087, Coef 1( 68) */
  -20615,                              /* Vector 1087, coef 2(175), 3(121) */
  19109,                               /* Vector 1088, coef 1( 74), 2(165) */
  -21947,                              /* Vector 1088, coef 3(170) Vector
                                        * 1089, Coef 1( 69) */
  -22121,                              /* Vector 1089, coef 2(169), 3(151) */
  16559,                               /* Vector 1090, coef 1( 64), 2(175) */
  -31929,                              /* Vector 1090, coef 3(131) Vector
                                        * 1091, Coef 1( 71) */
  -14436,                              /* Vector 1091, coef 2(199), 3(156) */
  18623,                               /* Vector 1092, coef 1( 72), 2(191) */
  -28092,                              /* Vector 1092, coef 3(146) Vector
                                        * 1093, Coef 1( 68) */
  -12653,                              /* Vector 1093, coef 2(206), 3(147) */
  16842,                               /* Vector 1094, coef 1( 65), 2(202) */
  -28606,                              /* Vector 1094, coef 3(144) Vector
                                        * 1095, Coef 1( 66) */
  -15724,                              /* Vector 1095, coef 2(194), 3(148) */
  16825,                               /* Vector 1096, coef 1( 65), 2(185) */
  -27836,                              /* Vector 1096, coef 3(147) Vector
                                        * 1097, Coef 1( 68) */
  -18806,                              /* Vector 1097, coef 2(182), 3(138) */
  16563,                               /* Vector 1098, coef 1( 64), 2(179) */
  -29889,                              /* Vector 1098, coef 3(139) Vector
                                        * 1099, Coef 1( 63) */
  -17274,                              /* Vector 1099, coef 2(188), 3(134) */
  15289,                               /* Vector 1100, coef 1( 59), 2(185) */
  -32192,                              /* Vector 1100, coef 3(130) Vector
                                        * 1101, Coef 1( 64) */
  -15971,                              /* Vector 1101, coef 2(193), 3(157) */
  15814,                               /* Vector 1102, coef 1( 61), 2(198) */
  -27330,                              /* Vector 1102, coef 3(149) Vector
                                        * 1103, Coef 1( 62) */
  -18793,                              /* Vector 1103, coef 2(182), 3(151) */
  15292,                               /* Vector 1104, coef 1( 59), 2(188) */
  -28098,                              /* Vector 1104, coef 3(146) Vector
                                        * 1105, Coef 1( 62) */
  -17496,                              /* Vector 1105, coef 2(187), 3(168) */
  15030,                               /* Vector 1106, coef 1( 58), 2(182) */
  -24008,                              /* Vector 1106, coef 3(162) Vector
                                        * 1107, Coef 1( 56) */
  -16485,                              /* Vector 1107, coef 2(191), 3(155) */
  14797,                               /* Vector 1108, coef 1( 57), 2(205) */
  -28617,                              /* Vector 1108, coef 3(144) Vector
                                        * 1109, Coef 1( 55) */
  -13940,                              /* Vector 1109, coef 2(201), 3(140) */
  13780,                               /* Vector 1110, coef 1( 53), 2(212) */
  -26829,                              /* Vector 1110, coef 3(151) Vector
                                        * 1111, Coef 1( 51) */
  -13677,                              /* Vector 1111, coef 2(202), 3(147) */
  14013,                               /* Vector 1112, coef 1( 54), 2(189) */
  -29634,                              /* Vector 1112, coef 3(140) Vector
                                        * 1113, Coef 1( 62) */
  -20850,                              /* Vector 1113, coef 2(174), 3(142) */
  14771,                               /* Vector 1114, coef 1( 57), 2(179) */
  -29641,                              /* Vector 1114, coef 3(140) Vector
                                        * 1115, Coef 1( 55) */
  -16720,                              /* Vector 1115, coef 2(190), 3(176) */
  13502,                               /* Vector 1116, coef 1( 52), 2(190) */
  -23233,                              /* Vector 1116, coef 3(165) Vector
                                        * 1117, Coef 1( 63) */
  -13181,                              /* Vector 1117, coef 2(204), 3(131) */
  15559,                               /* Vector 1118, coef 1( 60), 2(199) */
  -31175,                              /* Vector 1118, coef 3(134) Vector
                                        * 1119, Coef 1( 57) */
  -15743,                              /* Vector 1119, coef 2(194), 3(129) */
  12705,                               /* Vector 1120, coef 1( 49), 2(161) */
  31286,                               /* Vector 1120, coef 3(122) Vector
                                        * 1121, Coef 1( 54) */
  -19045,                              /* Vector 1121, coef 2(181), 3(155) */
  13239,                               /* Vector 1122, coef 1( 51), 2(183) */
  -28366,                              /* Vector 1122, coef 3(145) Vector
                                        * 1123, Coef 1( 50) */
  -15225,                              /* Vector 1123, coef 2(196), 3(135) */
  12476,                               /* Vector 1124, coef 1( 48), 2(188) */
  -30662,                              /* Vector 1124, coef 3(136) Vector
                                        * 1125, Coef 1( 58) */
  -22902,                              /* Vector 1125, coef 2(166), 3(138) */
  15273,                               /* Vector 1126, coef 1( 59), 2(169) */
  -31691,                              /* Vector 1126, coef 3(132) Vector
                                        * 1127, Coef 1( 53) */
  -22401,                              /* Vector 1127, coef 2(168), 3(127) */
  13481,                               /* Vector 1128, coef 1( 52), 2(169) */
  -30672,                              /* Vector 1128, coef 3(136) Vector
                                        * 1129, Coef 1( 48) */
  -23414,                              /* Vector 1129, coef 2(164), 3(138) */
  13748,                               /* Vector 1130, coef 1( 53), 2(180) */
  -31437,                              /* Vector 1130, coef 3(133) Vector
                                        * 1131, Coef 1( 51) */
  -18818,                              /* Vector 1131, coef 2(182), 3(126) */
  12465,                               /* Vector 1132, coef 1( 48), 2(177) */
  -30930,                              /* Vector 1132, coef 3(135) Vector
                                        * 1133, Coef 1( 46) */
  -20349,                              /* Vector 1133, coef 2(176), 3(131) */
  12197,                               /* Vector 1134, coef 1( 47), 2(165) */
  -32213,                              /* Vector 1134, coef 3(130) Vector
                                        * 1135, Coef 1( 43) */
  -23418,                              /* Vector 1135, coef 2(164), 3(134) */
  11432,                               /* Vector 1136, coef 1( 44), 2(168) */
  32046,                               /* Vector 1136, coef 3(125) Vector
                                        * 1137, Coef 1( 46) */
  -15981,                              /* Vector 1137, coef 2(193), 3(147) */
  11448,                               /* Vector 1138, coef 1( 44), 2(184) */
  -29142,                              /* Vector 1138, coef 3(142) Vector
                                        * 1139, Coef 1( 42) */
  -19828,                              /* Vector 1139, coef 2(178), 3(140) */
  10172,                               /* Vector 1140, coef 1( 39), 2(188) */
  -30933,                              /* Vector 1140, coef 3(135) Vector
                                        * 1141, Coef 1( 43) */
  -18814,                              /* Vector 1141, coef 2(182), 3(130) */
  10163,                               /* Vector 1142, coef 1( 39), 2(179) */
  -32472,                              /* Vector 1142, coef 3(129) Vector
                                        * 1143, Coef 1( 40) */
  -21628,                              /* Vector 1143, coef 2(171), 3(132) */
  12461,                               /* Vector 1144, coef 1( 48), 2(173) */
  31529,                               /* Vector 1144, coef 3(123) Vector
                                        * 1145, Coef 1( 41) */
  -20105,                              /* Vector 1145, coef 2(177), 3(119) */
  13474,                               /* Vector 1146, coef 1( 52), 2(162) */
  -25808,                              /* Vector 1146, coef 3(155) Vector
                                        * 1147, Coef 1( 48) */
  -21358,                              /* Vector 1147, coef 2(172), 3(146) */
  11692,                               /* Vector 1148, coef 1( 45), 2(172) */
  -27082,                              /* Vector 1148, coef 3(150) Vector
                                        * 1149, Coef 1( 54) */
  -16258,                              /* Vector 1149, coef 2(192), 3(126) */
  12218,                               /* Vector 1150, coef 1( 47), 2(186) */
  32299,                               /* Vector 1150, coef 3(126) Vector
                                        * 1151, Coef 1( 43) */
  -17284,                              /* Vector 1151, coef 2(188), 3(124) */
  22202,                               /* Vector 1152, coef 1( 86), 2(186) */
  27217,                               /* Vector 1152, coef 3(106) Vector
                                        * 1153, Coef 1( 81) */
  -18574,                              /* Vector 1153, coef 2(183), 3(114) */
  19416,                               /* Vector 1154, coef 1( 75), 2(216) */
  -24759,                              /* Vector 1154, coef 3(159) Vector
                                        * 1155, Coef 1( 73) */
  -11373,                              /* Vector 1155, coef 2(211), 3(147) */
  18638,                               /* Vector 1156, coef 1( 72), 2(206) */
  32583,                               /* Vector 1156, coef 3(127) Vector
                                        * 1157, Coef 1( 71) */
  -14980,                              /* Vector 1157, coef 2(197), 3(124) */
  17613,                               /* Vector 1158, coef 1( 68), 2(205) */
  -32169,                              /* Vector 1158, coef 3(130) Vector
                                        * 1159, Coef 1( 87) */
  -8332,                               /* Vector 1159, coef 2(223), 3(116) */
  21721,                               /* Vector 1160, coef 1( 84), 2(217) */
  32088,                               /* Vector 1160, coef 3(125) Vector
                                        * 1161, Coef 1( 88) */
  -14221,                              /* Vector 1161, coef 2(200), 3(115) */
  21948,                               /* Vector 1162, coef 1( 85), 2(188) */
  30805,                               /* Vector 1162, coef 3(120) Vector
                                        * 1163, Coef 1( 85) */
  -12927,                              /* Vector 1163, coef 2(205), 3(129) */
  20699,                               /* Vector 1164, coef 1( 80), 2(219) */
  30797,                               /* Vector 1164, coef 3(120) Vector
                                        * 1165, Coef 1( 77) */
  -9855,                               /* Vector 1165, coef 2(217), 3(129) */
  20684,                               /* Vector 1166, coef 1( 80), 2(204) */
  -31665,                              /* Vector 1166, coef 3(132) Vector
                                        * 1167, Coef 1( 79) */
  -14211,                              /* Vector 1167, coef 2(200), 3(125) */
  19653,                               /* Vector 1168, coef 1( 76), 2(197) */
  -29878,                              /* Vector 1168, coef 3(139) Vector
                                        * 1169, Coef 1( 74) */
  -16762,                              /* Vector 1169, coef 2(190), 3(134) */
  21704,                               /* Vector 1170, coef 1( 84), 2(200) */
  27472,                               /* Vector 1170, coef 3(107) Vector
                                        * 1171, Coef 1( 80) */
  -14486,                              /* Vector 1171, coef 2(199), 3(106) */
  20924,                               /* Vector 1172, coef 1( 81), 2(188) */
  26958,                               /* Vector 1172, coef 3(105) Vector
                                        * 1173, Coef 1( 78) */
  -16797,                              /* Vector 1173, coef 2(190), 3( 99) */
  20413,                               /* Vector 1174, coef 1( 79), 2(189) */
  31816,                               /* Vector 1174, coef 3(124) Vector
                                        * 1175, Coef 1( 72) */
  -18569,                              /* Vector 1175, coef 2(183), 3(119) */
  19658,                               /* Vector 1176, coef 1( 76), 2(202) */
  28744,                               /* Vector 1176, coef 3(112) Vector
                                        * 1177, Coef 1( 72) */
  -14228,                              /* Vector 1177, coef 2(200), 3(108) */
  18880,                               /* Vector 1178, coef 1( 73), 2(192) */
  25932,                               /* Vector 1178, coef 3(101) Vector
                                        * 1179, Coef 1( 76) */
  -8341,                               /* Vector 1179, coef 2(223), 3(107) */
  19169,                               /* Vector 1180, coef 1( 74), 2(225) */
  28999,                               /* Vector 1180, coef 3(113) Vector
                                        * 1181, Coef 1( 71) */
  -11416,                              /* Vector 1181, coef 2(211), 3(104) */
  20690,                               /* Vector 1182, coef 1( 80), 2(210) */
  25418,                               /* Vector 1182, coef 3( 99) Vector
                                        * 1183, Coef 1( 74) */
  -11428,                              /* Vector 1183, coef 2(211), 3( 92) */
  18355,                               /* Vector 1184, coef 1( 71), 2(179) */
  26689,                               /* Vector 1184, coef 3(104) Vector
                                        * 1185, Coef 1( 65) */
  -21398,                              /* Vector 1185, coef 2(172), 3(106) */
  16042,                               /* Vector 1186, coef 1( 62), 2(170) */
  -32702,                              /* Vector 1186, coef 3(128) Vector
                                        * 1187, Coef 1( 66) */
  -14977,                              /* Vector 1187, coef 2(197), 3(127) */
  18360,                               /* Vector 1188, coef 1( 71), 2(184) */
  28483,                               /* Vector 1188, coef 3(111) Vector
                                        * 1189, Coef 1( 67) */
  -17564,                              /* Vector 1189, coef 2(187), 3(100) */
  17089,                               /* Vector 1190, coef 1( 66), 2(193) */
  26942,                               /* Vector 1190, coef 3(105) Vector
                                        * 1191, Coef 1( 62) */
  -17814,                              /* Vector 1191, coef 2(186), 3(106) */
  15803,                               /* Vector 1192, coef 1( 61), 2(187) */
  24888,                               /* Vector 1192, coef 3( 97) Vector
                                        * 1193, Coef 1( 56) */
  -19101,                              /* Vector 1193, coef 2(181), 3( 99) */
  17595,                               /* Vector 1194, coef 1( 68), 2(187) */
  -32447,                              /* Vector 1194, coef 3(129) Vector
                                        * 1195, Coef 1( 65) */
  -18308,                              /* Vector 1195, coef 2(184), 3(124) */
  17597,                               /* Vector 1196, coef 1( 68), 2(189) */
  28992,                               /* Vector 1196, coef 3(113) Vector
                                        * 1197, Coef 1( 64) */
  -18318,                              /* Vector 1197, coef 2(184), 3(114) */
  16066,                               /* Vector 1198, coef 1( 62), 2(194) */
  30780,                               /* Vector 1198, coef 3(120) Vector
                                        * 1199, Coef 1( 60) */
  -18055,                              /* Vector 1199, coef 2(185), 3(121) */
  14519,                               /* Vector 1200, coef 1( 56), 2(183) */
  30270,                               /* Vector 1200, coef 3(118) Vector
                                        * 1201, Coef 1( 62) */
  -21388,                              /* Vector 1201, coef 2(172), 3(116) */
  14761,                               /* Vector 1202, coef 1( 57), 2(169) */
  28221,                               /* Vector 1202, coef 3(110) Vector
                                        * 1203, Coef 1( 61) */
  -19861,                              /* Vector 1203, coef 2(178), 3(107) */
  14514,                               /* Vector 1204, coef 1( 56), 2(178) */
  28214,                               /* Vector 1204, coef 3(110) Vector
                                        * 1205, Coef 1( 54) */
  -21398,                              /* Vector 1205, coef 2(172), 3(106) */
  16067,                               /* Vector 1206, coef 1( 62), 2(195) */
  26681,                               /* Vector 1206, coef 3(104) Vector
                                        * 1207, Coef 1( 57) */
  -16528,                              /* Vector 1207, coef 2(191), 3(112) */
  13757,                               /* Vector 1208, coef 1( 53), 2(189) */
  29748,                               /* Vector 1208, coef 3(116) Vector
                                        * 1209, Coef 1( 52) */
  -18833,                              /* Vector 1209, coef 2(182), 3(111) */
  14267,                               /* Vector 1210, coef 1( 55), 2(187) */
  25923,                               /* Vector 1210, coef 3(101) Vector
                                        * 1211, Coef 1( 67) */
  -19360,                              /* Vector 1211, coef 2(180), 3( 96) */
  14510,                               /* Vector 1212, coef 1( 56), 2(174) */
  31028,                               /* Vector 1212, coef 3(121) Vector
                                        * 1213, Coef 1( 52) */
  -21132,                              /* Vector 1213, coef 2(173), 3(116) */
  17098,                               /* Vector 1214, coef 1( 66), 2(202) */
  28995,                               /* Vector 1214, coef 3(113) Vector
                                        * 1215, Coef 1( 67) */
  -15014,                              /* Vector 1215, coef 2(197), 3( 90) */
  13753,                               /* Vector 1216, coef 1( 53), 2(185) */
  23857,                               /* Vector 1216, coef 3( 93) Vector
                                        * 1217, Coef 1( 49) */
  -19615,                              /* Vector 1217, coef 2(179), 3( 97) */
  11698,                               /* Vector 1218, coef 1( 45), 2(178) */
  23616,                               /* Vector 1218, coef 3( 92) Vector
                                        * 1219, Coef 1( 64) */
  -19366,                              /* Vector 1219, coef 2(180), 3( 90) */
  17326,                               /* Vector 1220, coef 1( 67), 2(174) */
  22334,                               /* Vector 1220, coef 3( 87) Vector
                                        * 1221, Coef 1( 62) */
  -21677,                              /* Vector 1221, coef 2(171), 3( 83) */
  15015,                               /* Vector 1222, coef 1( 58), 2(167) */
  23092,                               /* Vector 1222, coef 3( 90) Vector
                                        * 1223, Coef 1( 52) */
  -23210,                              /* Vector 1223, coef 2(165), 3( 86) */
  18111,                               /* Vector 1224, coef 1( 70), 2(191) */
  22591,                               /* Vector 1224, coef 3( 88) Vector
                                        * 1225, Coef 1( 63) */
  -17835,                              /* Vector 1225, coef 2(186), 3( 85) */
  14771,                               /* Vector 1226, coef 1( 57), 2(179) */
  22836,                               /* Vector 1226, coef 3( 89) Vector
                                        * 1227, Coef 1( 52) */
  -19882,                              /* Vector 1227, coef 2(178), 3( 86) */
  15288,                               /* Vector 1228, coef 1( 59), 2(184) */
  19765,                               /* Vector 1228, coef 3( 77) Vector
                                        * 1229, Coef 1( 53) */
  -19896,                              /* Vector 1229, coef 2(178), 3( 72) */
  11950,                               /* Vector 1230, coef 1( 46), 2(174) */
  20778,                               /* Vector 1230, coef 3( 81) Vector
                                        * 1231, Coef 1( 42) */
  -21178,                              /* Vector 1231, coef 2(173), 3( 70) */
  15278,                               /* Vector 1232, coef 1( 59), 2(174) */
  25142,                               /* Vector 1232, coef 3( 98) Vector
                                        * 1233, Coef 1( 54) */
  -21669,                              /* Vector 1233, coef 2(171), 3( 91) */
  12974,                               /* Vector 1234, coef 1( 50), 2(174) */
  24877,                               /* Vector 1234, coef 3( 97) Vector
                                        * 1235, Coef 1( 45) */
  -21413,                              /* Vector 1235, coef 2(172), 3( 91) */
  12198,                               /* Vector 1236, coef 1( 47), 2(166) */
  24107,                               /* Vector 1236, coef 3( 94) Vector
                                        * 1237, Coef 1( 43) */
  -22953,                              /* Vector 1237, coef 2(166), 3( 87) */
  10655,                               /* Vector 1238, coef 1( 41), 2(159) */
  22561,                               /* Vector 1238, coef 3( 88) Vector
                                        * 1239, Coef 1( 33) */
  -25259,                              /* Vector 1239, coef 2(157), 3( 85) */
  15793,                               /* Vector 1240, coef 1( 61), 2(177) */
  21294,                               /* Vector 1240, coef 3( 83) Vector
                                        * 1241, Coef 1( 46) */
  -22708,                              /* Vector 1241, coef 2(167), 3( 76) */
  14245,                               /* Vector 1242, coef 1( 55), 2(165) */
  19500,                               /* Vector 1242, coef 3( 76) Vector
                                        * 1243, Coef 1( 44) */
  -25016,                              /* Vector 1243, coef 2(158), 3( 72) */
  9371,                                /* Vector 1244, coef 1( 36), 2(155) */
  15413,                               /* Vector 1244, coef 3( 60) Vector
                                        * 1245, Coef 1( 53) */
  -18355,                              /* Vector 1245, coef 2(184), 3( 77) */
  9644,                                /* Vector 1246, coef 1( 37), 2(172) */
  23844,                               /* Vector 1246, coef 3( 93) Vector
                                        * 1247, Coef 1( 36) */
  -22704,                              /* Vector 1247, coef 2(167), 3( 80) */
  20153,                               /* Vector 1248, coef 1( 78), 2(185) */
  24137,                               /* Vector 1248, coef 3( 94) Vector
                                        * 1249, Coef 1( 73) */
  -19111,                              /* Vector 1249, coef 2(181), 3( 89) */
  15023,                               /* Vector 1250, coef 1( 58), 2(175) */
  19247,                               /* Vector 1250, coef 3( 75) Vector
                                        * 1251, Coef 1( 47) */
  -22463,                              /* Vector 1251, coef 2(168), 3( 65) */
  14504,                               /* Vector 1252, coef 1( 56), 2(168) */
  16224,                               /* Vector 1252, coef 3( 63) Vector
                                        * 1253, Coef 1( 96) */
  -10406,                              /* Vector 1253, coef 2(215), 3( 90) */
  23249,                               /* Vector 1254, coef 1( 90), 2(209) */
  24920,                               /* Vector 1254, coef 3( 97) Vector
                                        * 1255, Coef 1( 88) */
  -15266,                              /* Vector 1255, coef 2(196), 3( 94) */
  21693,                               /* Vector 1256, coef 1( 84), 2(189) */
  22613,                               /* Vector 1256, coef 3( 88) Vector
                                        * 1257, Coef 1( 85) */
  -13738,                              /* Vector 1257, coef 2(202), 3( 86) */
  20928,                               /* Vector 1258, coef 1( 81), 2(192) */
  20299,                               /* Vector 1258, coef 3( 79) Vector
                                        * 1259, Coef 1( 75) */
  -16815,                              /* Vector 1259, coef 2(190), 3( 81) */
  17847,                               /* Vector 1260, coef 1( 69), 2(183) */
  18767,                               /* Vector 1260, coef 3( 73) Vector
                                        * 1261, Coef 1( 79) */
  -18620,                              /* Vector 1261, coef 2(183), 3( 68) */
  18355,                               /* Vector 1262, coef 1( 71), 2(179) */
  18754,                               /* Vector 1262, coef 3( 73) Vector
                                        * 1263, Coef 1( 66) */
  -21951,                              /* Vector 1263, coef 2(170), 3( 65) */
  24013,                               /* Vector 1264, coef 1( 93), 2(205) */
  16213,                               /* Vector 1264, coef 3( 63) Vector
                                        * 1265, Coef 1( 85) */
  -16331,                              /* Vector 1265, coef 2(192), 3( 53) */
  18613,                               /* Vector 1266, coef 1( 72), 2(181) */
  15171,                               /* Vector 1266, coef 3( 59) Vector
                                        * 1267, Coef 1( 67) */
  -21203,                              /* Vector 1267, coef 2(173), 3( 45) */
  20689,                               /* Vector 1268, coef 1( 80), 2(209) */
  20039,                               /* Vector 1268, coef 3( 78) Vector
                                        * 1269, Coef 1( 71) */
  -15285,                              /* Vector 1269, coef 2(196), 3( 75) */
  19138,                               /* Vector 1270, coef 1( 74), 2(194) */
  16963,                               /* Vector 1270, coef 3( 66) Vector
                                        * 1271, Coef 1( 67) */
  -16831,                              /* Vector 1271, coef 2(190), 3( 65) */
  20931,                               /* Vector 1272, coef 1( 81), 2(195) */
  16465,                               /* Vector 1272, coef 3( 64) Vector
                                        * 1273, Coef 1( 81) */
  -14536,                              /* Vector 1273, coef 2(199), 3( 56) */
  16565,                               /* Vector 1274, coef 1( 64), 2(181) */
  15932,                               /* Vector 1274, coef 3( 62) Vector
                                        * 1275, Coef 1( 60) */
  -20174,                              /* Vector 1275, coef 2(177), 3( 50) */
  13488,                               /* Vector 1276, coef 1( 52), 2(176) */
  15680,                               /* Vector 1276, coef 3( 61) Vector
                                        * 1277, Coef 1( 64) */
  -16845,                              /* Vector 1277, coef 2(190), 3( 51) */
  11948,                               /* Vector 1278, coef 1( 46), 2(172) */
  9558,                                /* Vector 1278, coef 3( 37) Vector
                                        * 1279, Coef 1( 86) */
  -9122,                               /* Vector 1279, coef 2(220), 3( 94) */
  11206,                               /* Vector 1280, coef 1( 43), 2(198) */
  -28366,                              /* Vector 1280, coef 3(145) Vector
                                        * 1281, Coef 1( 50) */
  -11909,                              /* Vector 1281, coef 2(209), 3(123) */
  12241,                               /* Vector 1282, coef 1( 47), 2(209) */
  31276,                               /* Vector 1282, coef 3(122) Vector
                                        * 1283, Coef 1( 44) */
  -13965,                              /* Vector 1283, coef 2(201), 3(115) */
  12760,                               /* Vector 1284, coef 1( 49), 2(216) */
  -29648,                              /* Vector 1284, coef 3(140) Vector
                                        * 1285, Coef 1( 48) */
  -12663,                              /* Vector 1285, coef 2(206), 3(137) */
  11992,                               /* Vector 1286, coef 1( 46), 2(216) */
  -31188,                              /* Vector 1286, coef 3(134) Vector
                                        * 1287, Coef 1( 44) */
  -11896,                              /* Vector 1287, coef 2(209), 3(136) */
  11975,                               /* Vector 1288, coef 1( 46), 2(199) */
  -32213,                              /* Vector 1288, coef 3(130) Vector
                                        * 1289, Coef 1( 43) */
  -15230,                              /* Vector 1289, coef 2(196), 3(130) */
  10958,                               /* Vector 1290, coef 1( 42), 2(206) */
  -30680,                              /* Vector 1290, coef 3(136) Vector
                                        * 1291, Coef 1( 40) */
  -13181,                              /* Vector 1291, coef 2(204), 3(131) */
  12767,                               /* Vector 1292, coef 1( 49), 2(223) */
  31279,                               /* Vector 1292, coef 3(122) Vector
                                        * 1293, Coef 1( 47) */
  -10122,                              /* Vector 1293, coef 2(216), 3(118) */
  11743,                               /* Vector 1294, coef 1( 45), 2(223) */
  30508,                               /* Vector 1294, coef 3(119) Vector
                                        * 1295, Coef 1( 44) */
  -10377,                              /* Vector 1295, coef 2(215), 3(119) */
  10959,                               /* Vector 1296, coef 1( 42), 2(207) */
  28457,                               /* Vector 1296, coef 3(111) Vector
                                        * 1297, Coef 1( 41) */
  -11148,                              /* Vector 1297, coef 2(212), 3(116) */
  10191,                               /* Vector 1298, coef 1( 39), 2(207) */
  28715,                               /* Vector 1298, coef 3(112) Vector
                                        * 1299, Coef 1( 43) */
  -9600,                               /* Vector 1299, coef 2(218), 3(128) */
  10455,                               /* Vector 1300, coef 1( 40), 2(215) */
  -32217,                              /* Vector 1300, coef 3(130) Vector
                                        * 1301, Coef 1( 39) */
  -7810,                               /* Vector 1301, coef 2(225), 3(126) */
  9949,                                /* Vector 1302, coef 1( 38), 2(221) */
  -31706,                              /* Vector 1302, coef 3(132) Vector
                                        * 1303, Coef 1( 38) */
  -10375,                              /* Vector 1303, coef 2(215), 3(121) */
  10698,                               /* Vector 1304, coef 1( 41), 2(202) */
  31013,                               /* Vector 1304, coef 3(121) Vector
                                        * 1305, Coef 1( 37) */
  -13437,                              /* Vector 1305, coef 2(203), 3(131) */
  12493,                               /* Vector 1306, coef 1( 48), 2(205) */
  -26579,                              /* Vector 1306, coef 3(152) Vector
                                        * 1307, Coef 1( 45) */
  -13157,                              /* Vector 1307, coef 2(204), 3(155) */
  10713,                               /* Vector 1308, coef 1( 41), 2(217) */
  -27609,                              /* Vector 1308, coef 3(148) Vector
                                        * 1309, Coef 1( 39) */
  -10353,                              /* Vector 1309, coef 2(215), 3(143) */
  13518,                               /* Vector 1310, coef 1( 52), 2(206) */
  -31959,                              /* Vector 1310, coef 3(131) Vector
                                        * 1311, Coef 1( 41) */
  -8337,                               /* Vector 1311, coef 2(223), 3(111) */
  9928,                                /* Vector 1312, coef 1( 38), 2(200) */
  30759,                               /* Vector 1312, coef 3(120) Vector
                                        * 1313, Coef 1( 39) */
  -16001,                              /* Vector 1313, coef 2(193), 3(127) */
  9153,                                /* Vector 1314, coef 1( 35), 2(193) */
  31522,                               /* Vector 1314, coef 3(123) Vector
                                        * 1315, Coef 1( 34) */
  -13181,                              /* Vector 1315, coef 2(204), 3(131) */
  9154,                                /* Vector 1316, coef 1( 35), 2(194) */
  -31457,                              /* Vector 1316, coef 3(133) Vector
                                        * 1317, Coef 1( 31) */
  -15741,                              /* Vector 1317, coef 2(194), 3(131) */
  9419,                                /* Vector 1318, coef 1( 36), 2(203) */
  31009,                               /* Vector 1318, coef 3(121) Vector
                                        * 1319, Coef 1( 33) */
  -13705,                              /* Vector 1319, coef 2(202), 3(119) */
  8388,                                /* Vector 1320, coef 1( 32), 2(196) */
  31261,                               /* Vector 1320, coef 3(122) Vector
                                        * 1321, Coef 1( 29) */
  -14468,                              /* Vector 1321, coef 2(199), 3(124) */
  8380,                                /* Vector 1322, coef 1( 32), 2(188) */
  31517,                               /* Vector 1322, coef 3(123) Vector
                                        * 1323, Coef 1( 29) */
  -17284,                              /* Vector 1323, coef 2(188), 3(124) */
  7105,                                /* Vector 1324, coef 1( 27), 2(193) */
  31524,                               /* Vector 1324, coef 3(123) Vector
                                        * 1325, Coef 1( 36) */
  -15509,                              /* Vector 1325, coef 2(195), 3(107) */
  9155,                                /* Vector 1326, coef 1( 35), 2(195) */
  29472,                               /* Vector 1326, coef 3(115) Vector
                                        * 1327, Coef 1( 32) */
  -16786,                              /* Vector 1327, coef 2(190), 3(110) */
  8387,                                /* Vector 1328, coef 1( 32), 2(195) */
  27420,                               /* Vector 1328, coef 3(107) Vector
                                        * 1329, Coef 1( 28) */
  -16280,                              /* Vector 1329, coef 2(192), 3(104) */
  7107,                                /* Vector 1330, coef 1( 27), 2(195) */
  28703,                               /* Vector 1330, coef 3(112) Vector
                                        * 1331, Coef 1( 31) */
  -18061,                              /* Vector 1331, coef 2(185), 3(115) */
  7354,                                /* Vector 1332, coef 1( 28), 2(186) */
  29221,                               /* Vector 1332, coef 3(114) Vector
                                        * 1333, Coef 1( 37) */
  -17794,                              /* Vector 1333, coef 2(186), 3(126) */
  9142,                                /* Vector 1334, coef 1( 35), 2(182) */
  32545,                               /* Vector 1334, coef 3(127) Vector
                                        * 1335, Coef 1( 33) */
  -19849,                              /* Vector 1335, coef 2(178), 3(119) */
  9914,                                /* Vector 1336, coef 1( 38), 2(186) */
  29731,                               /* Vector 1336, coef 3(116) Vector
                                        * 1337, Coef 1( 35) */
  -18061,                              /* Vector 1337, coef 2(185), 3(115) */
  8884,                                /* Vector 1338, coef 1( 34), 2(180) */
  27940,                               /* Vector 1338, coef 3(109) Vector
                                        * 1339, Coef 1( 36) */
  -14236,                              /* Vector 1339, coef 2(200), 3(100) */
  8900,                                /* Vector 1340, coef 1( 34), 2(196) */
  24869,                               /* Vector 1340, coef 3( 97) Vector
                                        * 1341, Coef 1( 37) */
  -12951,                              /* Vector 1341, coef 2(205), 3(105) */
  8907,                                /* Vector 1342, coef 1( 34), 2(203) */
  27934,                               /* Vector 1342, coef 3(109) Vector
                                        * 1343, Coef 1( 30) */
  -13453,                              /* Vector 1343, coef 2(203), 3(115) */
  12450,                               /* Vector 1344, coef 1( 48), 2(162) */
  25899,                               /* Vector 1344, coef 3(101) Vector
                                        * 1345, Coef 1( 43) */
  -24993,                              /* Vector 1345, coef 2(158), 3( 95) */
  8343,                                /* Vector 1346, coef 1( 32), 2(151) */
  23594,                               /* Vector 1346, coef 3( 92) Vector
                                        * 1347, Coef 1( 42) */
  -24970,                              /* Vector 1347, coef 2(158), 3(118) */
  9632,                                /* Vector 1348, coef 1( 37), 2(160) */
  30760,                               /* Vector 1348, coef 3(120) Vector
                                        * 1349, Coef 1( 40) */
  -22406,                              /* Vector 1349, coef 2(168), 3(122) */
  12969,                               /* Vector 1350, coef 1( 50), 2(169) */
  26667,                               /* Vector 1350, coef 3(104) Vector
                                        * 1351, Coef 1( 43) */
  -20890,                              /* Vector 1351, coef 2(174), 3(102) */
  9646,                                /* Vector 1352, coef 1( 37), 2(174) */
  26157,                               /* Vector 1352, coef 3(102) Vector
                                        * 1353, Coef 1( 45) */
  -21644,                              /* Vector 1353, coef 2(171), 3(116) */
  13733,                               /* Vector 1354, coef 1( 53), 2(165) */
  29232,                               /* Vector 1354, coef 3(114) Vector
                                        * 1355, Coef 1( 48) */
  -23952,                              /* Vector 1355, coef 2(162), 3(112) */
  11173,                               /* Vector 1356, coef 1( 43), 2(165) */
  29222,                               /* Vector 1356, coef 3(114) Vector
                                        * 1357, Coef 1( 38) */
  -24210,                              /* Vector 1357, coef 2(161), 3(110) */
  12463,                               /* Vector 1358, coef 1( 48), 2(175) */
  28197,                               /* Vector 1358, coef 3(110) Vector
                                        * 1359, Coef 1( 37) */
  -20619,                              /* Vector 1359, coef 2(175), 3(117) */
  10157,                               /* Vector 1360, coef 1( 39), 2(173) */
  28450,                               /* Vector 1360, coef 3(111) Vector
                                        * 1361, Coef 1( 34) */
  -21394,                              /* Vector 1361, coef 2(172), 3(110) */
  9127,                                /* Vector 1362, coef 1( 35), 2(167) */
  27166,                               /* Vector 1362, coef 3(106) Vector
                                        * 1363, Coef 1( 30) */
  -22681,                              /* Vector 1363, coef 2(167), 3(103) */
  11688,                               /* Vector 1364, coef 1( 45), 2(168) */
  26407,                               /* Vector 1364, coef 3(103) Vector
                                        * 1365, Coef 1( 39) */
  -22939,                              /* Vector 1365, coef 2(166), 3(101) */
  10399,                               /* Vector 1366, coef 1( 40), 2(159) */
  26659,                               /* Vector 1366, coef 3(104) Vector
                                        * 1367, Coef 1( 35) */
  -24988,                              /* Vector 1367, coef 2(158), 3(100) */
  9382,                                /* Vector 1368, coef 1( 36), 2(166) */
  23835,                               /* Vector 1368, coef 3( 93) Vector
                                        * 1369, Coef 1( 27) */
  -23458,                              /* Vector 1369, coef 2(164), 3( 94) */
  7068,                                /* Vector 1370, coef 1( 27), 2(156) */
  26647,                               /* Vector 1370, coef 3(104) Vector
                                        * 1371, Coef 1( 23) */
  -25248,                              /* Vector 1371, coef 2(157), 3( 96) */
  11162,                               /* Vector 1372, coef 1( 43), 2(154) */
  27680,                               /* Vector 1372, coef 3(108) Vector
                                        * 1373, Coef 1( 32) */
  -25487,                              /* Vector 1373, coef 2(156), 3(113) */
  8869,                                /* Vector 1374, coef 1( 34), 2(165) */
  29470,                               /* Vector 1374, coef 3(115) Vector
                                        * 1375, Coef 1( 30) */
  -23440,                              /* Vector 1375, coef 2(164), 3(112) */
  8871,                                /* Vector 1376, coef 1( 34), 2(167) */
  31263,                               /* Vector 1376, coef 3(122) Vector
                                        * 1377, Coef 1( 31) */
  -23169,                              /* Vector 1377, coef 2(165), 3(127) */
  7861,                                /* Vector 1378, coef 1( 30), 2(181) */
  -31720,                              /* Vector 1378, coef 3(132) Vector
                                        * 1379, Coef 1( 24) */
  -16260,                              /* Vector 1379, coef 2(192), 3(124) */
  6836,                                /* Vector 1380, coef 1( 26), 2(180) */
  28181,                               /* Vector 1380, coef 3(110) Vector
                                        * 1381, Coef 1( 21) */
  -19346,                              /* Vector 1381, coef 2(180), 3(110) */
  6330,                                /* Vector 1382, coef 1( 24), 2(186) */
  29972,                               /* Vector 1382, coef 3(117) Vector
                                        * 1383, Coef 1( 20) */
  -17801,                              /* Vector 1383, coef 2(186), 3(119) */
  7855,                                /* Vector 1384, coef 1( 30), 2(175) */
  29728,                               /* Vector 1384, coef 3(116) Vector
                                        * 1385, Coef 1( 32) */
  -20609,                              /* Vector 1385, coef 2(175), 3(127) */
  7084,                                /* Vector 1386, coef 1( 27), 2(172) */
  32027,                               /* Vector 1386, coef 3(125) Vector
                                        * 1387, Coef 1( 27) */
  -19332,                              /* Vector 1387, coef 2(180), 3(124) */
  6321,                                /* Vector 1388, coef 1( 24), 2(177) */
  31513,                               /* Vector 1388, coef 3(123) Vector
                                        * 1389, Coef 1( 25) */
  -23950,                              /* Vector 1389, coef 2(162), 3(114) */
  5539,                                /* Vector 1390, coef 1( 21), 2(163) */
  27675,                               /* Vector 1390, coef 3(108) Vector
                                        * 1391, Coef 1( 27) */
  -22921,                              /* Vector 1391, coef 2(166), 3(119) */
  6051,                                /* Vector 1392, coef 1( 23), 2(163) */
  31254,                               /* Vector 1392, coef 3(122) Vector
                                        * 1393, Coef 1( 22) */
  -21386,                              /* Vector 1393, coef 2(172), 3(118) */
  5039,                                /* Vector 1394, coef 1( 19), 2(175) */
  30998,                               /* Vector 1394, coef 3(121) Vector
                                        * 1395, Coef 1( 22) */
  -20090,                              /* Vector 1395, coef 2(177), 3(134) */
  5289,                                /* Vector 1396, coef 1( 20), 2(169) */
  -31984,                              /* Vector 1396, coef 3(131) Vector
                                        * 1397, Coef 1( 16) */
  -22401,                              /* Vector 1397, coef 2(168), 3(127) */
  6585,                                /* Vector 1398, coef 1( 25), 2(185) */
  -31467,                              /* Vector 1398, coef 3(133) Vector
                                        * 1399, Coef 1( 21) */
  -18816,                              /* Vector 1399, coef 2(182), 3(128) */
  4792,                                /* Vector 1400, coef 1( 18), 2(184) */
  32020,                               /* Vector 1400, coef 3(125) Vector
                                        * 1401, Coef 1( 20) */
  -18037,                              /* Vector 1401, coef 2(185), 3(139) */
  4530,                                /* Vector 1402, coef 1( 17), 2(178) */
  -30952,                              /* Vector 1402, coef 3(135) Vector
                                        * 1403, Coef 1( 24) */
  -23677,                              /* Vector 1403, coef 2(163), 3(131) */
  5023,                                /* Vector 1404, coef 1( 19), 2(159) */
  -31980,                              /* Vector 1404, coef 3(131) Vector
                                        * 1405, Coef 1( 20) */
  -21617,                              /* Vector 1405, coef 2(171), 3(143) */
  7084,                                /* Vector 1406, coef 1( 27), 2(172) */
  28439,                               /* Vector 1406, coef 3(111) Vector
                                        * 1407, Coef 1( 23) */
  -21400,                              /* Vector 1407, coef 2(172), 3(104) */
  10690,                               /* Vector 1408, coef 1( 41), 2(194) */
  -29402,                              /* Vector 1408, coef 3(141) Vector
                                        * 1409, Coef 1( 38) */
  -17518,                              /* Vector 1409, coef 2(187), 3(146) */
  9146,                                /* Vector 1410, coef 1( 35), 2(186) */
  -29904,                              /* Vector 1410, coef 3(139) Vector
                                        * 1411, Coef 1( 48) */
  -16988,                              /* Vector 1411, coef 2(189), 3(164) */
  12463,                               /* Vector 1412, coef 1( 48), 2(175) */
  -24533,                              /* Vector 1412, coef 3(160) Vector
                                        * 1413, Coef 1( 43) */
  -13149,                              /* Vector 1413, coef 2(204), 3(163) */
  10447,                               /* Vector 1414, coef 1( 40), 2(207) */
  -24018,                              /* Vector 1414, coef 3(162) Vector
                                        * 1415, Coef 1( 46) */
  -18509,                              /* Vector 1415, coef 2(183), 3(179) */
  11192,                               /* Vector 1416, coef 1( 43), 2(184) */
  -21201,                              /* Vector 1416, coef 3(173) Vector
                                        * 1417, Coef 1( 47) */
  -24153,                              /* Vector 1417, coef 2(161), 3(167) */
  11424,                               /* Vector 1418, coef 1( 44), 2(160) */
  -24021,                              /* Vector 1418, coef 3(162) Vector
                                        * 1419, Coef 1( 43) */
  -17766,                              /* Vector 1419, coef 2(186), 3(154) */
  10678,                               /* Vector 1420, coef 1( 41), 2(182) */
  -25050,                              /* Vector 1420, coef 3(158) Vector
                                        * 1421, Coef 1( 38) */
  -18020,                              /* Vector 1421, coef 2(185), 3(156) */
  9142,                                /* Vector 1422, coef 1( 35), 2(182) */
  -25561,                              /* Vector 1422, coef 3(156) Vector
                                        * 1423, Coef 1( 39) */
  -13417,                              /* Vector 1423, coef 2(203), 3(151) */
  9673,                                /* Vector 1424, coef 1( 37), 2(201) */
  -27355,                              /* Vector 1424, coef 3(149) Vector
                                        * 1425, Coef 1( 37) */
  -9837,                               /* Vector 1425, coef 2(217), 3(147) */
  9175,                                /* Vector 1426, coef 1( 35), 2(215) */
  -26844,                              /* Vector 1426, coef 3(151) Vector
                                        * 1427, Coef 1( 36) */
  -12915,                              /* Vector 1427, coef 2(205), 3(141) */
  9929,                                /* Vector 1428, coef 1( 38), 2(201) */
  -23261,                              /* Vector 1428, coef 3(165) Vector
                                        * 1429, Coef 1( 35) */
  -15445,                              /* Vector 1429, coef 2(195), 3(171) */
  8910,                                /* Vector 1430, coef 1( 34), 2(206) */
  -23775,                              /* Vector 1430, coef 3(163) Vector
                                        * 1431, Coef 1( 33) */
  -13155,                              /* Vector 1431, coef 2(204), 3(157) */
  8378,                                /* Vector 1432, coef 1( 32), 2(186) */
  -24536,                              /* Vector 1432, coef 3(160) Vector
                                        * 1433, Coef 1( 40) */
  -21610,                              /* Vector 1433, coef 2(171), 3(150) */
  10149,                               /* Vector 1434, coef 1( 39), 2(165) */
  -24277,                              /* Vector 1434, coef 3(161) Vector
                                        * 1435, Coef 1( 43) */
  -23363,                              /* Vector 1435, coef 2(164), 3(189) */
  10665,                               /* Vector 1436, coef 1( 41), 2(169) */
  -21469,                              /* Vector 1436, coef 3(172) Vector
                                        * 1437, Coef 1( 35) */
  -17736,                              /* Vector 1437, coef 2(186), 3(184) */
  9134,                                /* Vector 1438, coef 1( 35), 2(174) */
  -21214,                              /* Vector 1438, coef 3(173) Vector
                                        * 1439, Coef 1( 34) */
  -16236,                              /* Vector 1439, coef 2(192), 3(148) */
  8593,                                /* Vector 1440, coef 1( 33), 2(145) */
  -29413,                              /* Vector 1440, coef 3(141) Vector
                                        * 1441, Coef 1( 27) */
  -29041,                              /* Vector 1441, coef 2(142), 3(143) */
  10645,                               /* Vector 1442, coef 1( 41), 2(149) */
  -27099,                              /* Vector 1442, coef 3(150) Vector
                                        * 1443, Coef 1( 37) */
  -27237,                              /* Vector 1443, coef 2(149), 3(155) */
  8847,                                /* Vector 1444, coef 1( 34), 2(143) */
  -27100,                              /* Vector 1444, coef 3(150) Vector
                                        * 1445, Coef 1( 36) */
  -21632,                              /* Vector 1445, coef 2(171), 3(128) */
  9392,                                /* Vector 1446, coef 1( 36), 2(176) */
  -29151,                              /* Vector 1446, coef 3(142) Vector
                                        * 1447, Coef 1( 33) */
  -20336,                              /* Vector 1447, coef 2(176), 3(144) */
  8883,                                /* Vector 1448, coef 1( 34), 2(179) */
  -30947,                              /* Vector 1448, coef 3(135) Vector
                                        * 1449, Coef 1( 29) */
  -20847,                              /* Vector 1449, coef 2(174), 3(145) */
  7089,                                /* Vector 1450, coef 1( 27), 2(177) */
  -30182,                              /* Vector 1450, coef 3(138) Vector
                                        * 1451, Coef 1( 26) */
  -19055,                              /* Vector 1451, coef 2(181), 3(145) */
  6827,                                /* Vector 1452, coef 1( 26), 2(171) */
  -30946,                              /* Vector 1452, coef 3(135) Vector
                                        * 1453, Coef 1( 30) */
  -18274,                              /* Vector 1453, coef 2(184), 3(158) */
  11426,                               /* Vector 1454, coef 1( 44), 2(162) */
  -27350,                              /* Vector 1454, coef 3(149) Vector
                                        * 1455, Coef 1( 42) */
  -25717,                              /* Vector 1455, coef 2(155), 3(139) */
  9881,                                /* Vector 1456, coef 1( 38), 2(153) */
  -30680,                              /* Vector 1456, coef 3(136) Vector
                                        * 1457, Coef 1( 40) */
  -22641,                              /* Vector 1457, coef 2(167), 3(143) */
  9638,                                /* Vector 1458, coef 1( 37), 2(166) */
  -26846,                              /* Vector 1458, coef 3(151) Vector
                                        * 1459, Coef 1( 34) */
  -24430,                              /* Vector 1459, coef 2(160), 3(146) */
  9123,                                /* Vector 1460, coef 1( 35), 2(163) */
  -24545,                              /* Vector 1460, coef 3(160) Vector
                                        * 1461, Coef 1( 31) */
  -23138,                              /* Vector 1461, coef 2(165), 3(158) */
  9635,                                /* Vector 1462, coef 1( 37), 2(163) */
  -31199,                              /* Vector 1462, coef 3(134) Vector
                                        * 1463, Coef 1( 33) */
  -23417,                              /* Vector 1463, coef 2(164), 3(135) */
  8346,                                /* Vector 1464, coef 1( 32), 2(154) */
  -30947,                              /* Vector 1464, coef 3(135) Vector
                                        * 1465, Coef 1( 29) */
  -27002,                              /* Vector 1465, coef 2(150), 3(134) */
  7842,                                /* Vector 1466, coef 1( 30), 2(162) */
  -28134,                              /* Vector 1466, coef 3(146) Vector
                                        * 1467, Coef 1( 26) */
  -23923,                              /* Vector 1467, coef 2(162), 3(141) */
  6811,                                /* Vector 1468, coef 1( 26), 2(155) */
  -27620,                              /* Vector 1468, coef 3(148) Vector
                                        * 1469, Coef 1( 28) */
  -23674,                              /* Vector 1469, coef 2(163), 3(134) */
  6042,                                /* Vector 1470, coef 1( 23), 2(154) */
  -31721,                              /* Vector 1470, coef 3(132) Vector
                                        * 1471, Coef 1( 23) */
  -22381,                              /* Vector 1471, coef 2(168), 3(147) */
  7821,                                /* Vector 1472, coef 1( 30), 2(141) */
  -27113,                              /* Vector 1472, coef 3(150) Vector
                                        * 1473, Coef 1( 23) */
  -19311,                              /* Vector 1473, coef 2(180), 3(145) */
  7332,                                /* Vector 1474, coef 1( 28), 2(164) */
  -25834,                              /* Vector 1474, coef 3(155) Vector
                                        * 1475, Coef 1( 22) */
  -26478,                              /* Vector 1475, coef 2(152), 3(146) */
  8359,                                /* Vector 1476, coef 1( 32), 2(167) */
  -17124,                              /* Vector 1476, coef 3(189) Vector
                                        * 1477, Coef 1( 28) */
  -19011,                              /* Vector 1477, coef 2(181), 3(189) */
  7852,                                /* Vector 1478, coef 1( 30), 2(172) */
  -21988,                              /* Vector 1478, coef 3(170) Vector
                                        * 1479, Coef 1( 28) */
  -22864,                              /* Vector 1479, coef 2(166), 3(176) */
  8598,                                /* Vector 1480, coef 1( 33), 2(150) */
  -23524,                              /* Vector 1480, coef 3(164) Vector
                                        * 1481, Coef 1( 28) */
  -26720,                              /* Vector 1481, coef 2(151), 3(160) */
  6811,                                /* Vector 1482, coef 1( 26), 2(155) */
  -21736,                              /* Vector 1482, coef 3(171) Vector
                                        * 1483, Coef 1( 24) */
  -25177,                              /* Vector 1483, coef 2(157), 3(167) */
  7052,                                /* Vector 1484, coef 1( 27), 2(140) */
  -25321,                              /* Vector 1484, coef 3(157) Vector
                                        * 1485, Coef 1( 23) */
  -29277,                              /* Vector 1485, coef 2(141), 3(163) */
  5793,                                /* Vector 1486, coef 1( 22), 2(161) */
  -24558,                              /* Vector 1486, coef 3(160) Vector
                                        * 1487, Coef 1( 18) */
  -23395,                              /* Vector 1487, coef 2(164), 3(157) */
  12167,                               /* Vector 1488, coef 1( 47), 2(135) */
  -22748,                              /* Vector 1488, coef 3(167) Vector
                                        * 1489, Coef 1( 36) */
  -31313,                              /* Vector 1489, coef 2(133), 3(175) */
  7812,                                /* Vector 1490, coef 1( 30), 2(132) */
  -21474,                              /* Vector 1490, coef 3(172) Vector
                                        * 1491, Coef 1( 30) */
  25775,                               /* Vector 1491, coef 2(100), 3(175) */
  5806,                                /* Vector 1492, coef 1( 22), 2(174) */
  -18921,                              /* Vector 1492, coef 3(182) Vector
                                        * 1493, Coef 1( 23) */
  -29243,                              /* Vector 1493, coef 2(141), 3(197) */
  5251,                                /* Vector 1494, coef 1( 20), 2(131) */
  -14312,                              /* Vector 1494, coef 3(200) Vector
                                        * 1495, Coef 1( 24) */
  -29002,                              /* Vector 1495, coef 2(142), 3(182) */
  5761,                                /* Vector 1496, coef 1( 22), 2(129) */
  -21997,                              /* Vector 1496, coef 3(170) Vector
                                        * 1497, Coef 1( 19) */
  -23373,                              /* Vector 1497, coef 2(164), 3(179) */
  4260,                                /* Vector 1498, coef 1( 16), 2(164) */
  -20711,                              /* Vector 1498, coef 3(175) Vector
                                        * 1499, Coef 1( 25) */
  -19043,                              /* Vector 1499, coef 2(181), 3(157) */
  6574,                                /* Vector 1500, coef 1( 25), 2(174) */
  -24042,                              /* Vector 1500, coef 3(162) Vector
                                        * 1501, Coef 1( 22) */
  -20581,                              /* Vector 1501, coef 2(175), 3(155) */
  6077,                                /* Vector 1502, coef 1( 23), 2(189) */
  -19946,                              /* Vector 1502, coef 3(178) Vector
                                        * 1503, Coef 1( 22) */
  -19540,                              /* Vector 1503, coef 2(179), 3(172) */
  4516,                                /* Vector 1504, coef 1( 17), 2(164) */
  -28404,                              /* Vector 1504, coef 3(145) Vector
                                        * 1505, Coef 1( 12) */
  -22631,                              /* Vector 1505, coef 2(167), 3(153) */
  4796,                                /* Vector 1506, coef 1( 18), 2(188) */
  -16103,                              /* Vector 1506, coef 3(193) Vector
                                        * 1507, Coef 1( 25) */
  -14936,                              /* Vector 1507, coef 2(197), 3(168) */
  6088,                                /* Vector 1508, coef 1( 23), 2(200) */
  -23274,                              /* Vector 1508, coef 3(165) Vector
                                        * 1509, Coef 1( 22) */
  -16485,                              /* Vector 1509, coef 2(191), 3(155) */
  5307,                                /* Vector 1510, coef 1( 20), 2(187) */
  -27630,                              /* Vector 1510, coef 3(148) Vector
                                        * 1511, Coef 1( 18) */
  -15722,                              /* Vector 1511, coef 2(194), 3(150) */
  5307,                                /* Vector 1512, coef 1( 20), 2(187) */
  -24046,                              /* Vector 1512, coef 3(162) Vector
                                        * 1513, Coef 1( 18) */
  -19805,                              /* Vector 1513, coef 2(178), 3(163) */
  4537,                                /* Vector 1514, coef 1( 17), 2(185) */
  -26098,                              /* Vector 1514, coef 3(154) Vector
                                        * 1515, Coef 1( 14) */
  -17769,                              /* Vector 1515, coef 2(186), 3(151) */
  6342,                                /* Vector 1516, coef 1( 24), 2(198) */
  -19948,                              /* Vector 1516, coef 3(178) Vector
                                        * 1517, Coef 1( 20) */
  -17492,                              /* Vector 1517, coef 2(187), 3(172) */
  5582,                                /* Vector 1518, coef 1( 21), 2(206) */
  -22765,                              /* Vector 1518, coef 3(167) Vector
                                        * 1519, Coef 1( 19) */
  -13141,                              /* Vector 1519, coef 2(204), 3(171) */
  4818,                                /* Vector 1520, coef 1( 18), 2(210) */
  -23792,                              /* Vector 1520, coef 3(163) Vector
                                        * 1521, Coef 1( 16) */
  -12380,                              /* Vector 1521, coef 2(207), 3(164) */
  4820,                                /* Vector 1522, coef 1( 18), 2(212) */
  -19696,                              /* Vector 1522, coef 3(179) Vector
                                        * 1523, Coef 1( 16) */
  -10063,                              /* Vector 1523, coef 2(216), 3(177) */
  4038,                                /* Vector 1524, coef 1( 15), 2(198) */
  -16883,                              /* Vector 1524, coef 3(190) Vector
                                        * 1525, Coef 1( 13) */
  -19521,                              /* Vector 1525, coef 2(179), 3(191) */
  4288,                                /* Vector 1526, coef 1( 16), 2(192) */
  -21235,                              /* Vector 1526, coef 3(173) Vector
                                        * 1527, Coef 1( 13) */
  -18773,                              /* Vector 1527, coef 2(182), 3(171) */
  3277,                                /* Vector 1528, coef 1( 12), 2(205) */
  -23275,                              /* Vector 1528, coef 3(165) Vector
                                        * 1529, Coef 1( 21) */
  -13160,                              /* Vector 1529, coef 2(204), 3(152) */
  5071,                                /* Vector 1530, coef 1( 19), 2(207) */
  -27629,                              /* Vector 1530, coef 3(148) Vector
                                        * 1531, Coef 1( 19) */
  -9575,                               /* Vector 1531, coef 2(218), 3(153) */
  4830,                                /* Vector 1532, coef 1( 18), 2(222) */
  -25840,                              /* Vector 1532, coef 3(155) Vector
                                        * 1533, Coef 1( 16) */
  -11112,                              /* Vector 1533, coef 2(212), 3(152) */
  3800,                                /* Vector 1534, coef 1( 14), 2(216) */
  -24305,                              /* Vector 1534, coef 3(161) Vector
                                        * 1535, Coef 1( 15) */
  -7269,                               /* Vector 1535, coef 2(227), 3(155) */
  15302,                               /* Vector 1536, coef 1( 59), 2(198) */
  30786,                               /* Vector 1536, coef 3(120) Vector
                                        * 1537, Coef 1( 66) */
  -13468,                              /* Vector 1537, coef 2(203), 3(100) */
  15302,                               /* Vector 1538, coef 1( 59), 2(198) */
  27191,                               /* Vector 1538, coef 3(106) Vector
                                        * 1539, Coef 1( 55) */
  -8844,                               /* Vector 1539, coef 2(221), 3(116) */
  13782,                               /* Vector 1540, coef 1( 53), 2(214) */
  28230,                               /* Vector 1540, coef 3(110) Vector
                                        * 1541, Coef 1( 70) */
  -10124,                              /* Vector 1541, coef 2(216), 3(116) */
  17364,                               /* Vector 1542, coef 1( 67), 2(212) */
  29759,                               /* Vector 1542, coef 3(116) Vector
                                        * 1543, Coef 1( 63) */
  -13709,                              /* Vector 1543, coef 2(202), 3(115) */
  17114,                               /* Vector 1544, coef 1( 66), 2(218) */
  23104,                               /* Vector 1544, coef 3( 90) Vector
                                        * 1545, Coef 1( 64) */
  -9879,                               /* Vector 1545, coef 2(217), 3(105) */
  16084,                               /* Vector 1546, coef 1( 62), 2(212) */
  26687,                               /* Vector 1546, coef 3(104) Vector
                                        * 1547, Coef 1( 63) */
  -12701,                              /* Vector 1547, coef 2(206), 3( 99) */
  15056,                               /* Vector 1548, coef 1( 58), 2(208) */
  24897,                               /* Vector 1548, coef 3( 97) Vector
                                        * 1549, Coef 1( 65) */
  -11384,                              /* Vector 1549, coef 2(211), 3(136) */
  15574,                               /* Vector 1550, coef 1( 60), 2(214) */
  -30401,                              /* Vector 1550, coef 3(137) Vector
                                        * 1551, Coef 1( 63) */
  -10376,                              /* Vector 1551, coef 2(215), 3(120) */
  15570,                               /* Vector 1552, coef 1( 60), 2(210) */
  30267,                               /* Vector 1552, coef 3(118) Vector
                                        * 1553, Coef 1( 59) */
  -9095,                               /* Vector 1553, coef 2(220), 3(121) */
  14806,                               /* Vector 1554, coef 1( 57), 2(214) */
  29242,                               /* Vector 1554, coef 3(114) Vector
                                        * 1555, Coef 1( 58) */
  -11392,                              /* Vector 1555, coef 2(211), 3(128) */
  14285,                               /* Vector 1556, coef 1( 55), 2(205) */
  31798,                               /* Vector 1556, coef 3(124) Vector
                                        * 1557, Coef 1( 54) */
  -10879,                              /* Vector 1557, coef 2(213), 3(129) */
  15569,                               /* Vector 1558, coef 1( 60), 2(209) */
  27705,                               /* Vector 1558, coef 3(108) Vector
                                        * 1559, Coef 1( 57) */
  -12693,                              /* Vector 1559, coef 2(206), 3(107) */
  14281,                               /* Vector 1560, coef 1( 55), 2(201) */
  29751,                               /* Vector 1560, coef 3(116) Vector
                                        * 1561, Coef 1( 55) */
  -14488,                              /* Vector 1561, coef 2(199), 3(104) */
  16097,                               /* Vector 1562, coef 1( 62), 2(225) */
  26684,                               /* Vector 1562, coef 3(104) Vector
                                        * 1563, Coef 1( 60) */
  -9372,                               /* Vector 1563, coef 2(219), 3(100) */
  15074,                               /* Vector 1564, coef 1( 58), 2(226) */
  24888,                               /* Vector 1564, coef 3( 97) Vector
                                        * 1565, Coef 1( 56) */
  -10654,                              /* Vector 1565, coef 2(214), 3( 98) */
  16868,                               /* Vector 1566, coef 1( 65), 2(228) */
  21053,                               /* Vector 1566, coef 3( 82) Vector
                                        * 1567, Coef 1( 61) */
  -9900,                               /* Vector 1567, coef 2(217), 3( 84) */
  14016,                               /* Vector 1568, coef 1( 54), 2(192) */
  25650,                               /* Vector 1568, coef 3(100) Vector
                                        * 1569, Coef 1( 50) */
  -18839,                              /* Vector 1569, coef 2(182), 3(105) */
  11959,                               /* Vector 1570, coef 1( 46), 2(183) */
  26408,                               /* Vector 1570, coef 3(103) Vector
                                        * 1571, Coef 1( 40) */
  -19352,                              /* Vector 1571, coef 2(180), 3(104) */
  12233,                               /* Vector 1572, coef 1( 47), 2(201) */
  29480,                               /* Vector 1572, coef 3(115) Vector
                                        * 1573, Coef 1( 40) */
  -14740,                              /* Vector 1573, coef 2(198), 3(108) */
  11698,                               /* Vector 1574, coef 1( 45), 2(178) */
  28214,                               /* Vector 1574, coef 3(110) Vector
                                        * 1575, Coef 1( 54) */
  -13220,                              /* Vector 1575, coef 2(204), 3( 92) */
  13510,                               /* Vector 1576, coef 1( 52), 2(198) */
  32049,                               /* Vector 1576, coef 3(125) Vector
                                        * 1577, Coef 1( 49) */
  -15748,                              /* Vector 1577, coef 2(194), 3(124) */
  13515,                               /* Vector 1578, coef 1( 52), 2(203) */
  29235,                               /* Vector 1578, coef 3(114) Vector
                                        * 1579, Coef 1( 51) */
  -15004,                              /* Vector 1579, coef 2(197), 3(100) */
  12735,                               /* Vector 1580, coef 1( 49), 2(191) */
  27950,                               /* Vector 1580, coef 3(109) Vector
                                        * 1581, Coef 1( 46) */
  -15767,                              /* Vector 1581, coef 2(194), 3(105) */
  13259,                               /* Vector 1582, coef 1( 51), 2(203) */
  27183,                               /* Vector 1582, coef 3(106) Vector
                                        * 1583, Coef 1( 47) */
  -13979,                              /* Vector 1583, coef 2(201), 3(101) */
  12488,                               /* Vector 1584, coef 1( 48), 2(200) */
  23851,                               /* Vector 1584, coef 3( 93) Vector
                                        * 1585, Coef 1( 43) */
  -15266,                              /* Vector 1585, coef 2(196), 3( 94) */
  12735,                               /* Vector 1586, coef 1( 49), 2(191) */
  24877,                               /* Vector 1586, coef 3( 97) Vector
                                        * 1587, Coef 1( 45) */
  -17057,                              /* Vector 1587, coef 2(189), 3( 95) */
  11961,                               /* Vector 1588, coef 1( 46), 2(185) */
  29738,                               /* Vector 1588, coef 3(116) Vector
                                        * 1589, Coef 1( 42) */
  -18574,                              /* Vector 1589, coef 2(183), 3(114) */
  11458,                               /* Vector 1590, coef 1( 44), 2(194) */
  29992,                               /* Vector 1590, coef 3(117) Vector
                                        * 1591, Coef 1( 40) */
  -16013,                              /* Vector 1591, coef 2(193), 3(115) */
  11197,                               /* Vector 1592, coef 1( 43), 2(189) */
  26919,                               /* Vector 1592, coef 3(105) Vector
                                        * 1593, Coef 1( 39) */
  -17046,                              /* Vector 1593, coef 2(189), 3(106) */
  14279,                               /* Vector 1594, coef 1( 55), 2(199) */
  23089,                               /* Vector 1594, coef 3( 90) Vector
                                        * 1595, Coef 1( 49) */
  -16039,                              /* Vector 1595, coef 2(193), 3( 89) */
  13244,                               /* Vector 1596, coef 1( 51), 2(188) */
  22575,                               /* Vector 1596, coef 3( 88) Vector
                                        * 1597, Coef 1( 47) */
  -15020,                              /* Vector 1597, coef 2(197), 3( 84) */
  11209,                               /* Vector 1598, coef 1( 43), 2(201) */
  27175,                               /* Vector 1598, coef 3(106) Vector
                                        * 1599, Coef 1( 39) */
  -15521,                              /* Vector 1599, coef 2(195), 3( 95) */
  17606,                               /* Vector 1600, coef 1( 68), 2(198) */
  17467,                               /* Vector 1600, coef 3( 68) Vector
                                        * 1601, Coef 1( 59) */
  -16828,                              /* Vector 1601, coef 2(190), 3( 68) */
  17095,                               /* Vector 1602, coef 1( 66), 2(199) */
  14911,                               /* Vector 1602, coef 3( 58) Vector
                                        * 1603, Coef 1( 63) */
  -14802,                              /* Vector 1603, coef 2(198), 3( 46) */
  14263,                               /* Vector 1604, coef 1( 55), 2(183) */
  16201,                               /* Vector 1604, coef 3( 63) Vector
                                        * 1605, Coef 1( 73) */
  -7320,                               /* Vector 1605, coef 2(227), 3(104) */
  17885,                               /* Vector 1606, coef 1( 69), 2(221) */
  24893,                               /* Vector 1606, coef 3( 97) Vector
                                        * 1607, Coef 1( 61) */
  -12965,                              /* Vector 1607, coef 2(205), 3( 91) */
  20439,                               /* Vector 1608, coef 1( 79), 2(215) */
  20810,                               /* Vector 1608, coef 3( 81) Vector
                                        * 1609, Coef 1( 74) */
  -12465,                              /* Vector 1609, coef 2(207), 3( 79) */
  17872,                               /* Vector 1610, coef 1( 69), 2(208) */
  21825,                               /* Vector 1610, coef 3( 85) Vector
                                        * 1611, Coef 1( 65) */
  -15281,                              /* Vector 1611, coef 2(196), 3( 79) */
  16068,                               /* Vector 1612, coef 1( 62), 2(196) */
  23097,                               /* Vector 1612, coef 3( 90) Vector
                                        * 1613, Coef 1( 57) */
  -16040,                              /* Vector 1613, coef 2(193), 3( 88) */
  15037,                               /* Vector 1614, coef 1( 58), 2(189) */
  21307,                               /* Vector 1614, coef 3( 83) Vector
                                        * 1615, Coef 1( 59) */
  -15285,                              /* Vector 1615, coef 2(196), 3( 75) */
  14017,                               /* Vector 1616, coef 1( 54), 2(193) */
  17459,                               /* Vector 1616, coef 3( 68) Vector
                                        * 1617, Coef 1( 51) */
  -17853,                              /* Vector 1617, coef 2(186), 3( 67) */
  10673,                               /* Vector 1618, coef 1( 41), 2(177) */
  16712,                               /* Vector 1618, coef 3( 65) Vector
                                        * 1619, Coef 1( 72) */
  -8362,                               /* Vector 1619, coef 2(223), 3( 86) */
  17361,                               /* Vector 1620, coef 1( 67), 2(209) */
  19775,                               /* Vector 1620, coef 3( 77) Vector
                                        * 1621, Coef 1( 63) */
  -11955,                              /* Vector 1621, coef 2(209), 3( 77) */
  14794,                               /* Vector 1622, coef 1( 57), 2(202) */
  17729,                               /* Vector 1622, coef 3( 69) Vector
                                        * 1623, Coef 1( 65) */
  -10174,                              /* Vector 1623, coef 2(216), 3( 66) */
  15827,                               /* Vector 1624, coef 1( 61), 2(211) */
  13371,                               /* Vector 1624, coef 3( 52) Vector
                                        * 1625, Coef 1( 59) */
  -14280,                              /* Vector 1625, coef 2(200), 3( 56) */
  13764,                               /* Vector 1626, coef 1( 53), 2(196) */
  11315,                               /* Vector 1626, coef 3( 44) Vector
                                        * 1627, Coef 1( 51) */
  -14270,                              /* Vector 1627, coef 2(200), 3( 66) */
  10940,                               /* Vector 1628, coef 1( 42), 2(188) */
  16182,                               /* Vector 1628, coef 3( 63) Vector
                                        * 1629, Coef 1( 54) */
  -15281,                              /* Vector 1629, coef 2(196), 3( 79) */
  12737,                               /* Vector 1630, coef 1( 49), 2(193) */
  19768,                               /* Vector 1630, coef 3( 77) Vector
                                        * 1631, Coef 1( 56) */
  -12209,                              /* Vector 1631, coef 2(208), 3( 79) */
  10686,                               /* Vector 1632, coef 1( 41), 2(190) */
  23844,                               /* Vector 1632, coef 3( 93) Vector
                                        * 1633, Coef 1( 36) */
  -17575,                              /* Vector 1633, coef 2(187), 3( 89) */
  9656,                                /* Vector 1634, coef 1( 37), 2(184) */
  26146,                               /* Vector 1634, coef 3(102) Vector
                                        * 1635, Coef 1( 34) */
  -19100,                              /* Vector 1635, coef 2(181), 3(100) */
  7859,                                /* Vector 1636, coef 1( 30), 2(179) */
  27177,                               /* Vector 1636, coef 3(106) Vector
                                        * 1637, Coef 1( 41) */
  -17335,                              /* Vector 1637, coef 2(188), 3( 73) */
  8120,                                /* Vector 1638, coef 1( 31), 2(184) */
  17444,                               /* Vector 1638, coef 3( 68) Vector
                                        * 1639, Coef 1( 36) */
  -19149,                              /* Vector 1639, coef 2(181), 3( 51) */
  12218,                               /* Vector 1640, coef 1( 47), 2(186) */
  20780,                               /* Vector 1640, coef 3( 81) Vector
                                        * 1641, Coef 1( 44) */
  -19373,                              /* Vector 1641, coef 2(180), 3( 83) */
  10163,                               /* Vector 1642, coef 1( 39), 2(179) */
  19243,                               /* Vector 1642, coef 3( 75) Vector
                                        * 1643, Coef 1( 43) */
  -16049,                              /* Vector 1643, coef 2(193), 3( 79) */
  9149,                                /* Vector 1644, coef 1( 35), 2(189) */
  19753,                               /* Vector 1644, coef 3( 77) Vector
                                        * 1645, Coef 1( 41) */
  -18089,                              /* Vector 1645, coef 2(185), 3( 87) */
  9911,                                /* Vector 1646, coef 1( 38), 2(183) */
  21022,                               /* Vector 1646, coef 3( 82) Vector
                                        * 1647, Coef 1( 30) */
  -18857,                              /* Vector 1647, coef 2(182), 3( 87) */
  6322,                                /* Vector 1648, coef 1( 24), 2(178) */
  20266,                               /* Vector 1648, coef 3( 79) Vector
                                        * 1649, Coef 1( 42) */
  -18593,                              /* Vector 1649, coef 2(183), 3( 95) */
  9138,                                /* Vector 1650, coef 1( 35), 2(178) */
  24093,                               /* Vector 1650, coef 3( 94) Vector
                                        * 1651, Coef 1( 29) */
  -20895,                              /* Vector 1651, coef 2(174), 3( 97) */
  9646,                                /* Vector 1652, coef 1( 37), 2(174) */
  21789,                               /* Vector 1652, coef 3( 85) Vector
                                        * 1653, Coef 1( 29) */
  -20652,                              /* Vector 1653, coef 2(175), 3( 84) */
  7079,                                /* Vector 1654, coef 1( 27), 2(167) */
  19491,                               /* Vector 1654, coef 3( 76) Vector
                                        * 1655, Coef 1( 35) */
  -16540,                              /* Vector 1655, coef 2(191), 3(100) */
  8125,                                /* Vector 1656, coef 1( 31), 2(189) */
  24863,                               /* Vector 1656, coef 3( 97) Vector
                                        * 1657, Coef 1( 31) */
  -18592,                              /* Vector 1657, coef 2(183), 3( 96) */
  7094,                                /* Vector 1658, coef 1( 27), 2(182) */
  25112,                               /* Vector 1658, coef 3( 98) Vector
                                        * 1659, Coef 1( 24) */
  -17817,                              /* Vector 1659, coef 2(186), 3(103) */
  5812,                                /* Vector 1660, coef 1( 22), 2(180) */
  24082,                               /* Vector 1660, coef 3( 94) Vector
                                        * 1661, Coef 1( 18) */
  -21413,                              /* Vector 1661, coef 2(172), 3( 91) */
  8641,                                /* Vector 1662, coef 1( 33), 2(193) */
  22040,                               /* Vector 1662, coef 3( 86) Vector
                                        * 1663, Coef 1( 24) */
  -17579,                              /* Vector 1663, coef 2(187), 3( 85) */
  13530,                               /* Vector 1664, coef 1( 52), 2(218) */
  31538,                               /* Vector 1664, coef 3(123) Vector
                                        * 1665, Coef 1( 50) */
  -10894,                              /* Vector 1665, coef 2(213), 3(114) */
  11734,                               /* Vector 1666, coef 1( 45), 2(214) */
  27703,                               /* Vector 1666, coef 3(108) Vector
                                        * 1667, Coef 1( 55) */
  -8604,                               /* Vector 1667, coef 2(222), 3(100) */
  13777,                               /* Vector 1668, coef 1( 53), 2(209) */
  24371,                               /* Vector 1668, coef 3( 95) Vector
                                        * 1669, Coef 1( 51) */
  -10654,                              /* Vector 1669, coef 2(214), 3( 98) */
  12496,                               /* Vector 1670, coef 1( 48), 2(208) */
  24117,                               /* Vector 1670, coef 3( 94) Vector
                                        * 1671, Coef 1( 53) */
  -7837,                               /* Vector 1671, coef 2(225), 3( 99) */
  13027,                               /* Vector 1672, coef 1( 50), 2(227) */
  23601,                               /* Vector 1672, coef 3( 92) Vector
                                        * 1673, Coef 1( 49) */
  -9109,                               /* Vector 1673, coef 2(220), 3(107) */
  12248,                               /* Vector 1674, coef 1( 47), 2(216) */
  25393,                               /* Vector 1674, coef 3( 99) Vector
                                        * 1675, Coef 1( 49) */
  -9639,                               /* Vector 1675, coef 2(218), 3( 89) */
  12251,                               /* Vector 1676, coef 1( 47), 2(219) */
  22061,                               /* Vector 1676, coef 3( 86) Vector
                                        * 1677, Coef 1( 45) */
  -11174,                              /* Vector 1677, coef 2(212), 3( 90) */
  10959,                               /* Vector 1678, coef 1( 42), 2(207) */
  23344,                               /* Vector 1678, coef 3( 91) Vector
                                        * 1679, Coef 1( 48) */
  -11925,                              /* Vector 1679, coef 2(209), 3(107) */
  11726,                               /* Vector 1680, coef 1( 45), 2(206) */
  26153,                               /* Vector 1680, coef 3(102) Vector
                                        * 1681, Coef 1( 41) */
  -13210,                              /* Vector 1681, coef 2(204), 3(102) */
  11222,                               /* Vector 1682, coef 1( 43), 2(214) */
  25639,                               /* Vector 1682, coef 3(100) Vector
                                        * 1683, Coef 1( 39) */
  -11161,                              /* Vector 1683, coef 2(212), 3(103) */
  10192,                               /* Vector 1684, coef 1( 39), 2(208) */
  24618,                               /* Vector 1684, coef 3( 96) Vector
                                        * 1685, Coef 1( 42) */
  -13986,                              /* Vector 1685, coef 2(201), 3( 94) */
  13269,                               /* Vector 1686, coef 1( 51), 2(213) */
  21295,                               /* Vector 1686, coef 3( 83) Vector
                                        * 1687, Coef 1( 47) */
  -12973,                              /* Vector 1687, coef 2(205), 3( 83) */
  11466,                               /* Vector 1688, coef 1( 44), 2(202) */
  22059,                               /* Vector 1688, coef 3( 86) Vector
                                        * 1689, Coef 1( 43) */
  -12207,                              /* Vector 1689, coef 2(208), 3( 81) */
  14819,                               /* Vector 1690, coef 1( 57), 2(227) */
  21813,                               /* Vector 1690, coef 3( 85) Vector
                                        * 1691, Coef 1( 53) */
  -8876,                               /* Vector 1691, coef 2(221), 3( 84) */
  13794,                               /* Vector 1692, coef 1( 53), 2(226) */
  18737,                               /* Vector 1692, coef 3( 73) Vector
                                        * 1693, Coef 1( 49) */
  -9911,                               /* Vector 1693, coef 2(217), 3( 73) */
  12003,                               /* Vector 1694, coef 1( 46), 2(227) */
  25644,                               /* Vector 1694, coef 3(100) Vector
                                        * 1695, Coef 1( 44) */
  -8348,                               /* Vector 1695, coef 2(223), 3(100) */
  12229,                               /* Vector 1696, coef 1( 47), 2(197) */
  14888,                               /* Vector 1696, coef 3( 58) Vector
                                        * 1697, Coef 1( 40) */
  -15547,                              /* Vector 1697, coef 2(195), 3( 69) */
  9150,                                /* Vector 1698, coef 1( 35), 2(190) */
  15656,                               /* Vector 1698, coef 3( 61) Vector
                                        * 1699, Coef 1( 40) */
  -15281,                              /* Vector 1699, coef 2(196), 3( 79) */
  9415,                                /* Vector 1700, coef 1( 36), 2(199) */
  22588,                               /* Vector 1700, coef 3( 88) Vector
                                        * 1701, Coef 1( 60) */
  -7856,                               /* Vector 1701, coef 2(225), 3( 80) */
  15064,                               /* Vector 1702, coef 1( 58), 2(216) */
  18483,                               /* Vector 1702, coef 3( 72) Vector
                                        * 1703, Coef 1( 51) */
  -13492,                              /* Vector 1703, coef 2(203), 3( 76) */
  13520,                               /* Vector 1704, coef 1( 52), 2(208) */
  17966,                               /* Vector 1704, coef 3( 70) Vector
                                        * 1705, Coef 1( 46) */
  -14009,                              /* Vector 1705, coef 2(201), 3( 71) */
  14558,                               /* Vector 1706, coef 1( 56), 2(222) */
  18483,                               /* Vector 1706, coef 3( 72) Vector
                                        * 1707, Coef 1( 51) */
  -10938,                              /* Vector 1707, coef 2(213), 3( 70) */
  13274,                               /* Vector 1708, coef 1( 51), 2(218) */
  15406,                               /* Vector 1708, coef 3( 60) Vector
                                        * 1709, Coef 1( 46) */
  -11984,                              /* Vector 1709, coef 2(209), 3( 48) */
  15062,                               /* Vector 1710, coef 1( 58), 2(214) */
  14383,                               /* Vector 1710, coef 3( 56) Vector
                                        * 1711, Coef 1( 47) */
  -13510,                              /* Vector 1711, coef 2(203), 3( 58) */
  12754,                               /* Vector 1712, coef 1( 49), 2(210) */
  13349,                               /* Vector 1712, coef 3( 52) Vector
                                        * 1713, Coef 1( 37) */
  -15312,                              /* Vector 1713, coef 2(196), 3( 48) */
  10440,                               /* Vector 1714, coef 1( 40), 2(200) */
  20771,                               /* Vector 1714, coef 3( 81) Vector
                                        * 1715, Coef 1( 35) */
  -14258,                              /* Vector 1715, coef 2(200), 3( 78) */
  8135,                                /* Vector 1716, coef 1( 31), 2(199) */
  21038,                               /* Vector 1716, coef 3( 82) Vector
                                        * 1717, Coef 1( 46) */
  -12218,                              /* Vector 1717, coef 2(208), 3( 70) */
  10184,                               /* Vector 1718, coef 1( 39), 2(200) */
  17706,                               /* Vector 1718, coef 3( 69) Vector
                                        * 1719, Coef 1( 42) */
  -12993,                              /* Vector 1719, coef 2(205), 3( 63) */
  8904,                                /* Vector 1720, coef 1( 34), 2(200) */
  15395,                               /* Vector 1720, coef 3( 60) Vector
                                        * 1721, Coef 1( 35) */
  -12986,                              /* Vector 1721, coef 2(205), 3( 70) */
  7873,                                /* Vector 1722, coef 1( 30), 2(193) */
  18196,                               /* Vector 1722, coef 3( 71) Vector
                                        * 1723, Coef 1( 20) */
  -16565,                              /* Vector 1723, coef 2(191), 3( 75) */
  11476,                               /* Vector 1724, coef 1( 44), 2(212) */
  18474,                               /* Vector 1724, coef 3( 72) Vector
                                        * 1725, Coef 1( 42) */
  -11199,                              /* Vector 1725, coef 2(212), 3( 65) */
  11992,                               /* Vector 1726, coef 1( 46), 2(216) */
  13605,                               /* Vector 1726, coef 3( 53) Vector
                                        * 1727, Coef 1( 37) */
  -12487,                              /* Vector 1727, coef 2(207), 3( 57) */
  9686,                                /* Vector 1728, coef 1( 37), 2(214) */
  29219,                               /* Vector 1728, coef 3(114) Vector
                                        * 1729, Coef 1( 35) */
  -11149,                              /* Vector 1729, coef 2(212), 3(115) */
  8401,                                /* Vector 1730, coef 1( 32), 2(209) */
  29219,                               /* Vector 1730, coef 3(114) Vector
                                        * 1731, Coef 1( 35) */
  -7824,                               /* Vector 1731, coef 2(225), 3(112) */
  8670,                                /* Vector 1732, coef 1( 33), 2(222) */
  27681,                               /* Vector 1732, coef 3(108) Vector
                                        * 1733, Coef 1( 33) */
  -8351,                               /* Vector 1733, coef 2(223), 3( 97) */
  7900,                                /* Vector 1734, coef 1( 30), 2(220) */
  25636,                               /* Vector 1734, coef 3(100) Vector
                                        * 1735, Coef 1( 36) */
  -7297,                               /* Vector 1735, coef 2(227), 3(127) */
  9179,                                /* Vector 1736, coef 1( 35), 2(219) */
  32547,                               /* Vector 1736, coef 3(127) Vector
                                        * 1737, Coef 1( 35) */
  -11137,                              /* Vector 1737, coef 2(212), 3(127) */
  8408,                                /* Vector 1738, coef 1( 32), 2(216) */
  32545,                               /* Vector 1738, coef 3(127) Vector
                                        * 1739, Coef 1( 33) */
  -8841,                               /* Vector 1739, coef 2(221), 3(119) */
  8154,                                /* Vector 1740, coef 1( 31), 2(218) */
  28958,                               /* Vector 1740, coef 3(113) Vector
                                        * 1741, Coef 1( 30) */
  -10884,                              /* Vector 1741, coef 2(213), 3(124) */
  7378,                                /* Vector 1742, coef 1( 28), 2(210) */
  32033,                               /* Vector 1742, coef 3(125) Vector
                                        * 1743, Coef 1( 33) */
  -5775,                               /* Vector 1743, coef 2(233), 3(113) */
  8423,                                /* Vector 1744, coef 1( 32), 2(231) */
  29983,                               /* Vector 1744, coef 3(117) Vector
                                        * 1745, Coef 1( 31) */
  -4239,                               /* Vector 1745, coef 2(239), 3(113) */
  7658,                                /* Vector 1746, coef 1( 29), 2(234) */
  29470,                               /* Vector 1746, coef 3(115) Vector
                                        * 1747, Coef 1( 30) */
  -7065,                               /* Vector 1747, coef 2(228), 3(103) */
  7904,                                /* Vector 1748, coef 1( 30), 2(224) */
  30748,                               /* Vector 1748, coef 3(120) Vector
                                        * 1749, Coef 1( 28) */
  -8332,                               /* Vector 1749, coef 2(223), 3(116) */
  7136,                                /* Vector 1750, coef 1( 27), 2(224) */
  32025,                               /* Vector 1750, coef 3(125) Vector
                                        * 1751, Coef 1( 25) */
  -8584,                               /* Vector 1751, coef 2(222), 3(120) */
  6613,                                /* Vector 1752, coef 1( 25), 2(213) */
  31008,                               /* Vector 1752, coef 3(121) Vector
                                        * 1753, Coef 1( 32) */
  -11159,                              /* Vector 1753, coef 2(212), 3(105) */
  7893,                                /* Vector 1754, coef 1( 30), 2(213) */
  26657,                               /* Vector 1754, coef 3(104) Vector
                                        * 1755, Coef 1( 33) */
  -7282,                               /* Vector 1755, coef 2(227), 3(142) */
  8163,                                /* Vector 1756, coef 1( 31), 2(227) */
  -31200,                              /* Vector 1756, coef 3(134) Vector
                                        * 1757, Coef 1( 32) */
  -11129,                              /* Vector 1757, coef 2(212), 3(135) */
  7653,                                /* Vector 1758, coef 1( 29), 2(229) */
  -30949,                              /* Vector 1758, coef 3(135) Vector
                                        * 1759, Coef 1( 27) */
  -10130,                              /* Vector 1759, coef 2(216), 3(110) */
  6080,                                /* Vector 1760, coef 1( 23), 2(192) */
  27922,                               /* Vector 1760, coef 3(109) Vector
                                        * 1761, Coef 1( 18) */
  -17297,                              /* Vector 1761, coef 2(188), 3(111) */
  3769,                                /* Vector 1762, coef 1( 14), 2(185) */
  29202,                               /* Vector 1762, coef 3(114) Vector
                                        * 1763, Coef 1( 18) */
  -17822,                              /* Vector 1763, coef 2(186), 3( 98) */
  7104,                                /* Vector 1764, coef 1( 27), 2(192) */
  23831,                               /* Vector 1764, coef 3( 93) Vector
                                        * 1765, Coef 1( 23) */
  -10893,                              /* Vector 1765, coef 2(213), 3(115) */
  8396,                                /* Vector 1766, coef 1( 32), 2(204) */
  26653,                               /* Vector 1766, coef 3(104) Vector
                                        * 1767, Coef 1( 29) */
  -13977,                              /* Vector 1767, coef 2(201), 3(103) */
  7120,                                /* Vector 1768, coef 1( 27), 2(208) */
  26648,                               /* Vector 1768, coef 3(104) Vector
                                        * 1769, Coef 1( 24) */
  -13209,                              /* Vector 1769, coef 2(204), 3(103) */
  8652,                                /* Vector 1770, coef 1( 33), 2(204) */
  24350,                               /* Vector 1770, coef 3( 95) Vector
                                        * 1771, Coef 1( 30) */
  -14244,                              /* Vector 1771, coef 2(200), 3( 92) */
  7113,                                /* Vector 1772, coef 1( 27), 2(201) */
  22550,                               /* Vector 1772, coef 3( 88) Vector
                                        * 1773, Coef 1( 22) */
  -15017,                              /* Vector 1773, coef 2(197), 3( 87) */
  6854,                                /* Vector 1774, coef 1( 26), 2(198) */
  25365,                               /* Vector 1774, coef 3( 99) Vector
                                        * 1775, Coef 1( 21) */
  -15520,                              /* Vector 1775, coef 2(195), 3( 96) */
  5323,                                /* Vector 1776, coef 1( 20), 2(203) */
  24848,                               /* Vector 1776, coef 3( 97) Vector
                                        * 1777, Coef 1( 16) */
  -15008,                              /* Vector 1777, coef 2(197), 3( 96) */
  7113,                                /* Vector 1778, coef 1( 27), 2(201) */
  29211,                               /* Vector 1778, coef 3(114) Vector
                                        * 1779, Coef 1( 27) */
  -11917,                              /* Vector 1779, coef 2(209), 3(115) */
  6093,                                /* Vector 1780, coef 1( 23), 2(205) */
  29977,                               /* Vector 1780, coef 3(117) Vector
                                        * 1781, Coef 1( 25) */
  -14215,                              /* Vector 1781, coef 2(200), 3(121) */
  5570,                                /* Vector 1782, coef 1( 21), 2(194) */
  30225,                               /* Vector 1782, coef 3(118) Vector
                                        * 1783, Coef 1( 17) */
  -15753,                              /* Vector 1783, coef 2(194), 3(119) */
  6090,                                /* Vector 1784, coef 1( 23), 2(202) */
  28180,                               /* Vector 1784, coef 3(110) Vector
                                        * 1785, Coef 1( 20) */
  -14228,                              /* Vector 1785, coef 2(200), 3(108) */
  5072,                                /* Vector 1786, coef 1( 19), 2(208) */
  28187,                               /* Vector 1786, coef 3(110) Vector
                                        * 1787, Coef 1( 27) */
  -12452,                              /* Vector 1787, coef 2(207), 3( 92) */
  6610,                                /* Vector 1788, coef 1( 25), 2(210) */
  23577,                               /* Vector 1788, coef 3( 92) Vector
                                        * 1789, Coef 1( 25) */
  -10130,                              /* Vector 1789, coef 2(216), 3(110) */
  6102,                                /* Vector 1790, coef 1( 23), 2(214) */
  26133,                               /* Vector 1790, coef 3(102) Vector
                                        * 1791, Coef 1( 21) */
  -11160,                              /* Vector 1791, coef 2(212), 3(104) */
  9421,                                /* Vector 1792, coef 1( 36), 2(205) */
  24102,                               /* Vector 1792, coef 3( 94) Vector
                                        * 1793, Coef 1( 38) */
  -12718,                              /* Vector 1793, coef 2(206), 3( 82) */
  10205,                               /* Vector 1794, coef 1( 39), 2(221) */
  27174,                               /* Vector 1794, coef 3(106) Vector
                                        * 1795, Coef 1( 38) */
  -6555,                               /* Vector 1795, coef 2(230), 3(101) */
  9442,                                /* Vector 1796, coef 1( 36), 2(226) */
  25648,                               /* Vector 1796, coef 3(100) Vector
                                        * 1797, Coef 1( 48) */
  -8116,                               /* Vector 1797, coef 2(224), 3( 76) */
  11481,                               /* Vector 1798, coef 1( 44), 2(217) */
  18736,                               /* Vector 1798, coef 3( 73) Vector
                                        * 1799, Coef 1( 48) */
  -8387,                               /* Vector 1799, coef 2(223), 3( 61) */
  11484,                               /* Vector 1800, coef 1( 44), 2(220) */
  13350,                               /* Vector 1800, coef 3( 52) Vector
                                        * 1801, Coef 1( 38) */
  -11201,                              /* Vector 1801, coef 2(212), 3( 63) */
  11233,                               /* Vector 1802, coef 1( 43), 2(225) */
  23081,                               /* Vector 1802, coef 3( 90) Vector
                                        * 1803, Coef 1( 41) */
  -9633,                               /* Vector 1803, coef 2(218), 3( 95) */
  11226,                               /* Vector 1804, coef 1( 43), 2(218) */
  22312,                               /* Vector 1804, coef 3( 87) Vector
                                        * 1805, Coef 1( 40) */
  -10925,                              /* Vector 1805, coef 2(213), 3( 83) */
  9682,                                /* Vector 1806, coef 1( 37), 2(210) */
  22818,                               /* Vector 1806, coef 3( 89) Vector
                                        * 1807, Coef 1( 34) */
  -12459,                              /* Vector 1807, coef 2(207), 3( 85) */
  10207,                               /* Vector 1808, coef 1( 39), 2(223) */
  23334,                               /* Vector 1808, coef 3( 91) Vector
                                        * 1809, Coef 1( 38) */
  -9900,                               /* Vector 1809, coef 2(217), 3( 84) */
  9179,                                /* Vector 1810, coef 1( 35), 2(219) */
  21796,                               /* Vector 1810, coef 3( 85) Vector
                                        * 1811, Coef 1( 36) */
  -10653,                              /* Vector 1811, coef 2(214), 3( 99) */
  8919,                                /* Vector 1812, coef 1( 34), 2(215) */
  24607,                               /* Vector 1812, coef 3( 96) Vector
                                        * 1813, Coef 1( 31) */
  -11428,                              /* Vector 1813, coef 2(211), 3( 92) */
  12003,                               /* Vector 1814, coef 1( 46), 2(227) */
  21548,                               /* Vector 1814, coef 3( 84) Vector
                                        * 1815, Coef 1( 44) */
  -8121,                               /* Vector 1815, coef 2(224), 3( 71) */
  11236,                               /* Vector 1816, coef 1( 43), 2(228) */
  20007,                               /* Vector 1816, coef 3( 78) Vector
                                        * 1817, Coef 1( 39) */
  -8629,                               /* Vector 1817, coef 2(222), 3( 75) */
  11233,                               /* Vector 1818, coef 1( 43), 2(225) */
  15657,                               /* Vector 1818, coef 3( 61) Vector
                                        * 1819, Coef 1( 41) */
  -8651,                               /* Vector 1819, coef 2(222), 3( 53) */
  10458,                               /* Vector 1820, coef 1( 40), 2(218) */
  18211,                               /* Vector 1820, coef 3( 71) Vector
                                        * 1821, Coef 1( 35) */
  -10681,                              /* Vector 1821, coef 2(214), 3( 71) */
  9171,                                /* Vector 1822, coef 1( 35), 2(211) */
  19495,                               /* Vector 1822, coef 3( 76) Vector
                                        * 1823, Coef 1( 39) */
  -6829,                               /* Vector 1823, coef 2(229), 3( 83) */
  7368,                                /* Vector 1824, coef 1( 28), 2(200) */
  17693,                               /* Vector 1824, coef 3( 69) Vector
                                        * 1825, Coef 1( 29) */
  -12993,                              /* Vector 1825, coef 2(205), 3( 63) */
  8915,                                /* Vector 1826, coef 1( 34), 2(211) */
  14109,                               /* Vector 1826, coef 3( 55) Vector
                                        * 1827, Coef 1( 29) */
  -12503,                              /* Vector 1827, coef 2(207), 3( 41) */
  8666,                                /* Vector 1828, coef 1( 33), 2(218) */
  22050,                               /* Vector 1828, coef 3( 86) Vector
                                        * 1829, Coef 1( 34) */
  -10434,                              /* Vector 1829, coef 2(215), 3( 62) */
  7636,                                /* Vector 1830, coef 1( 29), 2(212) */
  13864,                               /* Vector 1830, coef 3( 54) Vector
                                        * 1831, Coef 1( 40) */
  -6847,                               /* Vector 1831, coef 2(229), 3( 65) */
  9695,                                /* Vector 1832, coef 1( 37), 2(223) */
  15653,                               /* Vector 1832, coef 3( 61) Vector
                                        * 1833, Coef 1( 37) */
  -7630,                               /* Vector 1833, coef 2(226), 3( 50) */
  7900,                                /* Vector 1834, coef 1( 30), 2(220) */
  12321,                               /* Vector 1834, coef 3( 48) Vector
                                        * 1835, Coef 1( 33) */
  -9915,                               /* Vector 1835, coef 2(217), 3( 69) */
  7125,                                /* Vector 1836, coef 1( 27), 2(213) */
  17949,                               /* Vector 1836, coef 3( 70) Vector
                                        * 1837, Coef 1( 29) */
  -9667,                               /* Vector 1837, coef 2(218), 3( 61) */
  6101,                                /* Vector 1838, coef 1( 23), 2(213) */
  16159,                               /* Vector 1838, coef 3( 63) Vector
                                        * 1839, Coef 1( 31) */
  -11959,                              /* Vector 1839, coef 2(209), 3( 73) */
  8406,                                /* Vector 1840, coef 1( 32), 2(214) */
  19995,                               /* Vector 1840, coef 3( 78) Vector
                                        * 1841, Coef 1( 27) */
  -11694,                              /* Vector 1841, coef 2(210), 3( 82) */
  6860,                                /* Vector 1842, coef 1( 26), 2(204) */
  19988,                               /* Vector 1842, coef 3( 78) Vector
                                        * 1843, Coef 1( 20) */
  -13492,                              /* Vector 1843, coef 2(203), 3( 76) */
  7386,                                /* Vector 1844, coef 1( 28), 2(218) */
  23573,                               /* Vector 1844, coef 3( 92) Vector
                                        * 1845, Coef 1( 21) */
  -11686,                              /* Vector 1845, coef 2(210), 3( 90) */
  6357,                                /* Vector 1846, coef 1( 24), 2(213) */
  21014,                               /* Vector 1846, coef 3( 82) Vector
                                        * 1847, Coef 1( 22) */
  -10675,                              /* Vector 1847, coef 2(214), 3( 77) */
  4818,                                /* Vector 1848, coef 1( 18), 2(210) */
  21776,                               /* Vector 1848, coef 3( 85) Vector
                                        * 1849, Coef 1( 16) */
  -11957,                              /* Vector 1849, coef 2(209), 3( 75) */
  9442,                                /* Vector 1850, coef 1( 36), 2(226) */
  21796,                               /* Vector 1850, coef 3( 85) Vector
                                        * 1851, Coef 1( 36) */
  -7861,                               /* Vector 1851, coef 2(225), 3( 75) */
  8671,                                /* Vector 1852, coef 1( 33), 2(223) */
  18975,                               /* Vector 1852, coef 3( 74) Vector
                                        * 1853, Coef 1( 31) */
  -8875,                               /* Vector 1853, coef 2(221), 3( 85) */
  7387,                                /* Vector 1854, coef 1( 28), 2(219) */
  20005,                               /* Vector 1854, coef 3( 78) Vector
                                        * 1855, Coef 1( 37) */
  -6590,                               /* Vector 1855, coef 2(230), 3( 66) */
  7136,                                /* Vector 1856, coef 1( 27), 2(224) */
  25627,                               /* Vector 1856, coef 3(100) Vector
                                        * 1857, Coef 1( 27) */
  -6804,                               /* Vector 1857, coef 2(229), 3(108) */
  6616,                                /* Vector 1858, coef 1( 25), 2(216) */
  24355,                               /* Vector 1858, coef 3( 95) Vector
                                        * 1859, Coef 1( 35) */
  -6311,                               /* Vector 1859, coef 2(231), 3( 89) */
  8419,                                /* Vector 1860, coef 1( 32), 2(227) */
  21539,                               /* Vector 1860, coef 3( 84) Vector
                                        * 1861, Coef 1( 35) */
  -6072,                               /* Vector 1861, coef 2(232), 3( 72) */
  8417,                                /* Vector 1862, coef 1( 32), 2(225) */
  16924,                               /* Vector 1862, coef 3( 66) Vector
                                        * 1863, Coef 1( 28) */
  -8129,                               /* Vector 1863, coef 2(224), 3( 63) */
  8933,                                /* Vector 1864, coef 1( 34), 2(229) */
  14368,                               /* Vector 1864, coef 3( 56) Vector
                                        * 1865, Coef 1( 32) */
  -7376,                               /* Vector 1865, coef 2(227), 3( 48) */
  7133,                                /* Vector 1866, coef 1( 27), 2(221) */
  13334,                               /* Vector 1866, coef 3( 52) Vector
                                        * 1867, Coef 1( 22) */
  -9418,                               /* Vector 1867, coef 2(219), 3( 54) */
  8425,                                /* Vector 1868, coef 1( 32), 2(233) */
  23325,                               /* Vector 1868, coef 3( 91) Vector
                                        * 1869, Coef 1( 29) */
  -7080,                               /* Vector 1869, coef 2(228), 3( 88) */
  7917,                                /* Vector 1870, coef 1( 30), 2(237) */
  21788,                               /* Vector 1870, coef 3( 85) Vector
                                        * 1871, Coef 1( 28) */
  -6051,                               /* Vector 1871, coef 2(232), 3( 93) */
  6113,                                /* Vector 1872, coef 1( 23), 2(225) */
  24601,                               /* Vector 1872, coef 3( 96) Vector
                                        * 1873, Coef 1( 25) */
  -8870,                               /* Vector 1873, coef 2(221), 3( 90) */
  5849,                                /* Vector 1874, coef 1( 22), 2(217) */
  23318,                               /* Vector 1874, coef 3( 91) Vector
                                        * 1875, Coef 1( 22) */
  -8366,                               /* Vector 1875, coef 2(223), 3( 82) */
  8426,                                /* Vector 1876, coef 1( 32), 2(234) */
  19487,                               /* Vector 1876, coef 3( 76) Vector
                                        * 1877, Coef 1( 31) */
  -6331,                               /* Vector 1877, coef 2(231), 3( 69) */
  8425,                                /* Vector 1878, coef 1( 32), 2(233) */
  14619,                               /* Vector 1878, coef 3( 57) Vector
                                        * 1879, Coef 1( 27) */
  -7372,                               /* Vector 1879, coef 2(227), 3( 52) */
  7918,                                /* Vector 1880, coef 1( 30), 2(238) */
  15900,                               /* Vector 1880, coef 3( 62) Vector
                                        * 1881, Coef 1( 28) */
  -6086,                               /* Vector 1881, coef 2(232), 3( 58) */
  7139,                                /* Vector 1882, coef 1( 27), 2(227) */
  19736,                               /* Vector 1882, coef 3( 77) Vector
                                        * 1883, Coef 1( 24) */
  -8627,                               /* Vector 1883, coef 2(222), 3( 77) */
  6364,                                /* Vector 1884, coef 1( 24), 2(220) */
  17940,                               /* Vector 1884, coef 3( 70) Vector
                                        * 1885, Coef 1( 20) */
  -9911,                               /* Vector 1885, coef 2(217), 3( 73) */
  6882,                                /* Vector 1886, coef 1( 26), 2(226) */
  16408,                               /* Vector 1886, coef 3( 64) Vector
                                        * 1887, Coef 1( 24) */
  -8342,                               /* Vector 1887, coef 2(223), 3(106) */
  6631,                                /* Vector 1888, coef 1( 25), 2(231) */
  28700,                               /* Vector 1888, coef 3(112) Vector
                                        * 1889, Coef 1( 28) */
  -5554,                               /* Vector 1889, coef 2(234), 3( 78) */
  6631,                                /* Vector 1890, coef 1( 25), 2(231) */
  23061,                               /* Vector 1890, coef 3( 90) Vector
                                        * 1891, Coef 1( 21) */
  -7846,                               /* Vector 1891, coef 2(225), 3( 90) */
  5083,                                /* Vector 1892, coef 1( 19), 2(219) */
  22032,                               /* Vector 1892, coef 3( 86) Vector
                                        * 1893, Coef 1( 16) */
  -9127,                               /* Vector 1893, coef 2(220), 3( 89) */
  7147,                                /* Vector 1894, coef 1( 27), 2(235) */
  13082,                               /* Vector 1894, coef 3( 51) Vector
                                        * 1895, Coef 1( 26) */
  -6099,                               /* Vector 1895, coef 2(232), 3( 45) */
  5856,                                /* Vector 1896, coef 1( 22), 2(224) */
  15125,                               /* Vector 1896, coef 3( 59) Vector
                                        * 1897, Coef 1( 21) */
  -8635,                               /* Vector 1897, coef 2(222), 3( 69) */
  4313,                                /* Vector 1898, coef 1( 16), 2(217) */
  18713,                               /* Vector 1898, coef 3( 73) Vector
                                        * 1899, Coef 1( 25) */
  -5049,                               /* Vector 1899, coef 2(236), 3( 71) */
  6118,                                /* Vector 1900, coef 1( 23), 2(230) */
  20247,                               /* Vector 1900, coef 3( 79) Vector
                                        * 1901, Coef 1( 23) */
  -5293,                               /* Vector 1901, coef 2(235), 3( 83) */
  5610,                                /* Vector 1902, coef 1( 21), 2(234) */
  19734,                               /* Vector 1902, coef 3( 77) Vector
                                        * 1903, Coef 1( 22) */
  -7352,                               /* Vector 1903, coef 2(227), 3( 72) */
  5090,                                /* Vector 1904, coef 1( 19), 2(226) */
  19731,                               /* Vector 1904, coef 3( 77) Vector
                                        * 1905, Coef 1( 19) */
  -6583,                               /* Vector 1905, coef 2(230), 3( 73) */
  4321,                                /* Vector 1906, coef 1( 16), 2(225) */
  19483,                               /* Vector 1906, coef 3( 76) Vector
                                        * 1907, Coef 1( 27) */
  -4552,                               /* Vector 1907, coef 2(238), 3( 56) */
  6123,                                /* Vector 1908, coef 1( 23), 2(235) */
  12055,                               /* Vector 1908, coef 3( 47) Vector
                                        * 1909, Coef 1( 23) */
  -6340,                               /* Vector 1909, coef 2(231), 3( 60) */
  5093,                                /* Vector 1910, coef 1( 19), 2(229) */
  15381,                               /* Vector 1910, coef 3( 60) Vector
                                        * 1911, Coef 1( 21) */
  -4543,                               /* Vector 1911, coef 2(238), 3( 65) */
  4843,                                /* Vector 1912, coef 1( 18), 2(235) */
  16660,                               /* Vector 1912, coef 3( 65) Vector
                                        * 1913, Coef 1( 20) */
  -5330,                               /* Vector 1913, coef 2(235), 3( 46) */
  4846,                                /* Vector 1914, coef 1( 18), 2(238) */
  12567,                               /* Vector 1914, coef 3( 49) Vector
                                        * 1915, Coef 1( 23) */
  -5537,                               /* Vector 1915, coef 2(234), 3( 95) */
  5612,                                /* Vector 1916, coef 1( 21), 2(236) */
  25619,                               /* Vector 1916, coef 3(100) Vector
                                        * 1917, Coef 1( 19) */
  -6825,                               /* Vector 1917, coef 2(229), 3( 87) */
  5361,                                /* Vector 1918, coef 1( 20), 2(241) */
  21519,                               /* Vector 1918, coef 3( 84) Vector
                                        * 1919, Coef 1( 15) */
  -5565,                               /* Vector 1919, coef 2(234), 3( 67) */
  8651,                                /* Vector 1920, coef 1( 33), 2(203) */
  -29153,                              /* Vector 1920, coef 3(142) Vector
                                        * 1921, Coef 1( 31) */
  -11866,                              /* Vector 1921, coef 2(209), 3(166) */
  7882,                                /* Vector 1922, coef 1( 30), 2(202) */
  -22501,                              /* Vector 1922, coef 3(168) Vector
                                        * 1923, Coef 1( 27) */
  -13394,                              /* Vector 1923, coef 2(203), 3(174) */
  6860,                                /* Vector 1924, coef 1( 26), 2(204) */
  -23525,                              /* Vector 1924, coef 3(164) Vector
                                        * 1925, Coef 1( 27) */
  -18023,                              /* Vector 1925, coef 2(185), 3(153) */
  6336,                                /* Vector 1926, coef 1( 24), 2(192) */
  -26337,                              /* Vector 1926, coef 3(153) Vector
                                        * 1927, Coef 1( 31) */
  -9323,                               /* Vector 1927, coef 2(219), 3(149) */
  7899,                                /* Vector 1928, coef 1( 30), 2(219) */
  -28386,                              /* Vector 1928, coef 3(145) Vector
                                        * 1929, Coef 1( 30) */
  -11385,                              /* Vector 1929, coef 2(211), 3(135) */
  7133,                                /* Vector 1930, coef 1( 27), 2(221) */
  -29154,                              /* Vector 1930, coef 3(142) Vector
                                        * 1931, Coef 1( 30) */
  -11881,                              /* Vector 1931, coef 2(209), 3(151) */
  7383,                                /* Vector 1932, coef 1( 28), 2(215) */
  -26596,                              /* Vector 1932, coef 3(152) Vector
                                        * 1933, Coef 1( 28) */
  -14184,                              /* Vector 1933, coef 2(200), 3(152) */
  6857,                                /* Vector 1934, coef 1( 26), 2(201) */
  -27110,                              /* Vector 1934, coef 3(150) Vector
                                        * 1935, Coef 1( 26) */
  -8553,                               /* Vector 1935, coef 2(222), 3(151) */
  6369,                                /* Vector 1936, coef 1( 24), 2(225) */
  -27625,                              /* Vector 1936, coef 3(148) Vector
                                        * 1937, Coef 1( 23) */
  -9825,                               /* Vector 1937, coef 2(217), 3(159) */
  5855,                                /* Vector 1938, coef 1( 22), 2(223) */
  -26849,                              /* Vector 1938, coef 3(151) Vector
                                        * 1939, Coef 1( 31) */
  -16491,                              /* Vector 1939, coef 2(191), 3(149) */
  8123,                                /* Vector 1940, coef 1( 31), 2(187) */
  -29412,                              /* Vector 1940, coef 3(141) Vector
                                        * 1941, Coef 1( 28) */
  -16758,                              /* Vector 1941, coef 2(190), 3(138) */
  7882,                                /* Vector 1942, coef 1( 30), 2(202) */
  -29156,                              /* Vector 1942, coef 3(142) Vector
                                        * 1943, Coef 1( 28) */
  -13945,                              /* Vector 1943, coef 2(201), 3(135) */
  6856,                                /* Vector 1944, coef 1( 26), 2(200) */
  -32233,                              /* Vector 1944, coef 3(130) Vector
                                        * 1945, Coef 1( 23) */
  -13438,                              /* Vector 1945, coef 2(203), 3(130) */
  6857,                                /* Vector 1946, coef 1( 26), 2(201) */
  -29160,                              /* Vector 1946, coef 3(142) Vector
                                        * 1947, Coef 1( 24) */
  -15730,                              /* Vector 1947, coef 2(194), 3(142) */
  6872,                                /* Vector 1948, coef 1( 26), 2(216) */
  -31719,                              /* Vector 1948, coef 3(132) Vector
                                        * 1949, Coef 1( 25) */
  -10869,                              /* Vector 1949, coef 2(213), 3(139) */
  6360,                                /* Vector 1950, coef 1( 24), 2(216) */
  -31210,                              /* Vector 1950, coef 3(134) Vector
                                        * 1951, Coef 1( 22) */
  -11122,                              /* Vector 1951, coef 2(212), 3(142) */
  5822,                                /* Vector 1952, coef 1( 22), 2(190) */
  -31727,                              /* Vector 1952, coef 3(132) Vector
                                        * 1953, Coef 1( 17) */
  -17269,                              /* Vector 1953, coef 2(188), 3(139) */
  3261,                                /* Vector 1954, coef 1( 12), 2(189) */
  -31735,                              /* Vector 1954, coef 3(132) Vector
                                        * 1955, Coef 1(  9) */
  -13401,                              /* Vector 1955, coef 2(203), 3(167) */
  5575,                                /* Vector 1956, coef 1( 21), 2(199) */
  31762,                               /* Vector 1956, coef 3(124) Vector
                                        * 1957, Coef 1( 18) */
  -13193,                              /* Vector 1957, coef 2(204), 3(119) */
  4041,                                /* Vector 1958, coef 1( 15), 2(201) */
  30742,                               /* Vector 1958, coef 3(120) Vector
                                        * 1959, Coef 1( 22) */
  -13430,                              /* Vector 1959, coef 2(203), 3(138) */
  5317,                                /* Vector 1960, coef 1( 20), 2(197) */
  -29164,                              /* Vector 1960, coef 3(142) Vector
                                        * 1961, Coef 1( 20) */
  -14715,                              /* Vector 1961, coef 2(198), 3(133) */
  4809,                                /* Vector 1962, coef 1( 18), 2(201) */
  -30191,                              /* Vector 1962, coef 3(138) Vector
                                        * 1963, Coef 1( 17) */
  -15743,                              /* Vector 1963, coef 2(194), 3(129) */
  4039,                                /* Vector 1964, coef 1( 15), 2(199) */
  -31980,                              /* Vector 1964, coef 3(131) Vector
                                        * 1965, Coef 1( 20) */
  -10615,                              /* Vector 1965, coef 2(214), 3(137) */
  5088,                                /* Vector 1966, coef 1( 19), 2(224) */
  -29167,                              /* Vector 1966, coef 3(142) Vector
                                        * 1967, Coef 1( 17) */
  -10100,                              /* Vector 1967, coef 2(216), 3(140) */
  5584,                                /* Vector 1968, coef 1( 21), 2(208) */
  -32237,                              /* Vector 1968, coef 3(130) Vector
                                        * 1969, Coef 1( 19) */
  -12419,                              /* Vector 1969, coef 2(207), 3(125) */
  4307,                                /* Vector 1970, coef 1( 16), 2(211) */
  31761,                               /* Vector 1970, coef 3(124) Vector
                                        * 1971, Coef 1( 17) */
  -11643,                              /* Vector 1971, coef 2(210), 3(133) */
  4311,                                /* Vector 1972, coef 1( 16), 2(215) */
  -31218,                              /* Vector 1972, coef 3(134) Vector
                                        * 1973, Coef 1( 14) */
  -10620,                              /* Vector 1973, coef 2(214), 3(132) */
  3023,                                /* Vector 1974, coef 1( 11), 2(207) */
  -32240,                              /* Vector 1974, coef 3(130) Vector
                                        * 1975, Coef 1( 16) */
  -13428,                              /* Vector 1975, coef 2(203), 3(140) */
  3786,                                /* Vector 1976, coef 1( 14), 2(202) */
  -29168,                              /* Vector 1976, coef 3(142) Vector
                                        * 1977, Coef 1( 16) */
  -7795,                               /* Vector 1977, coef 2(225), 3(141) */
  4058,                                /* Vector 1978, coef 1( 15), 2(218) */
  -29427,                              /* Vector 1978, coef 3(141) Vector
                                        * 1979, Coef 1( 13) */
  -7278,                               /* Vector 1979, coef 2(227), 3(146) */
  3536,                                /* Vector 1980, coef 1( 13), 2(208) */
  -26869,                              /* Vector 1980, coef 3(151) Vector
                                        * 1981, Coef 1( 11) */
  -11885,                              /* Vector 1981, coef 2(209), 3(147) */
  2783,                                /* Vector 1982, coef 1( 10), 2(223) */
  -28910,                              /* Vector 1982, coef 3(143) Vector
                                        * 1983, Coef 1( 18) */
  -8831,                               /* Vector 1983, coef 2(221), 3(129) */
  6103,                                /* Vector 1984, coef 1( 23), 2(215) */
  31252,                               /* Vector 1984, coef 3(122) Vector
                                        * 1985, Coef 1( 20) */
  -11148,                              /* Vector 1985, coef 2(212), 3(116) */
  4561,                                /* Vector 1986, coef 1( 17), 2(209) */
  28175,                               /* Vector 1986, coef 3(110) Vector
                                        * 1987, Coef 1( 15) */
  -12689,                              /* Vector 1987, coef 2(206), 3(111) */
  3276,                                /* Vector 1988, coef 1( 12), 2(204) */
  27670,                               /* Vector 1988, coef 3(108) Vector
                                        * 1989, Coef 1( 22) */
  -9086,                               /* Vector 1989, coef 2(220), 3(130) */
  4823,                                /* Vector 1990, coef 1( 18), 2(215) */
  31510,                               /* Vector 1990, coef 3(123) Vector
                                        * 1991, Coef 1( 22) */
  -9111,                               /* Vector 1991, coef 2(220), 3(105) */
  5078,                                /* Vector 1992, coef 1( 19), 2(214) */
  26128,                               /* Vector 1992, coef 3(102) Vector
                                        * 1993, Coef 1( 16) */
  -11165,                              /* Vector 1993, coef 2(212), 3( 99) */
  5861,                                /* Vector 1994, coef 1( 22), 2(229) */
  30484,                               /* Vector 1994, coef 3(119) Vector
                                        * 1995, Coef 1( 20) */
  -8076,                               /* Vector 1995, coef 2(224), 3(116) */
  5081,                                /* Vector 1996, coef 1( 19), 2(217) */
  28945,                               /* Vector 1996, coef 3(113) Vector
                                        * 1997, Coef 1( 17) */
  -9357,                               /* Vector 1997, coef 2(219), 3(115) */
  4577,                                /* Vector 1998, coef 1( 17), 2(225) */
  27151,                               /* Vector 1998, coef 3(106) Vector
                                        * 1999, Coef 1( 15) */
  -7831,                               /* Vector 1999, coef 2(225), 3(105) */
  4311,                                /* Vector 2000, coef 1( 16), 2(215) */
  28685,                               /* Vector 2000, coef 3(112) Vector
                                        * 2001, Coef 1( 13) */
  -10381,                              /* Vector 2001, coef 2(215), 3(115) */
  5601,                                /* Vector 2002, coef 1( 21), 2(225) */
  32530,                               /* Vector 2002, coef 3(127) Vector
                                        * 2003, Coef 1( 18) */
  -5754,                               /* Vector 2003, coef 2(233), 3(134) */
  4323,                                /* Vector 2004, coef 1( 16), 2(227) */
  -31982,                              /* Vector 2004, coef 3(131) Vector
                                        * 2005, Coef 1( 18) */
  -7049,                               /* Vector 2005, coef 2(228), 3(119) */
  4325,                                /* Vector 2006, coef 1( 16), 2(229) */
  30480,                               /* Vector 2006, coef 3(119) Vector
                                        * 2007, Coef 1( 16) */
  -6533,                               /* Vector 2007, coef 2(230), 3(123) */
  4074,                                /* Vector 2008, coef 1( 15), 2(234) */
  30480,                               /* Vector 2008, coef 3(119) Vector
                                        * 2009, Coef 1( 16) */
  -9349,                               /* Vector 2009, coef 2(219), 3(123) */
  4064,                                /* Vector 2010, coef 1( 15), 2(224) */
  29710,                               /* Vector 2010, coef 3(116) Vector
                                        * 2011, Coef 1( 14) */
  -7294,                               /* Vector 2011, coef 2(227), 3(130) */
  3548,                                /* Vector 2012, coef 1( 13), 2(220) */
  31757,                               /* Vector 2012, coef 3(124) Vector
                                        * 2013, Coef 1( 13) */
  -6277,                               /* Vector 2013, coef 2(231), 3(123) */
  3301,                                /* Vector 2014, coef 1( 12), 2(229) */
  -32236,                              /* Vector 2014, coef 3(130) Vector
                                        * 2015, Coef 1( 20) */
  -7321,                               /* Vector 2015, coef 2(227), 3(103) */
  5341,                                /* Vector 2016, coef 1( 20), 2(221) */
  24849,                               /* Vector 2016, coef 3( 97) Vector
                                        * 2017, Coef 1( 17) */
  -8609,                               /* Vector 2017, coef 2(222), 3( 95) */
  3291,                                /* Vector 2018, coef 1( 12), 2(219) */
  24072,                               /* Vector 2018, coef 3( 94) Vector
                                        * 2019, Coef 1(  8) */
  -9588,                               /* Vector 2019, coef 2(218), 3(140) */
  3304,                                /* Vector 2020, coef 1( 12), 2(232) */
  29706,                               /* Vector 2020, coef 3(116) Vector
                                        * 2021, Coef 1( 10) */
  -7809,                               /* Vector 2021, coef 2(225), 3(127) */
  5094,                                /* Vector 2022, coef 1( 19), 2(230) */
  24593,                               /* Vector 2022, coef 3( 96) Vector
                                        * 2023, Coef 1( 17) */
  -6314,                               /* Vector 2023, coef 2(231), 3( 86) */
  4586,                                /* Vector 2024, coef 1( 17), 2(234) */
  24079,                               /* Vector 2024, coef 3( 94) Vector
                                        * 2025, Coef 1( 15) */
  -6561,                               /* Vector 2025, coef 2(230), 3( 95) */
  5098,                                /* Vector 2026, coef 1( 19), 2(234) */
  27153,                               /* Vector 2026, coef 3(106) Vector
                                        * 2027, Coef 1( 17) */
  -5780,                               /* Vector 2027, coef 2(233), 3(108) */
  4593,                                /* Vector 2028, coef 1( 17), 2(241) */
  25360,                               /* Vector 2028, coef 3( 99) Vector
                                        * 2029, Coef 1( 16) */
  -4506,                               /* Vector 2029, coef 2(238), 3(102) */
  4329,                                /* Vector 2030, coef 1( 16), 2(233) */
  28687,                               /* Vector 2030, coef 3(112) Vector
                                        * 2031, Coef 1( 15) */
  -5268,                               /* Vector 2031, coef 2(235), 3(108) */
  3562,                                /* Vector 2032, coef 1( 13), 2(234) */
  26891,                               /* Vector 2032, coef 3(105) Vector
                                        * 2033, Coef 1( 11) */
  -5524,                               /* Vector 2033, coef 2(234), 3(108) */
  4850,                                /* Vector 2034, coef 1( 18), 2(242) */
  18193,                               /* Vector 2034, coef 3( 71) Vector
                                        * 2035, Coef 1( 17) */
  -5298,                               /* Vector 2035, coef 2(235), 3( 78) */
  4594,                                /* Vector 2036, coef 1( 17), 2(242) */
  22032,                               /* Vector 2036, coef 3( 86) Vector
                                        * 2037, Coef 1( 16) */
  -3252,                               /* Vector 2037, coef 2(243), 3( 76) */
  4335,                                /* Vector 2038, coef 1( 16), 2(239) */
  23823,                               /* Vector 2038, coef 3( 93) Vector
                                        * 2039, Coef 1( 15) */
  -4007,                               /* Vector 2039, coef 2(240), 3( 89) */
  4337,                                /* Vector 2040, coef 1( 16), 2(241) */
  13839,                               /* Vector 2040, coef 3( 54) Vector
                                        * 2041, Coef 1( 15) */
  -3268,                               /* Vector 2041, coef 2(243), 3( 60) */
  3571,                                /* Vector 2042, coef 1( 13), 2(243) */
  15628,                               /* Vector 2042, coef 3( 61) Vector
                                        * 2043, Coef 1( 12) */
  -3772,                               /* Vector 2043, coef 2(241), 3( 68) */
  3821,                                /* Vector 2044, coef 1( 14), 2(237) */
  19981,                               /* Vector 2044, coef 3( 78) Vector
                                        * 2045, Coef 1( 13) */
  -6058,                               /* Vector 2045, coef 2(232), 3( 86) */
  3312,                                /* Vector 2046, coef 1( 12), 2(240) */
  21515,                               /* Vector 2046, coef 3( 84) Vector
                                        * 2047, Coef 1( 11) */
  -4772                                /* Vector 2047, coef 2(237), 3( 92) */
};

ShortwordRom psrQuant2[QUANT2_NUM_OF_WORDS] =
{
  -15195,                              /* Vector    0, coef 1(196), 2(165) */
  -24371,                              /* Vector    0, coef 3(160) Vector 1,
                                        * Coef 1(205) */
  -26463,                              /* Vector    1, coef 2(152), 3(161) */
  -15723,                              /* Vector    2, coef 1(194), 2(149) */
  -24381,                              /* Vector    2, coef 3(160) Vector 3,
                                        * Coef 1(195) */
  -28524,                              /* Vector    3, coef 2(144), 3(148) */
  -12896,                              /* Vector    4, coef 1(205), 2(160) */
  -28224,                              /* Vector    4, coef 3(145) Vector 5,
                                        * Coef 1(192) */
  -25457,                              /* Vector    5, coef 2(156), 3(143) */
  -15453,                              /* Vector    6, coef 1(195), 2(163) */
  32712,                               /* Vector    6, coef 3(127) Vector 7,
                                        * Coef 1(200) */
  -28494,                              /* Vector    7, coef 2(144), 3(178) */
  -14965,                              /* Vector    8, coef 1(197), 2(139) */
  -22574,                              /* Vector    8, coef 3(167) Vector 9,
                                        * Coef 1(210) */
  -30310,                              /* Vector    9, coef 2(137), 3(154) */
  -16503,                              /* Vector   10, coef 1(191), 2(137) */
  -24111,                              /* Vector   10, coef 3(161) Vector 11,
                                        * Coef 1(209) */
  -28027,                              /* Vector   11, coef 2(146), 3(133) */
  -18030,                              /* Vector   12, coef 1(185), 2(146) */
  -24905,                              /* Vector   12, coef 3(158) Vector 13,
                                        * Coef 1(183) */
  -25189,                              /* Vector   13, coef 2(157), 3(155) */
  -20583,                              /* Vector   14, coef 1(175), 2(153) */
  -26445,                              /* Vector   14, coef 3(152) Vector 15,
                                        * Coef 1(179) */
  -25209,                              /* Vector   15, coef 2(157), 3(135) */
  -18032,                              /* Vector   16, coef 1(185), 2(144) */
  -28992,                              /* Vector   16, coef 3(142) Vector 17,
                                        * Coef 1(192) */
  -28289,                              /* Vector   17, coef 2(145), 3(127) */
  -17020,                              /* Vector   18, coef 1(189), 2(132) */
  -24646,                              /* Vector   18, coef 3(159) Vector 19,
                                        * Coef 1(186) */
  -30058,                              /* Vector   19, coef 2(138), 3(150) */
  -18301,                              /* Vector   20, coef 1(184), 2(131) */
  -27961,                              /* Vector   20, coef 3(146) Vector 21,
                                        * Coef 1(199) */
  -30070,                              /* Vector   21, coef 2(138), 3(138) */
  -15487,                              /* Vector   22, coef 1(195), 2(129) */
  -30278,                              /* Vector   22, coef 3(137) Vector 23,
                                        * Coef 1(186) */
  -31354,                              /* Vector   23, coef 2(133), 3(134) */
  -18563,                              /* Vector   24, coef 1(183), 2(125) */
  -31821,                              /* Vector   24, coef 3(131) Vector 25,
                                        * Coef 1(179) */
  -28543,                              /* Vector   25, coef 2(144), 3(129) */
  -16771,                              /* Vector   26, coef 1(190), 2(125) */
  -26695,                              /* Vector   26, coef 3(151) Vector 27,
                                        * Coef 1(185) */
  31376,                               /* Vector   27, coef 2(122), 3(144) */
  -20091,                              /* Vector   28, coef 1(177), 2(133) */
  -28243,                              /* Vector   28, coef 3(145) Vector 29,
                                        * Coef 1(173) */
  -32374,                              /* Vector   29, coef 2(129), 3(138) */
  -19572,                              /* Vector   30, coef 1(179), 2(140) */
  -24912,                              /* Vector   30, coef 3(158) Vector 31,
                                        * Coef 1(176) */
  -28525,                              /* Vector   31, coef 2(144), 3(147) */
  -14952,                              /* Vector   32, coef 1(197), 2(152) */
  27829,                               /* Vector   32, coef 3(108) Vector 33,
                                        * Coef 1(181) */
  -25753,                              /* Vector   33, coef 2(155), 3(103) */
  -19319,                              /* Vector   34, coef 1(180), 2(137) */
  27823,                               /* Vector   34, coef 3(108) Vector 35,
                                        * Coef 1(175) */
  -31399,                              /* Vector   35, coef 2(133), 3( 89) */
  -16009,                              /* Vector   36, coef 1(193), 2(119) */
  -32046,                              /* Vector   36, coef 3(130) Vector 37,
                                        * Coef 1(210) */
  -32639,                              /* Vector   37, coef 2(128), 3(129) */
  -13433,                              /* Vector   38, coef 1(203), 2(135) */
  30656,                               /* Vector   38, coef 3(119) Vector 39,
                                        * Coef 1(192) */
  -32394,                              /* Vector   39, coef 2(129), 3(118) */
  -13710,                              /* Vector   40, coef 1(202), 2(114) */
  31424,                               /* Vector   40, coef 3(122) Vector 41,
                                        * Coef 1(192) */
  26745,                               /* Vector   41, coef 2(104), 3(121) */
  -12160,                              /* Vector   42, coef 1(208), 2(128) */
  27075,                               /* Vector   42, coef 3(105) Vector 43,
                                        * Coef 1(195) */
  30562,                               /* Vector   43, coef 2(119), 3( 98) */
  -12955,                              /* Vector   44, coef 1(205), 2(101) */
  28092,                               /* Vector   44, coef 3(109) Vector 45,
                                        * Coef 1(188) */
  25450,                               /* Vector   45, coef 2( 99), 3(106) */
  -18057,                              /* Vector   46, coef 1(185), 2(119) */
  30385,                               /* Vector   46, coef 3(118) Vector 47,
                                        * Coef 1(177) */
  30572,                               /* Vector   47, coef 2(119), 3(108) */
  -21135,                              /* Vector   48, coef 1(173), 2(113) */
  31655,                               /* Vector   48, coef 3(123) Vector 49,
                                        * Coef 1(167) */
  26981,                               /* Vector   49, coef 2(105), 3(101) */
  -20609,                              /* Vector   50, coef 1(175), 2(127) */
  31652,                               /* Vector   50, coef 3(123) Vector 51,
                                        * Coef 1(164) */
  -30345,                              /* Vector   51, coef 2(137), 3(119) */
  -23169,                              /* Vector   52, coef 1(165), 2(127) */
  28819,                               /* Vector   52, coef 3(112) Vector 53,
                                        * Coef 1(147) */
  -30865,                              /* Vector   53, coef 2(135), 3(111) */
  -27781,                              /* Vector   54, coef 1(147), 2(123) */
  25765,                               /* Vector   54, coef 3(100) Vector 55,
                                        * Coef 1(165) */
  32129,                               /* Vector   55, coef 2(125), 3(129) */
  -23434,                              /* Vector   56, coef 1(164), 2(118) */
  31393,                               /* Vector   56, coef 3(122) Vector 57,
                                        * Coef 1(161) */
  29060,                               /* Vector   57, coef 2(113), 3(132) */
  -25747,                              /* Vector   58, coef 1(155), 2(109) */
  32408,                               /* Vector   58, coef 3(126) Vector 59,
                                        * Coef 1(152) */
  30582,                               /* Vector   59, coef 2(119), 3(118) */
  -24979,                              /* Vector   60, coef 1(158), 2(109) */
  29092,                               /* Vector   60, coef 3(113) Vector 61,
                                        * Coef 1(164) */
  31371,                               /* Vector   61, coef 2(122), 3(139) */
  -25735,                              /* Vector   62, coef 1(155), 2(121) */
  -32366,                              /* Vector   62, coef 3(129) Vector 63,
                                        * Coef 1(146) */
  32635,                               /* Vector   63, coef 2(127), 3(123) */
  -14206,                              /* Vector   64, coef 1(200), 2(130) */
  -27180,                              /* Vector   64, coef 3(149) Vector 65,
                                        * Coef 1(212) */
  28811,                               /* Vector   65, coef 2(112), 3(139) */
  -14463,                              /* Vector   66, coef 1(199), 2(129) */
  -23358,                              /* Vector   66, coef 3(164) Vector 67,
                                        * Coef 1(194) */
  31390,                               /* Vector   67, coef 2(122), 3(158) */
  -12167,                              /* Vector   68, coef 1(208), 2(121) */
  -23608,                              /* Vector   68, coef 3(163) Vector 69,
                                        * Coef 1(200) */
  29083,                               /* Vector   69, coef 2(113), 3(155) */
  -14217,                              /* Vector   70, coef 1(200), 2(119) */
  -28736,                              /* Vector   70, coef 3(143) Vector 71,
                                        * Coef 1(192) */
  29071,                               /* Vector   71, coef 2(113), 3(143) */
  -15497,                              /* Vector   72, coef 1(195), 2(119) */
  -19261,                              /* Vector   72, coef 3(180) Vector 73,
                                        * Coef 1(195) */
  29609,                               /* Vector   73, coef 2(115), 3(169) */
  -16020,                              /* Vector   74, coef 1(193), 2(108) */
  -18756,                              /* Vector   74, coef 3(182) Vector 75,
                                        * Coef 1(188) */
  29602,                               /* Vector   75, coef 2(115), 3(162) */
  -17556,                              /* Vector   76, coef 1(187), 2(108) */
  -25392,                              /* Vector   76, coef 3(156) Vector 77,
                                        * Coef 1(208) */
  25505,                               /* Vector   77, coef 2( 99), 3(161) */
  -18571,                              /* Vector   78, coef 1(183), 2(117) */
  -26439,                              /* Vector   78, coef 3(152) Vector 79,
                                        * Coef 1(185) */
  31910,                               /* Vector   79, coef 2(124), 3(166) */
  -12448,                              /* Vector   80, coef 1(207), 2( 96) */
  -31546,                              /* Vector   80, coef 3(132) Vector 81,
                                        * Coef 1(198) */
  24471,                               /* Vector   81, coef 2( 95), 3(151) */
  -15771,                              /* Vector   82, coef 1(194), 2(101) */
  -22855,                              /* Vector   82, coef 3(166) Vector 83,
                                        * Coef 1(185) */
  25246,                               /* Vector   83, coef 2( 98), 3(158) */
  -13235,                              /* Vector   84, coef 1(204), 2( 77) */
  -24131,                              /* Vector   84, coef 3(161) Vector 85,
                                        * Coef 1(189) */
  20390,                               /* Vector   85, coef 2( 79), 3(166) */
  -18857,                              /* Vector   86, coef 1(182), 2( 87) */
  -25931,                              /* Vector   86, coef 3(154) Vector 87,
                                        * Coef 1(181) */
  18581,                               /* Vector   87, coef 2( 72), 3(149) */
  -20397,                              /* Vector   88, coef 1(176), 2( 83) */
  -20822,                              /* Vector   88, coef 3(174) Vector 89,
                                        * Coef 1(170) */
  20888,                               /* Vector   89, coef 2( 81), 3(152) */
  -20128,                              /* Vector   90, coef 1(177), 2( 96) */
  -26942,                              /* Vector   90, coef 3(150) Vector 91,
                                        * Coef 1(194) */
  26507,                               /* Vector   91, coef 2(103), 3(139) */
  -17826,                              /* Vector   92, coef 1(186), 2( 94) */
  -30010,                              /* Vector   92, coef 3(138) Vector 93,
                                        * Coef 1(198) */
  21130,                               /* Vector   93, coef 2( 82), 3(138) */
  -16299,                              /* Vector   94, coef 1(192), 2( 85) */
  31665,                               /* Vector   94, coef 3(123) Vector 95,
                                        * Coef 1(177) */
  21382,                               /* Vector   95, coef 2( 83), 3(134) */
  -18062,                              /* Vector   96, coef 1(185), 2(114) */
  -31053,                              /* Vector   96, coef 3(134) Vector 97,
                                        * Coef 1(179) */
  29057,                               /* Vector   97, coef 2(113), 3(129) */
  -21144,                              /* Vector   98, coef 1(173), 2(104) */
  -26191,                              /* Vector   98, coef 3(153) Vector 99,
                                        * Coef 1(177) */
  28818,                               /* Vector   99, coef 2(112), 3(146) */
  -18326,                              /* Vector  100, coef 1(184), 2(106) */
  -28496,                              /* Vector  100, coef 3(144) Vector
                                        * 101, Coef 1(176) */
  27020,                               /* Vector  101, coef 2(105), 3(140) */
  -21923,                              /* Vector  102, coef 1(170), 2( 93) */
  -28758,                              /* Vector  102, coef 3(143) Vector
                                        * 103, Coef 1(170) */
  29331,                               /* Vector  103, coef 2(114), 3(147) */
  -23185,                              /* Vector  104, coef 1(165), 2(111) */
  -27741,                              /* Vector  104, coef 3(147) Vector
                                        * 105, Coef 1(163) */
  27789,                               /* Vector  105, coef 2(108), 3(141) */
  -21394,                              /* Vector  106, coef 1(172), 2(110) */
  -30039,                              /* Vector  106, coef 3(138) Vector
                                        * 107, Coef 1(169) */
  26756,                               /* Vector  107, coef 2(104), 3(132) */
  -18844,                              /* Vector  108, coef 1(182), 2(100) */
  32679,                               /* Vector  108, coef 3(127) Vector
                                        * 109, Coef 1(167) */
  25746,                               /* Vector  109, coef 2(100), 3(146) */
  -20359,                              /* Vector  110, coef 1(176), 2(121) */
  -29525,                              /* Vector  110, coef 3(140) Vector
                                        * 111, Coef 1(171) */
  30343,                               /* Vector  111, coef 2(118), 3(135) */
  -20632,                              /* Vector  112, coef 1(175), 2(104) */
  29353,                               /* Vector  112, coef 3(114) Vector
                                        * 113, Coef 1(169) */
  17301,                               /* Vector  113, coef 2( 67), 3(149) */
  -24218,                              /* Vector  114, coef 1(161), 2(102) */
  -30562,                              /* Vector  114, coef 3(136) Vector
                                        * 115, Coef 1(158) */
  25218,                               /* Vector  115, coef 2( 98), 3(130) */
  -20643,                              /* Vector  116, coef 1(175), 2( 93) */
  32434,                               /* Vector  116, coef 3(126) Vector
                                        * 117, Coef 1(178) */
  20342,                               /* Vector  117, coef 2( 79), 3(118) */
  -24228,                              /* Vector  118, coef 1(161), 2( 92) */
  -28761,                              /* Vector  118, coef 3(143) Vector
                                        * 119, Coef 1(167) */
  22661,                               /* Vector  119, coef 2( 88), 3(133) */
  -24497,                              /* Vector  120, coef 1(160), 2( 79) */
  -30809,                              /* Vector  120, coef 3(135) Vector
                                        * 121, Coef 1(167) */
  26492,                               /* Vector  121, coef 2(103), 3(124) */
  -23199,                              /* Vector  122, coef 1(165), 2( 97) */
  29859,                               /* Vector  122, coef 3(116) Vector
                                        * 123, Coef 1(163) */
  22652,                               /* Vector  123, coef 2( 88), 3(124) */
  -24492,                              /* Vector  124, coef 1(160), 2( 84) */
  28567,                               /* Vector  124, coef 3(111) Vector
                                        * 125, Coef 1(151) */
  23174,                               /* Vector  125, coef 2( 90), 3(134) */
  -28086,                              /* Vector  126, coef 1(146), 2( 74) */
  -31338,                              /* Vector  126, coef 3(133) Vector
                                        * 127, Coef 1(150) */
  22650,                               /* Vector  127, coef 2( 88), 3(122) */
  -14973,                              /* Vector  128, coef 1(197), 2(131) */
  -18498,                              /* Vector  128, coef 3(183) Vector
                                        * 129, Coef 1(190) */
  31664,                               /* Vector  129, coef 2(123), 3(176) */
  -18291,                              /* Vector  130, coef 1(184), 2(141) */
  -19021,                              /* Vector  130, coef 3(181) Vector
                                        * 131, Coef 1(179) */
  -30281,                              /* Vector  131, coef 2(137), 3(183) */
  -17784,                              /* Vector  132, coef 1(186), 2(136) */
  -20557,                              /* Vector  132, coef 3(175) Vector
                                        * 133, Coef 1(179) */
  -31565,                              /* Vector  133, coef 2(132), 3(179) */
  -21116,                              /* Vector  134, coef 1(173), 2(132) */
  -20295,                              /* Vector  134, coef 3(176) Vector
                                        * 135, Coef 1(185) */
  -31041,                              /* Vector  135, coef 2(134), 3(191) */
  -19844,                              /* Vector  136, coef 1(178), 2(124) */
  -16454,                              /* Vector  136, coef 3(191) Vector
                                        * 137, Coef 1(186) */
  -32340,                              /* Vector  137, coef 2(129), 3(172) */
  -19334,                              /* Vector  138, coef 1(180), 2(122) */
  -20300,                              /* Vector  138, coef 3(176) Vector
                                        * 139, Coef 1(180) */
  -30808,                              /* Vector  139, coef 2(135), 3(168) */
  -21126,                              /* Vector  140, coef 1(173), 2(122) */
  -19276,                              /* Vector  140, coef 3(180) Vector
                                        * 141, Coef 1(180) */
  -29271,                              /* Vector  141, coef 2(141), 3(169) */
  -20861,                              /* Vector  142, coef 1(174), 2(131) */
  -14164,                              /* Vector  142, coef 3(200) Vector
                                        * 143, Coef 1(172) */
  32700,                               /* Vector  143, coef 2(127), 3(188) */
  -20872,                              /* Vector  144, coef 1(174), 2(120) */
  -26443,                              /* Vector  144, coef 3(152) Vector
                                        * 145, Coef 1(181) */
  31134,                               /* Vector  145, coef 2(121), 3(158) */
  -20877,                              /* Vector  146, coef 1(174), 2(115) */
  -24907,                              /* Vector  146, coef 3(158) Vector
                                        * 147, Coef 1(181) */
  -31585,                              /* Vector  147, coef 2(132), 3(159) */
  -21113,                              /* Vector  148, coef 1(173), 2(135) */
  -23381,                              /* Vector  148, coef 3(164) Vector
                                        * 149, Coef 1(171) */
  -32351,                              /* Vector  149, coef 2(129), 3(161) */
  -20096,                              /* Vector  150, coef 1(177), 2(128) */
  -25427,                              /* Vector  150, coef 3(156) Vector
                                        * 151, Coef 1(173) */
  32661,                               /* Vector  151, coef 2(127), 3(149) */
  -22405,                              /* Vector  152, coef 1(168), 2(123) */
  -25177,                              /* Vector  152, coef 3(157) Vector
                                        * 153, Coef 1(167) */
  30869,                               /* Vector  153, coef 2(120), 3(149) */
  -22912,                              /* Vector  154, coef 1(166), 2(128) */
  -28756,                              /* Vector  154, coef 3(143) Vector
                                        * 155, Coef 1(172) */
  32683,                               /* Vector  155, coef 2(127), 3(171) */
  -21895,                              /* Vector  156, coef 1(170), 2(121) */
  -22621,                              /* Vector  156, coef 3(167) Vector
                                        * 157, Coef 1(163) */
  32168,                               /* Vector  157, coef 2(125), 3(168) */
  -21879,                              /* Vector  158, coef 1(170), 2(137) */
  -25947,                              /* Vector  158, coef 3(154) Vector
                                        * 159, Coef 1(165) */
  -32103,                              /* Vector  159, coef 2(130), 3(153) */
  -18583,                              /* Vector  160, coef 1(183), 2(105) */
  -21325,                              /* Vector  160, coef 3(172) Vector
                                        * 161, Coef 1(179) */
  27554,                               /* Vector  161, coef 2(107), 3(162) */
  -18848,                              /* Vector  162, coef 1(182), 2( 96) */
  -18511,                              /* Vector  162, coef 3(183) Vector
                                        * 163, Coef 1(177) */
  30631,                               /* Vector  163, coef 2(119), 3(167) */
  -18313,                              /* Vector  164, coef 1(184), 2(119) */
  -16207,                              /* Vector  164, coef 3(192) Vector
                                        * 165, Coef 1(177) */
  29887,                               /* Vector  165, coef 2(116), 3(191) */
  -18317,                              /* Vector  166, coef 1(184), 2(115) */
  -20047,                              /* Vector  166, coef 3(177) Vector
                                        * 167, Coef 1(177) */
  28590,                               /* Vector  167, coef 2(111), 3(174) */
  -18835,                              /* Vector  168, coef 1(182), 2(109) */
  -16209,                              /* Vector  168, coef 3(192) Vector
                                        * 169, Coef 1(175) */
  26818,                               /* Vector  169, coef 2(104), 3(194) */
  -21144,                              /* Vector  170, coef 1(173), 2(104) */
  -19796,                              /* Vector  170, coef 3(178) Vector
                                        * 171, Coef 1(172) */
  29617,                               /* Vector  171, coef 2(115), 3(177) */
  -21649,                              /* Vector  172, coef 1(171), 2(111) */
  -22614,                              /* Vector  172, coef 3(167) Vector
                                        * 173, Coef 1(170) */
  28098,                               /* Vector  173, coef 2(109), 3(194) */
  -23705,                              /* Vector  174, coef 1(163), 2(103) */
  -16471,                              /* Vector  174, coef 3(191) Vector
                                        * 175, Coef 1(169) */
  29372,                               /* Vector  175, coef 2(114), 3(188) */
  -20642,                              /* Vector  176, coef 1(175), 2( 94) */
  -22105,                              /* Vector  176, coef 3(169) Vector
                                        * 177, Coef 1(167) */
  24249,                               /* Vector  177, coef 2( 94), 3(185) */
  -23208,                              /* Vector  178, coef 1(165), 2( 88) */
  -21594,                              /* Vector  178, coef 3(171) Vector
                                        * 179, Coef 1(166) */
  18602,                               /* Vector  179, coef 2( 72), 3(170) */
  -22938,                              /* Vector  180, coef 1(166), 2(102) */
  -20322,                              /* Vector  180, coef 3(176) Vector
                                        * 181, Coef 1(158) */
  24510,                               /* Vector  181, coef 2( 95), 3(190) */
  -24999,                              /* Vector  182, coef 1(158), 2( 89) */
  -20317,                              /* Vector  182, coef 3(176) Vector
                                        * 183, Coef 1(163) */
  26797,                               /* Vector  183, coef 2(104), 3(173) */
  -25761,                              /* Vector  184, coef 1(155), 2( 95) */
  -22615,                              /* Vector  184, coef 3(167) Vector
                                        * 185, Coef 1(169) */
  25764,                               /* Vector  185, coef 2(100), 3(164) */
  -23713,                              /* Vector  186, coef 1(163), 2( 95) */
  -24408,                              /* Vector  186, coef 3(160) Vector
                                        * 187, Coef 1(168) */
  22941,                               /* Vector  187, coef 2( 89), 3(157) */
  -24749,                              /* Vector  188, coef 1(159), 2( 83) */
  -26463,                              /* Vector  188, coef 3(152) Vector
                                        * 189, Coef 1(161) */
  20407,                               /* Vector  189, coef 2( 79), 3(183) */
  -25787,                              /* Vector  190, coef 1(155), 2( 69) */
  -20325,                              /* Vector  190, coef 3(176) Vector
                                        * 191, Coef 1(155) */
  20643,                               /* Vector  191, coef 2( 80), 3(163) */
  -23441,                              /* Vector  192, coef 1(164), 2(111) */
  -21848,                              /* Vector  192, coef 3(170) Vector
                                        * 193, Coef 1(168) */
  27808,                               /* Vector  193, coef 2(108), 3(160) */
  -23698,                              /* Vector  194, coef 1(163), 2(110) */
  -24925,                              /* Vector  194, coef 3(158) Vector
                                        * 195, Coef 1(163) */
  26779,                               /* Vector  195, coef 2(104), 3(155) */
  -25496,                              /* Vector  196, coef 1(156), 2(104) */
  -21602,                              /* Vector  196, coef 3(171) Vector
                                        * 197, Coef 1(158) */
  31139,                               /* Vector  197, coef 2(121), 3(163) */
  -22923,                              /* Vector  198, coef 1(166), 2(117) */
  -23902,                              /* Vector  198, coef 3(162) Vector
                                        * 199, Coef 1(162) */
  29851,                               /* Vector  199, coef 2(116), 3(155) */
  -24709,                              /* Vector  200, coef 1(159), 2(123) */
  -25188,                              /* Vector  200, coef 3(157) Vector
                                        * 201, Coef 1(156) */
  30361,                               /* Vector  201, coef 2(118), 3(153) */
  -24971,                              /* Vector  202, coef 1(158), 2(117) */
  -22115,                              /* Vector  202, coef 3(169) Vector
                                        * 203, Coef 1(157) */
  28579,                               /* Vector  203, coef 2(111), 3(163) */
  -26508,                              /* Vector  204, coef 1(152), 2(116) */
  -24425,                              /* Vector  204, coef 3(160) Vector
                                        * 205, Coef 1(151) */
  27555,                               /* Vector  205, coef 2(107), 3(163) */
  -25233,                              /* Vector  206, coef 1(157), 2(111) */
  -26724,                              /* Vector  206, coef 3(151) Vector
                                        * 207, Coef 1(156) */
  26525,                               /* Vector  207, coef 2(103), 3(157) */
  -27023,                              /* Vector  208, coef 1(150), 2(113) */
  -26223,                              /* Vector  208, coef 3(153) Vector
                                        * 209, Coef 1(145) */
  26785,                               /* Vector  209, coef 2(104), 3(161) */
  -27289,                              /* Vector  210, coef 1(149), 2(103) */
  -25971,                              /* Vector  210, coef 3(154) Vector
                                        * 211, Coef 1(141) */
  25238,                               /* Vector  211, coef 2( 98), 3(150) */
  -24717,                              /* Vector  212, coef 1(159), 2(115) */
  -29033,                              /* Vector  212, coef 3(142) Vector
                                        * 213, Coef 1(151) */
  27795,                               /* Vector  213, coef 2(108), 3(147) */
  -26766,                              /* Vector  214, coef 1(151), 2(114) */
  -29797,                              /* Vector  214, coef 3(139) Vector
                                        * 215, Coef 1(155) */
  27531,                               /* Vector  215, coef 2(107), 3(139) */
  -27799,                              /* Vector  216, coef 1(147), 2(105) */
  -29281,                              /* Vector  216, coef 3(141) Vector
                                        * 217, Coef 1(159) */
  26259,                               /* Vector  217, coef 2(102), 3(147) */
  -25760,                              /* Vector  218, coef 1(155), 2( 96) */
  -26475,                              /* Vector  218, coef 3(152) Vector
                                        * 219, Coef 1(149) */
  24467,                               /* Vector  219, coef 2( 95), 3(147) */
  -26270,                              /* Vector  220, coef 1(153), 2( 98) */
  -29804,                              /* Vector  220, coef 3(139) Vector
                                        * 221, Coef 1(148) */
  21905,                               /* Vector  221, coef 2( 85), 3(145) */
  -28821,                              /* Vector  222, coef 1(143), 2(107) */
  -27246,                              /* Vector  222, coef 3(149) Vector
                                        * 223, Coef 1(146) */
  27525,                               /* Vector  223, coef 2(107), 3(133) */
  -27300,                              /* Vector  224, coef 1(149), 2( 92) */
  -24174,                              /* Vector  224, coef 3(161) Vector
                                        * 225, Coef 1(146) */
  19357,                               /* Vector  225, coef 2( 75), 3(157) */
  -26270,                              /* Vector  226, coef 1(153), 2( 98) */
  -18542,                              /* Vector  226, coef 3(183) Vector
                                        * 227, Coef 1(146) */
  25534,                               /* Vector  227, coef 2( 99), 3(190) */
  -27813,                              /* Vector  228, coef 1(147), 2( 91) */
  -19562,                              /* Vector  228, coef 3(179) Vector
                                        * 229, Coef 1(150) */
  23236,                               /* Vector  229, coef 2( 90), 3(196) */
  -27570,                              /* Vector  230, coef 1(148), 2( 78) */
  -18031,                              /* Vector  230, coef 3(185) Vector
                                        * 231, Coef 1(145) */
  21168,                               /* Vector  231, coef 2( 82), 3(176) */
  -30129,                              /* Vector  232, coef 1(138), 2( 79) */
  -21102,                              /* Vector  232, coef 3(173) Vector
                                        * 233, Coef 1(146) */
  25514,                               /* Vector  233, coef 2( 99), 3(170) */
  -29600,                              /* Vector  234, coef 1(140), 2( 96) */
  -23410,                              /* Vector  234, coef 3(164) Vector
                                        * 235, Coef 1(142) */
  22941,                               /* Vector  235, coef 2( 89), 3(157) */
  -30888,                              /* Vector  236, coef 1(135), 2( 88) */
  -21376,                              /* Vector  236, coef 3(172) Vector
                                        * 237, Coef 1(128) */
  19109,                               /* Vector  237, coef 2( 74), 3(165) */
  -29345,                              /* Vector  238, coef 1(141), 2( 95) */
  -15739,                              /* Vector  238, coef 3(194) Vector
                                        * 239, Coef 1(133) */
  22463,                               /* Vector  239, coef 2( 87), 3(191) */
  -28565,                              /* Vector  240, coef 1(144), 2(107) */
  30605,                               /* Vector  240, coef 3(119) Vector
                                        * 241, Coef 1(141) */
  25707,                               /* Vector  241, coef 2(100), 3(107) */
  -30123,                              /* Vector  242, coef 1(138), 2( 85) */
  -27262,                              /* Vector  242, coef 3(149) Vector
                                        * 243, Coef 1(130) */
  21395,                               /* Vector  243, coef 2( 83), 3(147) */
  -30611,                              /* Vector  244, coef 1(136), 2(109) */
  -28280,                              /* Vector  244, coef 3(145) Vector
                                        * 245, Coef 1(136) */
  25997,                               /* Vector  245, coef 2(101), 3(141) */
  -32671,                              /* Vector  246, coef 1(128), 2( 97) */
  -28552,                              /* Vector  246, coef 3(144) Vector
                                        * 247, Coef 1(120) */
  23944,                               /* Vector  247, coef 2( 93), 3(136) */
  -28317,                              /* Vector  248, coef 1(145), 2( 99) */
  32647,                               /* Vector  248, coef 3(127) Vector
                                        * 249, Coef 1(135) */
  27523,                               /* Vector  249, coef 2(107), 3(131) */
  -32667,                              /* Vector  250, coef 1(128), 2(101) */
  31375,                               /* Vector  250, coef 3(122) Vector
                                        * 251, Coef 1(143) */
  24458,                               /* Vector  251, coef 2( 95), 3(138) */
  -30374,                              /* Vector  252, coef 1(137), 2( 90) */
  -31093,                              /* Vector  252, coef 3(134) Vector
                                        * 253, Coef 1(139) */
  21113,                               /* Vector  253, coef 2( 82), 3(121) */
  32620,                               /* Vector  254, coef 1(127), 2(108) */
  -30609,                              /* Vector  254, coef 3(136) Vector
                                        * 255, Coef 1(111) */
  24182,                               /* Vector  255, coef 2( 94), 3(118) */
  -14939,                              /* Vector  256, coef 1(197), 2(165) */
  -20545,                              /* Vector  256, coef 3(175) Vector
                                        * 257, Coef 1(191) */
  -25933,                              /* Vector  257, coef 2(154), 3(179) */
  -19049,                              /* Vector  258, coef 1(181), 2(151) */
  -22085,                              /* Vector  258, coef 3(169) Vector
                                        * 259, Coef 1(187) */
  -28481,                              /* Vector  259, coef 2(144), 3(191) */
  -19548,                              /* Vector  260, coef 1(179), 2(164) */
  -17488,                              /* Vector  260, coef 3(187) Vector
                                        * 261, Coef 1(176) */
  -25160,                              /* Vector  261, coef 2(157), 3(184) */
  -19560,                              /* Vector  262, coef 1(179), 2(152) */
  -15444,                              /* Vector  262, coef 3(195) Vector
                                        * 263, Coef 1(172) */
  -27207,                              /* Vector  263, coef 2(149), 3(185) */
  -20058,                              /* Vector  264, coef 1(177), 2(166) */
  -23120,                              /* Vector  264, coef 3(165) Vector
                                        * 265, Coef 1(176) */
  -25176,                              /* Vector  265, coef 2(157), 3(168) */
  -21864,                              /* Vector  266, coef 1(170), 2(152) */
  -20552,                              /* Vector  266, coef 3(175) Vector
                                        * 267, Coef 1(184) */
  -27986,                              /* Vector  267, coef 2(146), 3(174) */
  -20333,                              /* Vector  268, coef 1(176), 2(147) */
  -23378,                              /* Vector  268, coef 3(164) Vector
                                        * 269, Coef 1(174) */
  -29006,                              /* Vector  269, coef 2(142), 3(178) */
  -23902,                              /* Vector  270, coef 1(162), 2(162) */
  -16993,                              /* Vector  270, coef 3(189) Vector
                                        * 271, Coef 1(159) */
  -25166,                              /* Vector  271, coef 2(157), 3(178) */
  -19024,                              /* Vector  272, coef 1(181), 2(176) */
  -27987,                              /* Vector  272, coef 3(146) Vector
                                        * 273, Coef 1(173) */
  -22132,                              /* Vector  273, coef 2(169), 3(140) */
  -23624,                              /* Vector  274, coef 1(163), 2(184) */
  -27741,                              /* Vector  274, coef 3(147) Vector
                                        * 275, Coef 1(163) */
  -22898,                              /* Vector  275, coef 2(166), 3(142) */
  -19531,                              /* Vector  276, coef 1(179), 2(181) */
  32172,                               /* Vector  276, coef 3(125) Vector
                                        * 277, Coef 1(172) */
  -20306,                              /* Vector  277, coef 2(176), 3(174) */
  -23890,                              /* Vector  278, coef 1(162), 2(174) */
  -22375,                              /* Vector  278, coef 3(168) Vector
                                        * 279, Coef 1(153) */
  -19034,                              /* Vector  279, coef 2(181), 3(166) */
  -25945,                              /* Vector  280, coef 1(154), 2(167) */
  -23642,                              /* Vector  280, coef 3(163) Vector
                                        * 281, Coef 1(166) */
  -23393,                              /* Vector  281, coef 2(164), 3(159) */
  -23654,                              /* Vector  282, coef 1(163), 2(154) */
  -25703,                              /* Vector  282, coef 3(155) Vector
                                        * 283, Coef 1(153) */
  -23911,                              /* Vector  283, coef 2(162), 3(153) */
  -28749,                              /* Vector  284, coef 1(143), 2(179) */
  -26219,                              /* Vector  284, coef 3(153) Vector
                                        * 285, Coef 1(149) */
  -21116,                              /* Vector  285, coef 2(173), 3(132) */
  -26709,                              /* Vector  286, coef 1(151), 2(171) */
  -18027,                              /* Vector  286, coef 3(185) Vector
                                        * 287, Coef 1(149) */
  -22866,                              /* Vector  287, coef 2(166), 3(174) */
  -20340,                              /* Vector  288, coef 1(176), 2(140) */
  -15956,                              /* Vector  288, coef 3(193) Vector
                                        * 289, Coef 1(172) */
  -31045,                              /* Vector  289, coef 2(134), 3(187) */
  -22127,                              /* Vector  290, coef 1(169), 2(145) */
  -21336,                              /* Vector  290, coef 3(172) Vector
                                        * 291, Coef 1(168) */
  -27962,                              /* Vector  291, coef 2(146), 3(198) */
  -22642,                              /* Vector  292, coef 1(167), 2(142) */
  -17498,                              /* Vector  292, coef 3(187) Vector
                                        * 293, Coef 1(166) */
  -30268,                              /* Vector  293, coef 2(137), 3(196) */
  -24694,                              /* Vector  294, coef 1(159), 2(138) */
  -18009,                              /* Vector  294, coef 3(185) Vector
                                        * 295, Coef 1(167) */
  -30291,                              /* Vector  295, coef 2(137), 3(173) */
  -23420,                              /* Vector  296, coef 1(164), 2(132) */
  -19299,                              /* Vector  296, coef 3(180) Vector
                                        * 297, Coef 1(157) */
  -30775,                              /* Vector  297, coef 2(135), 3(201) */
  -26744,                              /* Vector  298, coef 1(151), 2(136) */
  -16988,                              /* Vector  298, coef 3(189) Vector
                                        * 299, Coef 1(164) */
  -31809,                              /* Vector  299, coef 2(131), 3(191) */
  -24683,                              /* Vector  300, coef 1(159), 2(149) */
  -15455,                              /* Vector  300, coef 3(195) Vector
                                        * 301, Coef 1(161) */
  -28231,                              /* Vector  301, coef 2(145), 3(185) */
  -25453,                              /* Vector  302, coef 1(156), 2(147) */
  -20078,                              /* Vector  302, coef 3(177) Vector
                                        * 303, Coef 1(146) */
  -31290,                              /* Vector  303, coef 2(133), 3(198) */
  -25463,                              /* Vector  304, coef 1(156), 2(137) */
  -20586,                              /* Vector  304, coef 3(175) Vector
                                        * 305, Coef 1(150) */
  -31830,                              /* Vector  305, coef 2(131), 3(170) */
  -28285,                              /* Vector  306, coef 1(145), 2(131) */
  -20082,                              /* Vector  306, coef 3(177) Vector
                                        * 307, Coef 1(142) */
  -31577,                              /* Vector  307, coef 2(132), 3(167) */
  -24426,                              /* Vector  308, coef 1(160), 2(150) */
  -22881,                              /* Vector  308, coef 3(166) Vector
                                        * 309, Coef 1(159) */
  -29274,                              /* Vector  309, coef 2(141), 3(166) */
  -25714,                              /* Vector  310, coef 1(155), 2(142) */
  -24669,                              /* Vector  310, coef 3(159) Vector
                                        * 311, Coef 1(163) */
  -30812,                              /* Vector  311, coef 2(135), 3(164) */
  -24702,                              /* Vector  312, coef 1(159), 2(130) */
  -23908,                              /* Vector  312, coef 3(162) Vector
                                        * 313, Coef 1(156) */
  -31331,                              /* Vector  313, coef 2(133), 3(157) */
  -28014,                              /* Vector  314, coef 1(146), 2(146) */
  -26220,                              /* Vector  314, coef 3(153) Vector
                                        * 315, Coef 1(148) */
  -30306,                              /* Vector  315, coef 2(137), 3(158) */
  -25984,                              /* Vector  316, coef 1(154), 2(128) */
  -22119,                              /* Vector  316, coef 3(169) Vector
                                        * 317, Coef 1(153) */
  32162,                               /* Vector  317, coef 2(125), 3(162) */
  -27776,                              /* Vector  318, coef 1(147), 2(128) */
  -24170,                              /* Vector  318, coef 3(161) Vector
                                        * 319, Coef 1(150) */
  -29266,                              /* Vector  319, coef 2(141), 3(174) */
  -20335,                              /* Vector  320, coef 1(176), 2(145) */
  30637,                               /* Vector  320, coef 3(119) Vector
                                        * 321, Coef 1(173) */
  -31102,                              /* Vector  321, coef 2(134), 3(130) */
  -22384,                              /* Vector  322, coef 1(168), 2(144) */
  -24670,                              /* Vector  322, coef 3(159) Vector
                                        * 323, Coef 1(162) */
  -30057,                              /* Vector  323, coef 2(138), 3(151) */
  -21848,                              /* Vector  324, coef 1(170), 2(168) */
  28832,                               /* Vector  324, coef 3(112) Vector
                                        * 325, Coef 1(160) */
  -25217,                              /* Vector  325, coef 2(157), 3(127) */
  -26471,                              /* Vector  326, coef 1(152), 2(153) */
  -27750,                              /* Vector  326, coef 3(147) Vector
                                        * 327, Coef 1(154) */
  -29290,                              /* Vector  327, coef 2(141), 3(150) */
  -22889,                              /* Vector  328, coef 1(166), 2(151) */
  -30804,                              /* Vector  328, coef 3(135) Vector
                                        * 329, Coef 1(172) */
  -29301,                              /* Vector  329, coef 2(141), 3(139) */
  -22904,                              /* Vector  330, coef 1(166), 2(136) */
  -28769,                              /* Vector  330, coef 3(143) Vector
                                        * 331, Coef 1(159) */
  -31350,                              /* Vector  331, coef 2(133), 3(138) */
  -23404,                              /* Vector  332, coef 1(164), 2(148) */
  -27745,                              /* Vector  332, coef 3(147) Vector
                                        * 333, Coef 1(159) */
  -28787,                              /* Vector  333, coef 2(143), 3(141) */
  -23669,                              /* Vector  334, coef 1(163), 2(139) */
  -32100,                              /* Vector  334, coef 3(130) Vector
                                        * 335, Coef 1(156) */
  -31613,                              /* Vector  335, coef 2(132), 3(131) */
  -25195,                              /* Vector  336, coef 1(157), 2(149) */
  29582,                               /* Vector  336, coef 3(115) Vector
                                        * 337, Coef 1(142) */
  -29803,                              /* Vector  337, coef 2(139), 3(149) */
  -26990,                              /* Vector  338, coef 1(150), 2(146) */
  -29034,                              /* Vector  338, coef 3(142) Vector
                                        * 339, Coef 1(150) */
  -28027,                              /* Vector  339, coef 2(146), 3(133) */
  -26742,                              /* Vector  340, coef 1(151), 2(138) */
  31880,                               /* Vector  340, coef 3(124) Vector
                                        * 341, Coef 1(136) */
  -22888,                              /* Vector  341, coef 2(166), 3(152) */
  -28257,                              /* Vector  342, coef 1(145), 2(159) */
  -29818,                              /* Vector  342, coef 3(139) Vector
                                        * 343, Coef 1(134) */
  -25212,                              /* Vector  343, coef 2(157), 3(132) */
  -30058,                              /* Vector  344, coef 1(138), 2(150) */
  -28022,                              /* Vector  344, coef 3(146) Vector
                                        * 345, Coef 1(138) */
  -29302,                              /* Vector  345, coef 2(141), 3(138) */
  -27998,                              /* Vector  346, coef 1(146), 2(162) */
  30856,                               /* Vector  346, coef 3(120) Vector
                                        * 347, Coef 1(136) */
  -22932,                              /* Vector  347, coef 2(166), 3(108) */
  32149,                               /* Vector  348, coef 1(125), 2(149) */
  29323,                               /* Vector  348, coef 3(114) Vector
                                        * 349, Coef 1(139) */
  -28546,                              /* Vector  349, coef 2(144), 3(126) */
  -30586,                              /* Vector  350, coef 1(136), 2(134) */
  32636,                               /* Vector  350, coef 3(127) Vector
                                        * 351, Coef 1(124) */
  -19824,                              /* Vector  351, coef 2(178), 3(144) */
  -24966,                              /* Vector  352, coef 1(158), 2(122) */
  -30560,                              /* Vector  352, coef 3(136) Vector
                                        * 353, Coef 1(160) */
  31123,                               /* Vector  353, coef 2(121), 3(147) */
  -26250,                              /* Vector  354, coef 1(153), 2(118) */
  -28265,                              /* Vector  354, coef 3(145) Vector
                                        * 355, Coef 1(151) */
  -32359,                              /* Vector  355, coef 2(129), 3(153) */
  -27515,                              /* Vector  356, coef 1(148), 2(133) */
  -27762,                              /* Vector  356, coef 3(147) Vector
                                        * 357, Coef 1(142) */
  -32104,                              /* Vector  357, coef 2(130), 3(152) */
  -24449,                              /* Vector  358, coef 1(160), 2(127) */
  -27237,                              /* Vector  358, coef 3(149) Vector
                                        * 359, Coef 1(155) */
  -31855,                              /* Vector  359, coef 2(131), 3(145) */
  -26755,                              /* Vector  360, coef 1(151), 2(125) */
  -29031,                              /* Vector  360, coef 3(142) Vector
                                        * 361, Coef 1(153) */
  31899,                               /* Vector  361, coef 2(124), 3(155) */
  -27526,                              /* Vector  362, coef 1(148), 2(122) */
  -26475,                              /* Vector  362, coef 3(152) Vector
                                        * 363, Coef 1(149) */
  -30323,                              /* Vector  363, coef 2(137), 3(141) */
  -28028,                              /* Vector  364, coef 1(146), 2(132) */
  -31092,                              /* Vector  364, coef 3(134) Vector
                                        * 365, Coef 1(140) */
  -31860,                              /* Vector  365, coef 2(131), 3(140) */
  -28037,                              /* Vector  366, coef 1(146), 2(123) */
  -30318,                              /* Vector  366, coef 3(137) Vector
                                        * 367, Coef 1(146) */
  30083,                               /* Vector  367, coef 2(117), 3(131) */
  -30081,                              /* Vector  368, coef 1(138), 2(127) */
  -27521,                              /* Vector  368, coef 3(148) Vector
                                        * 369, Coef 1(127) */
  -32391,                              /* Vector  369, coef 2(129), 3(121) */
  30072,                               /* Vector  370, coef 1(117), 2(120) */
  29582,                               /* Vector  370, coef 3(115) Vector
                                        * 371, Coef 1(142) */
  29583,                               /* Vector  371, coef 2(115), 3(143) */
  -31362,                              /* Vector  372, coef 1(133), 2(126) */
  -30837,                              /* Vector  372, coef 3(135) Vector
                                        * 373, Coef 1(139) */
  30851,                               /* Vector  373, coef 2(120), 3(131) */
  -31627,                              /* Vector  374, coef 1(132), 2(117) */
  31614,                               /* Vector  374, coef 3(123) Vector
                                        * 375, Coef 1(126) */
  32406,                               /* Vector  375, coef 2(126), 3(150) */
  -32122,                              /* Vector  376, coef 1(130), 2(134) */
  -29838,                              /* Vector  376, coef 3(139) Vector
                                        * 377, Coef 1(114) */
  -32380,                              /* Vector  377, coef 2(129), 3(132) */
  -30599,                              /* Vector  378, coef 1(136), 2(121) */
  -28026,                              /* Vector  378, coef 3(146) Vector
                                        * 379, Coef 1(134) */
  29837,                               /* Vector  379, coef 2(116), 3(141) */
  -32651,                              /* Vector  380, coef 1(128), 2(117) */
  -29576,                              /* Vector  380, coef 3(140) Vector
                                        * 381, Coef 1(120) */
  30858,                               /* Vector  381, coef 2(120), 3(138) */
  30063,                               /* Vector  382, coef 1(117), 2(111) */
  -32133,                              /* Vector  382, coef 3(130) Vector
                                        * 383, Coef 1(123) */
  29847,                               /* Vector  383, coef 2(116), 3(151) */
  -23180,                              /* Vector  384, coef 1(165), 2(116) */
  -20321,                              /* Vector  384, coef 3(176) Vector
                                        * 385, Coef 1(159) */
  -32337,                              /* Vector  385, coef 2(129), 3(175) */
  -25471,                              /* Vector  386, coef 1(156), 2(129) */
  -15210,                              /* Vector  386, coef 3(196) Vector
                                        * 387, Coef 1(150) */
  -32324,                              /* Vector  387, coef 2(129), 3(188) */
  -22917,                              /* Vector  388, coef 1(166), 2(123) */
  -15707,                              /* Vector  388, coef 3(194) Vector
                                        * 389, Coef 1(165) */
  31925,                               /* Vector  389, coef 2(124), 3(181) */
  -24194,                              /* Vector  390, coef 1(161), 2(126) */
  -17766,                              /* Vector  390, coef 3(186) Vector
                                        * 391, Coef 1(154) */
  31925,                               /* Vector  391, coef 2(124), 3(181) */
  -26249,                              /* Vector  392, coef 1(153), 2(119) */
  -20822,                              /* Vector  392, coef 3(174) Vector
                                        * 393, Coef 1(170) */
  30405,                               /* Vector  393, coef 2(118), 3(197) */
  -23697,                              /* Vector  394, coef 1(163), 2(111) */
  -15971,                              /* Vector  394, coef 3(193) Vector
                                        * 395, Coef 1(157) */
  27831,                               /* Vector  395, coef 2(108), 3(183) */
  -24203,                              /* Vector  396, coef 1(161), 2(117) */
  -17509,                              /* Vector  396, coef 3(187) Vector
                                        * 397, Coef 1(155) */
  29879,                               /* Vector  397, coef 2(116), 3(183) */
  -24455,                              /* Vector  398, coef 1(160), 2(121) */
  -14182,                              /* Vector  398, coef 3(200) Vector
                                        * 399, Coef 1(154) */
  31173,                               /* Vector  399, coef 2(121), 3(197) */
  -28561,                              /* Vector  400, coef 1(144), 2(111) */
  -25967,                              /* Vector  400, coef 3(154) Vector
                                        * 401, Coef 1(145) */
  30357,                               /* Vector  401, coef 2(118), 3(149) */
  -27271,                              /* Vector  402, coef 1(149), 2(121) */
  -22125,                              /* Vector  402, coef 3(169) Vector
                                        * 403, Coef 1(147) */
  30625,                               /* Vector  403, coef 2(119), 3(161) */
  -29570,                              /* Vector  404, coef 1(140), 2(126) */
  -24178,                              /* Vector  404, coef 3(161) Vector
                                        * 405, Coef 1(142) */
  30885,                               /* Vector  405, coef 2(120), 3(165) */
  -29578,                              /* Vector  406, coef 1(140), 2(118) */
  -25710,                              /* Vector  406, coef 3(155) Vector
                                        * 407, Coef 1(146) */
  28841,                               /* Vector  407, coef 2(112), 3(169) */
  -29073,                              /* Vector  408, coef 1(142), 2(111) */
  -24181,                              /* Vector  408, coef 3(161) Vector
                                        * 409, Coef 1(139) */
  27048,                               /* Vector  409, coef 2(105), 3(168) */
  -30613,                              /* Vector  410, coef 1(136), 2(107) */
  -24441,                              /* Vector  410, coef 3(160) Vector
                                        * 411, Coef 1(135) */
  29347,                               /* Vector  411, coef 2(114), 3(163) */
  -31626,                              /* Vector  412, coef 1(132), 2(118) */
  -26230,                              /* Vector  412, coef 3(153) Vector
                                        * 413, Coef 1(138) */
  28844,                               /* Vector  413, coef 2(112), 3(172) */
  -30083,                              /* Vector  414, coef 1(138), 2(125) */
  -21371,                              /* Vector  414, coef 3(172) Vector
                                        * 415, Coef 1(133) */
  31143,                               /* Vector  415, coef 2(121), 3(167) */
  -26007,                              /* Vector  416, coef 1(154), 2(105) */
  -15208,                              /* Vector  416, coef 3(196) Vector
                                        * 417, Coef 1(152) */
  28845,                               /* Vector  417, coef 2(112), 3(173) */
  -27287,                              /* Vector  418, coef 1(149), 2(105) */
  -20075,                              /* Vector  418, coef 3(177) Vector
                                        * 419, Coef 1(149) */
  30404,                               /* Vector  419, coef 2(118), 3(196) */
  -27532,                              /* Vector  420, coef 1(148), 2(116) */
  -18542,                              /* Vector  420, coef 3(183) Vector
                                        * 421, Coef 1(146) */
  31420,                               /* Vector  421, coef 2(122), 3(188) */
  -28548,                              /* Vector  422, coef 1(144), 2(124) */
  -19573,                              /* Vector  422, coef 3(179) Vector
                                        * 423, Coef 1(139) */
  30130,                               /* Vector  423, coef 2(117), 3(178) */
  -26255,                              /* Vector  424, coef 1(153), 2(113) */
  -14958,                              /* Vector  424, coef 3(197) Vector
                                        * 425, Coef 1(146) */
  27589,                               /* Vector  425, coef 2(107), 3(197) */
  -28562,                              /* Vector  426, coef 1(144), 2(110) */
  -17780,                              /* Vector  426, coef 3(186) Vector
                                        * 427, Coef 1(140) */
  26550,                               /* Vector  427, coef 2(103), 3(182) */
  -29068,                              /* Vector  428, coef 1(142), 2(116) */
  -14714,                              /* Vector  428, coef 3(198) Vector
                                        * 429, Coef 1(134) */
  29634,                               /* Vector  429, coef 2(115), 3(194) */
  -32152,                              /* Vector  430, coef 1(130), 2(104) */
  -15988,                              /* Vector  430, coef 3(193) Vector
                                        * 431, Coef 1(140) */
  32197,                               /* Vector  431, coef 2(125), 3(197) */
  30797,                               /* Vector  432, coef 1(120), 2( 77) */
  -26745,                              /* Vector  432, coef 3(151) Vector
                                        * 433, Coef 1(135) */
  26011,                               /* Vector  433, coef 2(101), 3(155) */
  -32163,                              /* Vector  434, coef 1(130), 2( 93) */
  -25213,                              /* Vector  434, coef 3(157) Vector
                                        * 435, Coef 1(131) */
  27804,                               /* Vector  435, coef 2(108), 3(156) */
  32362,                               /* Vector  436, coef 1(126), 2(106) */
  -27003,                              /* Vector  436, coef 3(150) Vector
                                        * 437, Coef 1(133) */
  24753,                               /* Vector  437, coef 2( 96), 3(177) */
  32088,                               /* Vector  438, coef 1(125), 2( 88) */
  -20860,                              /* Vector  438, coef 3(174) Vector
                                        * 439, Coef 1(132) */
  28595,                               /* Vector  439, coef 2(111), 3(179) */
  -32409,                              /* Vector  440, coef 1(129), 2(103) */
  -21638,                              /* Vector  440, coef 3(171) Vector
                                        * 441, Coef 1(122) */
  24508,                               /* Vector  441, coef 2( 95), 3(188) */
  29539,                               /* Vector  442, coef 1(115), 2( 99) */
  -17555,                              /* Vector  442, coef 3(187) Vector
                                        * 443, Coef 1(109) */
  21420,                               /* Vector  443, coef 2( 83), 3(172) */
  31340,                               /* Vector  444, coef 1(122), 2(108) */
  -23689,                              /* Vector  444, coef 3(163) Vector
                                        * 445, Coef 1(119) */
  24997,                               /* Vector  445, coef 2( 97), 3(165) */
  29792,                               /* Vector  446, coef 1(116), 2( 96) */
  -25744,                              /* Vector  446, coef 3(155) Vector
                                        * 447, Coef 1(112) */
  22421,                               /* Vector  447, coef 2( 87), 3(149) */
  -26472,                              /* Vector  448, coef 1(152), 2(152) */
  -23919,                              /* Vector  448, coef 3(162) Vector
                                        * 449, Coef 1(145) */
  -28252,                              /* Vector  449, coef 2(145), 3(164) */
  -30071,                              /* Vector  450, coef 1(138), 2(137) */
  -24689,                              /* Vector  450, coef 3(159) Vector
                                        * 451, Coef 1(143) */
  -24898,                              /* Vector  451, coef 2(158), 3(190) */
  -27750,                              /* Vector  452, coef 1(147), 2(154) */
  -20331,                              /* Vector  452, coef 3(176) Vector
                                        * 453, Coef 1(149) */
  -27456,                              /* Vector  453, coef 2(148), 3(192) */
  -29043,                              /* Vector  454, coef 1(142), 2(141) */
  -16498,                              /* Vector  454, coef 3(191) Vector
                                        * 455, Coef 1(142) */
  -28239,                              /* Vector  455, coef 2(145), 3(177) */
  -31347,                              /* Vector  456, coef 1(133), 2(141) */
  -20086,                              /* Vector  456, coef 3(177) Vector
                                        * 457, Coef 1(138) */
  -30294,                              /* Vector  457, coef 2(137), 3(170) */
  -31059,                              /* Vector  458, coef 1(134), 2(173) */
  -19070,                              /* Vector  458, coef 3(181) Vector
                                        * 459, Coef 1(130) */
  -24395,                              /* Vector  459, coef 2(160), 3(181) */
  -29540,                              /* Vector  460, coef 1(140), 2(156) */
  -23414,                              /* Vector  460, coef 3(164) Vector
                                        * 461, Coef 1(138) */
  -31816,                              /* Vector  461, coef 2(131), 3(184) */
  -30827,                              /* Vector  462, coef 1(135), 2(149) */
  -16001,                              /* Vector  462, coef 3(193) Vector
                                        * 463, Coef 1(127) */
  -29000,                              /* Vector  463, coef 2(142), 3(184) */
  32662,                               /* Vector  464, coef 1(127), 2(150) */
  -29323,                              /* Vector  464, coef 3(141) Vector
                                        * 465, Coef 1(117) */
  -28027,                              /* Vector  465, coef 2(146), 3(133) */
  -32376,                              /* Vector  466, coef 1(129), 2(136) */
  -25723,                              /* Vector  466, coef 3(155) Vector
                                        * 467, Coef 1(133) */
  -30061,                              /* Vector  467, coef 2(138), 3(147) */
  31108,                               /* Vector  468, coef 1(121), 2(132) */
  -28797,                              /* Vector  468, coef 3(143) Vector
                                        * 469, Coef 1(131) */
  -27742,                              /* Vector  469, coef 2(147), 3(162) */
  32412,                               /* Vector  470, coef 1(126), 2(156) */
  -25735,                              /* Vector  470, coef 3(155) Vector
                                        * 471, Coef 1(121) */
  -28523,                              /* Vector  471, coef 2(144), 3(149) */
  30366,                               /* Vector  472, coef 1(118), 2(158) */
  -19854,                              /* Vector  472, coef 3(178) Vector
                                        * 473, Coef 1(114) */
  -25181,                              /* Vector  473, coef 2(157), 3(163) */
  26774,                               /* Vector  474, coef 1(104), 2(150) */
  -20110,                              /* Vector  474, coef 3(177) Vector
                                        * 475, Coef 1(114) */
  -22133,                              /* Vector  475, coef 2(169), 3(139) */
  25759,                               /* Vector  476, coef 1(100), 2(159) */
  -29838,                              /* Vector  476, coef 3(139) Vector
                                        * 477, Coef 1(114) */
  -29256,                              /* Vector  477, coef 2(141), 3(184) */
  30091,                               /* Vector  478, coef 1(117), 2(139) */
  -24211,                              /* Vector  478, coef 3(161) Vector
                                        * 479, Coef 1(109) */
  -30824,                              /* Vector  479, coef 2(135), 3(152) */
  -31873,                              /* Vector  480, coef 1(131), 2(127) */
  -24962,                              /* Vector  480, coef 3(158) Vector
                                        * 481, Coef 1(126) */
  29104,                               /* Vector  481, coef 2(113), 3(176) */
  29806,                               /* Vector  482, coef 1(116), 2(110) */
  -21373,                              /* Vector  482, coef 3(172) Vector
                                        * 483, Coef 1(131) */
  -31831,                              /* Vector  483, coef 2(131), 3(169) */
  32136,                               /* Vector  484, coef 1(125), 2(136) */
  -22151,                              /* Vector  484, coef 3(169) Vector
                                        * 485, Coef 1(121) */
  -32093,                              /* Vector  485, coef 2(130), 3(163) */
  -31622,                              /* Vector  486, coef 1(132), 2(122) */
  -18299,                              /* Vector  486, coef 3(184) Vector
                                        * 487, Coef 1(133) */
  -31291,                              /* Vector  487, coef 2(133), 3(197) */
  31871,                               /* Vector  488, coef 1(124), 2(127) */
  -18049,                              /* Vector  488, coef 3(185) Vector
                                        * 489, Coef 1(127) */
  30402,                               /* Vector  489, coef 2(118), 3(194) */
  30066,                               /* Vector  490, coef 1(117), 2(114) */
  -16513,                              /* Vector  490, coef 3(191) Vector
                                        * 491, Coef 1(127) */
  31661,                               /* Vector  491, coef 2(123), 3(173) */
  32629,                               /* Vector  492, coef 1(127), 2(117) */
  -23943,                              /* Vector  492, coef 3(162) Vector
                                        * 493, Coef 1(121) */
  31401,                               /* Vector  493, coef 2(122), 3(169) */
  30328,                               /* Vector  494, coef 1(118), 2(120) */
  -24972,                              /* Vector  494, coef 3(158) Vector
                                        * 495, Coef 1(116) */
  32704,                               /* Vector  495, coef 2(127), 3(192) */
  25967,                               /* Vector  496, coef 1(101), 2(111) */
  -31891,                              /* Vector  496, coef 3(131) Vector
                                        * 497, Coef 1(109) */
  26786,                               /* Vector  497, coef 2(104), 3(162) */
  26459,                               /* Vector  498, coef 1(103), 2( 91) */
  -29601,                              /* Vector  498, coef 3(140) Vector
                                        * 499, Coef 1( 95) */
  -27230,                              /* Vector  499, coef 2(149), 3(162) */
  23943,                               /* Vector  500, coef 1( 93), 2(135) */
  -29068,                              /* Vector  500, coef 3(142) Vector
                                        * 501, Coef 1(116) */
  27538,                               /* Vector  501, coef 2(107), 3(146) */
  28546,                               /* Vector  502, coef 1(111), 2(130) */
  -21908,                              /* Vector  502, coef 3(170) Vector
                                        * 503, Coef 1(108) */
  30132,                               /* Vector  503, coef 2(117), 3(180) */
  26727,                               /* Vector  504, coef 1(104), 2(103) */
  -18849,                              /* Vector  504, coef 3(182) Vector
                                        * 505, Coef 1( 95) */
  26795,                               /* Vector  505, coef 2(104), 3(171) */
  28537,                               /* Vector  506, coef 1(111), 2(121) */
  -26007,                              /* Vector  506, coef 3(154) Vector
                                        * 507, Coef 1(105) */
  29329,                               /* Vector  507, coef 2(114), 3(145) */
  25984,                               /* Vector  508, coef 1(101), 2(128) */
  -19116,                              /* Vector  508, coef 3(181) Vector
                                        * 509, Coef 1( 84) */
  -31577,                              /* Vector  509, coef 2(132), 3(167) */
  25463,                               /* Vector  510, coef 1( 99), 2(119) */
  -24489,                              /* Vector  510, coef 3(160) Vector
                                        * 511, Coef 1( 87) */
  27538                                /* Vector  511, coef 2(107), 3(146) */
};

ShortwordRom psrQuant3[QUANT3_NUM_OF_WORDS] =
{
  -26192,                              /* Vector    0, coef 1(153), 2(176) */
  -32358,                              /* Vector    0, coef 3(129), 4(154) */
  -27991,                              /* Vector    1, coef 1(146), 2(169) */
  31890,                               /* Vector    1, coef 3(124), 4(146) */
  -27731,                              /* Vector    2, coef 1(147), 2(173) */
  30625,                               /* Vector    2, coef 3(119), 4(161) */
  -28246,                              /* Vector    3, coef 1(145), 2(170) */
  29078,                               /* Vector    3, coef 3(113), 4(150) */
  -28501,                              /* Vector    4, coef 1(144), 2(171) */
  -31610,                              /* Vector    4, coef 3(132), 4(134) */
  -30302,                              /* Vector    5, coef 1(137), 2(162) */
  30347,                               /* Vector    5, coef 3(118), 4(139) */
  -25161,                              /* Vector    6, coef 1(157), 2(183) */
  29326,                               /* Vector    6, coef 3(114), 4(142) */
  -26959,                              /* Vector    7, coef 1(150), 2(177) */
  27784,                               /* Vector    7, coef 3(108), 4(136) */
  -27468,                              /* Vector    8, coef 1(148), 2(180) */
  26514,                               /* Vector    8, coef 3(103), 4(146) */
  -30561,                              /* Vector    9, coef 1(136), 2(159) */
  29088,                               /* Vector    9, coef 3(113), 4(160) */
  -28484,                              /* Vector   10, coef 1(144), 2(188) */
  31120,                               /* Vector   10, coef 3(121), 4(144) */
  -29789,                              /* Vector   11, coef 1(139), 2(163) */
  -30559,                              /* Vector   11, coef 3(136), 4(161) */
  -30302,                              /* Vector   12, coef 1(137), 2(162) */
  -32361,                              /* Vector   12, coef 3(129), 4(151) */
  -30548,                              /* Vector   13, coef 1(136), 2(172) */
  31646,                               /* Vector   13, coef 3(123), 4(158) */
  -25687,                              /* Vector   14, coef 1(155), 2(169) */
  29589,                               /* Vector   14, coef 3(115), 4(149) */
  -29286,                              /* Vector   15, coef 1(141), 2(154) */
  30359,                               /* Vector   15, coef 3(118), 4(151) */
  -31069,                              /* Vector   16, coef 1(134), 2(163) */
  -32127,                              /* Vector   16, coef 3(130), 4(129) */
  -31074,                              /* Vector   17, coef 1(134), 2(158) */
  29819,                               /* Vector   17, coef 3(116), 4(123) */
  -29016,                              /* Vector   18, coef 1(142), 2(168) */
  23178,                               /* Vector   18, coef 3( 90), 4(138) */
  -30027,                              /* Vector   19, coef 1(138), 2(181) */
  30600,                               /* Vector   19, coef 3(119), 4(136) */
  -28995,                              /* Vector   20, coef 1(142), 2(189) */
  27787,                               /* Vector   20, coef 3(108), 4(139) */
  -31305,                              /* Vector   21, coef 1(133), 2(183) */
  27527,                               /* Vector   21, coef 3(107), 4(135) */
  -31049,                              /* Vector   22, coef 1(134), 2(183) */
  29808,                               /* Vector   22, coef 3(116), 4(112) */
  32171,                               /* Vector   23, coef 1(125), 2(171) */
  26486,                               /* Vector   23, coef 3(103), 4(118) */
  -30295,                              /* Vector   24, coef 1(137), 2(169) */
  27026,                               /* Vector   24, coef 3(105), 4(146) */
  -32096,                              /* Vector   25, coef 1(130), 2(160) */
  27014,                               /* Vector   25, coef 3(105), 4(134) */
  32677,                               /* Vector   26, coef 1(127), 2(165) */
  24203,                               /* Vector   26, coef 3( 94), 4(139) */
  31132,                               /* Vector   27, coef 1(121), 2(156) */
  22399,                               /* Vector   27, coef 3( 87), 4(127) */
  -26968,                              /* Vector   28, coef 1(150), 2(168) */
  26231,                               /* Vector   28, coef 3(102), 4(119) */
  -30302,                              /* Vector   29, coef 1(137), 2(162) */
  24437,                               /* Vector   29, coef 3( 95), 4(117) */
  31391,                               /* Vector   30, coef 1(122), 2(159) */
  28304,                               /* Vector   30, coef 3(110), 4(144) */
  31400,                               /* Vector   31, coef 1(122), 2(168) */
  30599,                               /* Vector   31, coef 3(119), 4(135) */
  -31076,                              /* Vector   32, coef 1(134), 2(156) */
  27796,                               /* Vector   32, coef 3(108), 4(148) */
  -31335,                              /* Vector   33, coef 1(133), 2(153) */
  32672,                               /* Vector   33, coef 3(127), 4(160) */
  32153,                               /* Vector   34, coef 1(125), 2(153) */
  31896,                               /* Vector   34, coef 3(124), 4(152) */
  -31853,                              /* Vector   35, coef 1(131), 2(147) */
  30613,                               /* Vector   35, coef 3(119), 4(149) */
  31892,                               /* Vector   36, coef 1(124), 2(148) */
  29840,                               /* Vector   36, coef 3(116), 4(144) */
  -32100,                              /* Vector   37, coef 1(130), 2(156) */
  32654,                               /* Vector   37, coef 3(127), 4(142) */
  -32616,                              /* Vector   38, coef 1(128), 2(152) */
  29576,                               /* Vector   38, coef 3(115), 4(136) */
  32140,                               /* Vector   39, coef 1(125), 2(140) */
  31375,                               /* Vector   39, coef 3(122), 4(143) */
  32408,                               /* Vector   40, coef 1(126), 2(152) */
  30369,                               /* Vector   40, coef 3(118), 4(161) */
  31890,                               /* Vector   41, coef 1(124), 2(146) */
  28062,                               /* Vector   41, coef 3(109), 4(158) */
  31373,                               /* Vector   42, coef 1(122), 2(141) */
  30619,                               /* Vector   42, coef 3(119), 4(155) */
  31129,                               /* Vector   43, coef 1(121), 2(153) */
  28826,                               /* Vector   43, coef 3(112), 4(154) */
  32390,                               /* Vector   44, coef 1(126), 2(134) */
  27555,                               /* Vector   44, coef 3(107), 4(163) */
  32159,                               /* Vector   45, coef 1(125), 2(159) */
  30099,                               /* Vector   45, coef 3(117), 4(147) */
  30362,                               /* Vector   46, coef 1(118), 2(154) */
  32674,                               /* Vector   46, coef 3(127), 4(162) */
  -32103,                              /* Vector   47, coef 1(130), 2(153) */
  -28774,                              /* Vector   47, coef 3(143), 4(154) */
  32660,                               /* Vector   48, coef 1(127), 2(148) */
  22937,                               /* Vector   48, coef 3( 89), 4(153) */
  30599,                               /* Vector   49, coef 1(119), 2(135) */
  22164,                               /* Vector   49, coef 3( 86), 4(148) */
  29847,                               /* Vector   50, coef 1(116), 2(151) */
  24465,                               /* Vector   50, coef 3( 95), 4(145) */
  27534,                               /* Vector   51, coef 1(107), 2(142) */
  21905,                               /* Vector   51, coef 3( 85), 4(145) */
  30859,                               /* Vector   52, coef 1(120), 2(139) */
  27788,                               /* Vector   52, coef 3(108), 4(140) */
  28803,                               /* Vector   53, coef 1(112), 2(131) */
  26763,                               /* Vector   53, coef 3(104), 4(139) */
  30092,                               /* Vector   54, coef 1(117), 2(140) */
  29333,                               /* Vector   54, coef 3(114), 4(149) */
  28295,                               /* Vector   55, coef 1(110), 2(135) */
  29081,                               /* Vector   55, coef 3(113), 4(153) */
  28545,                               /* Vector   56, coef 1(111), 2(129) */
  26786,                               /* Vector   56, coef 3(104), 4(162) */
  29843,                               /* Vector   57, coef 1(116), 2(147) */
  27283,                               /* Vector   57, coef 3(106), 4(147) */
  26249,                               /* Vector   58, coef 1(102), 2(137) */
  26261,                               /* Vector   58, coef 3(102), 4(149) */
  24707,                               /* Vector   59, coef 1( 96), 2(131) */
  29600,                               /* Vector   59, coef 3(115), 4(160) */
  31618,                               /* Vector   60, coef 1(123), 2(130) */
  26772,                               /* Vector   60, coef 3(104), 4(148) */
  27790,                               /* Vector   61, coef 1(108), 2(142) */
  27778,                               /* Vector   61, coef 3(108), 4(130) */
  31880,                               /* Vector   62, coef 1(124), 2(136) */
  27518,                               /* Vector   62, coef 3(107), 4(126) */
  27022,                               /* Vector   63, coef 1(105), 2(142) */
  29840,                               /* Vector   63, coef 3(116), 4(144) */
  -23384,                              /* Vector   64, coef 1(164), 2(168) */
  31896,                               /* Vector   64, coef 3(124), 4(152) */
  -25701,                              /* Vector   65, coef 1(155), 2(155) */
  31373,                               /* Vector   65, coef 3(122), 4(141) */
  -27495,                              /* Vector   66, coef 1(148), 2(153) */
  29317,                               /* Vector   66, coef 3(114), 4(133) */
  -23127,                              /* Vector   67, coef 1(165), 2(169) */
  27534,                               /* Vector   67, coef 3(107), 4(142) */
  -25176,                              /* Vector   68, coef 1(157), 2(168) */
  24974,                               /* Vector   68, coef 3( 97), 4(142) */
  -27231,                              /* Vector   69, coef 1(149), 2(161) */
  27285,                               /* Vector   69, coef 3(106), 4(149) */
  -23917,                              /* Vector   70, coef 1(162), 2(147) */
  22157,                               /* Vector   70, coef 3( 86), 4(141) */
  -26979,                              /* Vector   71, coef 1(150), 2(157) */
  30625,                               /* Vector   71, coef 3(119), 4(161) */
  -27245,                              /* Vector   72, coef 1(149), 2(147) */
  28569,                               /* Vector   72, coef 3(111), 4(153) */
  -30062,                              /* Vector   73, coef 1(138), 2(146) */
  29580,                               /* Vector   73, coef 3(115), 4(140) */
  -22890,                              /* Vector   74, coef 1(166), 2(150) */
  28565,                               /* Vector   74, coef 3(111), 4(149) */
  -24436,                              /* Vector   75, coef 1(160), 2(140) */
  27277,                               /* Vector   75, coef 3(106), 4(141) */
  -27509,                              /* Vector   76, coef 1(148), 2(139) */
  28814,                               /* Vector   76, coef 3(112), 4(142) */
  -26725,                              /* Vector   77, coef 1(151), 2(155) */
  -30573,                              /* Vector   77, coef 3(136), 4(147) */
  -25961,                              /* Vector   78, coef 1(154), 2(151) */
  -31877,                              /* Vector   78, coef 3(131), 4(123) */
  -26739,                              /* Vector   79, coef 1(151), 2(141) */
  30071,                               /* Vector   79, coef 3(117), 4(119) */
  -30829,                              /* Vector   80, coef 1(135), 2(147) */
  26777,                               /* Vector   80, coef 3(104), 4(153) */
  -32626,                              /* Vector   81, coef 1(128), 2(142) */
  25999,                               /* Vector   81, coef 3(101), 4(143) */
  -28775,                              /* Vector   82, coef 1(143), 2(153) */
  25992,                               /* Vector   82, coef 3(101), 4(136) */
  -26988,                              /* Vector   83, coef 1(150), 2(148) */
  22913,                               /* Vector   83, coef 3( 89), 4(129) */
  -28007,                              /* Vector   84, coef 1(146), 2(153) */
  21903,                               /* Vector   84, coef 3( 85), 4(143) */
  -30575,                              /* Vector   85, coef 1(136), 2(145) */
  21132,                               /* Vector   85, coef 3( 82), 4(140) */
  -31594,                              /* Vector   86, coef 1(132), 2(150) */
  24707,                               /* Vector   86, coef 3( 96), 4(131) */
  -31851,                              /* Vector   87, coef 1(131), 2(149) */
  21629,                               /* Vector   87, coef 3( 84), 4(125) */
  -30068,                              /* Vector   88, coef 1(138), 2(140) */
  27542,                               /* Vector   88, coef 3(107), 4(150) */
  -31604,                              /* Vector   89, coef 1(132), 2(140) */
  28807,                               /* Vector   89, coef 3(112), 4(135) */
  -31358,                              /* Vector   90, coef 1(133), 2(130) */
  24708,                               /* Vector   90, coef 3( 96), 4(132) */
  -27509,                              /* Vector   91, coef 1(148), 2(139) */
  25240,                               /* Vector   91, coef 3( 98), 4(152) */
  -28542,                              /* Vector   92, coef 1(144), 2(130) */
  21907,                               /* Vector   92, coef 3( 85), 4(147) */
  -30328,                              /* Vector   93, coef 1(137), 2(136) */
  21915,                               /* Vector   93, coef 3( 85), 4(155) */
  -31872,                              /* Vector   94, coef 1(131), 2(128) */
  21907,                               /* Vector   94, coef 3( 85), 4(147) */
  -31092,                              /* Vector   95, coef 1(134), 2(140) */
  28271,                               /* Vector   95, coef 3(110), 4(111) */
  -30846,                              /* Vector   96, coef 1(135), 2(130) */
  27787,                               /* Vector   96, coef 3(108), 4(139) */
  -28028,                              /* Vector   97, coef 1(146), 2(132) */
  27522,                               /* Vector   97, coef 3(107), 4(130) */
  -26494,                              /* Vector   98, coef 1(152), 2(130) */
  32399,                               /* Vector   98, coef 3(126), 4(143) */
  -28809,                              /* Vector   99, coef 1(143), 2(119) */
  30350,                               /* Vector   99, coef 3(118), 4(142) */
  -28047,                              /* Vector  100, coef 1(146), 2(113) */
  32382,                               /* Vector  100, coef 3(126), 4(126) */
  -30860,                              /* Vector  101, coef 1(135), 2(116) */
  28546,                               /* Vector  101, coef 3(111), 4(130) */
  -24456,                              /* Vector  102, coef 1(160), 2(120) */
  25747,                               /* Vector  102, coef 3(100), 4(147) */
  -27019,                              /* Vector  103, coef 1(150), 2(117) */
  22680,                               /* Vector  103, coef 3( 88), 4(152) */
  -29831,                              /* Vector  104, coef 1(139), 2(121) */
  26776,                               /* Vector  104, coef 3(104), 4(152) */
  -31633,                              /* Vector  105, coef 1(132), 2(111) */
  23964,                               /* Vector  105, coef 3( 93), 4(156) */
  -28818,                              /* Vector  106, coef 1(143), 2(110) */
  31384,                               /* Vector  106, coef 3(122), 4(152) */
  -28829,                              /* Vector  107, coef 1(143), 2( 99) */
  28053,                               /* Vector  107, coef 3(109), 4(149) */
  -30841,                              /* Vector  108, coef 1(135), 2(135) */
  30365,                               /* Vector  108, coef 3(118), 4(157) */
  -32382,                              /* Vector  109, coef 1(129), 2(130) */
  29587,                               /* Vector  109, coef 3(115), 4(147) */
  -31875,                              /* Vector  110, coef 1(131), 2(125) */
  32660,                               /* Vector  110, coef 3(127), 4(148) */
  32628,                               /* Vector  111, coef 1(127), 2(116) */
  30862,                               /* Vector  111, coef 3(120), 4(142) */
  27002,                               /* Vector  112, coef 1(105), 2(122) */
  25753,                               /* Vector  112, coef 3(100), 4(153) */
  31356,                               /* Vector  113, coef 1(122), 2(124) */
  23711,                               /* Vector  113, coef 3( 92), 4(159) */
  31606,                               /* Vector  114, coef 1(123), 2(118) */
  24978,                               /* Vector  114, coef 3( 97), 4(146) */
  32361,                               /* Vector  115, coef 1(126), 2(105) */
  30369,                               /* Vector  115, coef 3(118), 4(161) */
  31844,                               /* Vector  116, coef 1(124), 2(100) */
  29333,                               /* Vector  116, coef 3(114), 4(149) */
  30588,                               /* Vector  117, coef 1(119), 2(124) */
  29838,                               /* Vector  117, coef 3(116), 4(142) */
  30324,                               /* Vector  118, coef 1(118), 2(116) */
  28294,                               /* Vector  118, coef 3(110), 4(134) */
  30081,                               /* Vector  119, coef 1(117), 2(129) */
  30870,                               /* Vector  119, coef 3(120), 4(150) */
  32121,                               /* Vector  120, coef 1(125), 2(121) */
  29598,                               /* Vector  120, coef 3(115), 4(158) */
  29297,                               /* Vector  121, coef 1(114), 2(113) */
  26785,                               /* Vector  121, coef 3(104), 4(161) */
  28029,                               /* Vector  122, coef 1(109), 2(125) */
  32672,                               /* Vector  122, coef 3(127), 4(160) */
  25720,                               /* Vector  123, coef 1(100), 2(120) */
  30616,                               /* Vector  123, coef 3(119), 4(152) */
  28784,                               /* Vector  124, coef 1(112), 2(112) */
  30359,                               /* Vector  124, coef 3(118), 4(151) */
  26733,                               /* Vector  125, coef 1(104), 2(109) */
  28305,                               /* Vector  125, coef 3(110), 4(145) */
  26988,                               /* Vector  126, coef 1(105), 2(108) */
  -31083,                              /* Vector  126, coef 3(134), 4(149) */
  27774,                               /* Vector  127, coef 1(108), 2(126) */
  31370,                               /* Vector  127, coef 3(122), 4(138) */
  -32092,                              /* Vector  128, coef 1(130), 2(164) */
  31128,                               /* Vector  128, coef 3(121), 4(152) */
  -30794,                              /* Vector  129, coef 1(135), 2(182) */
  -31079,                              /* Vector  129, coef 3(134), 4(153) */
  -32336,                              /* Vector  130, coef 1(129), 2(176) */
  -32370,                              /* Vector  130, coef 3(129), 4(142) */
  -32575,                              /* Vector  131, coef 1(128), 2(193) */
  -32631,                              /* Vector  131, coef 3(128), 4(137) */
  31932,                               /* Vector  132, coef 1(124), 2(188) */
  29069,                               /* Vector  132, coef 3(113), 4(141) */
  -31561,                              /* Vector  133, coef 1(132), 2(183) */
  30364,                               /* Vector  133, coef 3(118), 4(156) */
  32427,                               /* Vector  134, coef 1(126), 2(171) */
  29590,                               /* Vector  134, coef 3(115), 4(150) */
  30141,                               /* Vector  135, coef 1(117), 2(189) */
  -31342,                              /* Vector  135, coef 3(133), 4(146) */
  29874,                               /* Vector  136, coef 1(116), 2(178) */
  32406,                               /* Vector  136, coef 3(126), 4(150) */
  27327,                               /* Vector  137, coef 1(106), 2(191) */
  -31858,                              /* Vector  137, coef 3(131), 4(142) */
  29877,                               /* Vector  138, coef 1(116), 2(181) */
  31872,                               /* Vector  138, coef 3(124), 4(128) */
  31654,                               /* Vector  139, coef 1(123), 2(166) */
  -29297,                              /* Vector  139, coef 3(141), 4(143) */
  31654,                               /* Vector  140, coef 1(123), 2(166) */
  -30813,                              /* Vector  140, coef 3(135), 4(163) */
  30369,                               /* Vector  141, coef 1(118), 2(161) */
  -31077,                              /* Vector  141, coef 3(134), 4(155) */
  30881,                               /* Vector  142, coef 1(120), 2(161) */
  32147,                               /* Vector  142, coef 3(125), 4(147) */
  27820,                               /* Vector  143, coef 1(108), 2(172) */
  -29032,                              /* Vector  143, coef 3(142), 4(152) */
  32159,                               /* Vector  144, coef 1(125), 2(159) */
  -31889,                              /* Vector  144, coef 3(131), 4(111) */
  29849,                               /* Vector  145, coef 1(116), 2(153) */
  28785,                               /* Vector  145, coef 3(112), 4(113) */
  31896,                               /* Vector  146, coef 1(124), 2(152) */
  32135,                               /* Vector  146, coef 3(125), 4(135) */
  31892,                               /* Vector  147, coef 1(124), 2(148) */
  27775,                               /* Vector  147, coef 3(108), 4(127) */
  27821,                               /* Vector  148, coef 1(108), 2(173) */
  29557,                               /* Vector  148, coef 3(115), 4(117) */
  28840,                               /* Vector  149, coef 1(112), 2(168) */
  27017,                               /* Vector  149, coef 3(105), 4(137) */
  26270,                               /* Vector  150, coef 1(102), 2(158) */
  25731,                               /* Vector  150, coef 3(100), 4(131) */
  30623,                               /* Vector  151, coef 1(119), 2(159) */
  -31352,                              /* Vector  151, coef 3(133), 4(136) */
  28582,                               /* Vector  152, coef 1(111), 2(166) */
  -30079,                              /* Vector  152, coef 3(138), 4(129) */
  27801,                               /* Vector  153, coef 1(108), 2(153) */
  -31622,                              /* Vector  153, coef 3(132), 4(122) */
  29327,                               /* Vector  154, coef 1(114), 2(143) */
  -28299,                              /* Vector  154, coef 3(145), 4(117) */
  29600,                               /* Vector  155, coef 1(115), 2(160) */
  30337,                               /* Vector  155, coef 3(118), 4(129) */
  29848,                               /* Vector  156, coef 1(116), 2(152) */
  28552,                               /* Vector  156, coef 3(111), 4(136) */
  27291,                               /* Vector  157, coef 1(106), 2(155) */
  32392,                               /* Vector  157, coef 3(126), 4(136) */
  30864,                               /* Vector  158, coef 1(120), 2(144) */
  31871,                               /* Vector  158, coef 3(124), 4(127) */
  29574,                               /* Vector  159, coef 1(115), 2(134) */
  30331,                               /* Vector  159, coef 3(118), 4(123) */
  29326,                               /* Vector  160, coef 1(114), 2(142) */
  30373,                               /* Vector  160, coef 3(118), 4(165) */
  27538,                               /* Vector  161, coef 1(107), 2(146) */
  27553,                               /* Vector  161, coef 3(107), 4(161) */
  27566,                               /* Vector  162, coef 1(107), 2(174) */
  31119,                               /* Vector  162, coef 3(121), 4(143) */
  29336,                               /* Vector  163, coef 1(114), 2(152) */
  31388,                               /* Vector  163, coef 3(122), 4(156) */
  29342,                               /* Vector  164, coef 1(114), 2(158) */
  31632,                               /* Vector  164, coef 3(123), 4(144) */
  29845,                               /* Vector  165, coef 1(116), 2(149) */
  31379,                               /* Vector  165, coef 3(122), 4(147) */
  30630,                               /* Vector  166, coef 1(119), 2(166) */
  31391,                               /* Vector  166, coef 3(122), 4(159) */
  29859,                               /* Vector  167, coef 1(116), 2(163) */
  28314,                               /* Vector  167, coef 3(110), 4(154) */
  28058,                               /* Vector  168, coef 1(109), 2(154) */
  30359,                               /* Vector  168, coef 3(118), 4(151) */
  27288,                               /* Vector  169, coef 1(106), 2(152) */
  28560,                               /* Vector  169, coef 3(111), 4(144) */
  26788,                               /* Vector  170, coef 1(104), 2(164) */
  31388,                               /* Vector  170, coef 3(122), 4(156) */
  25249,                               /* Vector  171, coef 1( 98), 2(161) */
  29074,                               /* Vector  171, coef 3(113), 4(146) */
  25511,                               /* Vector  172, coef 1( 99), 2(167) */
  -29540,                              /* Vector  172, coef 3(140), 4(156) */
  26782,                               /* Vector  173, coef 1(104), 2(158) */
  -31088,                              /* Vector  173, coef 3(134), 4(144) */
  26777,                               /* Vector  174, coef 1(104), 2(153) */
  -32093,                              /* Vector  174, coef 3(130), 4(163) */
  27538,                               /* Vector  175, coef 1(107), 2(146) */
  31126,                               /* Vector  175, coef 3(121), 4(150) */
  25242,                               /* Vector  176, coef 1( 98), 2(154) */
  30850,                               /* Vector  176, coef 3(120), 4(130) */
  23434,                               /* Vector  177, coef 1( 91), 2(138) */
  30335,                               /* Vector  177, coef 3(118), 4(127) */
  24982,                               /* Vector  178, coef 1( 97), 2(150) */
  28572,                               /* Vector  178, coef 3(111), 4(156) */
  23183,                               /* Vector  179, coef 1( 90), 2(143) */
  27791,                               /* Vector  179, coef 3(108), 4(143) */
  24728,                               /* Vector  180, coef 1( 96), 2(152) */
  -32115,                              /* Vector  180, coef 3(130), 4(141) */
  25009,                               /* Vector  181, coef 1( 97), 2(177) */
  -30841,                              /* Vector  181, coef 3(135), 4(135) */
  24230,                               /* Vector  182, coef 1( 94), 2(166) */
  -32389,                              /* Vector  182, coef 3(129), 4(123) */
  23720,                               /* Vector  183, coef 1( 92), 2(168) */
  31885,                               /* Vector  183, coef 3(124), 4(141) */
  21663,                               /* Vector  184, coef 1( 84), 2(159) */
  30090,                               /* Vector  184, coef 3(117), 4(138) */
  24978,                               /* Vector  185, coef 1( 97), 2(146) */
  32668,                               /* Vector  185, coef 3(127), 4(156) */
  23450,                               /* Vector  186, coef 1( 91), 2(154) */
  32153,                               /* Vector  186, coef 3(125), 4(153) */
  22926,                               /* Vector  187, coef 1( 89), 2(142) */
  31126,                               /* Vector  187, coef 3(121), 4(150) */
  22432,                               /* Vector  188, coef 1( 87), 2(160) */
  -29809,                              /* Vector  188, coef 3(139), 4(143) */
  21143,                               /* Vector  189, coef 1( 82), 2(151) */
  -30078,                              /* Vector  189, coef 3(138), 4(130) */
  20885,                               /* Vector  190, coef 1( 81), 2(149) */
  -31343,                              /* Vector  190, coef 3(133), 4(145) */
  19344,                               /* Vector  191, coef 1( 75), 2(144) */
  32394,                               /* Vector  191, coef 3(126), 4(138) */
  -29301,                              /* Vector  192, coef 1(141), 2(139) */
  -32102,                              /* Vector  192, coef 3(130), 4(154) */
  -30839,                              /* Vector  193, coef 1(135), 2(137) */
  32145,                               /* Vector  193, coef 3(125), 4(145) */
  32663,                               /* Vector  194, coef 1(127), 2(151) */
  -29057,                              /* Vector  194, coef 3(142), 4(127) */
  -32631,                              /* Vector  195, coef 1(128), 2(137) */
  -30602,                              /* Vector  195, coef 3(136), 4(118) */
  -29292,                              /* Vector  196, coef 1(141), 2(148) */
  -30074,                              /* Vector  196, coef 3(138), 4(134) */
  -29803,                              /* Vector  197, coef 1(139), 2(149) */
  -32627,                              /* Vector  197, coef 3(128), 4(141) */
  -30064,                              /* Vector  198, coef 1(138), 2(144) */
  32131,                               /* Vector  198, coef 3(125), 4(131) */
  -30834,                              /* Vector  199, coef 1(135), 2(142) */
  31608,                               /* Vector  199, coef 3(123), 4(120) */
  -32370,                              /* Vector  200, coef 1(129), 2(142) */
  -31863,                              /* Vector  200, coef 3(131), 4(137) */
  -32631,                              /* Vector  201, coef 1(128), 2(137) */
  32132,                               /* Vector  201, coef 3(125), 4(132) */
  -30078,                              /* Vector  202, coef 1(138), 2(130) */
  31877,                               /* Vector  202, coef 3(124), 4(133) */
  -32642,                              /* Vector  203, coef 1(128), 2(126) */
  31103,                               /* Vector  203, coef 3(121), 4(127) */
  31101,                               /* Vector  204, coef 1(121), 2(125) */
  27507,                               /* Vector  204, coef 3(107), 4(115) */
  -29816,                              /* Vector  205, coef 1(139), 2(136) */
  -27509,                              /* Vector  205, coef 3(148), 4(139) */
  -29563,                              /* Vector  206, coef 1(140), 2(133) */
  -30592,                              /* Vector  206, coef 3(136), 4(128) */
  32657,                               /* Vector  207, coef 1(127), 2(145) */
  -30061,                              /* Vector  207, coef 3(138), 4(147) */
  -32126,                              /* Vector  208, coef 1(130), 2(130) */
  -30070,                              /* Vector  208, coef 3(138), 4(138) */
  -32645,                              /* Vector  209, coef 1(128), 2(123) */
  -31611,                              /* Vector  209, coef 3(132), 4(133) */
  -30600,                              /* Vector  210, coef 1(136), 2(120) */
  -26506,                              /* Vector  210, coef 3(152), 4(118) */
  30310,                               /* Vector  211, coef 1(118), 2(102) */
  -25216,                              /* Vector  211, coef 3(157), 4(128) */
  30328,                               /* Vector  212, coef 1(118), 2(120) */
  -29310,                              /* Vector  212, coef 3(141), 4(130) */
  29550,                               /* Vector  213, coef 1(115), 2(110) */
  -31618,                              /* Vector  213, coef 3(132), 4(126) */
  -30603,                              /* Vector  214, coef 1(136), 2(117) */
  -27501,                              /* Vector  214, coef 3(148), 4(147) */
  -32401,                              /* Vector  215, coef 1(129), 2(111) */
  -30834,                              /* Vector  215, coef 3(135), 4(142) */
  -30881,                              /* Vector  216, coef 1(135), 2( 95) */
  -29045,                              /* Vector  216, coef 3(142), 4(139) */
  32096,                               /* Vector  217, coef 1(125), 2( 96) */
  -32632,                              /* Vector  217, coef 3(128), 4(136) */
  30064,                               /* Vector  218, coef 1(117), 2(112) */
  -29031,                              /* Vector  218, coef 3(142), 4(153) */
  29539,                               /* Vector  219, coef 1(115), 2( 99) */
  -31852,                              /* Vector  219, coef 3(131), 4(148) */
  29567,                               /* Vector  220, coef 1(115), 2(127) */
  -32636,                              /* Vector  220, coef 3(128), 4(132) */
  29314,                               /* Vector  221, coef 1(114), 2(130) */
  -25726,                              /* Vector  221, coef 3(155), 4(130) */
  27512,                               /* Vector  222, coef 1(107), 2(120) */
  -24714,                              /* Vector  222, coef 3(159), 4(118) */
  29053,                               /* Vector  223, coef 1(113), 2(125) */
  -31088,                              /* Vector  223, coef 3(134), 4(144) */
  30598,                               /* Vector  224, coef 1(119), 2(134) */
  30346,                               /* Vector  224, coef 3(118), 4(138) */
  25993,                               /* Vector  225, coef 1(101), 2(137) */
  32159,                               /* Vector  225, coef 3(125), 4(159) */
  25221,                               /* Vector  226, coef 1( 98), 2(133) */
  29841,                               /* Vector  226, coef 3(116), 4(145) */
  31623,                               /* Vector  227, coef 1(123), 2(135) */
  32672,                               /* Vector  227, coef 3(127), 4(160) */
  29072,                               /* Vector  228, coef 1(113), 2(144) */
  30857,                               /* Vector  228, coef 3(120), 4(137) */
  30869,                               /* Vector  229, coef 1(120), 2(149) */
  -31854,                              /* Vector  229, coef 3(131), 4(146) */
  29841,                               /* Vector  230, coef 1(116), 2(145) */
  -31862,                              /* Vector  230, coef 3(131), 4(138) */
  32137,                               /* Vector  231, coef 1(125), 2(137) */
  -30318,                              /* Vector  231, coef 3(137), 4(146) */
  31365,                               /* Vector  232, coef 1(122), 2(133) */
  -32369,                              /* Vector  232, coef 3(129), 4(143) */
  30344,                               /* Vector  233, coef 1(118), 2(136) */
  -30844,                              /* Vector  233, coef 3(135), 4(132) */
  29330,                               /* Vector  234, coef 1(114), 2(146) */
  -30563,                              /* Vector  234, coef 3(136), 4(157) */
  28556,                               /* Vector  235, coef 1(111), 2(140) */
  32665,                               /* Vector  235, coef 3(127), 4(153) */
  28554,                               /* Vector  236, coef 1(111), 2(138) */
  -32368,                              /* Vector  236, coef 3(129), 4(144) */
  26759,                               /* Vector  237, coef 1(104), 2(135) */
  32141,                               /* Vector  237, coef 3(125), 4(141) */
  29076,                               /* Vector  238, coef 1(113), 2(148) */
  -27763,                              /* Vector  238, coef 3(147), 4(141) */
  29570,                               /* Vector  239, coef 1(115), 2(130) */
  -28777,                              /* Vector  239, coef 3(143), 4(151) */
  22143,                               /* Vector  240, coef 1( 86), 2(127) */
  29588,                               /* Vector  240, coef 3(115), 4(148) */
  28038,                               /* Vector  241, coef 1(109), 2(134) */
  -31625,                              /* Vector  241, coef 3(132), 4(119) */
  24717,                               /* Vector  242, coef 1( 96), 2(141) */
  -32634,                              /* Vector  242, coef 3(128), 4(134) */
  25447,                               /* Vector  243, coef 1( 99), 2(103) */
  -31349,                              /* Vector  243, coef 3(133), 4(139) */
  25979,                               /* Vector  244, coef 1(101), 2(123) */
  30084,                               /* Vector  244, coef 3(117), 4(132) */
  27275,                               /* Vector  245, coef 1(106), 2(139) */
  -29559,                              /* Vector  245, coef 3(140), 4(137) */
  24724,                               /* Vector  246, coef 1( 96), 2(148) */
  -28027,                              /* Vector  246, coef 3(146), 4(133) */
  24202,                               /* Vector  247, coef 1( 94), 2(138) */
  -29063,                              /* Vector  247, coef 3(142), 4(121) */
  23929,                               /* Vector  248, coef 1( 93), 2(121) */
  -26230,                              /* Vector  248, coef 3(153), 4(138) */
  24437,                               /* Vector  249, coef 1( 95), 2(117) */
  -29059,                              /* Vector  249, coef 3(142), 4(125) */
  26490,                               /* Vector  250, coef 1(103), 2(122) */
  -28786,                              /* Vector  250, coef 3(143), 4(142) */
  24693,                               /* Vector  251, coef 1( 96), 2(117) */
  -31602,                              /* Vector  251, coef 3(132), 4(142) */
  24971,                               /* Vector  252, coef 1( 97), 2(139) */
  -29802,                              /* Vector  252, coef 3(139), 4(150) */
  22405,                               /* Vector  253, coef 1( 87), 2(133) */
  -31337,                              /* Vector  253, coef 3(133), 4(151) */
  22917,                               /* Vector  254, coef 1( 89), 2(133) */
  -30072,                              /* Vector  254, coef 3(138), 4(136) */
  20861,                               /* Vector  255, coef 1( 81), 2(125) */
  -30842                               /* Vector  255, coef 3(135), 4(134) */
};


/* lpc pre-quantizer */
/*-------------------*/

ShortwordRom psrPreQ1[PREQ1_NUM_OF_WORDS] =
{
  -13624,                              /* Vector    0, coef 1(202), 2(200) */
  -31561,                              /* Vector    0, coef 3(132) Vector 1,
                                        * Coef 1(183) */
  -11886,                              /* Vector    1, coef 2(209), 3(146) */
  -22329,                              /* Vector    2, coef 1(168), 2(199) */
  -28759,                              /* Vector    2, coef 3(143) Vector 3,
                                        * Coef 1(169) */
  -22416,                              /* Vector    3, coef 2(168), 3(112) */
  -20112,                              /* Vector    4, coef 1(177), 2(112) */
  28822,                               /* Vector    4, coef 3(112) Vector 5,
                                        * Coef 1(150) */
  24392,                               /* Vector    5, coef 2( 95), 3( 72) */
  21340,                               /* Vector    6, coef 1( 83), 2( 92) */
  26761,                               /* Vector    6, coef 3(104) Vector 7,
                                        * Coef 1(137) */
  32323,                               /* Vector    7, coef 2(126), 3( 67) */
  -25919,                              /* Vector    8, coef 1(154), 2(193) */
  -29300,                              /* Vector    8, coef 3(141) Vector 9,
                                        * Coef 1(140) */
  -14455,                              /* Vector    9, coef 2(199), 3(137) */
  -29788,                              /* Vector   10, coef 1(139), 2(164) */
  -32648,                              /* Vector   10, coef 3(128) Vector 11,
                                        * Coef 1(120) */
  -26510,                              /* Vector   11, coef 2(152), 3(114) */
  25500,                               /* Vector   12, coef 1( 99), 2(156) */
  27209,                               /* Vector   12, coef 3(106) Vector 13,
                                        * Coef 1( 73) */
  -30113,                              /* Vector   13, coef 2(138), 3( 95) */
  31917,                               /* Vector   14, coef 1(124), 2(173) */
  22884,                               /* Vector   14, coef 3( 89) Vector 15,
                                        * Coef 1(100) */
  -29633,                              /* Vector   15, coef 2(140), 3( 63) */
  31429,                               /* Vector   16, coef 1(122), 2(197) */
  -29072,                              /* Vector   16, coef 3(142) Vector 17,
                                        * Coef 1(112) */
  -18827,                              /* Vector   17, coef 2(182), 3(117) */
  26052,                               /* Vector   18, coef 1(101), 2(196) */
  26961,                               /* Vector   18, coef 3(105) Vector 19,
                                        * Coef 1( 81) */
  -21177,                              /* Vector   19, coef 2(173), 3( 71) */
  16031,                               /* Vector   20, coef 1( 62), 2(159) */
  25903,                               /* Vector   20, coef 3(101) Vector 21,
                                        * Coef 1( 47) */
  -29092,                              /* Vector   21, coef 2(142), 3( 92) */
  22691,                               /* Vector   22, coef 1( 88), 2(163) */
  21594,                               /* Vector   22, coef 3( 84) Vector 23,
                                        * Coef 1( 90) */
  -26817,                              /* Vector   23, coef 2(151), 3( 63) */
  14738,                               /* Vector   24, coef 1( 57), 2(146) */
  -30686,                              /* Vector   24, coef 3(136) Vector 25,
                                        * Coef 1( 34) */
  -28552,                              /* Vector   25, coef 2(144), 3(120) */
  10352,                               /* Vector   26, coef 1( 40), 2(112) */
  -28111,                              /* Vector   26, coef 3(146) Vector 27,
                                        * Coef 1( 49) */
  30576,                               /* Vector   27, coef 2(119), 3(112) */
  21631,                               /* Vector   28, coef 1( 84), 2(127) */
  29521,                               /* Vector   28, coef 3(115) Vector 29,
                                        * Coef 1( 81) */
  30041,                               /* Vector   29, coef 2(117), 3( 89) */
  -27988,                              /* Vector   30, coef 1(146), 2(172) */
  22897,                               /* Vector   30, coef 3( 89) Vector 31,
                                        * Coef 1(113) */
  32583,                               /* Vector   31, coef 2(127), 3( 71) */
  23991,                               /* Vector   32, coef 1( 93), 2(183) */
  -28340,                              /* Vector   32, coef 3(145) Vector 33,
                                        * Coef 1( 76) */
  -23428,                              /* Vector   33, coef 2(164), 3(124) */
  15550,                               /* Vector   34, coef 1( 60), 2(190) */
  -28626,                              /* Vector   34, coef 3(144) Vector 35,
                                        * Coef 1( 46) */
  -20091,                              /* Vector   35, coef 2(177), 3(133) */
  19913,                               /* Vector   36, coef 1( 77), 2(201) */
  29757,                               /* Vector   36, coef 3(116) Vector 37,
                                        * Coef 1( 61) */
  -18579,                              /* Vector   37, coef 2(183), 3(109) */
  13228,                               /* Vector   38, coef 1( 51), 2(172) */
  21831,                               /* Vector   38, coef 3( 85) Vector 39,
                                        * Coef 1( 71) */
  -18366,                              /* Vector   39, coef 2(184), 3( 66) */
  11219,                               /* Vector   40, coef 1( 43), 2(211) */
  32289,                               /* Vector   40, coef 3(126) Vector 41,
                                        * Coef 1( 33) */
  -16012,                              /* Vector   41, coef 2(193), 3(116) */
  9892,                                /* Vector   42, coef 1( 38), 2(164) */
  27415,                               /* Vector   42, coef 3(107) Vector 43,
                                        * Coef 1( 23) */
  -20612,                              /* Vector   43, coef 2(175), 3(124) */
  9919,                                /* Vector   44, coef 1( 38), 2(191) */
  -24801,                              /* Vector   44, coef 3(159) Vector 45,
                                        * Coef 1( 31) */
  -23921,                              /* Vector   45, coef 2(162), 3(143) */
  6302,                                /* Vector   46, coef 1( 24), 2(158) */
  -22510,                              /* Vector   46, coef 3(168) Vector 47,
                                        * Coef 1( 18) */
  -14174,                              /* Vector   47, coef 2(200), 3(162) */
  15572,                               /* Vector   48, coef 1( 60), 2(212) */
  27438,                               /* Vector   48, coef 3(107) Vector 49,
                                        * Coef 1( 46) */
  -16281,                              /* Vector   49, coef 2(192), 3(103) */
  15046,                               /* Vector   50, coef 1( 58), 2(198) */
  17698,                               /* Vector   50, coef 3( 69) Vector 51,
                                        * Coef 1( 34) */
  -18601,                              /* Vector   51, coef 2(183), 3( 87) */
  12246,                               /* Vector   52, coef 1( 47), 2(214) */
  23595,                               /* Vector   52, coef 3( 92) Vector 53,
                                        * Coef 1( 43) */
  -13248,                              /* Vector   53, coef 2(204), 3( 64) */
  8157,                                /* Vector   54, coef 1( 31), 2(221) */
  29463,                               /* Vector   54, coef 3(115) Vector 55,
                                        * Coef 1( 23) */
  -13721,                              /* Vector   55, coef 2(202), 3(103) */
  10458,                               /* Vector   56, coef 1( 40), 2(218) */
  19998,                               /* Vector   56, coef 3( 78) Vector 57,
                                        * Coef 1( 30) */
  -10173,                              /* Vector   57, coef 2(216), 3( 67) */
  7395,                                /* Vector   58, coef 1( 28), 2(227) */
  18453,                               /* Vector   58, coef 3( 72) Vector 59,
                                        * Coef 1( 21) */
  -6332,                               /* Vector   59, coef 2(231), 3( 68) */
  6862,                                /* Vector   60, coef 1( 26), 2(206) */
  -28399,                              /* Vector   60, coef 3(145) Vector 61,
                                        * Coef 1( 17) */
  -12154,                              /* Vector   61, coef 2(208), 3(134) */
  4575,                                /* Vector   62, coef 1( 17), 2(223) */
  29455,                               /* Vector   62, coef 3(115) Vector 63,
                                        * Coef 1( 15) */
  -5029                                /* Vector   63, coef 2(236), 3( 91) */
};

ShortwordRom psrPreQ2[PREQ2_NUM_OF_WORDS] =
{
  -14954,                              /* Vector    0, coef 1(197), 2(150) */
  -26695,                              /* Vector    0, coef 3(151) Vector 1,
                                        * Coef 1(185) */
  -31091,                              /* Vector    1, coef 2(134), 3(141) */
  -15239,                              /* Vector    2, coef 1(196), 2(121) */
  29089,                               /* Vector    2, coef 3(113) Vector 3,
                                        * Coef 1(161) */
  31099,                               /* Vector    3, coef 2(121), 3(123) */
  -14987,                              /* Vector    4, coef 1(197), 2(117) */
  -25155,                              /* Vector    4, coef 3(157) Vector 5,
                                        * Coef 1(189) */
  23445,                               /* Vector    5, coef 2( 91), 3(149) */
  -20883,                              /* Vector    6, coef 1(174), 2(109) */
  -29788,                              /* Vector    6, coef 3(139) Vector 7,
                                        * Coef 1(164) */
  23424,                               /* Vector    7, coef 2( 91), 3(128) */
  -18813,                              /* Vector    8, coef 1(182), 2(131) */
  -19541,                              /* Vector    8, coef 3(179) Vector 9,
                                        * Coef 1(171) */
  32412,                               /* Vector    9, coef 2(126), 3(156) */
  -20114,                              /* Vector   10, coef 1(177), 2(110) */
  -19805,                              /* Vector   10, coef 3(178) Vector 11,
                                        * Coef 1(163) */
  23464,                               /* Vector   11, coef 2( 91), 3(168) */
  -24719,                              /* Vector   12, coef 1(159), 2(113) */
  -24682,                              /* Vector   12, coef 3(159) Vector 13,
                                        * Coef 1(150) */
  27026,                               /* Vector   13, coef 2(105), 3(146) */
  -28839,                              /* Vector   14, coef 1(143), 2( 89) */
  -20857,                              /* Vector   14, coef 3(174) Vector 15,
                                        * Coef 1(135) */
  25479,                               /* Vector   15, coef 2( 99), 3(135) */
  -19558,                              /* Vector   16, coef 1(179), 2(154) */
  -20062,                              /* Vector   16, coef 3(177) Vector 17,
                                        * Coef 1(162) */
  -22120,                              /* Vector   17, coef 2(169), 3(152) */
  -23669,                              /* Vector   18, coef 1(163), 2(139) */
  -18023,                              /* Vector   18, coef 3(185) Vector 19,
                                        * Coef 1(153) */
  -30813,                              /* Vector   19, coef 2(135), 3(163) */
  -23666,                              /* Vector   20, coef 1(163), 2(142) */
  -29810,                              /* Vector   20, coef 3(139) Vector 21,
                                        * Coef 1(142) */
  -27514,                              /* Vector   21, coef 2(148), 3(134) */
  -27010,                              /* Vector   22, coef 1(150), 2(126) */
  -28797,                              /* Vector   22, coef 3(143) Vector 23,
                                        * Coef 1(131) */
  31371,                               /* Vector   23, coef 2(122), 3(139) */
  -24711,                              /* Vector   24, coef 1(159), 2(121) */
  -18292,                              /* Vector   24, coef 3(184) Vector 25,
                                        * Coef 1(140) */
  29857,                               /* Vector   25, coef 2(116), 3(161) */
  -28558,                              /* Vector   26, coef 1(144), 2(114) */
  -18051,                              /* Vector   26, coef 3(185) Vector 27,
                                        * Coef 1(125) */
  25508,                               /* Vector   27, coef 2( 99), 3(164) */
  -29550,                              /* Vector   28, coef 1(140), 2(146) */
  -20871,                              /* Vector   28, coef 3(174) Vector 29,
                                        * Coef 1(121) */
  -28262,                              /* Vector   29, coef 2(145), 3(154) */
  32123,                               /* Vector   30, coef 1(125), 2(123) */
  -21144,                              /* Vector   30, coef 3(173) Vector 31,
                                        * Coef 1(104) */
  30111                                /* Vector   31, coef 2(117), 3(159) */
};

ShortwordRom psrPreQ3[PREQ3_NUM_OF_WORDS] =
{
  -28246,                              /* Vector    0, coef 1(145), 2(170) */
  30356,                               /* Vector    0, coef 3(118), 4(148) */
  -31320,                              /* Vector    1, coef 1(133), 2(168) */
  27523,                               /* Vector    1, coef 3(107), 4(131) */
  32405,                               /* Vector    2, coef 1(126), 2(149) */
  30613,                               /* Vector    2, coef 3(119), 4(149) */
  29066,                               /* Vector    3, coef 1(113), 2(138) */
  27281,                               /* Vector    3, coef 3(106), 4(145) */
  -26216,                              /* Vector    4, coef 1(153), 2(152) */
  28557,                               /* Vector    4, coef 3(111), 4(141) */
  -30322,                              /* Vector    5, coef 1(137), 2(142) */
  24972,                               /* Vector    5, coef 3( 97), 4(140) */
  -29576,                              /* Vector    6, coef 1(140), 2(120) */
  29072,                               /* Vector    6, coef 3(113), 4(144) */
  29301,                               /* Vector    7, coef 1(114), 2(117) */
  29589,                               /* Vector    7, coef 3(115), 4(149) */
  31407,                               /* Vector    8, coef 1(122), 2(175) */
  -32622,                              /* Vector    8, coef 3(128), 4(146) */
  29594,                               /* Vector    9, coef 1(115), 2(154) */
  31359,                               /* Vector    9, coef 3(122), 4(127) */
  28060,                               /* Vector   10, coef 1(109), 2(156) */
  30870,                               /* Vector   10, coef 3(120), 4(150) */
  23193,                               /* Vector   11, coef 1( 90), 2(153) */
  32141,                               /* Vector   11, coef 3(125), 4(141) */
  -31349,                              /* Vector   12, coef 1(133), 2(139) */
  -32379,                              /* Vector   12, coef 3(129), 4(133) */
  31093,                               /* Vector   13, coef 1(121), 2(117) */
  -29817,                              /* Vector   13, coef 3(139), 4(135) */
  29323,                               /* Vector   14, coef 1(114), 2(139) */
  -32369,                              /* Vector   14, coef 3(129), 4(143) */
  24450,                               /* Vector   15, coef 1( 95), 2(130) */
  -30584                               /* Vector   15, coef 3(136), 4(136) */
};


/* size of the vq subset in the kth segment */
/*------------------------------------------*/

ShortwordRom psrQuantSz[QUANT_NUM_OF_TABLES] =
{
  1 << (QUANT1_NUM_OF_BITS - PREQ1_NUM_OF_BITS),
  1 << (QUANT2_NUM_OF_BITS - PREQ2_NUM_OF_BITS),
  1 << (QUANT3_NUM_OF_BITS - PREQ3_NUM_OF_BITS)
};


/* pre-quantizer size */
/*--------------------*/

ShortwordRom psrPreQSz[QUANT_NUM_OF_TABLES] =
{
  1 << PREQ1_NUM_OF_BITS,
  1 << PREQ2_NUM_OF_BITS,
  1 << PREQ3_NUM_OF_BITS
};


/* reflection coeff scalar quantizer */
/*-----------------------------------*/

ShortwordRom psrSQuant[SQUANT_NUM_OF_ROWS] =
{
  -32766, -32758, -32746, -32729,
  -32707, -32680, -32648, -32611,
  -32570, -32523, -32472, -32416,
  -32355, -32289, -32219, -32143,
  -32063, -31978, -31888, -31794,
  -31694, -31590, -31481, -31368,
  -31250, -31127, -30999, -30867,
  -30731, -30589, -30443, -30293,
  -30138, -29978, -29814, -29646,
  -29473, -29296, -29114, -28928,
  -28738, -28543, -28344, -28141,
  -27934, -27723, -27507, -27287,
  -27063, -26836, -26604, -26368,
  -26128, -25884, -25637, -25386,
  -25130, -24871, -24609, -24342,
  -24072, -23799, -23522, -23241,
  -22957, -22670, -22379, -22084,
  -21787, -21486, -21182, -20875,
  -20564, -20251, -19935, -19615,
  -19293, -18968, -18640, -18309,
  -17975, -17639, -17300, -16959,
  -16615, -16268, -15919, -15568,
  -15214, -14859, -14500, -14140,
  -13778, -13413, -13047, -12678,
  -12308, -11936, -11562, -11186,
  -10809, -10430, -10050, -9668,
  -9284, -8899, -8513, -8126,
  -7737, -7347, -6956, -6565,
  -6172, -5778, -5383, -4988,
  -4591, -4194, -3797, -3399,
  -3000, -2601, -2201, -1802,
  -1402, -1001, -601, -200,
  200, 601, 1001, 1402,
  1802, 2201, 2601, 3000,
  3399, 3797, 4194, 4591,
  4988, 5383, 5778, 6172,
  6565, 6956, 7347, 7737,
  8126, 8513, 8899, 9284,
  9668, 10050, 10430, 10809,
  11186, 11562, 11936, 12308,
  12678, 13047, 13413, 13778,
  14140, 14500, 14859, 15214,
  15568, 15919, 16268, 16615,
  16959, 17300, 17639, 17975,
  18309, 18640, 18968, 19293,
  19615, 19935, 20251, 20564,
  20875, 21182, 21486, 21787,
  22084, 22379, 22670, 22957,
  23241, 23522, 23799, 24072,
  24342, 24609, 24871, 25130,
  25386, 25637, 25884, 26128,
  26368, 26604, 26836, 27063,
  27287, 27507, 27723, 27934,
  28141, 28344, 28543, 28738,
  28928, 29114, 29296, 29473,
  29646, 29814, 29978, 30138,
  30293, 30443, 30589, 30731,
  30867, 30999, 31127, 31250,
  31368, 31481, 31590, 31694,
  31794, 31888, 31978, 32063,
  32143, 32219, 32289, 32355,
  32416, 32472, 32523, 32570,
  32611, 32648, 32680, 32707,
  32729, 32746, 32758, 32766
};


/* index structure for LPC Vector Quantizer */
/*------------------------------------------*/

struct IsubLHn psvqIndex[QUANT_NUM_OF_TABLES] =
{
  {1, 3, 3},                           /* Table 1 - low, high, length */
  {4, 6, 3},                           /* Table 2 - low, high, length */
  {7, 10, 4}                           /* Table 3 - low, high, length */
};


/* square root of p0 table */
/*-------------------------*/

ShortwordRom ppsrSqrtP0[SQRTP0_NUM_OF_MODES][SQRTP0_NUM_OF_ROWS] =
{
  {
    23853, 24682, 21337, 16698,
    27731, 24699, 8804, 16227,
    29760, 24223, 30248, 30903,
    7969, 17600, 5559, 16866,
    25609, 24727, 30483, 31688,
    27583, 28959, 31514, 31903,
    14035, 23660, 13761, 25661,
    28863, 31627, 30350, 32153
  },
  {
    26490, 19929, 28693, 23708,
    30266, 28196, 31577, 31714,
    20642, 20179, 27963, 29109,
    30776, 32054, 31076, 32241,
    23111, 28343, 25233, 29710,
    30403, 31915, 31466, 32360,
    28533, 30856, 31921, 32327,
    29569, 31489, 32348, 32114
  },
  {
    24318, 31095, 31943, 31912,
    27251, 30541, 32336, 32254,
    28448, 29139, 32149, 32413,
    31155, 31480, 32388, 32561,
    28647, 31172, 32084, 32512,
    30927, 32000, 32435, 32571,
    30455, 31973, 32487, 32564,
    31487, 32472, 31877, 32586
  }
};


/* interpolation filter used for C and G */
/*---------------------------------------*/

ShortwordRom ppsrCGIntFilt[CGINTFILT_MACS][OS_FCTR] =
{
  {0, 126, 0, -299, -778, -1381},
  {-1980, -2375, -2326, -1596, 0, 2535},
  {5936, 9968, 14254, 18322, 21678, 23889},
  {24661, 23889, 21678, 18322, 14254, 9968},
  {5936, 2535, 0, -1596, -2326, -2375},
  {-1980, -1381, -778, -299, 0, 126}
};


/* interpolation filter used pitch */
/*---------------------------------*/

ShortwordRom ppsrPVecIntFilt[PVECINTFILT_MACS][OS_FCTR] =
{
  {0, 183, 249, 251, 149, -67},
  {-368, -681, -897, -904, -622, -38},
  {762, 1595, 2214, 2360, 1842, 611},
  {-1186, -3195, -4898, -5708, -5089, -2696},
  {1515, 7239, 13835, 20424, 26040, 29814},
  {31144, 29814, 26040, 20424, 13835, 7239},
  {1515, -2696, -5089, -5708, -4898, -3195},
  {-1186, 611, 1842, 2360, 2214, 1595},
  {762, -38, -622, -904, -897, -681},
  {-368, -67, 149, 251, 249, 183}
};


/* fractional pitch lag table lag*OS_FCTR */
/*----------------------------------------*/

ShortwordRom psrLagTbl[LAGTBL_NUM_OF_ROWS] =
{
  126, 128, 130, 132, 134, 136, 138, 139,
  140, 141, 142, 143, 144, 145, 146, 147,
  148, 149, 150, 151, 152, 153, 154, 155,
  156, 157, 158, 159, 160, 161, 162, 163,
  164, 165, 166, 167, 168, 169, 170, 171,
  172, 173, 174, 175, 176, 177, 178, 179,
  180, 181, 182, 183, 184, 185, 186, 187,
  188, 189, 190, 191, 192, 193, 194, 195,
  196, 197, 198, 199, 200, 201, 202, 203,
  204, 205, 206, 207, 208, 209, 210, 212,
  214, 216, 218, 220, 222, 224, 226, 228,
  230, 232, 234, 236, 238, 240, 242, 244,
  246, 248, 250, 252, 254, 256, 258, 260,
  262, 264, 266, 268, 270, 272, 274, 276,
  278, 280, 282, 284, 286, 288, 290, 292,
  294, 296, 298, 300, 303, 306, 309, 312,
  315, 318, 321, 324, 327, 330, 333, 336,
  339, 342, 345, 348, 351, 354, 357, 360,
  363, 366, 369, 372, 375, 378, 381, 384,
  387, 390, 393, 396, 399, 402, 405, 408,
  411, 414, 417, 420, 423, 426, 429, 432,
  435, 438, 441, 444, 447, 450, 453, 456,
  459, 462, 465, 468, 471, 474, 477, 480,
  483, 486, 489, 492, 495, 498, 501, 504,
  507, 510, 513, 516, 519, 522, 525, 528,
  531, 534, 537, 540, 546, 552, 558, 564,
  570, 576, 582, 588, 594, 600, 606, 612,
  618, 624, 630, 636, 642, 648, 654, 660,
  666, 672, 678, 684, 690, 696, 702, 708,
  714, 720, 726, 732, 738, 744, 750, 756,
  762, 768, 774, 780, 786, 792, 798, 804,
  810, 816, 822, 828, 834, 840, 846, 852
};


/* R0 decision value table defines range (not the levels themselves */
/*------------------------------------------------------------------*/

ShortwordRom psrR0DecTbl[R0DECTBL_NUM_OF_ROWS] =
{
  6, 7, 8, 9, 10, 12, 13, 15,
  16, 18, 21, 23, 26, 29, 33, 37,
  41, 46, 52, 58, 65, 73, 82, 92,
  103, 115, 130, 145, 163, 183, 205, 230,
  258, 290, 325, 365, 410, 460, 516, 579,
  649, 728, 817, 917, 1029, 1154, 1295, 1453,
  1631, 1830, 2053, 2303, 2584, 2900, 3254, 3651,
  4096, 4596, 5157, 5786, 6492, 7284, 8173
};

/* high pass filter coefficients */
/*-------------------------------*/

ShortwordRom psrHPFCoefs[HPFCOEFS_NUM_OF_CODES] =
{
  10979, -21954, 10979, -14071, 30347,
  10979, -21936, 10979, -15385, 31632
};


/* spectral smoothing coefficients */
/*---------------------------------*/

ShortwordRom psrNWCoefs[NWCOEFS_NUM_OF_CODES] =
{
  30474, 28341, 26357, 24512, 22796,
  21201, 19717, 18336, 17053, 15859,
  22938, 16056, 11239, 7868, 5507,
  3855, 2699, 1889, 1322, 926
};


/* spectral smoothing coefficients for FLAT */
/*------------------------------------------*/

LongwordRom pL_rFlatSstCoefs[FLATSSTCOEFS_NUM_OF_CODES] =
{
  2145262551, 2138972761, 2129579484, 2118274784, 2106083684,
  2093615080, 2081026796, 2068156528, 2054704628, 2040382328
};

ShortwordRom psrOldCont[4] = {
  0x599a,                              /* 0.70 */
  0x30a4,                              /* 0.38 */
  0x0a3d,                              /* 0.08 */
  0x0000,                              /* 0.00 */
};

ShortwordRom psrNewCont[4] = {
  0x2666,                              /* 0.30 */
  0x4f5c,                              /* 0.62 */
  0x75c3,                              /* 0.92 */
  0x7fff,                              /* 1.00 */
};
