/* ###########################################################################
#
# Copyright (c) 2000-2016 Telchemy, Incorporated. All Rights Reserved.
#
# Telchemy Confidential and Proprietary
#
# The following software source code ("Software") is strictly confidential
# and is proprietary to Telchemy, Incorporated ("Telchemy").  It may only
# be read, used, copied, adapted, modified or otherwise utilized by parties
# (individuals, corporations, or organizations) that have entered into a
# license agreement or confidentiality agreement with Telchemy, and are thus
# subject to the terms of that license agreement or confidentiality agreement
# and any other applicable agreement between the party and Telchemy.  If
# there is any doubt as to whether a party is entitled to access, read, use,
# copy, adapt, modify or otherwise utilize the Software, or whether a party
# is entitled to disclose the Software to any other party, you should contact
# Telchemy.  If you, as a party, have not entered into a license agreement or
# confidentiality agreement with Telchemy granting access to this Software,
# all media, copies and printed listings containing the Software should be
# forthwith returned to Telchemy.
#
# This Software is protected by Granted and Pending Patents owned by Telchemy,
# Incorporated.
#
# Telchemy reserves the right to take legal action against any party that
# violates Telchemy's rights to the Software, including without limitation a
# party's breach of the above conditions.
#
# If you have questions about any of the above conditions, or wish to report
# violations, please contact:  support@telchemy.com
#
# ##########################################################################*/

/* 
 * vqmon.h
 *
 * VQmon Common Header File
 *
 * Copyright (c) 2000-2016 Telchemy, Incorporated
 */

#ifndef _TELCHEMY_VQMON_H_
#define _TELCHEMY_VQMON_H_

#include "vqmoncfg.h"
#include "vqmondef.h"
#include "vqmontyp.h"

#ifdef __cplusplus
extern "C" {
#endif


/* --------------------------------------------------------------------------
 *
 *      VQmon Public Constant Definitions
 *
 * --------------------------------------------------------------------------
 */

/*
 * VQmon stream management notification flags
 */
#define VQMON_STREAMMGMTNOTIFY_CREATED          (0x0001)
#define VQMON_STREAMMGMTNOTIFY_ACTIVATING       (0x0002)
#define VQMON_STREAMMGMTNOTIFY_CODECUPDATE      (0x0004)
#define VQMON_STREAMMGMTNOTIFY_CODECREQUEST     (0x0008)
#define VQMON_STREAMMGMTNOTIFY_DELAYUPDATE      (0x0010)
#define VQMON_STREAMMGMTNOTIFY_PLAYOUTBUFUPDATE (0x0020)
#define VQMON_STREAMMGMTNOTIFY_PROGINFOUPDATE   (0x0040)
#define VQMON_STREAMMGMTNOTIFY_INTERVALUPDATE   (0x0080)
#define VQMON_STREAMMGMTNOTIFY_DEACTIVATING     (0x0100)
#define VQMON_STREAMMGMTNOTIFY_TERMINATED       (0x0200)
#define VQMON_STREAMMGMTNOTIFY_RTPEXTHEADERRCVD (0x0400)
#define VQMON_STREAMMGMTNOTIFY_RTCPAPPRCVD      (0x0800)
#define VQMON_STREAMMGMTNOTIFY_RTCPSRRCVD       (0x1000)
#define VQMON_STREAMMGMTNOTIFY_RTCPXRRCVD       (0x2000)
#define VQMON_STREAMMGMTNOTIFY_CONFIGERROR      (0x4000)
#define VQMON_STREAMMGMTNOTIFY_RESET            (0x8000)
#define VQMON_STREAMMGMTNOTIFY_RTPSSRCCHANGING  (0x00010000)
#define VQMON_STREAMMGMTNOTIFY_RTPSSRCCHANGED   (0x00020000)
#define VQMON_STREAMMGMTNOTIFY_RTCPSDESPACKET   (0x00040000)
#define VQMON_STREAMMGMTNOTIFY_RTCPBYEPACKET    (0x00080000)

/*
 * VQmon stream management notification flags for streaming video.
 */
#define VQMON_STREAMMGMTNOTIFY_STRMVIDSAMPLEAVAIL   (0x100000)
#define VQMON_STREAMMGMTNOTIFY_STRMVIDSAMPLEFULL    (0x200000)


#define VQMON_STREAMMGMTNOTIFY_DEFAULT  (\
    VQMON_STREAMMGMTNOTIFY_CREATED | \
    VQMON_STREAMMGMTNOTIFY_ACTIVATING | \
    VQMON_STREAMMGMTNOTIFY_CODECUPDATE | \
    VQMON_STREAMMGMTNOTIFY_CODECREQUEST | \
    VQMON_STREAMMGMTNOTIFY_INTERVALUPDATE | \
    VQMON_STREAMMGMTNOTIFY_DEACTIVATING | \
    VQMON_STREAMMGMTNOTIFY_CONFIGERROR | \
    VQMON_STREAMMGMTNOTIFY_RESET | \
    VQMON_STREAMMGMTNOTIFY_TERMINATED )

/*
 * VQmon interface revision definition.
 */
#define VQMON_IFVERSION_3                   (0x0300)
#define VQMON_IFVERSION_3_1                 (0x0310)
#define VQMON_IFVERSION_CURRENT             VQMON_IFVERSION_3_1

/*
 * Packet descriptor flags - Ethernet
 */
#define VQMON_ETHERNETINFO_VALID            (0x00000001)

/*
 * Packet descriptor flags - VLAN
 */
#define VQMON_VLANINFO_VALID                (0x00000001)
#define VQMON_VLANINFO_TAGVALID             (0x00000002)
#define VQMON_VLANINFO_SERVICETAGVALID      (0x00000004)
#define VQMON_VLANINFO_ADDITIONALTAGVALID   (0x00000008)

/*
 * Flags to generate flow hash using VLAN tags
 */
#ifdef VQMON_BUILDCFG_FLOWHASH
#define VQMON_VLANINFO_FLOWHASHVLAN         (0x00000020)
#define VQMON_VLANINFO_FLOWHASHSERVICEVLAN  (0x00000040)
#define VQMON_VLANINFO_FLOWHASHADDITIONAL   (0x00000080)
#endif

/*
 * Packet descriptor flags - IP layer information
 */
#define VQMON_IPINFO_VALID                  (0x00000001)
#define VQMON_IPINFO_TTLVALID               (0x00000010)
#define VQMON_IPINFO_TOSVALID               (0x00000020)
#define VQMON_IPINFO_FRAGMENT               (0x00000080)

/*
 * Packet descriptor flags - TCP/UDP transport layer information
 */
#define VQMON_TRANSINFO_VALID               (0x00000001)
#define VQMON_TRANSINFO_TCP                 (0x00000010)
#define VQMON_TRANSINFO_UDP                 (0x00000020)
#define VQMON_TRANSINFO_TCPFIN              (0x00000100)
#define VQMON_TRANSINFO_TCPSYN              (0x00000200)
#define VQMON_TRANSINFO_TCPRST              (0x00000400)
#define VQMON_TRANSINFO_TCPPSH              (0x00000800)
#define VQMON_TRANSINFO_TCPACK              (0x00001000)
#define VQMON_TRANSINFO_TCPURG              (0x00002000)

/*
 * Packet descriptor flags - RTP layer information
 */
#define VQMON_RTPINFO_VALID                 (0x00000001)
#define VQMON_RTPFLAG_MARKBITSET            (0x00000010)
#define VQMON_RTPFLAG_PADBITSET             (0x00000020)
#define VQMON_RTPFLAG_EXTBITSET             (0x00000040)
#define VQMON_RTPFLAG_ENCAPSTRM             (0x00000080)
#define VQMON_RTPFLAG_ENVELOPESTRM          (0x00000100)

/*
 * Packet descriptor flags - multi-media payload layer information
 */
#define VQMON_PAYLOADINFO_VALID             (0x80000000)


/*#if defined(VQMON_BUILDCFG_FLOWHASH) || defined(VQMON_BUILDCFG_HAVEDATA)*/
#if defined(VQMON_BUILDCFG_FLOWHASH)
#define VQMON_MAX_VLAN_TAGS             (3)
#else
#define VQMON_MAX_VLAN_TAGS             (2)
#endif


/* --------------------------------------------------------------------------
 *
 *      VQmon Public Enumerations
 *
 * --------------------------------------------------------------------------
 */

/*
 * VQmon object identifiers.
 */
typedef enum
{
    vqmonObjectIdUnknown                = 0x00000000,
    vqmonObjectIdInterface              = 0x56514946,
    vqmonObjectIdStream                 = 0x56515354,
    vqmonObjectIdPlayoutBuffer          = 0x56515042,
    vqmonObjectIdCODEC                  = 0x56514344,
    vqmonObjectIdMisc                   = 0x56514D53

} vqmon_objectid_t;


/*
 * VQmon perceptual quality models.
 */
typedef enum
{
    vqmonModelTypeUnknown               = 0,
    vqmonModelTypeAverage,
    vqmonModelTypeFourstate,
    vqmonModelTypeRealtime

} vqmon_model_t;


/*
 * IP address versioning.
 */
typedef enum
{
    vqmonAddressIPv4                    = 4,
    vqmonAddressIPv6                    = 6

} vqmon_ipaddrver_t;


/*
 * Packet descriptor payload start indicator.
 */
typedef enum
{
    vqmonPktDescStartNone               = 0x00,
    vqmonPktDescStartIPHeader           = 0x01,
    vqmonPktDescStartUDPHeader,
    vqmonPktDescStartTCPHeader,
    vqmonPktDescStartRTPHeader,
    vqmonPktDescStartRTCPHeader,
    vqmonPktDescStartMP2TSHeader,
    vqmonPktDescStartPayload,
    vqmonPktDescStartPESHeader,
    vqmonPktDescStartHTTPMessage,
    vqmonPktDescStartRTMPChunk,
    vqmonPktDescStartHTTPSMessage,
    vqmonPktDescStartQUICHeader

} vqmon_pktdescstartindic_t;


/*
 * Calculate Engine Model type
 */
typedef enum
{
    vqmonCMEModelLinear                 = 0x01,
    vqmonCMEModelNonLinear              = 0x02

} vqmon_cmemodeltype_t;


/* --------------------------------------------------------------------------
 *
 *      VQmon Public Simple Data Types
 *
 * --------------------------------------------------------------------------
 */

/*
 * Memory management functions.
 */
typedef void* (*vqmon_func_malloc_t)(vqmon_size_t, vqmon_objectid_t);
typedef void* (*vqmon_func_realloc_t)(void*, vqmon_size_t, vqmon_objectid_t);
typedef void  (*vqmon_func_free_t)(void*, vqmon_objectid_t);


/*
 * Thread synchronization object management functions.
 */
typedef vqmon_result_t (*vqmon_func_mtxcreate_t)(void**, tcmyU16);
typedef void (*vqmon_func_mtxdestroy_t)(void*);
typedef vqmon_result_t (*vqmon_func_mtxlock_t)(void*);
typedef vqmon_result_t (*vqmon_func_mtxunlock_t)(void*);


/* --------------------------------------------------------------------------
 *
 *      VQmon Public Complex Data Structures
 *
 * --------------------------------------------------------------------------
 */

/*
 * VQmon IP address storage structure.
 */
struct _vqmon_ipaddr_s
{
    vqmon_ipaddrver_t               eVersion;

    union _ip_u
    {
        struct _v4_s
        {
            tcmyU8                  octet[4];
        } v4;

        struct _v6_s
        {
            tcmyU8                  octet[16];
        } v6;
    } ip;
};

typedef struct _vqmon_ipaddr_s vqmon_ipaddr_t;


/*
 * VQmon packet descriptor.
 */
struct _vqmon_pktdesc_s
{
    vqmon_pktdescstartindic_t       ePktStartIndicator;
    tcmyU8*                         pPktBuffer;
    tcmyU16                         nCapturedPktLen;      /* Length of the captured data */
    tcmyU16                         nTotalPktLen;         /* Length of the packet on the wire. */
                                                          /* It is >= nCapturedPktLen */
    tcmyU16                         nCurrPktBufferLen;    /* Scratch memory used by vqmon */
    tcmyU16                         nCurrPDULen;          /* Scratch memory used by vqmon */
#ifdef VQMON_BUILDCFG_FLOWHASH
    tcmyU64                         nFlowHash;
#endif
#ifdef VQMON_BUILDCFG_HAVEDATA
    tcmyU16                         nFrameLength;       /* Length of layer 2 frame */
    tcmyU16                         nFrameBufferLen;    /* Captured length of layer 2 frame */
    tcmyU8*                         pFrameBuffer;
#endif
    struct _pktdesc_ethernetinfo_s
    {
        tcmyU8                      aSrcMACAddress[6];
        tcmyU8                      aDstMACAddress[6];
        tcmyU32                     fHdrFlags;

    } ethernetinfo;

    struct _pktdesc_vlaninfo_s
    {
        tcmyU16                     nVlanTag;
        tcmyU16                     nServiceVlanTag;
        tcmyU16                     nAdditionalTag;
        tcmyU32                     fHdrFlags;

    } vlaninfo;

    struct _pktdesc_ipinfo_s
    {
        vqmon_ipaddr_t              tSrcAddress;
        vqmon_ipaddr_t              tDstAddress;
        tcmyU8                      nPacketTTL;
        tcmyU8                      nPacketTOS;
        tcmyU8                      nProtocolId;
        tcmyU32                     fHdrFlags;

    } ipinfo;

    struct _pktdesc_transportinfo_s
    {
        tcmyU16                     nSrcPort;
        tcmyU16                     nDstPort;
        tcmyU32                     nSeqNumber;
        tcmyU32                     nAckNumber;
        tcmyU16                     nWindow;
        tcmyU16                     nUrgPointer;
        tcmyU32                     fHdrFlags;
      
    } transportinfo;

    struct _pktdesc_rtpinfo_s
    {
        tcmyU8                      nPayloadType;
        tcmyU16                     nPayloadSize;
        tcmyU16                     nSeqNumber;
        tcmyU32                     nSSRC;
        tcmyU32                     nTimestamp;
        tcmyU32                     fHdrFlags;

    } rtpinfo;

    struct
    {
        tcmyU8                      nContinuityCount;
        tcmyU16                     nPID;
        tcmyU16                     nPCR_27MHz;
        tcmyU64                     nPCR_90kHz;
        tcmyU64                     nPTS;
        tcmyU32                     fHdrFlags;

    } mp2tsinfo;

    struct 
    {
        tcmyU32                     fPayloadFlags;
        tcmyU32                     nReserved;

    } payloadinfo;
};

typedef struct _vqmon_pktdesc_s vqmon_pktdesc_t;

/*
 * Include the extension header files to support the specific VQmon objects.
 */
#if (defined(VQMON_BUILDCFG_HAVEVOICE) || defined(VQMON_BUILDCFG_HAVEVIDEO))
#include "vqmoncdc.h"
#include "vqmonalt.h"
#include "vqmonstm.h"
#endif
#include "vqmonif.h"
#ifdef VQMON_BUILDCFG_HAVEDATA
#include "vqmondat.h"
#endif

/*
 * VQmon instance initialization block.
 */
struct _vqmon_initparams_s
{
    tcmyU32                             nVersion;
    tcmyCHAR*                           pstrLicense;

    vqmon_func_malloc_t                 pfMalloc;
    vqmon_func_realloc_t                pfRealloc;
    vqmon_func_free_t                   pfFree;

    vqmon_func_mtxcreate_t              pfMutexCreate;
    vqmon_func_mtxdestroy_t             pfMutexDestroy;
    vqmon_func_mtxlock_t                pfMutexLock;
    vqmon_func_mtxunlock_t              pfMutexUnlock;
#if (defined(VQMON_BUILDCFG_HAVEVOICE) || defined(VQMON_BUILDCFG_HAVEVIDEO))
    vqmon_func_streammgmtnotifyhdlrcb_t pfStreamNotifyHdlr;
    tcmyU32                             fStreamNotifyMask;

    vqmon_func_alerthdlrcb_t            pfAlertHdlr;
    vqmon_cmemodeltype_t                eCMEModelType;
#endif    
#ifdef VQMON_BUILDCFG_HAVEDATA
    vqmon_func_pipemgmtnotifyhdlrcb_t   pfPipeNotifyHdlr;
    tcmyU32                             fPipeNotifyMask;
#endif
};

typedef struct _vqmon_initparams_s vqmon_initparams_t;


/* --------------------------------------------------------------------------
 *
 *      VQmon Public Functions
 *
 * --------------------------------------------------------------------------
 */

/* 
 * VQmon instance control functions.
 */
VQMON_PUBLIC(vqmon_result_t)
VQmonInitialize(
    const vqmon_initparams_t* pInitParams
    );

VQMON_PUBLIC(void)
VQmonCleanup(void);

VQMON_PUBLIC(tcmyU32)
VQmonVersionInfo(
    tcmyU32* const pBuildCfgMap
    );

VQMON_PUBLIC(vqmon_result_t)
VQmonHostIDGet(
    tcmyCHAR    *pstrHostIDBuf,
    tcmyU32     *pnHostIDBufSz
    );

VQMON_PUBLIC(tcmyU8)
VQmonDebugSetTraceLevel(
    const tcmyU8 nNewLevel
    );

VQMON_PUBLIC(vqmon_result_t)
VQmonDefaultsGet(
    const tcmyU32 idDefConfigBlock,
    tcmyU32* const pnConfigBlockSz,
    void* const pConfigBlock
    );

VQMON_PUBLIC(vqmon_result_t)
VQmonDefaultsSet(
    const tcmyU32 idDefConfigBlock,
    const tcmyU32 nConfigBlockSz,
    const void* pConfigBlock
    );

#ifdef __cplusplus
}
#endif

#endif  /* _TELCHEMY_VQMON_H_ */
