/* ###########################################################################
#
# Copyright (c) 2017 Telchemy, Incorporated. All Rights Reserved.
#
# Telchemy Confidential and Proprietary
#
# The following software source code ("Software") is strictly confidential
# and is proprietary to Telchemy, Incorporated ("Telchemy").  It may only
# be read, used, copied, adapted, modified or otherwise utilized by parties
# (individuals, corporations, or organizations) that have entered into a
# license agreement or confidentiality agreement with Telchemy, and are thus
# subject to the terms of that license agreement or confidentiality agreement
# and any other applicable agreement between the party and Telchemy.  If
# there is any doubt as to whether a party is entitled to access, read, use,
# copy, adapt, modify or otherwise utilize the Software, or whether a party
# is entitled to disclose the Software to any other party, you should contact
# Telchemy.  If you, as a party, have not entered into a license agreement or
# confidentiality agreement with Telchemy granting access to this Software,
# all media, copies and printed listings containing the Software should be
# forthwith returned to Telchemy.
#
# Telchemy reserves the right to take legal action against any party that
# violates Telchemy's rights to the Software, including without limitation a
# party's breach of the above conditions.
#
# If you have questions about any of the above conditions, or wish to report
# violations, please contact:  support@telchemy.com
#
# ##########################################################################*/

/* 
 * exmshost.c
 *
 * VQmon E-XMS Host Sample Application - Voice Stream Simulator
 *
 * Copyright (c) 2017 Telchemy, Incorporated
 */

/*
 * This application demonstrates the usage of VQmon in monitoring a voice
 * call using the stream-based and the E-XMS extended APIs.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "vqmonmmd.h"


/* --------------------------------------------------------------------------
 *
 *      Constants
 *
 * --------------------------------------------------------------------------
 */

#define EXMS_STRMHANDLESMAX                 (20)

#define EXMS_EVENTSMAX                      (10)
#define EXMS_EVTITVLUSMAX                   (100000)

#define EXMS_CALLDURSECMAX                  (180)

#define EXMS_SIGLVLRANGE                    (50)
#define EXMS_NOISELVLRANGE                  (70)
#define EXMS_RERLRANGE                      (60)
#define EXMS_DELAYMAX                       (325)


/* --------------------------------------------------------------------------
 *
 *      Global Data
 *
 * --------------------------------------------------------------------------
 */

vqmon_handle_t g_hInterface = NULL;

int g_iStrmIdx = 0;
vqmon_handle_t g_aStrmHandles[EXMS_STRMHANDLESMAX];

vqmon_time_t g_tNullTS = { 0 };


/* --------------------------------------------------------------------------
 *
 *      Private Functions
 *
 * --------------------------------------------------------------------------
 */

/* --------------------------------------------------------------------------
 *
 *      EXMS_SetMMCallInfo
 *
 *      Stores some data in specified Markov Model that would be 
 *      representative of data collected on the XMS for VoIP calls.
 *
 * --------------------------------------------------------------------------
 */

static int
EXMS_SetMMCallInfo(
    vqmon_mmcntrsv1_t *pMMv1Cntrs,
    unsigned int* pJitterMax
    )
{
    tcmyU32 _nSz;
    int _nBitrate = 0;
    vqmon_result_t _result;
    unsigned int _i, _dpp;
    unsigned int _nPackets = 0;
    unsigned int _nDuration = 0;
    vqmon_codecgenprops_t _tCodecProps;
    vqmon_codecrateprops_t _tCodecRate;
    vqmon_codec_voice_extprops_t _tCodecExtProps;
    
    /*
     * Build a random call model. Start by selecting a random voice 
     *   codec and call duration.
     */
    VQMON_MEMCLEAR(pMMv1Cntrs, sizeof(vqmon_mmcntrsv1_t));
    VQMON_MEMCLEAR(&_tCodecProps, sizeof(vqmon_codecgenprops_t));
    VQMON_MEMCLEAR(&_tCodecExtProps, sizeof(vqmon_codec_voice_extprops_t));
    
    while ((vqmonCODECTableEntryInvalid == _tCodecProps.eValidity) ||
           (0 == _tCodecExtProps.nFrameDuration))
    {
        pMMv1Cntrs->eCODECType = (tcmyU16)(lrand48() %
                                    VQmonCODECCount(vqmonCODECMediaFormatVoice))
                                    + vqmonCODECMediaFormatVoice;
    
        _nSz = sizeof(vqmon_codecgenprops_t);
        _result = VQmonCODECPropertiesGet(
                    (vqmon_codecid_t)pMMv1Cntrs->eCODECType,
                    VQMON_CODECPROPBLOCKID_GENERAL,
                    NULL,
                    &_nSz,
                    &_tCodecProps
                    );
        if (VQMON_ESUCCESS != _result)
        {
            continue;
        }

        /*
         * Grab the extended properties to determine packets.
         */
        _nSz = sizeof(_tCodecExtProps);
        VQMON_MEMCLEAR(&_tCodecExtProps, _nSz);
        _result = VQmonCODECPropertiesGet(
                    (vqmon_codecid_t)pMMv1Cntrs->eCODECType,
                    VQMON_CODECPROPBLOCKID_EXTPROPS,
                    NULL,
                    &_nSz,
                    &_tCodecExtProps
                    );
        if ((VQMON_ESUCCESS != _result) ||
            (0 == _tCodecExtProps.nFrameDuration))
        {
            continue;
        }
    }
    
    /*
     * Calculate the number of frames per packet.
     */
    pMMv1Cntrs->nFramesPerPkt = 1;          /* <===== MUST BE SET */
    _dpp = _tCodecExtProps.nFrameDuration;
    while (_dpp < 20)
    {
        _dpp += _tCodecExtProps.nFrameDuration;
        pMMv1Cntrs->nFramesPerPkt++;
    }
    *pJitterMax = _dpp;
    
    /*
     * Select a rate and then grab the encoded bit rate - possibly modeling
     *    a variable bitrate codec running at a constant rate.
     */
    _nSz = sizeof(_tCodecRate);
    VQMON_MEMCLEAR(&_tCodecRate, _nSz);
    _result = VQmonCODECPropertiesGet(
                    (vqmon_codecid_t)pMMv1Cntrs->eCODECType,
                    VQMON_CODECPROPBLOCKID_RATEDATA,
                    NULL,
                    &_nSz,
                    &_tCodecRate
                    );
    if ((VQMON_ESUCCESS != _result) && (VQMON_ENOTSUPPORTED != _result))
    {
        fprintf(stderr, "VQmonCODECPropertiesGet (RATE) failed with code %d\n",
            _result);
        exit(EXIT_FAILURE);
    }
    _nBitrate = _tCodecRate.nEncodedBitRate;

    _nDuration = 30 + (unsigned int)(lrand48() % EXMS_CALLDURSECMAX);
    _nPackets = (_nDuration*(1000/_dpp));
    
    for (_i = 1; _i <= _nPackets; _i++)
    {
        /* Model simple, non-bursty loss. */
        if (drand48() < 0.01)
        {
            pMMv1Cntrs->c13++;
            pMMv1Cntrs->nLostPackets++;
        }
        else
        {
            pMMv1Cntrs->c11++;
            pMMv1Cntrs->nRcvPktCount++;
        }
    }
    return (_nBitrate);
}


/* --------------------------------------------------------------------------
 *
 *      EXMS_TerminateStream
 *
 *      Marks the stream as terminated.
 *
 * --------------------------------------------------------------------------
 */

static void
EXMS_TerminateStream(
    vqmon_handle_t                  hStream,
    vqmon_streamcfg_transprotoext_t *pEndPointDescExt
    )
{
    vqmon_result_t _result;

    /*
     * Terminate the stream
     */
    _result = VQmonStreamIndicateEvent(
                hStream,
                vqmonStreamProgEventTerminate,
                g_tNullTS,
                0,
                NULL
                );
    if (VQMON_ESUCCESS != _result)
    {
        fprintf(stderr,
                "VQmonStreamIndicateEvent failed with code %d\n",
                _result);
        exit(EXIT_FAILURE);
    }
}


/* --------------------------------------------------------------------------
 *
 *      EXMS_OutputKPIs
 *
 *      Prints a subset of VQmon stream KPIs to stdout.
 *
 * --------------------------------------------------------------------------
 */

static void
EXMS_OutputKPIs(
    vqmon_handle_t hStream,
    int nEventIdx
    )
{
    tcmyU32 _nSize;
    vqmon_result_t _result;
    vqmon_streamcfg_codec_t _tCodec;
    vqmon_codecgenprops_t _tCodecProps;
    vqmon_streammetrics_delay_t _tDelay;
    vqmon_streammetrics_pkttrans_t _tPktTrans;
    vqmon_streammetrics_voicequal_t _tVoiceQual;
    vqmon_streammetrics_voiceanalog_t _tAnalog;

    /*
     * Fetch packet counts and quality metrics
     */
    _nSize = sizeof(_tPktTrans);
    VQMON_MEMCLEAR(&_tPktTrans, _nSize);
    _result = VQmonStreamMetricsGet(
            hStream,
            VQMON_STREAMMETRICBLOCKID_PKTTRANS,
            &_nSize,
            &_tPktTrans
            );
    if (VQMON_ESUCCESS != _result)
    {
        fprintf(stderr, "VQmonStreamMetricsGet failed with code %d\n", _result);
        exit(EXIT_FAILURE);
    }

    _nSize = sizeof(_tVoiceQual);
    VQMON_MEMCLEAR(&_tVoiceQual, _nSize);
    _result = VQmonStreamMetricsGet(
            hStream,
            VQMON_STREAMMETRICBLOCKID_VOICEQUAL,
            &_nSize,
            &_tVoiceQual
            );
    if (VQMON_ESUCCESS != _result)
    {
        fprintf(stderr, "VQmonStreamMetricsGet failed with code %d\n", _result);
        exit(EXIT_FAILURE);
    }

    _nSize = sizeof(vqmon_streamcfg_codec_t);
    VQMON_MEMCLEAR(&_tCodec, _nSize);
    VQmonStreamConfigGet(
        hStream,
        VQMON_STREAMCFGBLOCKID_CODECINFO,
        &_nSize,
        &_tCodec
        );

    _nSize = sizeof(vqmon_codecgenprops_t);
    VQMON_MEMCLEAR(&_tCodecProps, _nSize);
    VQmonCODECPropertiesGet(
            _tCodec.idCODECType,
            VQMON_CODECPROPBLOCKID_GENERAL,
            NULL,
            &_nSize,
            &_tCodecProps
            );
    
    _nSize = sizeof(vqmon_streammetrics_delay_t);
    VQMON_MEMCLEAR(&_tDelay, _nSize);
    _result = VQmonStreamMetricsGet(
            hStream,
            VQMON_STREAMMETRICBLOCKID_DELAY,
            &_nSize,
            &_tDelay
            );
    if (VQMON_ESUCCESS != _result)
    {
        fprintf(stderr, "VQmonStreamMetricsGet (DELAY) failed with code %d\n",
            _result);
        exit(EXIT_FAILURE);
    }

    _nSize = sizeof(vqmon_streammetrics_voiceanalog_t);
    VQMON_MEMCLEAR(&_tAnalog, _nSize);
    _result = VQmonStreamMetricsGet(
            hStream,
            VQMON_STREAMMETRICBLOCKID_VOICEANALOG,
            &_nSize,
            &_tAnalog
            );
    if (VQMON_ESUCCESS != _result)
    {
        fprintf(stderr, "VQmonStreamMetricsGet (ANALOG) failed with code %d\n",
            _result);
        exit(EXIT_FAILURE);
    }

    printf("E-XMS event %d:\n", nEventIdx);
    printf("   Codec: %s\n",_tCodecProps.strName);
    printf("   Transport: rcvd/lost/discarded %d/%d/%d\n",
            _tPktTrans.nPktsRcvd,
            _tPktTrans.nPktsLost,
            _tPktTrans.nPktsDiscarded
          );
    printf("   Delay: rt/sowd %d ms/%d ms\n",
            _tDelay.nAvgRTDelayMs,
            _tDelay.nAvgOWDelayMs
          );
    printf("   Analog: sglvl/nslvl/rerl %d/%d/%d\n",
            _tAnalog.nSignalLevel_dBm,
            _tAnalog.nNoiseLevel_dBm,
            _tAnalog.nLocalRERL_dB
            );
    /* MOS is represented as a Q8 fixed-point type */
    printf("   Quality: MOS-LQ=%.02f,MOS-CQ=%.02f,MOS-NOM=%.02f\n",
            _tVoiceQual.nMOS_LQ / 256.0,
            _tVoiceQual.nMOS_CQ / 256.0,
            _tVoiceQual.nMOS_Nom / 256.0
          );
}


/* --------------------------------------------------------------------------
 *
 *      EXMS_EventHandler
 *
 *      Simulates an E-XMS end-of-call event that transfers MM data from
 *      XMS to host.
 *
 * --------------------------------------------------------------------------
 */

void EXMS_EventHandler(void)
{
    tcmyU32 _nSz;
    unsigned int _nJitMax;
    static int _evtIdx = 0;
    vqmon_result_t _result;
    tcmyU32 _nActiveBitrate;
    vqmon_mmcntrsv1_t _tVqmMMData;
    
    /*
     * Grab the handle of the next stream from the array.
     */
    vqmon_handle_t _hStream = g_aStrmHandles[g_iStrmIdx];
    _result = VQmonStreamReset(_hStream);
    if (VQMON_ESUCCESS == _result)
    {
        /*
         * Populate the MMv1 counters with data from XMS.
         */
        _nActiveBitrate = EXMS_SetMMCallInfo(&_tVqmMMData, &_nJitMax);

        _result = VQmonStreamUpdateMMSession(
                        _hStream,
                        g_tNullTS,
                        &_tVqmMMData,
                        _nActiveBitrate
                        );
        if (VQMON_ESUCCESS != _result)
        {
            fprintf(stderr, "VQmonStreamUpdateMMSession failed with code %d\n",
                _result);
        }
        while (VQMON_ESUCCESS == _result)
        {
            vqmon_streamcfg_rtdelay_t _tDelay;
            vqmon_streamcfg_voiceanalogmetrics_t _tAnalog;
            
            g_iStrmIdx++;
            if (g_iStrmIdx >= EXMS_STRMHANDLESMAX) { g_iStrmIdx = 0; }
            
            /*
             * Simulate updating analog metric levels and other 
             *   characteristics of the call.
             */
            _nSz = sizeof(vqmon_streamcfg_voiceanalogmetrics_t);
            VQMON_MEMCLEAR(&_tAnalog, _nSz);
            _tAnalog.nNoiseLevel_dBm =
                -80 + (int)(lrand48() % EXMS_NOISELVLRANGE);
            _tAnalog.nLocalSignalLevel_dBm =
                -40 + (int)(lrand48() % EXMS_SIGLVLRANGE);
            _tAnalog.nLocalRERL_dB =
                (int)(lrand48() % EXMS_RERLRANGE);
            _tAnalog.nRemoteRERL_dB = _tAnalog.nRemoteRERL_dB;
            _tAnalog.nRemoteSignalLevel_dBm = 127;
            _result = VQmonStreamConfigSet(
                            _hStream,
                            VQMON_STREAMCFGBLOCKID_VOICEANALOG,
                            sizeof(vqmon_streamcfg_voiceanalogmetrics_t),
                            &_tAnalog
                            );
            if (VQMON_ESUCCESS != _result)
            {
                fprintf(stderr,
                    "VQmonStreamConfigSet (ANALOG) failed with code %d\n",
                    _result);
            }
            
            _nSz = sizeof(vqmon_streamcfg_rtdelay_t);
            VQMON_MEMCLEAR(&_tDelay, _nSz);
            _tDelay.eDelaySource = vqmonStreamMetricSourceUserConfig;
            _tDelay.nDelayMs = (20 + (unsigned int)(lrand48() % EXMS_DELAYMAX)) << 4;
            _result = VQmonStreamConfigSet(
                            _hStream,
                            VQMON_STREAMCFGBLOCKID_ROUNDTRIPDELAY,
                            sizeof(vqmon_streamcfg_rtdelay_t),
                            &_tDelay
                            );
            if (VQMON_ESUCCESS != _result)
            {
                fprintf(stderr,
                    "VQmonStreamConfigSet (DELAY) failed with code %d\n",
                    _result);
            }
            
            /*
             * Finally, terminate the stream and output KPIs.
             */
            EXMS_TerminateStream(_hStream, NULL);
            EXMS_OutputKPIs(_hStream, ++_evtIdx);
            break;
        }
    }
}


/* --------------------------------------------------------------------------
 *
 *      main
 *
 *      Setup the VQmon logical interface and array of stream handles
 *      that will be used to process incoming E-XMS events.
 *
 * --------------------------------------------------------------------------
 */

int main(int argc, char** argv)
{
    unsigned long _seed;
    int _i, _events, _slpitvl;
    
    vqmon_result_t _result;

    vqmon_initparams_t _tInitParams;
    vqmon_interfaceprops_t _tIFProps;

    /*
     * Dump some version info.
     */
    _seed = (unsigned long)time(NULL);
    _seed = (_seed * 256) + _seed;
    srand48(_seed);
    _i = VQmonVersionInfo(NULL);
    printf("\nVQmon E-XMS Host Simulation\nUsing: VQmon %d.%d.%d\n\n",
        VQMON_VERSION_MAJOR(_i), VQMON_VERSION_MINOR(_i), VQMON_VERSION_BUILD(_i));

    /*
     * Initialize VQmon.
     * This example does not use the stream management or alert callbacks,
     * so that all that needs to be set is nVersion.
     */
    VQMON_MEMCLEAR(&_tInitParams, sizeof(_tInitParams));
    _tInitParams.nVersion = VQMON_IFVERSION_CURRENT;
    _result = VQmonInitialize(&_tInitParams);
    if (VQMON_ESUCCESS != _result)
    {
        fprintf(stderr, "VQmonInitialize failed with code %d\n", _result);
        exit(EXIT_FAILURE);
    }

    /*
     * Create an interface handle with the minimal number of bins.
     */
    VQMON_MEMCLEAR(&_tIFProps, sizeof(_tIFProps));
    _tIFProps.nStreamDemuxBins = 1;
    _result = VQmonInterfaceCreate(&g_hInterface, NULL);
    if (VQMON_ESUCCESS != _result)
    {
        fprintf(stderr, "VQmonInterfaceCreate failed with code %d\n",
            _result);
        exit(EXIT_FAILURE);
    }

    /*
     * Setup the array of stream handles used when handling E-XMS events.
     */
    for (_i = 0; _i < EXMS_STRMHANDLESMAX; _i++)
    {
        _result = VQmonStreamCreate(&g_aStrmHandles[_i], g_hInterface,
                        vqmonStreamTypeVoice);
        if (VQMON_ESUCCESS != _result)
        {
            fprintf(stderr, "VQmonStreamCreate failed with code %d\n",
                _result);
            exit(EXIT_FAILURE);
        }
    }
    
    /*
     * Randomly loop for some number of E-XMS events.
     */
    _events = (int)(lrand48() % EXMS_EVENTSMAX);
    if (_events < 1) { _events = 1; }
    printf("Simulating generation of %d E-XMS events.\n\n", _events);
    for (_i = 0; _i < _events; _i++)
    {
        EXMS_EventHandler();
        _slpitvl = (int)(lrand48() % EXMS_EVTITVLUSMAX);
        usleep(_slpitvl);
    }

    /*
     * Cleanup and exit.
     */
    printf("\n");
    for (_i = 0; _i < EXMS_STRMHANDLESMAX; _i++)
    {
        VQmonStreamDestroy(g_aStrmHandles[_i]);
        g_aStrmHandles[_i] = NULL;
    }
    VQmonInterfaceDestroy(g_hInterface);
    g_hInterface = NULL;
    VQmonCleanup();
    return 0;
}
