/* ###########################################################################
#
# Copyright (c) 2003-2016 Telchemy, Incorporated. All Rights Reserved.
#
# Telchemy Confidential and Proprietary
#
# The following software source code ("Software") is strictly confidential
# and is proprietary to Telchemy, Incorporated ("Telchemy").  It may only
# be read, used, copied, adapted, modified or otherwise utilized by parties
# (individuals, corporations, or organizations) that have entered into a
# license agreement or confidentiality agreement with Telchemy, and are thus
# subject to the terms of that license agreement or confidentiality agreement
# and any other applicable agreement between the party and Telchemy.  If
# there is any doubt as to whether a party is entitled to access, read, use,
# copy, adapt, modify or otherwise utilize the Software, or whether a party
# is entitled to disclose the Software to any other party, you should contact
# Telchemy.  If you, as a party, have not entered into a license agreement or
# confidentiality agreement with Telchemy granting access to this Software,
# all media, copies and printed listings containing the Software should be
# forthwith returned to Telchemy.
#
# Telchemy reserves the right to take legal action against any party that
# violates Telchemy's rights to the Software, including without limitation a
# party's breach of the above conditions.
#
# If you have questions about any of the above conditions, or wish to report
# violations, please contact:  support@telchemy.com
#
# ##########################################################################*/

/* voipsim.c
 *
 * VQmon Sample Host Application - Voice Stream Simulator
 *
 * Copyright (c) 2003-2016 Telchemy, Incorporated
 */


/*
 * This application demonstrates the usage of VQmon in monitoring a voice
 * call.  This application uses the stream-based API,
 * VQmonStreamIndicatePacket, to indicate RTP traffic to VQmon.  This means
 * that the task of managing the stream handles for multiple call endpoints is
 * left to the application.
 */


#include <stdio.h>
#include <stdlib.h>

#include "vqmon.h"

/*
#define __BUILD_WITH_SIP_QR
#define __BUILD_WITH_INTERVAL_REPORTS
*/

#if defined(__BUILD_WITH_SIP_QR)
#include "vqmonrpt.h"
#endif

static vqmon_handle_t
SetupVoiceStream(
    vqmon_handle_t hInterface,
    vqmon_streamcfg_netepdesc_t *pEndPointDesc,
    vqmon_streamcfg_rtpptmap_table_t *pRTPMap
    )
{
    vqmon_result_t _result;
    tcmyU32 _nSize;
    vqmon_handle_t  _hStream;

    /*
     * Create a stream handle.
     */
    _result = VQmonStreamCreate(&_hStream, hInterface, vqmonStreamTypeVoice);
    if (VQMON_ESUCCESS != _result)
    {
        fprintf(stderr, "VQmonStreamCreate failed with code %d\n", _result);
        exit(EXIT_FAILURE);
    }

    /*
     * Set the stream's endpoint configuration.
     */
    _nSize = sizeof(vqmon_streamcfg_netepdesc_t);
    _result = VQmonStreamConfigSet(
        _hStream,
        VQMON_STREAMCFGBLOCKID_NETEPDESC,
        _nSize,
        pEndPointDesc
        );
    if (VQMON_ESUCCESS != _result)
    {
        fprintf(stderr, "VQmonStreamConfigSet failed with code %d\n", _result);
        exit(EXIT_FAILURE);
    }

    /*
     * Set the stream's endpoint configuration.
     */
    _nSize = sizeof(vqmon_streamcfg_rtpptmap_table_t); /* Note we are only assuming one value here!! */
    _result = VQmonStreamConfigSet(
        _hStream,
        VQMON_STREAMCFGBLOCKID_RTPPTMAP,
        _nSize,
        pRTPMap
        );
    if (VQMON_ESUCCESS != _result)
    {
        fprintf(stderr, "VQmonStreamConfigSet failed with code %d\n", _result);
        exit(EXIT_FAILURE);
    }

    return _hStream;
}


static vqmon_time_t
IndicatePackets(
    vqmon_handle_t  hStream,
    tcmyU32         nPackets,
    tcmyU32         nLossRatePercent
    )
{
    vqmon_result_t _result;
    vqmon_time_t    _tTimestamp;
    vqmon_pktdesc_t _tPktDesc;
    tcmyU32 _i;
    tcmyU16 _nSeqNum = 0;
    tcmyU32 _nTimestamp = 0;

    tcmyU32 _nLossRateConv = 0;

    if(nLossRatePercent > 0)
    {
        _nLossRateConv = 100/nLossRatePercent;
    }

    /*
     * VQmon can decode packets if given a buffer beginning at the IP, UDP,
     * or RTP header, or the application can parse a packet into a
     * vqmon_pktdesc_t structure and indicate either the voice payload or no
     * payload to VQmon.  This example creates articifial traffic to indicate
     * to VQmon as packets packets parsed up to the payload and uses a NULL
     * pointer in place of the payload buffer.
     */

    VQMON_MEMCLEAR(&_tPktDesc, sizeof(_tPktDesc));
    VQMON_MEMCLEAR(&_tTimestamp, sizeof(_tTimestamp));
    for (_i = 1; _i <= nPackets; _i++)
    {
        _tPktDesc.ePktStartIndicator = vqmonPktDescStartPayload;
        _tPktDesc.pPktBuffer = NULL;
        _tPktDesc.nCapturedPktLen = 0;  /* amount of data available in
                                           pPktBuffer */
        _tPktDesc.nTotalPktLen = 200;   /* total number of bytes transmitted */

        _tPktDesc.rtpinfo.nPayloadType = 96;     /* dynamic payload type */
        _tPktDesc.rtpinfo.nPayloadSize = 150;    /* (bytes*8)/20ms == bitrate */
        _tPktDesc.rtpinfo.nSeqNumber = _nSeqNum;
        _tPktDesc.rtpinfo.nTimestamp = _nTimestamp;
        _tPktDesc.rtpinfo.fHdrFlags = VQMON_RTPINFO_VALID;

#if 1
        /* Low loss rate */
        if( !(0 == _i%100 && _i >=100 && _i <= 200) &&
        /* High loss rate */
           !(0 == _i%2 && _i >= 300 && _i <= 400))
        {
            _result = VQmonStreamIndicatePacket(
                    hStream,
                    _tTimestamp,
                    &_tPktDesc
                    );
            if (VQMON_ESUCCESS != _result)
            {
                fprintf(stderr,
                        "VQmonIndicatePacket failed with code %d\n",
                        _result);
                break;
            }
        }
#else
        _result = VQmonStreamIndicatePacket(
                hStream,
                _tTimestamp,
                &_tPktDesc
                );
        if (VQMON_ESUCCESS != _result)
        {
            fprintf(stderr,
                    "VQmonIndicatePacket failed with code %d\n",
                    _result);
            break;
        }
#endif

#if defined(__BUILD_WITH_INTERVAL_REPORTS)
         /* Create a interval report every 5 seconds. */
        if(_i > 1 && (_i % 100) == 0)
        {
            vqmon_streammetrics_voiceinterval_t _tVoiceIntvl;
            vqmon_streammetrics_voicequal_t _tVoiceQual;
            vqmon_streammetrics_pkttrans_t _tPktTrans;


            tcmyU32 _nSize = sizeof(_tVoiceIntvl);
            VQMON_MEMCLEAR(&_tVoiceIntvl, _nSize);
            
            _result = VQmonStreamIndicateEvent(hStream,
                vqmonStreamProgEventIntervalEnd,_tTimestamp,
                0,NULL);

           _result = VQmonStreamMetricsGet(hStream,
               VQMON_STREAMMETRICBLOCKID_VOICEINTERVAL,
               &_nSize, &_tVoiceIntvl);


            _nSize = sizeof(vqmon_streammetrics_voicequal_t);
            VQMON_MEMCLEAR(&_tVoiceQual, _nSize);
            _result = VQmonStreamMetricsGet(
                hStream,
                VQMON_STREAMMETRICBLOCKID_VOICEQUAL,
                &_nSize,
                &_tVoiceQual
                );

            _nSize = sizeof(_tPktTrans);
            VQMON_MEMCLEAR(&_tPktTrans, _nSize);
            _result = VQmonStreamMetricsGet(
                    hStream,
                    VQMON_STREAMMETRICBLOCKID_PKTTRANS,
                    &_nSize,
                    &_tPktTrans
                    );

            /* Packet statistics */
            printf("\n%-12s%-23s%u/%u/%u\n", "Packets:",
                "rcvd/lost/dscd", _tVoiceIntvl.nPacketsReceived,
                _tVoiceIntvl.nPacketsLost, _tVoiceIntvl.nPacketsDiscarded);
            printf( "%-12s%-23s%.02f\n", "", "loss rate",
                _tVoiceIntvl.nAvgLossRate / 256.0);

            printf("%-12s%-23s%.02f\n", "", "PPDV (ms)",
                _tVoiceIntvl.nPPDVMs / 16.0f);

            /* MOS */
            printf("%-35s%-3d/%-3d\n",
                    "R-factor (CQ/LQ):",
                    _tVoiceIntvl.nR_CQ,
                    _tVoiceIntvl.nR_LQ
                    );
            printf("%-35s%3.2f/%3.2f\n",
                    "MOS (CQ/LQ):",
                    _tVoiceIntvl.nMOS_CQ / 256.0,
                    _tVoiceIntvl.nMOS_LQ / 256.0
                    );
        }
#endif
        /*
         * Increase the sequence number and timestamp for the next packet
         */
        _nSeqNum++;
        _nTimestamp += 960; /* 20ms @ 8000Hz */

        /*
         * Increase the next packet arrival time by 20ms
         */
        _tTimestamp.tv_usec += 20000;
        if (_tTimestamp.tv_usec > 1000000)
        {
            _tTimestamp.tv_usec -= 1000000;
            _tTimestamp.tv_sec++;
        }
    }

    /* Last interval */
#if defined(__BUILD_WITH_INTERVAL_REPORTS)

    {
        vqmon_streammetrics_voiceinterval_t _tVoiceIntvl;
        vqmon_streammetrics_voicequal_t _tVoiceQual;
        vqmon_streammetrics_pkttrans_t _tPktTrans;


        tcmyU32 _nSize = sizeof(_tVoiceIntvl);
        VQMON_MEMCLEAR(&_tVoiceIntvl, _nSize);
        
        _result = VQmonStreamIndicateEvent(hStream,
            vqmonStreamProgEventIntervalEnd,_tTimestamp,
            0,NULL);

       _result = VQmonStreamMetricsGet(hStream,
           VQMON_STREAMMETRICBLOCKID_VOICEINTERVAL,
           &_nSize, &_tVoiceIntvl);


        _nSize = sizeof(vqmon_streammetrics_voicequal_t);
        VQMON_MEMCLEAR(&_tVoiceQual, _nSize);
        _result = VQmonStreamMetricsGet(
            hStream,
            VQMON_STREAMMETRICBLOCKID_VOICEQUAL,
            &_nSize,
            &_tVoiceQual
            );

        _nSize = sizeof(_tPktTrans);
        VQMON_MEMCLEAR(&_tPktTrans, _nSize);
        _result = VQmonStreamMetricsGet(
                hStream,
                VQMON_STREAMMETRICBLOCKID_PKTTRANS,
                &_nSize,
                &_tPktTrans
                );

        /* Packet statistics */
        printf("\n%-12s%-23s%u/%u/%u\n", "Packets:",
            "rcvd/lost/dscd", _tVoiceIntvl.nPacketsReceived,
            _tVoiceIntvl.nPacketsLost, _tVoiceIntvl.nPacketsDiscarded);
        printf( "%-12s%-23s%.02f\n", "", "loss rate",
            _tVoiceIntvl.nAvgLossRate / 256.0);

        printf("%-12s%-23s%.02f\n", "", "PPDV (ms)",
            _tVoiceIntvl.nPPDVMs / 16.0f);

        /* MOS */
        printf("%-35s%-3d/%-3d\n",
                "R-factor (CQ/LQ):",
                _tVoiceIntvl.nR_CQ,
                _tVoiceIntvl.nR_LQ
                );
        printf("%-35s%3.2f/%3.2f\n",
                "MOS (CQ/LQ):",
                _tVoiceIntvl.nMOS_CQ / 256.0,
                _tVoiceIntvl.nMOS_LQ / 256.0
                );
    }
#endif

    return _tTimestamp;
}

static void
TerminateStream(
    vqmon_handle_t                  hStream,
    vqmon_time_t                    nTerminationTimestamp,
    vqmon_streamcfg_transprotoext_t *pEndPointDescExt
    )
{
    vqmon_result_t _result;
    tcmyU32 _nSize;

    /*
     * Set the stream's SSRC, which is done here to make sure the stream was
     *  active or otherwise the SSRC will be overriden. Typically the SSRC
     *  would be set (if required) via a stream activation/creation callback.
     */
    _nSize = sizeof(vqmon_streamcfg_transprotoext_t);
    _result = VQmonStreamConfigSet(
        hStream,
        VQMON_STREAMCFGBLOCKID_TRANSPORTINFO,
        _nSize,
        pEndPointDescExt
        );
    if (VQMON_ESUCCESS != _result)
    {
        fprintf(stderr, "VQmonStreamConfigSet failed with code %d\n", _result);
        exit(EXIT_FAILURE);
    }

    /*
     * Terminate the stream
     */
    _result = VQmonStreamIndicateEvent(
        hStream,
        vqmonStreamProgEventTerminate,
        nTerminationTimestamp,
        0,
        NULL
        );
    if (VQMON_ESUCCESS != _result)
    {
        fprintf(stderr,
                "VQmonStreamIndicateEvent failed with code %d\n",
                _result);
        exit(EXIT_FAILURE);
    }
}


static void PrintResults(
    vqmon_handle_t hStream,
    char *pStreamDesc
    )
{
    vqmon_result_t _result;
    vqmon_streammetrics_pkttrans_t _tPktTrans;
    vqmon_streammetrics_voicequal_t _tVoiceQual;
    vqmon_streamcfg_codec_t   _tCodec;
    vqmon_codecgenprops_t   _tCodecProps;
    tcmyU32 _nSize;

    /*
     * Fetch packet counts and quality metrics
     */
    _nSize = sizeof(_tPktTrans);
    VQMON_MEMCLEAR(&_tPktTrans, _nSize);
    _result = VQmonStreamMetricsGet(
            hStream,
            VQMON_STREAMMETRICBLOCKID_PKTTRANS,
            &_nSize,
            &_tPktTrans
            );
    if (VQMON_ESUCCESS != _result)
    {
        fprintf(stderr, "VQmonStreamMetricsGet failed with code %d\n", _result);
        exit(EXIT_FAILURE);
    }

    _nSize = sizeof(_tVoiceQual);
    VQMON_MEMCLEAR(&_tVoiceQual, _nSize);
    _result = VQmonStreamMetricsGet(
            hStream,
            VQMON_STREAMMETRICBLOCKID_VOICEQUAL,
            &_nSize,
            &_tVoiceQual
            );
    if (VQMON_ESUCCESS != _result)
    {
        fprintf(stderr,
                "VQmonStreamMetricsGet failed with code %d\n",
                _result);
        exit(EXIT_FAILURE);
    }

    _nSize = sizeof(vqmon_streamcfg_codec_t);
    VQMON_MEMCLEAR(&_tCodec, _nSize);
    VQmonStreamConfigGet(
        hStream,
        VQMON_STREAMCFGBLOCKID_CODECINFO,
        &_nSize,
        &_tCodec
        );

    _nSize = sizeof(vqmon_codecgenprops_t);
    VQMON_MEMCLEAR(&_tCodecProps, _nSize);
    VQmonCODECPropertiesGet(
            _tCodec.idCODECType,
            VQMON_CODECPROPBLOCKID_GENERAL,
            NULL,
            &_nSize,
            &_tCodecProps
            );

    printf("%s Voice Stream results\n", pStreamDesc);

    printf("   Codec: %s\n",_tCodecProps.strName);
    printf("   Transport: rcvd/lost/discarded %d/%d/%d\n",
            _tPktTrans.nPktsRcvd,
            _tPktTrans.nPktsLost,
            _tPktTrans.nPktsDiscarded
          );
    /* MOS is represented as a Q8 fixed-point type */
    printf("   Quality: MOS-LQ=%.02f,MOS-CQ=%.02f,MOS-NOM=%.02f\n",
            _tVoiceQual.nMOS_LQ / 256.0,
            _tVoiceQual.nMOS_CQ / 256.0,
            _tVoiceQual.nMOS_Nom / 256.0
          );
}

#if defined(__BUILD_WITH_SIP_QR)
static void
PrintSIPSessionReport(
    vqmon_handle_t          hLocalStream,
    vqmon_sipreportinfo_t   *pSipReportInfo,
    vqmon_time_t            tTimestamp
    )
{
    vqmon_result_t _result;
    tcmyCHAR _aBuffer[4096] = {0};
    tcmyCHAR *_pBuffer = &(_aBuffer[0]);
    tcmyU32 _nBuffSize = sizeof(_aBuffer);
    tcmyU32 _nTmpBuffSize = _nBuffSize;

    do
    {
        /*
         * The first call to this function creates the SIP session report
         *  block and the local metrics block.
         */
        _result = VQmonStreamReportGenerate(
            hLocalStream,
            vqmonStreamReportTypeSIPVQSession,
            tTimestamp,
            pSipReportInfo,
            &_nTmpBuffSize,
            _pBuffer
            );
        if(VQMON_ESUCCESS != _result)
        {
            fprintf(stderr,"ERROR: Failed to create SIP-report (session + local)");
            break;
        }

        if(VQMON_ESUCCESS == _result)
        {
            printf("\n\nSIP-QR Report:\n\n%s\n",_aBuffer);
        }

    }while(0);

}
#endif

int main(int argc, char** argv)
{
    vqmon_initparams_t _tInitParams;
    vqmon_handle_t _hInterface,_hLocalStream;

    vqmon_time_t _nLastPacketTSLocal;
    vqmon_result_t _result;

    tcmyU32 _nPackets = 1000;

    /*
     * Test stream configurations for a local and remote stream.
     */
    vqmon_streamcfg_netepdesc_t _tEndPointDesc = {
        /* tSrcAddress */       {4,{{{172,168,1,22}}}},
        /* tDstAddress */       {4,{{{172,168,1,23}}}},
        /* nSrcPort */          33000,
        /* nDstPort */          44000,
    
        /* nVlanTag */          0,
        /* nServiceVlanTag */   0,
        /* nPacketTOS */        0,
    
        /* aSrcMACAddress[6] */ {0xa,0xb,0xc,0xd,0xe,0xf},
        /* aDstMACAddress[6] */ {0x1,0x2,0x3,0x4,0x5,0x6},
    
        /* eType */             vqmonStreamTypeVoice,
        /* eTransportProtos */  vqmonStreamTransProtoRTP
    };
    
    vqmon_streamcfg_transprotoext_t _tEndPointDescExt = {
        /* eTransportProto */   vqmonStreamTransProtoRTP,
        /* proto.rtp.nSSRC */   {{0x1a2b3c4d}}
    };

#if defined(__BUILD_WITH_SIP_QR)
    /*
     * SIP-Session call information.
     * NOTE: nRemoteSSRC can be derived VQmon-internally if the streams
     *       of a call had been associated. This is required actively in
     *       the VQmon stream API and done automatically using VQmon
     *       interface API.
     */
    vqmon_sipreportinfo_t _tSipReportInfoLocal = {
        /* nSipSessionInformationFlag */ (
                vqmonSIPSessionInfoCallId |
                vqmonSIPSessionInfoLocalId |
                vqmonSIPSessionInfoRemoteId |
                vqmonSIPSessionInfoOrigId |
                vqmonSIPSessionInfoLocalGroupId |
                vqmonSIPSessionInfoRemoteGroupId |
                vqmonSIPSessionInfoRemoteSSRC
                ),
        /* strCallID */ "VQC-CallerID@123456",

        /* strLocalID */ "VQCA <sip:daffodile@example.org>",
        /* strRemoteID */ "VQCB <sip:crocodile@example.org>",
        /* strOrigID */ "VQCA <sip:daffodile@example.org>",

        /* strLocalGroupID */ "example-phone-123",
        /* strRemoteGroupID */ "example-phone-456",

        /* nRemoteSSRC */   0x4d3c2b1a
     };
#endif
     vqmon_streamcfg_rtpptmap_table_t _tRTPMap =
     {
        1,
        {{
            96,
            vqmonCODECVoiceEvsFB
        }}
     };

    /*
     * Initialize VQmon.
     * This example does not use the stream management or alert callbacks,
     * so that all that needs to be set is nVersion.
     */
    VQMON_MEMCLEAR(&_tInitParams, sizeof(_tInitParams));
    _tInitParams.nVersion = VQMON_IFVERSION_CURRENT;

#ifdef VQMON_BUILDCFG_LICENSING
    {
    tcmyCHAR _strHostID[46] = {'\0'};
    tcmyU32  _nHostIDSz = sizeof(_strHostID);

    _result = VQmonHostIDGet(
            _strHostID,
            &_nHostIDSz
            );
    if(VQMON_ESUCCESS == _result)
    {
        printf(" Your HostID string is: %s\n", _strHostID);
    }
    else
    {
        printf("ERROR: Could not retrieve your HostID string.\n");
    }

    /* Add your VQmon license key string to vqmon_initparams_t. */
    _tInitParams.pstrLicense = "";
    }
#endif

    _result = VQmonInitialize(&_tInitParams);
    if (VQMON_ESUCCESS != _result)
    {
        fprintf(stderr, "VQmonInitialize failed with code %d\n", _result);
        exit(EXIT_FAILURE);
    }

    /*
     * Create an interface handle with the default settings.
     */
    _result = VQmonInterfaceCreate(&_hInterface, NULL);
    if (VQMON_ESUCCESS != _result)
    {
        fprintf(stderr, "VQmonInterfaceCreate failed with code %d\n", _result);
        exit(EXIT_FAILURE);
    }

    /*
     * Setup a voice stream.
     */
    _hLocalStream = SetupVoiceStream(_hInterface,&_tEndPointDesc,&_tRTPMap);

    /*
     * Indidicate the packets with some losses.
     */
    _nLastPacketTSLocal = IndicatePackets(_hLocalStream,_nPackets,10);

    /*
     * Terminate the stream.
     */
    TerminateStream(_hLocalStream,_nLastPacketTSLocal,&_tEndPointDescExt);

    /*
     * Print the quality and transport results.
     */
    PrintResults(_hLocalStream,"Local");

#if defined(__BUILD_WITH_SIP_QR)
    /*
     * Print the SIP-Session report.
     */
    PrintSIPSessionReport(_hLocalStream,&_tSipReportInfoLocal,_nLastPacketTSLocal);
#endif
    /*
     * Cleanup
     */
    VQmonStreamDestroy(_hLocalStream);
    VQmonInterfaceDestroy(_hInterface);
    VQmonCleanup();

    return 0;
}
